/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME: 	dbroot.c
 *
 * SCCSINFO:		@(#)dbroot.c	1.6 6/2/94
 *
 * ORIGINAL AUTHOR(S):  Ralph R|nnquist, 18 Sep 1987
 *
 * MODIFICATIONS:
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 * This file creates an empty system root node.
 */
/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */
#include "sunlincks.h"
#include "xconfig.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
extern void LL_releasemol( /* molecule *mol */ );
extern int CH_init();

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern molecule *LL_moltbl;
extern int global_comhistoff;
extern char *optarg;
extern int optind;
extern int opterr;

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
#define STRLEN 64

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static void lookatmol P_(( int mol ));
static void safedisplay P_(( int len, char *s ));
static void print_version P_((void));
/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
/* none */

/*  */
/**********************************************************************
 * Function: void main( int argc, char **argv )
 *
 * Modifications:
 *      <list mods with name and date>
 */
void main(argc, argv)
  int argc;
  char **argv;
{
  int ERRCODE;
  label lbl;
  attrval vp;
  char indexfile[MAXPATHLEN];
  struct stat buf;
  int c = 0, user = 0, pword = 0;
  char dbdir[MAXPATHLEN];
  char username[STRLEN];
  char password[STRLEN];
  int noninteractive = 0;

  /* command history should not be recorded by dbroot */
  global_comhistoff = 1;	/* OFF WITH 'IS 'EAD! */

  /* Get arguments */
  while ((c = getopt(argc, argv, "u:p:")) != EOF) {
    switch (c) {
      case 'u':
        (void)strcpy(username, optarg);
        user = 1;
        noninteractive = 1;
        break;
      case 'p':
	(void)strcpy(password, optarg);
	pword = 1;
	break;
      default:
	break;
    }
  }
  if (user != pword) {
    print_version();
    (void)fprintf(stderr, "You must use both -u and -p if you use either.\n");
    exit(1);
  }

  switch (argc - optind) {
  case 1:
    (void)strcpy(dbdir, argv[argc - 1]);
    break;
  default:
    print_version();
    (void)fprintf(stderr, "Usage: %s [-u user -p passwd] dbdir\n", argv[0]);
    exit(1);
  }

  print_version();
  
  if (noninteractive) {
    if (NOPROMPT_LOGIN(username, password, dbdir) != SUCCESS) {
      (void)fprintf(stderr, "NOPROMPT_LOGIN: login failure\n");
      exit(1);
    }
  } else {
    if (LOGIN(dbdir) != SUCCESS) {
      (void)fprintf(stderr, "LOGIN: login failure\n");
      exit(1);
    }
  }

  (void)sprintf(indexfile, "%s/%s", DBDIR, INDEXFILE);

  if ((stat(indexfile, &buf)) == -1) {
    perror("stat call");
    exit(1);
  }

  if (buf.st_size != 0) {
    /*
     * Get previous system root version structure molecule if there is any.
     * This call is done for the database to know that the workspace has
     * loaded the system root version structure molecule. We immediately
     * drop the molecule contents, resetting the molecule as if it were
     * released, although the database "thinks" that we have it.
     */
    (void)LL_DBRetrieve(0, 0 /* UNKNOWN_TYPE */);

    /* Drop molecule contents */
    LL_releasemol(LL_moltbl);
  }

  /* Set up an application level label to a fresh system root object. */
  if ((ERRCODE = CO_CREATEOBJ(&lbl)))
    exit(ERRCODE);

  /*
   * Make the new object to be the new system root object, and store it.
   * Note: the version structure molecule is forced to associate to the
   * system root version structure molecule. The database will "think"
   * that it gets the system root object previously retrieved.
   */
  LL_moltbl[lbl.vs].ITix = 0;
  if ((ERRCODE = SO_STOREOBJ(&lbl)))
    exit(ERRCODE);
  if ((ERRCODE = RO_RELEASEOBJ(&lbl)))
    exit(ERRCODE);
  lbl.vs = 0;

  /* Inspect the database contents */
  if ((ERRCODE = GA_GETATTR(&lbl, "SYSTEM", "Created", &vp)))
    exit(ERRCODE);
  lookatmol(lbl.vs);
  lookatmol(lbl.inst);
  
  
  global_comhistoff = 0;

  (void)CH_init("dbroot");

  global_comhistoff = 1;	/* OFF WITH 'IS 'EAD! */

  (void)LOGOUT();

  /* since this is run from a makefile .... */
  exit(0);
}

/*  */
/**********************************************************************
 * Function: static void lookatmol(int mol)
 *
 * Display all of a molecule
 * 
 * Modifications:
 *      <list mods with name and date>
 */
static void lookatmol(mol)	/* Display all of a molecule */
  int mol;
{
  attrgroup *agp;
  attribute *afp;
  linkgroup *lgp;
  linktag   *lfp;
  linkitem  *ip;

  (void)printf("\n\nMolecule #%d attributes:\n",mol);
  mapgroups(agp,LL_moltbl[mol].attributes) {
    (void)printf("%s\n",agp->tag);
    mapfields(afp,agp) {
      (void)printf("   %s ",afp->tag);
      safedisplay(afp->size,afp->value);
      (void)printf("\n");
      }
  }
  (void)printf("Molecule #%d links:\n",mol);
  mapgroups(lgp,LL_moltbl[mol].links) {
    (void)printf("%s\n",lgp->tag);
    mapfields(lfp,lgp) {
      (void)printf("   %s ",lfp->tag);
      mapitems(ip,lfp) {
	(void)printf("<%d:%d> ",ip->link.vs,ip->link.inst);
      }
      (void)printf("\n");
    }
  }
  (void)printf("Molecule #%d image:\n",mol);
  if (!LL_moltbl[mol].image)
    (void)printf("   --- No image ---");
  else safedisplay(LL_moltbl[mol].image->size,LL_moltbl[mol].image->value);
  (void)printf("\n");
  (void)printf("inWS=%d  Edited=%d  ",LL_moltbl[mol].inWS,
	       LL_moltbl[mol].edited);
  (void)printf("ITix=%d\n",LL_moltbl[mol].ITix);
}

/*  */
/**********************************************************************
 * Function: static void safedisplay(int len, char *s)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void safedisplay(len,s)
  int len;
  char *s;
{
  (void)printf("(%d)",len);
  if (!s) return;
  while (len--) {
    if (*s<' ') (void)printf(".");
    else (void)printf("%c",*s);
    ++s;
  };
}

#define VERSION "1.3"
#define DATE "1994-06-01"
#define VERSION_INFO "\n"

/*  */
/**********************************************************************
 * Function: static void print_version()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void print_version()
{
  (void)fprintf(stdout, "\tdbroot version %s (%s)\n%s", VERSION, DATE,
                VERSION_INFO);

}
