/*
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Martin Sjlin
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE:		dbpasswd.c
 *
 * SCCSINFO:		@(#)dbpasswd.c	1.9 6/2/94
 *
 * AUTHOR:		Martin Sjlin, 1993-10-29
 *
 * DESCRIPTION:
 *	This file contains the dbpasswd program which a lincks user
 *      should use when adding a password or changing a password in
 *      database password file
 *
 * MODIFICATIONS:
 *	<list mods with name and date>
 */

/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"		/* includes system dependent includes */

#include <sys/ioctl.h>

#include "lincks.h"
#include "libshared.h"
#include "xconfig.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
void main( /* int argc, char *(*argv)[] */ );

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern char *optarg;
extern int optind;
extern int opterr;
extern int errno;

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
#ifndef CRYPT_PASSWD
#define crypt(passwd,salt) (passwd)
#endif /* n CRYPT_PASSWD */

#define STRLEN 64

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static int addnewpasswd P_(( char *user, char *passwd ));
static void arginit P_(( int argc, char **argv ));
static char *getnewpass P_(( char *oldpw ));
static int getoldpasswd P_(( char *user, char *password ));
static char *ourgetpass P_(( char *prompt ));
static void to64 P_(( char *s, t_32bits v, int n ));
static void usage P_(( void ));
static void print_version P_(( void ));

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
static char passwdfile[MAXPATHLEN];
static char line[512];
static char dbdir[MAXPATHLEN];
static char username[STRLEN];

static int silent = 0;		/* no output */

/*  */
/**********************************************************************
 * Function: static char *ourgetpass(char *prompt)
 *
 * Read password from stdin after switching off echo. Return ptr to
 * (static) buffer, prompt with '*prompt' on stderr before reading.
 * Removes ending newline from password.
 *
 * (This should be moved to libshared/misc.c and also called from
 *  dbcomm.c:DBConnect - do it later).
 *
 * Modifications:
 *      <list mods with name and date>
 */
static char *ourgetpass(prompt)
  char *prompt;
{
  static char passwd[PASSWD_LENGTH];
#ifdef HAVE_TERMIOS
  struct termios status;
#else
  struct sgttyb status;
#endif

  if (isatty(fileno(stdin))) {
    /* Get status for stdin */
#ifdef HAVE_TERMIOS
    if (tcgetattr(fileno(stdin), &status) < 0) {
      perror("tcgetattr call failed");
#else
    if (ioctl(fileno(stdin), TIOCGETP, &status) < 0) {
      perror("ioctl call (1) failed");
#endif
      return NULL;
    }
    /* Disable echo on stdin */
#ifdef HAVE_TERMIOS
    status.c_lflag &= ~ECHO;
    if (tcsetattr(fileno(stdin), TCSANOW, &status) < 0) {
      perror("tcsetattr call (1) failed");
#else
    status.sg_flags &= ~(short)ECHO;
    if (ioctl(fileno(stdin), TIOCSETP, &status) < 0) {
      perror("ioctl call (2) failed");
#endif
      return NULL;
    }
  }				/* isatty(fileno(stdin)) */
  /* Get password (without echo) */
  if (!silent)
    (void)fprintf(stderr, prompt);
  (void)fgets(passwd, sizeof(passwd), stdin);
  passwd[strlen(passwd) - 1] = '\0';	/* Zap trailing newline */

  if (isatty(fileno(stdin))) {
    /* Enable echo on stdin */
#ifdef HAVE_TERMIOS
    status.c_lflag |= ECHO;
    if (tcsetattr(fileno(stdin), TCSANOW, &status) < 0) {
      perror("tcsetattr call (2) failed");
#else
    status.sg_flags |= (short)ECHO;
    if (ioctl(fileno(stdin), TIOCSETP, &status) < 0) {
      perror("ioctl call (3) failed");
#endif
      return NULL;
    }
  }				/* isatty(fileno(stdin)) */
  return (passwd);
}

/*  */
/**********************************************************************
 * Function: static char * getnewpass(char *oldpw)
 *
 * Prompt for old password, if any using 'ourgetpass'. Then reads the
 * new password and it re-read for consistency. If ok, encrypt the
 * password and return ptr to static buffer with password.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static char *getnewpass(oldpw)
  char *oldpw;
{
  char salt[4];
  char passwd[PASSWD_LENGTH];
  char *p = NULL;

  /* Prompt for old password if exists */
  if (oldpw[0] != '\0') {
    salt[0] = oldpw[0];
    salt[1] = oldpw[1];
    salt[2] = 0;

    if (!(p = ourgetpass("Old password:")) 
	|| strcmp(crypt(p, salt), oldpw) != 0)
      return (NULL);
  }
  /* Prompt twice for a new password */
  p = ourgetpass(p == NULL ? "New password:" : "\nNew password:");
  if (p == NULL)
    return (NULL);
  (void)strcpy(passwd, p);	/* static in getpass */
  if (strcmp(passwd, ourgetpass("\nRetype new password:")) != 0)
    return (NULL);

  /* OK, got two password */

  /* Grab a random printable character that isn't a colon */
#ifdef HAVE_RANDOM
  to64(&salt[0], random(), 2);
#else
  to64(&salt[0], lrand48(), 2);
#endif /* n HAVE_RANDOM */

  /* crypt it and return ptr 2 enscrypted static buffer */
  return (crypt(passwd, salt));
}

/*  */
/**********************************************************************
 * Function: static int addnewpasswd(char * user, char * passwd)
 *
 * Create a new file (with exclusive), write lock the old password,
 * copy line to line from old to new file, insert new password,
 * copy to end. Close new file, unlink old file, move new file to old
 * and unlock write lock.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int addnewpasswd(user, passwd)
  char *user;
  char *passwd;
{
  FILE *usersfp = NULL;
  FILE *newfp = NULL;
  char filename[MAXPATHLEN];
  char f_username[256];
  char f_password[256];
  int ok = 0;
  int f_uid;

  /* Open password file */
  if ((usersfp = fopen(passwdfile, "r+")) == NULL) {
    perror("Failed to open password file");
    return (ok);
  }
  /* Set lock */
  if (writelockit(fileno(usersfp)) == -1) {
    perror("Failed to obtain write lock on password file");
    (void)fclose(usersfp);
    return (ok);
  }
  (void)sprintf(filename, "%s.new", passwdfile);
  if ((newfp = fopen(filename, "w")) == NULL) {
    perror("Failed to create new password file");
    unlockit(fileno(usersfp), (off_t)0, (off_t)0);
    (void)fclose(usersfp);
    return (ok);
  }
  /* Search for user name */
  while (fgets(line, sizeof(line) - 1, usersfp) != NULL) {
    f_username[0] = f_password[0] = '\0';
    f_uid = 0;
    ok = sscanf(line, PASSWD_FORMAT, f_username, &f_uid, f_password);
    if (ok > 1 && strcmp(f_username, user) == 0)
      (void)fprintf(newfp, PASSWD_FORMAT_PR, f_username, f_uid, passwd);
    else
      (void)fprintf(newfp, PASSWD_FORMAT_PR, f_username, f_uid, f_password);
  }
  ok = 1;

  /* make the old password invisble, before unlocking the proctection */
  if (ok) {
    if (unlink(passwdfile) == -1) {
      (void)fprintf(stderr, "Faild to unlink old password file %s: %d\n",
		    passwdfile, errno);
      ok = 0;
    }
  }
  /* unlock the old passwd file */
  unlockit(fileno(usersfp), (off_t)0, (off_t)0);

  /* and close the old file. If we succeded in unlinking, it will
   * disapperar now and it will be impossible to lock it */
  (void)fclose(usersfp);
  (void)fclose(newfp);

  /* Finally, move the new file to the password file */
  if (ok) {
    if (rename(filename, passwdfile) == -1) {
      (void)fprintf(stderr, "Failed to rename %s to %s:%d\n",
		    filename, passwdfile, errno);
      ok = 0;
    }
  } else
    (void)unlink(filename);	/* do not keep it around! */

  return (ok);
}

/*  */
/**********************************************************************
 * Function: static int getoldpasswd(char *user,char *password)
 *
 * Read the password, checking for user name. Return
 *  0 - no match on user id
 *  1 - found user id, but no uid nor password
 *  2 - found user id & uid, but no password
 *  3 - found user, id, and password (in password ... )
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int getoldpasswd(user, password)
  char *user;
  char *password;
{
  FILE *usersfp;
  char f_username[256];
  int ok = 0;
  int f_uid;

  /* Open password file */
  if ((usersfp = fopen(passwdfile, "r")) == NULL) {
    perror("Failed to open password file");
    return (ok);
  }
  /* Set lock */
  if ((readlockit(fileno(usersfp), (off_t)0, (off_t)0)) == -1) {
    perror("Failed to set read lock on password file");
    (void)fclose(usersfp);
    return (ok);
  }
  /* Search for user name */
  while (fgets(line, sizeof(line) - 1, usersfp) != NULL) {
    f_username[0] = password[0] = '\0';
    ok = sscanf(line, PASSWD_FORMAT, f_username, &f_uid, password);
    if (ok > 0 && strcmp(f_username, user) == 0)
      goto exit;
  }

  ok = 0;
  (void)fprintf(stderr, "User %s unknown.\n", user);

exit:
  /* Remove lock and close file */
  unlockit(fileno(usersfp), (off_t)0, (off_t)0);
  (void)fclose(usersfp);
  return (ok);
}

/*  */
/**********************************************************************
 * Function: void main(int argc, char *argv[])
 *
 * main should be called with two arguments - DB and a lincks user
 * name for that database. The user name should exist in the
 * 'DB/passwd' file.
 *
 * Modifications:
 *      <list mods with name and date>
 */
void main(argc, argv)
  int argc;
  char *argv[];
{
  char oldpw[PASSWD_LENGTH];
  char *passwd = NULL;

  /* initialize the random number generator */
#ifdef HAVE_RANDOM
  (void)srandom(getpid());
#else
  srand48(getpid());
#endif /* n HAVE_RANDOM */

  arginit(argc, argv);

  (void)memset((char *)oldpw, 0, (int)sizeof(oldpw));

  /* read in configuration file - we need to get password file later */
  if (configuration(dbdir) == 0) {
    (void)fprintf(stderr, "Couldn't configure %s\n", dbdir);
    exit(1);
  }
  /* setup passwd file name ... */
  (void)sprintf(passwdfile, "%s/%s", DBDIR, PASSWDFILE);

  /* get old password ... */
  if (getoldpasswd(username, oldpw) > 1) {
    if ((passwd = getnewpass(oldpw)) && addnewpasswd(username, passwd)) {
      if (!silent)
        (void)fprintf(stderr, "\n");
      exit(0);
    } else
      if (!silent)
        (void)fprintf(stderr, "\nSorry!\n");
  }
  exit(1);
}

/*  */
/**********************************************************************
 * Function: static void arginit(int argc, char **argv)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void arginit(argc, argv)
  int argc;
  char **argv;
{
  int c = 0;
  int dbfound = 0;
  char *db = NULL;

  (void)memset((char *)dbdir, 0, (int)sizeof(dbdir));
  (void)memset((char *)username, 0, (int)sizeof(username));

  while ((c = getopt(argc, argv, "sd:")) != EOF) {
    switch (c) {
    case 's':
      silent = 1;
      break;
    case 'd':
      (void)strcpy(dbdir, optarg);
      dbfound = 1;
      break;
    default:
      break;
    }
  }

  if (!dbfound) {		/* DBDIR not on command line */
    if ((db = (char *)getenv("LINCKSDBDIR")) == (char *)NULL) {
      print_version();
      usage();
    }
    (void)strcpy(dbdir, db);
  }
  switch (argc - optind) {
  case 1:
    (void)strcpy(username, argv[argc - 1]);
    break;
  case 0:			/* didn't give us username */
  default:			/* fall through */
    print_version();
    usage();
    break;
  }
}

/*  */
/**********************************************************************
 * Function: static void usage()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void usage()
{
  (void)fprintf(stderr, "Usage: dbpasswd [-d database-dir] username\n");
  exit(1);
}

/*  */
/*-
 * Copyright (c) 1990 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/* 0 ... 63 => ascii - 64 */
static unsigned char itoa64[] =
"./0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";

static void to64(s, v, n)
  char *s;
  t_32bits v;
  int n;
{
  while (--n >= 0) {
    *s++ = itoa64[v & 0x3f];
    v >>= 6;
  }
}

#define VERSION "1.3"
#define DATE "1994-06-01"
#define VERSION_INFO "\n"

/*  */
/**********************************************************************
 * Function: static void print_version()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void print_version()
{
  (void)fprintf(stdout, "\tdbpasswd version %s (%s)\n%s", VERSION, DATE,
                VERSION_INFO);

}
