/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


/*
 * MODULE NAME: 	traverse.c
 *
 * SCCSINFO:		@(#)traverse.c	1.10 6/7/94
 *
 * ORIGINAL AUTHOR(S):  Tom Nilsson, ????
 *
 * MODIFICATIONS:
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 *  File containing Traverse. Cutoff2 calls traverse.
 *  Traverse searches the nodestructure and sets the CO_REACHED flag to
 *  true for the nodes that can be reached from the systemnode and shall
 *  be moved to the new version of the database.
 */
/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */
#include "xconfig.h"
#include "cutoff.h"
#include "dbserver.h"
#include "monitor.h"
#include "libshared.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_traverse.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_files.h"
#include "f_flag.h"
#include "f_pointers.h"
#include "f_packunpack.h"

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
/* cutoffmain.c */
extern off_t global_idxlen;

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
/* none */

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static int DoFetchNodesToCheck P_(( LABEL node, LABEL *next, PACKEDBUFF
                                     *pbuff, LINKGROUP **group ));
static int FetchNodesToCheck P_(( LABEL node, LABEL *next ));
static int SetReached P_(( LABEL cur, LABEL node, LABEL *next ));

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
/* none */

/*  */
/**********************************************************************
 * Function: int Traverse()
 *
 * set CO_REACHED flag to true for all nodes that can be
 * reached from the systemnode and shall be moved.
 *
 * returns SUCCESS or FAIL
 *
 * FetchNodesToCheck,ConvToLabel
 *
 * Modifications:
 *      <list mods with name and date>
 */
int Traverse()
{
  LABEL label = IT_SYSTEM_MOLECULE_LABEL;
  LABEL nextlabel = 0;
  long max_diff = 0;
  long n_back_step = 0;
  long sum_diff = 0;
  int return_code = SUCCESS;

  /* system node is reached */
  set_flag(IT_SYSTEM_MOLECULE_LABEL, CO_REACHED);
  while (label < ConvToLabel(global_idxlen)) {
    nextlabel = 0;
    if (is_flag_set(label, CO_REACHED) && !is_flag_set(label, CO_CHECK)) {
      set_flag(label, CO_CHECK);
      if ((return_code = FetchNodesToCheck(label, &nextlabel)) == FAIL) {
	(void)printf("Traverse: couldn't fetch nodes to check.\n");
	break;
      }
    }
    if (nextlabel != 0) {
      n_back_step++;
      if ((label - nextlabel) > max_diff)
	max_diff = label - nextlabel;
       sum_diff += (label - nextlabel);
      label = nextlabel;
    } else {
      label += sizeof(INDEX);
    }
  }
  if (return_code == FAIL)
    (void)printf("Last node processed: %ld\n", (long) label);
  (void)printf("Maximal back step length: %ld\n",max_diff);
  (void)printf("Sum of back step lengths: %ld\n",sum_diff);
  (void)printf("Number of back step times: %ld\n",n_back_step);
  return return_code;
}


/*  */
/**********************************************************************
 * Function: static int FetchNodesToCheck(LABEL node,LABEL *next)
 *
 * A wrapper for DoFetchNodesToCheck to handle memory reclaim on all
 * returns.
 * Returns FAIL on fails and SUCCESS on success.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int FetchNodesToCheck(node, next)
  LABEL node;
  LABEL *next;
{
  PACKEDBUFF pbuff;
  LINKGROUP *group;
  int return_code;

  pbuff.buff = (char *)NULL;
  pbuff.size = 0;
  pbuff.curp = 0;
  group = (LINKGROUP *) NULL;

  return_code = DoFetchNodesToCheck(node, next, &pbuff, &group);
  if (pbuff.buff != (char *)NULL)
    free(pbuff.buff);
  if (group != (LINKGROUP *) NULL)
    FreeLinks(group);

  return return_code;
}

/*  */
/**********************************************************************
 * Function: static int SetReached(LABEL cur, LABEL node, LABEL *next)
 *
 * set CO_REACHED for node and keep least non-zero label in next.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int SetReached(cur, node, next)
  LABEL cur;
  LABEL node;
  LABEL *next;
{
  if ((node == 0) || is_flag_set(node, CO_REACHED))
    return SUCCESS;

  set_flag(node, CO_REACHED);
  if (node >= cur)
    return SUCCESS;

  if (((*next) == 0) || (node < (*next)))
    (*next) = node;
  return SUCCESS;
}

/*  */
/**********************************************************************
 * Function: static int DoFetchNodesToCheck(FOUR PARAMETERS)
 * Parameters:
 *	LABEL node
 *	LABEL *next
 *	PACKEDBUFF *pbuff
 *	LINKGROUP **group
 *
 * Follow the forward links from the node and mark all nodes referred to
 * as CO_REACHED. The least label of next and newly reached nodes is
 * assigned to next.
 * Return FAIL for fails, and SUCCESS on success.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int DoFetchNodesToCheck(node, next, pbuff, group)
  LABEL node;
  LABEL *next;
  PACKEDBUFF *pbuff;
  LINKGROUP **group;
{
  off_t filpos;
  int filno;
  int molfd;
  int datfd;
  off_t fowptr;
  int size;
  LINKGROUP *gp;
  LINKS *fp;
  LINKITEM *ip;

  if ((filpos = GetFilPos(node)) == UNDEFINED_ENTRY)
    return SUCCESS;

  if (filpos < 0)
    return FAIL;

  if ((filno = GetFilNo(node)) == FAIL)
    return FAIL;

  molfd = molfileOfIndex(MOL, filno, ARCHIVE);
  datfd = datfileOfIndex(DAT, filno, ARCHIVE);

  pbuff->buff = NULL;
  if (((fowptr = GetDatPtr(molfd, filpos, FORWARD)) == FAIL) ||
      ((size = GetDatSize(molfd, filpos, FORWARD)) < 0) ||
      ( size > 0 && ((pbuff->buff = malloc((ALLOC_T)size)) == NULL)))
    return FAIL;
  pbuff->size = size;

  if (((GetEntry(datfd, fowptr, pbuff->buff, (unsigned)size)) == FAIL) ||
      (UnpackGroup(group, pbuff) == FAIL))
    return FAIL;

  for (gp = *group; gp != (LINKGROUP *) NULL; gp = gp->nextgroup) {
    for (fp = gp->links; fp != (LINKS *) NULL; fp = fp->nextlinks) {
      for (ip = fp->linkitem; ip != (LINKITEM *) NULL; ip = ip->nextitem) {
	if ((SetReached(node, ip->label, next) == FAIL) ||
	    (SetReached(node, ip->vs, next) == FAIL))
	  return FAIL;
      }
    }
  }
  return SUCCESS;
}
