/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME: 	pointers.c
 *
 * SCCSINFO:		@(#)pointers.c	1.6 5/3/94
 *
 * ORIGINAL AUTHOR(S):  Jens M Hansen, ????
 *
 * MODIFICATIONS:
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 *  File containing all functions of CUTOFF POINTERS module.
 *  POINTERS consists of a set of functions that retrieve/change file entries.
 *  All names of POINTERS functions contain the word "Get" or "Change".
 */
/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */
#include "cutoff.h"
#include "monitor.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_pointers.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_cutoffmain.h"

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
/* cutoffmain.c */
extern int global_indexfd;	/* index file descriptor */
extern off_t global_idxlen;	/* length of the index file */

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
#ifndef MOLCACHESIZE
#define MOLCACHESIZE	500
#endif /* MOLCACHESIZE */
#define NUMINDEXPAGES	4
#define OURPAGEMASK	((long)0x1FF)

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static INDEX *ReadIndexEntry P_(( LABEL label, int chg ));
static FILEMOL *ReadMolEntry P_(( int fd, off_t pos, int chg ));
static int WriteIndexPage P_(( int pg ));
static int WriteMolEntry P_(( int ix ));

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
static struct {
  int changed;
  int fd;
  off_t pos;
  FILEMOL entry;
} mol_cache[MOLCACHESIZE];

static int molcacheix = -1;

struct {
  off_t base;
  int chg;
  int size;
} index_page_head[NUMINDEXPAGES];

struct {
  INDEX entry[OURPAGEMASK + 1];
} index_page[NUMINDEXPAGES];

static int next_index_page = -1;

/*  */
/**********************************************************************
 * Function: int GetEntry(int fd, off_t pos, char *entry, unsigned size)
 *
 * read file entry of size from position of file
 * into entry
 *
 * Returns SUCCESS on success,
 * FAIL on error
 *
 * Modifications:
 *      <list mods with name and date>
 */
int GetEntry(fd, pos, entry, size)
  int fd;
  off_t pos;
  char *entry;
  unsigned size;
{
  if ((lseek(fd, pos, L_SET)) != pos)
    fatal_error(FATAL_LSEEK, "GetEntry", (char *)NULL);

  if ((read(fd, entry, (IOSIZE_T)size)) != size) {
    (void)fprintf(stderr, "GetEntry: EOF reached unexpectedly\n");
    return FAIL;
  }
  return SUCCESS;
}

/*  */
/**********************************************************************
 * Function: int ChangeEntry(int fd, off_t pos, char *entry, unsigned size)
 *
 * write file entry of size
 * to position of file
 *
 * Returns SUCCESS
 *
 * Modifications:
 *      <list mods with name and date>
 */
int ChangeEntry(fd, pos, entry, size)
  int fd;
  off_t pos;
  char *entry;
  unsigned size;
{
  if ((lseek(fd, pos, L_SET)) != pos)
    fatal_error(FATAL_LSEEK, "ChangeEntry", (char *)NULL);

  if ((write(fd, (char *)entry, (IOSIZE_T)size)) != size)
    fatal_error(FATAL_IO, "ChangeEntry", (char *)NULL);

  return SUCCESS;
}


/*  */
/*
 * Addition that handles cashing.
 */
/**********************************************************************
 * Function: static int WriteMolEntry(int ix)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int WriteMolEntry(ix)
  int ix;
{
  int fd = mol_cache[ix].fd;
  off_t pos = mol_cache[ix].pos;
  char *entry = (char *)&(mol_cache[ix].entry);
  unsigned size = sizeof(FILEMOL);

  return (ChangeEntry(fd, pos, entry, size) == FAIL);
}

/*  */
/**********************************************************************
 * Function: static FILEMOL *ReadMolEntry(int fd, off_t pos, int chg)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static FILEMOL *ReadMolEntry(fd, pos, chg)
  int fd;
  off_t pos;
  int chg;
{
  int ix;

  if (molcacheix == -1) {
    for (ix = 0; ix < MOLCACHESIZE; ix++)
      mol_cache[ix].fd = -1;
    molcacheix = 0;
  }
  for (ix = 0; ix < MOLCACHESIZE; ix++) {
    if (mol_cache[ix].fd == fd && mol_cache[ix].pos == pos) {
      if (chg)
	mol_cache[ix].changed = chg;
      return (&(mol_cache[ix].entry));
    }
  }

  ix = molcacheix++;
  if (molcacheix == MOLCACHESIZE)
    molcacheix = 0;

  if (mol_cache[ix].changed) {
    mol_cache[ix].changed = 0;
    if (WriteMolEntry(ix) == FAIL)
      return (FILEMOL *) NULL;
  }
  if (GetEntry(fd, pos, (char *)&(mol_cache[ix].entry), sizeof(FILEMOL))
      == FAIL)
    return (FILEMOL *) NULL;

  mol_cache[ix].fd = fd;
  mol_cache[ix].pos = pos;
  mol_cache[ix].changed = chg;
  return (&(mol_cache[ix].entry));
}

/*  */
/**********************************************************************
 * Function: static int WriteIndexPage(int pg)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int WriteIndexPage(pg)
  int pg;
{
  off_t pos = (index_page_head[pg].base * sizeof(INDEX));
  char *buf = (char *)&(index_page[pg].entry[0]);
  unsigned size = index_page_head[pg].size;

  return (ChangeEntry(global_indexfd, pos, buf, size));
}


/*  */
/**********************************************************************
 * Function: static INDEX *ReadIndexEntry(LABEL label, int chg)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static INDEX *ReadIndexEntry(label, chg)
  LABEL label;
  int chg;
{
  off_t addr;
  off_t base;
  off_t offset;
  unsigned size;
  char *buf;
  int pg;

  if (next_index_page == -1) {
    for (pg = 0; pg < NUMINDEXPAGES; pg++)
      index_page_head[pg].base = -1;
    next_index_page = 0;
  }
  /* convert label to seek address */
  if (((addr = ConvToSeek(label)) < 0) || (addr >= global_idxlen)) {
    (void)fprintf(stderr, "ReadIndexEntry: illegal seek address\n");
    return (INDEX *) NULL;
  }
  base = addr / sizeof(INDEX);
  offset = (base & OURPAGEMASK);
  base  -= offset;
  for (pg = 0; pg < NUMINDEXPAGES; pg++) {
    if (index_page_head[pg].base == base) {
      if (chg)
	index_page_head[pg].chg = chg;
      return &(index_page[pg].entry[offset]);
    }
  }

  pg = next_index_page++;
  if (next_index_page == NUMINDEXPAGES)
    next_index_page = 0;

  if (index_page_head[pg].chg) {
    index_page_head[pg].chg = 0;
    if (WriteIndexPage(pg) == FAIL)
      return (INDEX *) NULL;
  }
  if ((lseek(global_indexfd, (off_t)(base * sizeof(INDEX)), L_SET)) < 0) {
    (void)fprintf(stderr, "ReadIndexEntry: unsuccessful seek attempt\n");
    return (INDEX *) NULL;
  }
  size = (unsigned)(OURPAGEMASK + 1) * sizeof(INDEX);
  buf = (char *)&(index_page[pg].entry[0]);
  if ((size = read(global_indexfd, buf, (IOSIZE_T)size)) <= 0) {
    (void)fprintf(stderr, "ReadIndexEntry: EOF reached unexpectedly\n");
    return (INDEX *) NULL;
  }
  index_page_head[pg].base = base;
  index_page_head[pg].chg = chg;
  index_page_head[pg].size = size;
  return &(index_page[pg].entry[offset]);
}


/*  */
/**********************************************************************
 * Function: void EndFileCache()
 *
 * Modifications:
 *      <list mods with name and date>
 */
void EndFileCache()
{
  int ix;

  for (ix = 0; ix < MOLCACHESIZE; ix++) {
    if (mol_cache[ix].changed) {
      mol_cache[ix].changed = 0;
      (void)WriteMolEntry(ix);
    }
  }

  for (ix = 0; ix < NUMINDEXPAGES; ix++) {
    if (index_page_head[ix].chg) {
      index_page_head[ix].chg = 0;
      (void)WriteIndexPage(ix);
    }
  }
}

/*  */
/**********************************************************************
 * Function: off_t GetDatPtr(int fd, off_t pos, int field)
 *
 * fetch pointer to data file field
 * from molecule file fd position pos
 *
 * Returns pointer to field on success,
 * FAIL on error
 *
 * Modifications:
 *      <list mods with name and date>
 */
off_t GetDatPtr(fd, pos, field)
  int fd;
  off_t pos;
  MOLENTRYFIELD field;
{
  FILEMOL *entry;		/* molecule entry to extract pointer from */

  /* read molecule entry */
  if ((entry = ReadMolEntry(fd, pos, 0)) == (FILEMOL *) NULL)
    return FAIL;

  /* return pointer in field */
  switch (field) {
  case FORWARD:
    return entry->flink_filepos;
  case BACKWARD:
    return entry->blink_filepos;
  case ATTRIBUTES:
    return entry->attr_filepos;
  case IMAGE:
    return entry->image_filepos;
  default:
    (void)fprintf(stderr, "GetDatPtr: access to field not granted\n");
    return FAIL;
  }
}

/*  */
/**********************************************************************
 * Function: int GetDatSize(int fd, off_t pos, int field)
 *
 * fetch size of data file field
 * of molecule file fd position pos
 *
 * Returns size of field on success
 * FAIL on error
 *
 * Modifications:
 *      <list mods with name and date>
 */
int GetDatSize(fd, pos, field)
  int fd;
  off_t pos;
  MOLENTRYFIELD field;
{
  FILEMOL *entry;		/* molecule entry to extract pointer from */

  /* read molecule entry */
  if ((entry = ReadMolEntry(fd, pos, 0)) == (FILEMOL *) NULL)
    return FAIL;

  /* return size of field */
  switch (field) {
  case FORWARD:
    return entry->flink_size;
  case BACKWARD:
    return entry->blink_size;
  case ATTRIBUTES:
    return entry->attr_size;
  case IMAGE:
    return entry->image_size;
  default:
    (void)fprintf(stderr, "GetDatSize: access to field not granted.\n");
    return FAIL;
  }
}

/*  */
/**********************************************************************
 * Function: int GetFilNo(LABEL label)
 *
 * fetch molecule/data file # of label
 *
 * Returns file number on success,
 * FAIL on error
 *
 * Modifications:
 *      <list mods with name and date>
 */
int GetFilNo(label)
  LABEL label;
{
  INDEX *entry;			/* index entry to extract file number from */

  if ((entry = ReadIndexEntry(label, 0)) == (INDEX *) NULL)
    return FAIL;

  return entry->filenum;
}


/*  */
/**********************************************************************
 * Function: int ChangeFilNo(LABEL label, int no)
 *
 * Modifications:
 *      <list mods with name and date>
 */
int ChangeFilNo(label, no)
  LABEL label;
  int no;
{
  INDEX *entry;

  if ((entry = ReadIndexEntry(label, 1)) == (INDEX *) NULL)
    return FAIL;

  entry->filenum = no;

  return SUCCESS;
}


/*  */
/**********************************************************************
 * Function: off_t GetFilPos(LABEL label)
 *
 * fetch molecule file position of label
 *
 * Returns file position on success
 * FAIL on error
 *
 * Modifications:
 *      <list mods with name and date>
 */
off_t GetFilPos(label)
  LABEL label;
{
  INDEX *entry;			/* index entry to extract file position from */

  if ((entry = ReadIndexEntry(label, 0)) == (INDEX *) NULL)
    return FAIL;

  return entry->filepos;
}

/*  */
/**********************************************************************
 * Function: int ChangeFilPos(LABEL label, off_t pos)
 *
 *
 * Modifications:
 *      <list mods with name and date>
 */
int ChangeFilPos(label, pos)
  LABEL label;
  off_t pos;
{
  INDEX *entry;

  if ((entry = ReadIndexEntry(label, 1)) == (INDEX *) NULL)
    return FAIL;

  entry->filepos = pos;

  return SUCCESS;
}

/*  */
/**********************************************************************
 * Function: void reset_cache()
 *
 * Modifications:
 *      <list mods with name and date>
 */
void reset_cache()
{
  EndFileCache();
  molcacheix = -1;
}
