/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME: 	packunpack.c
 *
 * SCCSINFO:		@(#)packunpack.c	1.8 5/3/94
 *
 * ORIGINAL AUTHOR(S):  ???, 27 Aug, 1987
 *
 * MODIFICATIONS:
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 *	This file contains the functions PackGroup and UnpackGroup
 */
/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */

#include <netinet/in.h>

#include "lincks.h"
#include "dbserver.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_packunpack.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
/*
 * can't change this since it's different depending on what's being
 * linked with libshared.a
 */
#ifdef lint
static void Error();
#else
extern void Error( /* va_alist */ );
#endif

#include "f_links.h"

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
#define PACKBUFFSIZE 32768	/* Default size of pack buffer */

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static inline int DoPackGroup P_(( LINKGROUP *group, PACKEDBUFF *pbuff ));
static inline int DoUnpackGroup P_(( LINKGROUP **group, PACKEDBUFF *pbuff ));
static int PackChars P_(( char *str, int count, PACKEDBUFF *pbuff ));
static int PackItems P_(( LINKITEM *linkitem, PACKEDBUFF *pbuff ));
static int PackLabel P_(( LABEL lbl, PACKEDBUFF *pbuff ));
static int PackLinks P_(( LINKS *links, PACKEDBUFF *pbuff ));
static int PackString P_(( char *string, PACKEDBUFF *pbuff ));
static int PackWSLabel P_(( LABEL lbl, PACKEDBUFF *pbuff ));
static int UnpackItems P_(( LINKITEM **item, PACKEDBUFF *pbuff ));
static int UnpackLabel P_(( LABEL *lblp, PACKEDBUFF *pbuff ));
static int UnpackLinks P_(( LINKS **links, PACKEDBUFF *pbuff ));
static int UnpackString P_(( char **string, PACKEDBUFF *pbuff ));
static int UnpackWSLabel P_(( LABEL *lblp, PACKEDBUFF *pbuff ));
static inline int _PackGroup P_(( LINKGROUP *group, PACKEDBUFF *pbuff ));
static inline int _UnpackGroup P_(( LINKGROUP **group, PACKEDBUFF *pbuff ));

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
static int (* packlbl)()  = PackLabel;
static int (*unpacklbl)()  = UnpackLabel;

/*  */
/**********************************************************************
 * Function: int PackGroup(LINKGROUP *group, PACKEDBUFF *pbuff)
 *
 * Make a packed buffer from link group structure.
 *
 * Modifications:
 *      <list mods with name and date>
 */
int PackGroup(group, pbuff)
  LINKGROUP *group;
  PACKEDBUFF *pbuff;
{
  packlbl = PackLabel;
  unpacklbl = UnpackLabel;
  return(_PackGroup(group, pbuff));
}

/*  */
/**********************************************************************
 * Function: int PackWSGroup(LINKGROUP *group, PACKEDBUFF *pbuff)
 *
 * Make a packed buffer from link group structure.
 *
 * Modifications:
 *      <list mods with name and date>
 */
int PackWSGroup(group, pbuff)
  LINKGROUP *group;
  PACKEDBUFF *pbuff;
{
  packlbl = PackWSLabel;
  unpacklbl = UnpackWSLabel;
  return(_PackGroup(group, pbuff));
}

/*  */
/**********************************************************************
 * Function: int UnpackGroup(LINKGROUP **group,PACKEDBUFF *pbuff)
 *
 * Modifications:
 *      <list mods with name and date>
 */
int UnpackGroup(group, pbuff)
  LINKGROUP **group;
  PACKEDBUFF *pbuff;
{
  packlbl   = PackLabel;
  unpacklbl = UnpackLabel;
  return(_UnpackGroup(group, pbuff));
}

/*  */
/**********************************************************************
 * Function: int UnpackWSGroup(LINKGROUP **group,PACKEDBUFF *pbuff)
 *
 * Modifications:
 *      <list mods with name and date>
 */
int UnpackWSGroup(group, pbuff)
  LINKGROUP **group;
  PACKEDBUFF *pbuff;
{
  packlbl   = PackWSLabel;
  unpacklbl = UnpackWSLabel;
  return(_UnpackGroup(group, pbuff));
}

/*  */
/**********************************************************************
 * Function: static int _PackGroup(LINKGROUP *group, PACKEDBUFF *pbuff)
 *
 * Make a packed buffer from link group structure.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static inline int _PackGroup(group, pbuff)
  LINKGROUP *group;
  PACKEDBUFF *pbuff;
{
  if (pbuff->buff == (char *)NULL) {
    if ((pbuff->buff = (char *)malloc((ALLOC_T)PACKBUFFSIZE)) == NULL) {
      Error(ER_MALLOC, "_PackGroup: no more memory");
      return (FAIL);
    }
    pbuff->size = PACKBUFFSIZE;
  }
  pbuff->curp = 0;

  if (DoPackGroup(group, pbuff) != FAIL) {
    /*
     * The caller should see only the used portion of the packed buffer.
     * The memory manager knows real size.
     */
    pbuff->size = pbuff->curp;
    return (SUCCESS);
  }
  free(pbuff->buff);
  pbuff->buff = (char *)NULL;
  pbuff->size = 0;
  pbuff->curp = 0;

  return FAIL;
}

/*  */
/**********************************************************************
 * Function: static int DoPackGroup(LINKGROUP *group,PACKEDBUFF *pbuff)
 *
 * Packs all link groups into pbuff.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static inline int DoPackGroup(group, pbuff)
  LINKGROUP *group;
  PACKEDBUFF *pbuff;
{

  while (group != (LINKGROUP *) NULL) {

    if ((PackString(group->name, pbuff) == FAIL) ||
	(PackLinks(group->links, pbuff) == FAIL))
      return (FAIL);

    group = group->nextgroup;
  }

  /*
   * The group list is terminated by a zero-length group tag.
   */
  return (PackString((char *)NULL, pbuff));
}


/*  */
/**********************************************************************
 * Function: static int PackLinks(LINKS *links, PACKEDBUFF *pbuff)
 *
 * Packs all link fields.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int PackLinks(links, pbuff)
  LINKS *links;
  PACKEDBUFF *pbuff;
{

  while (links != (LINKS *) NULL) {

    if ((PackString(links->tag, pbuff) == FAIL) ||
	(PackItems(links->linkitem, pbuff) == FAIL))
      return (FAIL);

    links = links->nextlinks;
  }

  /*
   * The field list is terminated by a zero-length field tag.
   */

  return (PackString((char *)NULL, pbuff));
}


/*  */
/**********************************************************************
 * Function: static int PackItems(LINKITEM *linkitem, PACKEDBUFF *pbuff)
 *
 * Packs all link items.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int PackItems(linkitem, pbuff)
  LINKITEM *linkitem;
  PACKEDBUFF *pbuff;
{
  while (linkitem != (LINKITEM *) NULL) {

    if (((*packlbl)(linkitem->vs, pbuff) == FAIL) ||
	((*packlbl)(linkitem->label, pbuff) == FAIL))
      return (FAIL);

    linkitem = linkitem->nextitem;
  }
  /*
   * The link item list is terminated by a NULLABEL vs entry.
   */

  return ((*packlbl)(NULLABEL, pbuff));
}


/*  */
/**********************************************************************
 * Function: static int PackString(char *string, PACKEDBUFF *pbuff)
 *
 * Packs a tag string into the packbuff.
 *
 * Modifications:
 *      1993-08-18 Martin Sjolin. Added htonl/htons
 *      <list mods with name and date>
 */
static int PackString(string, pbuff)
  char *string;
  PACKEDBUFF *pbuff;
{
  unsigned short len;
#if (TAGSIZESIZE == 2)
  t_u16bits netlen;
#else
#if (TAGSIZESIZE == 4)
  t_u32bits netlen;
#endif /* TAGSIZESIZE == 4 */
#endif /* TAGSIZESIZE == 2 */

  if (string == (char *)NULL)
    len = 0;
  else
    len = strlen(string);

#if (TAGSIZESIZE == 2)
    netlen = htons(len);
#else
#if (TAGSIZESIZE == 4)
    netlen = htonl(len);
#else
  {
    Error(0, "PackString: Size/type mismatch");
    return(FAIL); 
  }
#endif
#endif

  if ((PackChars((char *)&netlen, TAGSIZESIZE, pbuff) == FAIL) ||
      (PackChars(string, (int)len, pbuff) == FAIL))
    return (FAIL);
  return (SUCCESS);
}


/*  */
/**********************************************************************
 * Function: static int PackLabel(LABEL lbl, PACKEDBUFF *pbuff)
 *
 * Packs a node label into the packbuff. Used internally within database server
 *
 * Modifications:
 *      1993-08-18 Martin Sjolin. Added htonl/htons
 *      <list mods with name and date>
 */
static int PackLabel(lbl, pbuff)
  LABEL lbl;
  PACKEDBUFF *pbuff;
{
/* in the case server is running with "Big" label, shrink before converting */
#ifdef LABEL_64_BITS
  t_u64bits netlonglong = lbl;
  return (PackChars((char *)&netlonglong, sizeof(netlonglong), pbuff));
#else
  t_u32bits netlong = htonl(((t_u32bits)lbl)); 
  return (PackChars((char *)&netlong, sizeof(netlong), pbuff));
#endif /* n LABEL_64_BITS */
}


/*  */
/**********************************************************************
 * Function: static int PackWSLabel(LABEL lbl, PACKEDBUFF *pbuff)
 *
 * Packs a node label into the packbuff. For communication with ws.
 *
 * Modifications:
 *      1993-08-18 Martin Sjolin. Added htonl/htons
 *      <list mods with name and date>
 */
static int PackWSLabel(lbl, pbuff)
  LABEL lbl;
  PACKEDBUFF *pbuff;
{
/* in the case server is running with "Big" label, shrink before converting */
#if (LABELSIZE == 4)
  t_u32bits netlong = htonl(((t_u32bits)lbl)); 
  return (PackChars((char *)&netlong, LABELSIZE, pbuff));
#else
#if (LABELSIZE == 2)
  t_u16bits netshort = htons(((t_u16bits)lbl));
  return (PackChars((char *)&netshort, LABELSIZE, pbuff));
#else
#if (LABELSIZE == 8)
  t_u64bits netlonglong = lbl;
  return (PackChars((char *)&netlonglong, LABELSIZE, pbuff));
#else
  Error(0, "PackWSLabel: Size/type mismatch");
  return(FAIL);
#endif /* LABELSIZE == 8 */
#endif /* LABELSIZE == 2 */
#endif /* LABELSIZE == 4 */
}



/*  */
/**********************************************************************
 * Function: static int PackChars(char *str,int count, PACKEDBUFF *pbuff)
 *
 * Packs characters into the pack buffer
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int PackChars(str, count, pbuff)
  char *str;
  int count;
  PACKEDBUFF *pbuff;
{
  char *buff;

  if (count == 0)
    return (SUCCESS);

  buff = pbuff->buff;
  if (pbuff->curp + count > pbuff->size) {
    if ((buff = (char *)realloc((FREEPTR *)buff, 
				(ALLOC_T)pbuff->size + count)) == (char *)NULL) {
      Error(ER_MALLOC, "PackChars: no more memory");
      return (FAIL);
    }
    pbuff->buff = buff;
    pbuff->size += count;
  }
  (void)memcpy((buff + pbuff->curp), str, count);
  pbuff->curp += count;

  return (SUCCESS);
}


/*  */
/**********************************************************************
 * Function: static int _UnpackGroup(LINKGROUP **group,PACKEDBUFF *pbuff)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static inline int _UnpackGroup(group, pbuff)
  LINKGROUP **group;
  PACKEDBUFF *pbuff;
{
  pbuff->curp = 0;

  if (DoUnpackGroup(group, pbuff) == SUCCESS)
    return SUCCESS;

  if (*group != (LINKGROUP *) NULL)
    FreeLinks(*group);

  *group = (LINKGROUP *) NULL;
  return (FAIL);
}


/*  */
/**********************************************************************
 * Function: static int DoUnpackGroup(LINKGROUP **group,PACKEDBUFF *pbuff)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static inline int DoUnpackGroup(group, pbuff)
  LINKGROUP **group;
  PACKEDBUFF *pbuff;
{
  static char *buff;
  LINKGROUP *last, *this;

  *group = last = this = (LINKGROUP *) NULL;

  while (pbuff->curp < pbuff->size) {
    /*
     * The pbuff->curp is moved in UnpackString and UnpackLinks.
     */
    if (UnpackString(&buff, pbuff) == FAIL)
      return (FAIL);

    /*
     * A zero-length group tag terminates the list of groups.
     */
    if (buff == (char *)NULL)
      return (SUCCESS);

    if ((this = (LINKGROUP *) malloc(sizeof(LINKGROUP))) == NULL) {
      /*
       * Note: buff holds malloc-ed memory.
       */
      free(buff);
      Error(ER_MALLOC, "UnpackGroup: no more memory");
      return (FAIL);
    }
    this->name = buff;
    this->nextgroup = (LINKGROUP *) NULL;
    this->links = (LINKS *) NULL;

    if (last == (LINKGROUP *) NULL)
      *group = this;
    else
      last->nextgroup = this;
    last = this;

    /*
     * At this point, the malloc-ed memory is held in the (partial but
     * clean) link group structure.
     */
    if (UnpackLinks(&(this->links), pbuff) == FAIL)
      return (FAIL);
  }

  return (FAIL);
}


/*  */
/**********************************************************************
 * Function: static int UnpackLinks(LINKS **links, PACKEDBUFF *pbuff)
 *
 * Unpacks links from pack buffer.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int UnpackLinks(links, pbuff)
  LINKS **links;
  PACKEDBUFF *pbuff;
{
  char *buff;
  LINKS *last, *this;

  *links = last = this = (LINKS *) NULL;

  while (pbuff->curp < pbuff->size) {
    /*
     * The pbuff->curp is moved in UnpackString and UnpackItems.
     */
    if (UnpackString(&buff, pbuff) == FAIL)
      return (FAIL);

    /*
     * A zero-length field tag terminates the list of fields.
     */
    if (buff == (char *)NULL)
      return (SUCCESS);

    if ((this = (LINKS *) malloc(sizeof(LINKS))) == NULL) {
      /*
       * Note: buff holds malloc-ed memory.
       */
      free(buff);
      Error(ER_MALLOC, "UnpackLinks: no more memory");
      return (FAIL);
    }
    this->tag = buff;
    this->nextlinks = (LINKS *) NULL;
    this->linkitem = (LINKITEM *) NULL;

    if (last == (LINKS *) NULL)
      *links = this;
    else
      last->nextlinks = this;
    last = this;

    /*
     * At this point, the malloc-ed memory is held in the (partial but
     * clean) link group structure.
     */
    if (UnpackItems(&(this->linkitem), pbuff) == FAIL)
      return (FAIL);
  }

  return (FAIL);
}


/*  */
/**********************************************************************
 * Function: static int UnpackItems(LINKITEM **item, PACKEDBUFF *pbuff)
 *
 * Unpacks link items from pack buffer.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int UnpackItems(item, pbuff)
  LINKITEM **item;
  PACKEDBUFF *pbuff;
{
  LABEL lbl;
  LINKITEM *last, *this;

  *item = last = (LINKITEM *) NULL;

  while (pbuff->curp < pbuff->size) {
    /*
     * The pbuff->curp is moved in UnpackLabel.
     */
    if ((*unpacklbl)(&lbl, pbuff) == FAIL)
      return (FAIL);

    /*
     * A NULLABEL history node (vs) terminates the link item list.
     */
    if (lbl == NULLABEL)
      return (SUCCESS);

    if ((this = (LINKITEM *) malloc(sizeof(LINKITEM))) == NULL) {
      Error(ER_MALLOC, "UnpackLinks: no more memory");
      return (FAIL);
    }
    this->nextitem = (LINKITEM *) NULL;
    this->vs = lbl;

    if (last == (LINKITEM *) NULL)
      *item = this;
    else
      last->nextitem = this;
    last = this;

    /*
     * At this point, the malloc-ed memory is held in the (partial but
     * clean) link group structure.
     */
    if ((*unpacklbl)(&(this->label), pbuff) == FAIL)
      return (FAIL);
  }

  return (FAIL);
}


/*  */
/**********************************************************************
 * Function: static int UnpackString(char **string, PACKEDBUFF *pbuff)
 *
 * Unpacks a tag string from the pack buffer.
 * Malloc-s tag memory for null-terminated string.
 *
 * Modifications:
 *      1993-08-18 Martin Sjolin. Added ntohl/ntohs
 *      <list mods with name and date>
 */
static int UnpackString(string, pbuff)
  char **string;
  PACKEDBUFF *pbuff;
{
  unsigned short len;
#if (TAGSIZESIZE == 2)
  t_u16bits netlen;
#else
#if (TAGSIZESIZE == 4)
  t_u32bits netlen;
#endif
#endif

  *string = (char *)NULL;

  if ((pbuff->size - pbuff->curp) < TAGSIZESIZE)
    return (FAIL);

  (void)memcpy((char *)&netlen, (pbuff->buff + pbuff->curp), TAGSIZESIZE);
  /* I'm not sure about the alignment => i will not try a direct cast. */
#if (TAGSIZESIZE == 2)
  len = ntohs(netlen);
#else
#if (TAGSIZESIZE == 4)
  len = ntohl(netlen);
#else
  Error(0, "UnpackString: Size/type mismatch");
  return(FAIL);
#endif
#endif

  pbuff->curp += TAGSIZESIZE;

  if (len == 0)
    return (SUCCESS);

  if ((pbuff->size - pbuff->curp) < (int)len)
    return (FAIL);

  if ((*string = (char *)malloc((ALLOC_T)len + 1)) == NULL) {
    Error(ER_MALLOC, "UnpackString: no more memory");
    return (FAIL);
  }
  (void)memcpy((*string), (pbuff->buff + pbuff->curp), (int)len);
  pbuff->curp += (int)len;
  *((*string) + (int)len) = '\0';

  return (SUCCESS);
}


/*  */
/**********************************************************************
 * Function: static int UnpackLabel(LABEL *lblp, PACKEDBUFF *pbuff)
 *
 * Unpacks node label from buffer. Used internally within database server
 *
 * Modifications:
 *      1993-08-18 Martin Sjolin. Added ntohl/ntohs
 *      <list mods with name and date>
 */
static int UnpackLabel(lblp, pbuff)
  LABEL *lblp;
  PACKEDBUFF *pbuff;
{
#ifdef LABEL_64_BITS
  t_u64bits netlbl;
#else
  t_u32bits netlbl;
#endif /* n LABEL_64_BITS */

  *lblp = NULLABEL;

  if ((pbuff->size - pbuff->curp) < sizeof(netlbl))
    return (FAIL);

  (void)memcpy((char *)&netlbl, (pbuff->buff + pbuff->curp), sizeof(netlbl));

#ifdef LABEL_64_BITS
  *lblp = netlbl;                  /* don't care yet - just inside the server */
#else
  *lblp = ntohl(netlbl);	
#endif /* n LABEL_64_BITS */

  pbuff->curp += sizeof(netlbl);

  return (SUCCESS);
}
/*  */
/**********************************************************************
 * Function: static int UnpackWSLabel(LABEL *lblp, PACKEDBUFF *pbuff)
 *
 * Unpacks node label from buffer. For communication with ws.
 *
 * Modifications:
 *      1993-08-18 Martin Sjolin. Added ntohl/ntohs
 *      <list mods with name and date>
 */
static int UnpackWSLabel(lblp, pbuff)
  LABEL *lblp;
  PACKEDBUFF *pbuff;
{
#if (LABELSIZE == 2)
  t_u16bits netlbl;
#else
#if (LABELSIZE == 4)
  t_u32bits netlbl;
#else
#if (LABELSIZE == 8)
  t_u64bits netlbl;
#endif
#endif
#endif

  *lblp = NULLABEL;

  if ((pbuff->size - pbuff->curp) < LABELSIZE)
    return (FAIL);

  (void)memcpy((char *)&netlbl, (pbuff->buff + pbuff->curp), LABELSIZE);
  /* extension it done by the compiler */
#if (LABELSIZE == 4)
  *lblp = (t_32bits) ntohl(netlbl);
#else
#if (LABELSIZE == 2)
  *lblp = (t_16bits) ntohs(netlbl);
#else
#if (LABELSIZE == 8)
  *lblp = (t_64bits) netlbl;
#else
  Error(0, "UnpackLabel: Size/type mismatch");
  return(FAIL);
#endif
#endif
#endif

  pbuff->curp += LABELSIZE;

  return (SUCCESS);
}

#ifdef lint	/* just so lint thinks it's here */
/* ARGSUSED */
static void Error(i, c)
  int i;
  char *c;
{
  return;
}
#endif
