/*
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME: 	parser.c
 *
 * SCCSINFO:		@(#)parser.c	1.11 6/7/94
 *
 * ORIGINAL AUTHOR(S): Ralph R\"onnquist, 1989
 *
 * MODIFICATIONS:
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 *
 * A simple syntax-driven interpretive LALR(k) parser with
 * complete backtrack.
 *
 */

/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */
#include "parser.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
/* all of these are in parser.h */

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
/* various array sizes */
#ifndef DSTACKLIMIT
#define DSTACKLIMIT	1024
#endif	/* DSTACKLIMIT */

#define INBUFLIMIT	5000
#define ARRAYSIZE	20

#define TERMTOK 0
#define NONTERM 1
#define WTOKEN  2
#define UTOKEN  3
#define ENDING  4

/* convenience macros */
#define New(record)     (record *) malloc((ALLOC_T) sizeof(record))
#define Maplist(p,lst)  for(p=lst;p!=0;p=p->nextp)

#define Acceptable(n)   ((n)>0)

#define Poprule rulestack[rulesp--]
#define Pushrule(r)     rulestack[++rulesp] = (r)
#define Toprule rulestack[rulesp]

#define Poptoken tokenstack[toksp--]
#define Pushtoken(tok)  tokenstack[++toksp] = (tok)
#define Toptoken tokenstack[toksp]

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static int blword P_(( char **str ));
static void bootgrammar P_(( void ));
static int callsystem P_(( void ));
static int catchline P_(( char **str ));
static void catchnonterms P_(( bnftoken *re ));
static void catchprint P_(( word *w ));
static int changegoal P_(( void ));
static int checkexecute P_(( funcell *fexpr ));
static int checktoken P_(( char *t, char **str ));
static int clearscreen P_(( void ));
static int currentgoal P_(( void ));
static int definerule P_(( void ));
static void doexecute P_(( funcell *fexpr ));
static int doprimary P_(( word *goal, char **str ));
static int drop1 P_(( void ));
static int empty P_(( char **str ));
static bnftoken *enlisttokens P_(( int n ));
static int eoln P_(( char **str ));
static int fliptrace P_(( void ));
static int identifier P_(( char **str ));
static void indent P_(( int i ));
static int integer P_(( char **str ));
static argcell *makeaction P_(( word *fn, argcell *args ));
static int makealt P_(( void ));
static int makeending P_(( void ));
static bnfrule *makerule P_(( bnfrule *nxt, word *semfn, bnftoken *elems ));
static bnftoken *maketoken P_(( bnftoken *nxt, int flg, word *w ));
static int makeuntil P_(( void ));
static int makewhile P_(( void ));
static int namedgoal P_(( void ));
static int nontermelem P_(( void ));
static int noop P_(( void ));
static int parseprod P_(( bnftoken *prod, char **str ));
static int parsetoken P_(( bnftoken *tok, char **str ));
static void pcatch P_(( word *w ));
static int primaryrule P_(( void ));
static int printcurrentgrammar P_(( void ));
static int printcurrentsyntax P_(( void ));
static int printgrammar P_(( void ));
static void printrelems P_(( bnftoken *re ));
static void printrules P_(( bnfrule *r, int ind ));
static void printsyntax P_(( word *r ));
static int printvoc P_(( void ));
static int quiet P_(( void ));
static int quotedstring P_(( char **str ));
static int repeatelem P_(( void ));
static void report_malloc_fail P_(( char *Pfunc_name ));
static char *strduplicate P_(( char *input ));
static int takeinputfile P_(( void ));
static int termelem P_(( void ));
static int verbose P_(( void ));

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
/*### Parser State Variables ###*/

static bnftoken *tokenstack[100];
static bnfrule *rulestack[100];

static argcell *funargs;

static FILE *infile = stdin, *outfile = stdout;

static word *vocabulary = (word *) NULL;
static word *goal = (word *) NULL;
static word *metagoal = (word *) NULL;
static word *predefinedwords = (word *) NULL;
static word *bnfgoal = (word *) NULL;

static char *dstack[DSTACKLIMIT];
static char tempint[ARRAYSIZE];
static char temp[INBUFLIMIT];

static char *inbuf = (char *)NULL;
static char *last = (char *)NULL;
static char *limit = (char *)NULL;
static char *lastfirst = (char *)NULL;

static int dsp = 0;

static int rulesp = 0;
static int toksp = 0;

static int displayflag = 0;
int exectrace = 0;

static int run = 0;
static int stopexec = 0;

extern char *sys_errlist[];
extern int sys_nerr;
extern int errno;

/*### Functions operating on strings ###*/
/*  */
/**********************************************************************
 * Function: int skipbl(char **str)
 *
 * Moves the string ptr *str over "space" characters (as defined by
 * the isspace function
 *
 * Modifications:
 *      <list mods with name and date>
 */
int skipbl(str)			/*EXPORTED*/
  char **str;
{
  if (*str == (char *)NULL)
    return (0);
  while (isspace(**str))
    (*str)++;
  return (**str != '\0');
}


/*### Functions to create records ###*/

/*  */
/**********************************************************************
 * Function: word *makesymbol(char *pn)
 *
 * Installs a new word record with pname pn at the top of the
 * current vocabulary.
 *
 * Modifications:
 *      <list mods with name and date>
 */
word *makesymbol(pn)		/*EXPORTED*/
  char *pn;
{
  word *r;

  r = New(word);
  r->nextp = vocabulary;
  r->primary = 0;
  r->actionfn = 0;
  r->pname = strduplicate(pn);
  r->syntax = 0;
  vocabulary = r;
  return (r);
}

/*  */
/**********************************************************************
 * Function: static bnftoken *maketoken(bnftoken *nxt, int flg, word *w)
 *
 * Returns ptr to new bnftoken record linked to nxt, with type flag
 * flg, and associated to word w.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static bnftoken *maketoken(nxt, flg, w)
  bnftoken *nxt;
  int flg;
  word *w;
{
  bnftoken *p;

  p = New(bnftoken);
  p->nextp = nxt;
  p->extra = (bnftoken *) NULL;
  p->termtoken = flg;
  p->token = w;
  return (p);
}

/*  */
/**********************************************************************
 * Function: static bnfrule *makerule(bnfrule *nxt,word *semfn,bnftoken *elems)
 *
 * Returns ptr to new bnfrule record linked to nxt, with semantic
 * semfn, and elements elems.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static bnfrule *makerule(nxt, semfn, elems)	/* Create a rule record */
  bnfrule *nxt;
  word *semfn;
  bnftoken *elems;
{
  bnfrule *p;

  p = New(bnfrule);
  p->nextp = nxt;
  p->semantic = semfn;
  p->elements = elems;
  return (p);
}

/*### Functions for free-ing memory ###*/

/*  */
/**********************************************************************
 * Function: void freeargcells(argcell *ap)
 *
 * Modifications:
 *      <list mods with name and date>
 */
void freeargcells(ap)
  argcell *ap;
{
  argcell *p;

  while (ap != (argcell *) NULL) {
    p = ap;
    ap = ap->nextp;
    if (p->data != (char *)NULL) {
      free((FREEPTR *)p->data);
      p->data = NULL;
    }
    if (p->fnexpr != (funcell *) NULL)
      freefuncell(p->fnexpr);
    free((FREEPTR *)p);
    p = NULL;
  }
}

/*  */
/**********************************************************************
 * Function: void freefuncell(funcell *fp)
 *
 * Modifications:
 *      <list mods with name and date>
 */
void freefuncell(fp)
  funcell *fp;
{
  if (fp == (funcell *) NULL)
    return;
  freeargcells(fp->args);
  free((FREEPTR *)fp);
  fp = NULL;
}

/*### Vocabulary access functions ###*/

/*  */
/**********************************************************************
 * Function: word *findsymbol(char *s)
 *
 * Returns ptr to first word record in vocabulary with pname s, or a
 * zero ptr if no such record is found.
 *
 * Modifications:
 *      <list mods with name and date>
 */
word *findsymbol(s)		/*EXPORTED*/
  char *s;
{
  word *p;

  Maplist(p, vocabulary)
    if (strcmp(p->pname, s) == 0)
    return (p);
  return (0);
}

/*  */
/**********************************************************************
 * Function: word *symbol(char *s, int flg)
 *
 * Locates first word record in vocabulary with pname s, or creates
 * such. Returns pointer to it. The argument flg marks (if non-zero)
 * that the argument string should be free-ed before return.
 *
 * Modifications:
 *      <list mods with name and date>
 */
word *symbol(s, flg)		/*EXPORTED*/
  char *s;
  int flg;
{
  word *p;

  if ((p = findsymbol(s)) == (word *) NULL)
    p = makesymbol(s);
  if (flg) {
    free((FREEPTR *)s);
  }
  return (p);
}

/*  */
/**********************************************************************
 * Function: void bindaction(char *s, int (*fn)())
 *
 * Define an action fn to an action name s, and install in the
 * current vocabulary. Note: every /bindaction/ makes a new
 * vocabulary entry when necessary (i.e. existent actions are not
 * redefined)
 *
 * Modifications:
 *      <list mods with name and date>
 */
void bindaction(s, fn)		/*EXPORTED*/
  char *s;
  int (*fn) ();
{
  word *p;

  p = symbol(s, 0);
  if (p->actionfn != 0)
    p = makesymbol(s);
  p->actionfn = fn;
}

/*  */
/**********************************************************************
 * Function: void defineactions(actionentry *tbl)
 *
 * Modifications:
 *      <list mods with name and date>
 */
void defineactions(tbl)		/*EXPORTED*/
  actionentry *tbl;
{
  while (tbl->fnname != (char *)NULL) {
    bindaction(tbl->fnname, tbl->fn);
    tbl++;
  }
}

/*  */
/**********************************************************************
 * Function: void bindprimary(char *s, int (*fn)())
 *
 * Defines a primary function fn for the name s and installs in the
 * current vocabulary. The primary field of word s is set to word s,
 * and the action of word s is set to fn. That is, there is a double
 * in-direction.
 *
 * Modifications:
 *      <list mods with name and date>
 */
void bindprimary(s, fn)		/*EXPORTED*/
  char *s;
  int (*fn) ();
{
  word *p;

  p = symbol(s, 0);
  if (p->primary != 0 || p->actionfn != 0)
    p = makesymbol(s);
  p->primary = p;
  p->actionfn = fn;
}

/*  */
/**********************************************************************
 * Function: void defineprimaries(actionentry *tbl)
 *
 * Modifications:
 *      <list mods with name and date>
 */
void defineprimaries(tbl)	/*EXPORTED*/
  actionentry *tbl;
{
  while (tbl->fnname != (char *)NULL) {
    bindprimary(tbl->fnname, tbl->fn);
    tbl++;
  }
}

/*  */
/**********************************************************************
 * Function: word *setvocabulary(word *w)
 *
 * Reassigns the vocabulary to w (if not zero). Returns previous
 * vocabulary. Note: setvocabulary((word *) NULL) retrieves current
 * vocabulary.
 *
 * Modifications:
 *      <list mods with name and date>
 */
word *setvocabulary(w)		/*EXPORTED*/
  word *w;
{
  word *p;

  p = vocabulary;
  if (w != (word *) NULL)
    vocabulary = w;
  return (p);
}

/*  */
/**********************************************************************
 * Function: word *hidevocabulary()
 *
 * Reassigns the vocabulary to the predfinedwords vocabulary.
 *
 * Modifications:
 *      <list mods with name and date>
 */
word *hidevocabulary()
{				/*EXPORTED*/
  return (setvocabulary(predefinedwords));
}

/*### Data stack functions ###*/

/*  */
/**********************************************************************
 * Function: char *popdata()
 *
 * Pops and returns the top element from the stack. Returns zero ptr
 * if the stack is empty.
 *
 * Modifications:
 *      <list mods with name and date>
 */
char *popdata()
{				/*EXPORTED*/
  if (dsp == 0)
    return((char *) NULL);
  return(dstack[--dsp]);
}

/*  */
/**********************************************************************
 * Function: int popint()
 *
 * Pops the top of stack element and interprets it as a number,
 * which is returned after that the stacked element is free-ed.
 * Returns zero if the stack is empty.
 *
 * Modifications:
 *      <list mods with name and date>
 */
int popint()
{				/*EXPORTED*/
  int x = 0;

  if (dsp == 0)
    return (0);

  dsp--;
  (void)sscanf(dstack[dsp], "%d", &x);
  free((FREEPTR *)dstack[dsp]);
  dstack[dsp] = NULL;

  return (x);
}

/*  */
/**********************************************************************
 * Function: void pushdata(char *s)
 *
 * Pushes string s onto the data stack.
 *
 * Modifications:
 *      <list mods with name and date>
 */
void pushdata(s)		/*EXPORTED*/
  char *s;
{
  if (dsp >= DSTACKLIMIT) {
    (void)fprintf(stderr, "%s%d%s\n%s\n%s%d\n%s\n",
		  "ERROR:  Cannot push more data.  Array size ",DSTACKLIMIT,
		  " is too small.",
		  "Recompile libparser.a with -DDSTACKLIMIT=N, where N",
		  "is some number greater than ", DSTACKLIMIT,
		  "and relink this program with libparser.a");
    exit(0);
  }
  dstack[dsp] = s;
  dsp++;
}

/*  */
/**********************************************************************
 * Function: void cleardata(int i, int flg)
 *
 * Clears the data stack down to the i:th level. If flg is non-zero,
 * all stack elements are free-ed.
 *
 * Modifications:
 *      <list mods with name and date>
 */
void cleardata(i, flg)		/*EXPORTED*/
  int i;
  int flg;
{

  if (flg) {
    while (i < dsp) {
      free((FREEPTR *)popdata());
      /* dsp is decreased in popdata */
    }
  }
  dsp = i;
}

/*  */
/**********************************************************************
 * Function: int datadepth()
 *
 * Modifications:
 *      <list mods with name and date>
 */
int datadepth()
{				/*EXPORTED*/
  return (dsp);
}

/*### Execution stack functions ###*/

/*  */
/**********************************************************************
 * Function: argcell *appendargs(argcell *a1, argcell *a2)
 *
 * Modifications:
 *      <list mods with name and date>
 */
argcell *appendargs(a1, a2)
  argcell *a1;
  argcell *a2;
{
  argcell *temp = a1;		/* Fix for ultrix which pukes on &a1 */
  argcell **p;

  if (a1 == 0)
    return (a2);
  for (p = &temp; *p != (argcell *) NULL; p = &((*p)->nextp));
  *p = a2;
  return (a1);
}

/*  */
/**********************************************************************
 * Function: static argcell *makeaction(word *fn, argcell *args)
 *
 * Make an argcell that applies fn to args
 *
 * Modifications:
 *      <list mods with name and date>
 */
static argcell *makeaction(fn, args)
  word *fn;
  argcell *args;
{
  argcell *ac;

  ac = New(argcell);
  ac->nextp = (argcell *) NULL;
  ac->fnexpr = New(funcell);
  ac->fnexpr->fn = fn;
  ac->fnexpr->args = args;
  ac->data = (char *)NULL;
  return (ac);
}

/*  */
/**********************************************************************
 * Function: void delaypushdata(char *s)
 *
 * Put the action to produce copy of string s at end of current funargs
 *
 * Modifications:
 *      <list mods with name and date>
 */
void delaypushdata(s)		/*EXPORTED*/
  char *s;
{

  argcell *ac;

  ac = New(argcell);
  ac->nextp = (argcell *) NULL;
  ac->fnexpr = (funcell *) NULL;
  ac->data = strduplicate(s);

  funargs = appendargs(funargs, ac);
}

/*### Execution control functions ###*/

/*  */
/**********************************************************************
 * Function: void printfunexpr(funcell *fexpr, int n)
 *
 * Modifications:
 *      <list mods with name and date>
 */
void printfunexpr(fexpr, n)
  funcell *fexpr;
  int n;
{
  argcell *ap;
  int i;

  if (fexpr == 0)
    return;
  (void)fprintf(stderr, "(%s ", fexpr->fn->pname);

  for (ap = fexpr->args; ap != 0; ap = ap->nextp) {
    (void)fprintf(stderr, "\n");
    for (i = n; i >= 0; i--)
      (void)fprintf(stderr, "  ");
    if (ap->fnexpr != 0)
      (void)printfunexpr(ap->fnexpr, n + 1);
    else
      (void)fprintf(stderr, "\"%s\"", ap->data);
  }
  (void)fprintf(stderr, ")");
  if (n == 0)
    (void)fprintf(stderr, "\n");
}

/*  */
/**********************************************************************
 * Function: funcell *funexpr()
 *
 * This function is called after a successful parsing to retrieve the
 * resulting function expression.
 *
 * Modifications:
 *      <list mods with name and date>
 */
funcell *funexpr()
{
  funcell *fp;

  if (funargs == (argcell *) NULL)
    return ((funcell *) NULL);
  fp = funargs->fnexpr;
  funargs->fnexpr = (funcell *) NULL;
  free((FREEPTR *)funargs);
  funargs = (argcell *) NULL;
  return (fp);
}

/*  */
/**********************************************************************
 * Function: static int checkexecute(funcell *fexpr)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int checkexecute(fexpr)
  funcell *fexpr;
{
  argcell *args;
  int flag;

  flag = 0;

  for (args = fexpr->args; args != (argcell *) NULL; args = args->nextp) {
    if (args->fnexpr != (funcell *) NULL) {
      flag = checkexecute(args->fnexpr);
    }
  }

  if (fexpr->fn->actionfn == 0) {
    (void)fprintf(stderr, "%%UNDEFINED ACTION %s\n", fexpr->fn->pname);
    flag = 1;
  }
  return (flag);
}

/*  */
/**********************************************************************
 * Function: static doexecute(funcell *fexpr)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void doexecute(fexpr)
  funcell *fexpr;
{
  argcell *args;

  for (args = fexpr->args; args != (argcell *) NULL; args = args->nextp) {
    if (args->fnexpr != (funcell *) NULL)
      doexecute(args->fnexpr);
    else
      pushdata(strduplicate(args->data));
    if (stopexec)
      return;
  }

  (*(fexpr->fn->actionfn)) ();
}

/*  */
/**********************************************************************
 * Function: void execute(funcell *fexpr)
 *
 * Modifications:
 *      <list mods with name and date>
 */
void execute(fexpr)		/*EXPORTED*/
  funcell *fexpr;
{
  stopexec = 0;
  if (exectrace)
    (void)printfunexpr(fexpr, 0);
  if (checkexecute(fexpr))
    return;
  doexecute(fexpr);
}

/*  */
/**********************************************************************
 * Function: void abortexec()
 *
 * Aborts current (on-going) execution.
 *
 * Modifications:
 *      <list mods with name and date>
 */
void abortexec()
{				/*EXPORTED*/
  stopexec = 1;
}

/*### The parser functions ###*/
/*  */
/**********************************************************************
 * Function: static int checktoken(char *t, char **str)
 *
 * Modifications:
 *      <list mods with name and date>
 */

static int checktoken(t, str)
  char *t;
  char **str;
{
  if (!skipbl(str))
    return (EXHAUST);

  while (*t != '\0' && **str != '\0' && *t == **str) {
    t++;
    (*str)++;
  }
  if (*t == '\0')
    return (ACCEPT);
  if (**str == '\0')
    return (EXHAUST);
  return (REJECT);
}

/*  */
/**********************************************************************
 * Function: static int parsetoken(bnftoken *tok, char **str)
 *
 * Parses a token.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int parsetoken(tok, str)
  bnftoken *tok;
  char **str;
{
  int count, flag;
  switch (tok->termtoken) {
  case TERMTOK:
    flag = checktoken(tok->token->pname, str);
    return (flag);
  case NONTERM:
    return (parse(tok->token, str));
  case UTOKEN:
    if (!Acceptable(flag = parseprod(tok->extra->nextp, str)))
      return (flag);
    count = 1;
    while ((flag = parseprod(tok->extra->extra, str)) == REJECT) {
      if (!Acceptable(flag = parseprod(tok->extra->nextp, str)))
	return (flag);
      count += 1;
    }
    break;
  case WTOKEN:
    flag = parseprod(tok->extra->nextp, str);
    if (!Acceptable(flag))
      return (flag);
    count = 1;
    while (Acceptable(flag = parseprod(tok->extra->extra, str))) {
      if (!Acceptable(flag = parseprod(tok->extra->nextp, str)))
	return (flag);
      count += 1;
    }
    if (flag == EXHAUST)
      flag = ACCEPTX;
    else
      flag = ACCEPT;
    break;
  default:
    return (REJECT);
  }
  if (flag != EXHAUST) {
    (void)sprintf(tempint, "%d", count);
    delaypushdata(tempint);
  }
  return (flag);
}

/*  */
/**********************************************************************
 * Function: static int parseprod(bnftoken *prod, char **str)
 *
 * Parses a production alternative.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int parseprod(prod, str)
  bnftoken *prod;
  char **str;
{
  char *s;
  int fail;

  s = *str;
  fail = REJECT;
  while (prod != (bnftoken *) NULL) {
    switch (parsetoken(prod, str)) {
    case EXHAUST:
      fail = EXHAUST;
    case REJECT:
      prod = (bnftoken *) NULL;
      *str = s;
      return (fail);
      break;
    case ACCEPTX:
      fail = EXHAUST;
    case ACCEPT:
      prod = prod->nextp;
    }
  }
  if (fail == EXHAUST)
    fail = ACCEPTX;
  else
    fail = ACCEPT;
  return (fail);
}

/*  */
/**********************************************************************
 * Function: static int doprimary(word *goal, char **str)
 *
 * Parses a primary non-terminal
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int doprimary(goal, str)
  word *goal;
  char **str;
{
  if (goal->primary == 0) {
    (void)fprintf(stderr, "No syntax defined for %s; mismatch assumed.\n", goal->pname);
    return (REJECT);
  }
  if (goal->primary->actionfn == 0) {
    (void)fprintf(stderr, "Primary action %s is undefined; mismatch assumed\n",
		  goal->primary->pname);
    return (REJECT);
  }
  return ((*(goal->primary->actionfn)) (str));
}

/*  */
/**********************************************************************
 * Function: int parse(word *goal, char **str)
 *
 *   Parses the string *str towards the given goal. If successful,
 *   return value is 1, str is moved to after last consumed character,
 *   and the semantics execution environment is setup accordingly (see
 *   execute). If parsing is unsuccessful, return value is 0, str is
 *   unchanged, and the semantics execution environment is empty.
 *
 *   Return value from the parse function is one of the following:
 *
 *   ACCEPT  meaning that the parsing of the string str towards the
 *   goal is accepted, and there is no potential longer acceptable
 *   parse. When parsing is accepted (ACCEPT or ACCEPTX), the internal
 *   execution stack has been set up with action calls in accordance
 *   to the accepted parse (see the execute function).
 *
 *   ACCEPTX meaning that the parsing is accepted, but there is a
 *   potential longer acceptable parse.
 *
 *   EXHAUST meaning that the string was exhausted before a complete
 *   acceptable parse was obtained. This is a reject condition.
 *
 *   REJECT meaning that the parsing failed.
 *
 *   Further, the argument str is moved to the character immediately
 *   succeding the portion of the string that was "consumed" by the
 *   parse.
 *
 * Modifications:
 *      <list mods with name and date>
 */
int parse(goal, str)		/*EXPORTED*/
  word *goal;
  char **str;
{
  char *s;
  int code, temp;
  bnfrule *alts;

  argcell *oldargs;

  if (goal == 0)
    return (REJECT);

  s = *str;
  oldargs = funargs;
  funargs = (argcell *) NULL;

  alts = goal->syntax;
  if (alts == 0) {
    if (((code = doprimary(goal, str)) == ACCEPT) || code == ACCEPTX) {
      /* Successful primary => pushdelayed */
      funargs = appendargs(oldargs, funargs);
      return (code);
    }
    /* Failing primary */
    *str = s;
    freeargcells(funargs);
    funargs = oldargs;
    return (code);
  }
  code = REJECT;
  while (alts) {
    switch (temp = parseprod(alts->elements, str)) {
    case ACCEPT:
      if (code == EXHAUST)
	temp = ACCEPTX;
    case ACCEPTX:
      /* Make action expression */
      funargs = appendargs(oldargs, makeaction(alts->semantic, funargs));
      return (temp);
    case EXHAUST:
      code = EXHAUST;
    case REJECT:
      /* Throw present args and resume */
      freeargcells(funargs);
      funargs = (argcell *) NULL;
      alts = alts->nextp;
    }
  }

  freeargcells(funargs);
  funargs = oldargs;
  return (code);
}

/*### Input functions ###*/

/*  */
/**********************************************************************
 * Function: char *readline(char *first, char *prompt, int *count)
 *
 * /readline/ "appends" next line from infile to current input. You
 * give as argument a string pointer to the first character of the
 * input buffer that should be kept. A NULL pointer indicates that
 * the whole buffer should be thrown, so that the new input line
 * will be the first text fragment. Otherwise, the portion from the
 * given pointer to the end of the current buffer is kept (but moved
 * to the buffer beginning), and the new input line is appended
 * after it.  Note: the newline characters, '\n', appear in the
 * text. A NULL character is added after the buffer and overwritten
 * at next /readline/.
 *
 * The second argument is a prompt string that is display on stdout
 * in the format "prompt>" when the first textline for the buffer is
 * asked for. Successive lines are prompted with "...>". The
 * prompting occurs when infile is stdin or when the input tracing
 * is enabled. You enable tracing by a call to /traceinput/ with a
 * non-zero argument, and disable it with a zero argument to
 * /traceinput/.
 *
 * When tracing is enabled and infile is not stdin, then the prompts
 * occur as above, and in addition, the input line read is also
 * displayed on stdout.
 *
 * Input is taken from an infile, which defaults to stdin. You
 * redirect input by a call to /usefile/, which takes an opened
 * input file ptr as argument and returns the previous file pointer
 * (without closing the file). /usefile/ sets tracing mode so that
 * tracing is disabled for stdin and enabled otherwise.
 *
 * Modifications:
 *      <list mods with name and date>
 */
char *readline(first, prompt, count)	/*EXPORTED*/
  char *first;
  char *prompt;
  int *count;
{
  char *p;
  int ch = EOF;

  if (first == (char *)NULL || first < inbuf || first > last) {
    last = first = inbuf;
    *last = '\0';
  }
  if (first != inbuf) {
    p = inbuf;
    while (first < last)
      *(p++) = *(first++);
    last = p;
    first = inbuf;
    *last = '\0';
  }
  if ((outfile != NULL) && (displayflag || infile == stdin)) {
    if (last == inbuf)
      (void)fprintf(outfile, "%s>", prompt);
    else
      (void)fprintf(outfile, "...>");
  }
  lastfirst = last;
  *count = 0;
  while ((last < limit) && (ch = getc(infile)) != EOF) {
    *(last++) = (char)ch;
    if ((char)ch == '\n') {
      *last = '\0';
      if (displayflag)
	if ((infile != stdin) || (outfile == NULL))
	  (void)fprintf(stderr, "%s", lastfirst);
      return (first);
    }
    *count += 1;
  }

  *last = '\0';
  if (displayflag)
    if ((infile != stdin) || (outfile == NULL)) {
      (void)fprintf(stderr, "%s\n", lastfirst);
      if (last == limit)
	(void)fprintf(stderr, "[** input buffer full **]\n");
      if (ch == EOF)
	(void)fprintf(stderr, "[** input reaching end of file **]\n");
    }
  if (first == last)
    return ((char *)NULL);
  return (first);
}

/*  */
/**********************************************************************
 * Function: void traceinput(int flg)
 *
 * Enables/disables tracing of input
 *
 * Modifications:
 *      <list mods with name and date>
 */
void traceinput(flg)		/*EXPORTED*/
  int flg;
{
  displayflag = flg;
}

/*  */
/**********************************************************************
 * Function: FILE *usefile(FILE *f)
 *
 *  Redirects input
 *
 * Modifications:
 *      <list mods with name and date>
 */
FILE *usefile(f)		/*EXPORTED*/
  FILE *f;
{
  FILE *fx;

  fx = infile;
  infile = f;
  return (fx);
}

/*  */
/**********************************************************************
 * Function: FILE *promptfile(FILE *f)
 *
 * Redirects prompt output
 *
 * Modifications:
 *      <list mods with name and date>
 */
FILE *promptfile(f)		/*EXPORTED*/
  FILE *f;
{
  FILE *fx;

  fx = outfile;
  outfile = f;
  return (fx);
}

/*  */
/**********************************************************************
 * Function: void droplastline()
 *
 * Drops the last input line from the buffer.
 *
 * Modifications:
 *      <list mods with name and date>
 */
void droplastline()
{				/*EXPORTED*/
  last = lastfirst;
  *last = '\0';
}

/*### Dialogue state control functions ###*/

/*  */
/**********************************************************************
 * Function: word *setmeta(char *s)
 *
 * Sets the secondary parsing goal.
 *
 * Modifications:
 *      <list mods with name and date>
 */
word *setmeta(s)		/*EXPORTED*/
  char *s;
{
  word *p, *g;

  g = metagoal;

  if ((s != (char *)NULL) && ((p = findsymbol(s)) != (word *) NULL))
    metagoal = p;
  return (g);
}

/*  */
/**********************************************************************
 * Function: word *setgoal(char *s)
 *
 * Sets the main parsing goal.
 *
 * Modifications:
 *      <list mods with name and date>
 */
word *setgoal(s)		/*EXPORTED*/
  char *s;
{
  word *p, *g;

  g = goal;

  if ((s != (char *)NULL) && ((p = findsymbol(s)) != (word *) NULL))
    goal = p;
  return (g);
}

/*  */
/**********************************************************************
 * Function: void interact(char *g)
 *
 * Repeatedly calls /readline/ and parses towards goal g. The
 * function is exited by a semantic action that calls
 * /quitinteract/.
 *
 * Modifications:
 *      <list mods with name and date>
 */
void interact(g)		/*EXPORTED*/
  char *g;
{
  char *str, *instr;
  word *oldgoal;
  int cnt;
  funcell *fp;
  argcell *fa;

  fa = funargs;
  funargs = (argcell *) NULL;

  oldgoal = setgoal(g);
  run = 1;
  str = (char *)NULL;

  while (run) {
    (void)skipbl(&str);
    if (goal == (word *) NULL)
      str = readline(str, (char *)NULL, &cnt);
    else
      str = readline(str, goal->pname, &cnt);
    if (str == (char *)NULL)
      run = 0;
    else {
      instr = str;
      switch (parse(goal, &str)) {
      case ACCEPT:
      case ACCEPTX:
	fp = funexpr();
	execute(fp);
	freefuncell(fp);
	break;
      case REJECT:
	str = instr;
	switch (parse(metagoal, &str)) {
	case ACCEPT:
	case ACCEPTX:
	  fp = funexpr();
	  execute(fp);
	  freefuncell(fp);
	  break;
	case REJECT:
	  if (cnt > 0) {
	    (void)fprintf(stderr, "[** Couldn't parse that; last line is dismissed **]\n");
	    droplastline();
	    str = instr;
	  } else {
	    (void)fprintf(stderr, "[** Couldn't parse that; input phrase dismissed **]\n");
	    str = (char *)NULL;
	  }
	  break;
	case EXHAUST:
	  if (cnt > 0) {
	    str = instr;
	  } else {
	    str = (char *)NULL;
	  }
	}
	break;
      case EXHAUST:
	if (cnt > 0) {
	  str = instr;
	} else {
	  str = (char *)NULL;
	}
	break;
      }
    }
  }

  funargs = fa;

  goal = oldgoal;
  run = 1;
}

/*  */
/**********************************************************************
 * Function: void takefile(char *fn, char *g)
 *
 * Calls /interact/ with goal g, taking input from file fn.
 *
 * Modifications:
 *      <list mods with name and date>
 */
void takefile(fn, g)		/*EXPORTED*/
  char *fn;
  char *g;
{
  FILE *oldin = infile;
  FILE *oldout = outfile;

  if (!fn)
    return;
  if (strcmp(fn, "-") != 0) {
    if ((infile = fopen(fn, "r")) == 0) {
      (void)fprintf(stderr, "[** File %s won't open **]\n", fn);
      infile = oldin;
      return;
    }
  } else {
    infile = stdin;
    outfile = NULL;
  }
  interact(g);
  if (infile != stdin)
    (void)fclose(infile);
  outfile = oldout;
  infile = oldin;
}

/*### actions ###*/

/*  */
/**********************************************************************
 * Function: static int noop()
 *
 * does nothing at all
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(noop)
{
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: int quitinteract()
 *
 * Forces an exit from /interact/ after completed treatment of
 * current input.
 *
 * Modifications:
 *      <list mods with name and date>
 */
ACTION(quitinteract)
{				/*EXPORTED*/
  run = 0;
  END_ACTION;
}

/*### primaries ###*/

/*  */
/**********************************************************************
 * Function: static int eoln(char **str)
 *
 * Checks that rest of line is blank
 *
 * Modifications:
 *      <list mods with name and date>
 */
static PRIMARY(eoln)
  p = NULL;
  if (skipbl(str))
    return (REJECT);
  return (ACCEPT);
}

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static int empty(char **str)
 *
 * Accepts always without checking anything.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static PRIMARY(empty)
  p = NULL;
  return (ACCEPT);
}

/*  */
/**********************************************************************
 * Function: static int quotedstring(char **str)
 *
 * Checks and pushes text within double-quotes
 *
 * Modifications:
 *      <list mods with name and date>
 */
static PRIMARY(quotedstring)
  Ensuretoken;
  Failif(Pick != '\"');
  Consume(temp, (Peep != '\"') && (Peep != '\\' || Pick != '\0'));
  Exhaustif(Peep == '\0');
  Acceptif(temp, (Pick == '\"'));
}

/*  */
/**********************************************************************
 * Function: static int identifier(char **str)
 *
 * Checks and pushes symbol of letters and digits
 *
 * Modifications:
 *      <list mods with name and date>
 */
static PRIMARY(identifier)
  Ensuretoken;
  Consume(temp, isalpha(Peep) || isdigit(Peep) || Peep == '_');
  Acceptif(temp, *temp != '\0');
}

/*  */
/**********************************************************************
 * Function: static int blword(char **str)
 *
 * Checks and pushes symbol surrounded by blanks
 * Modifications:
 *      <list mods with name and date>
 */
static PRIMARY(blword)
  Ensuretoken;
  Consume(temp, Peep != '\0' && !isspace(Peep));
  Acceptif(temp, *temp != '\0');
}

/*  */
/**********************************************************************
 * Function: static int integer(char **str)
 *
 * Checks and pushes symbol of digits
 *
 * Modifications:
 *      <list mods with name and date>
 */
static PRIMARY(integer)
  Ensuretoken;
  Consume(temp, isdigit(Peep));
  Acceptif(temp, *temp != '\0');
}

/*  */
/**********************************************************************
 * Function: static int catchline(char **str)
 *
 * Pushes rest of line without check, and accepts
 *
 * Modifications:
 *      <list mods with name and date>
 */
static PRIMARY(catchline)
  p = NULL;
  delaypushdata(*str);
  while ((Peep != '\0') && (Peep != '\n'))
  Pick;
  return (ACCEPT);
}

/*### Boot-strap grammar ###*/

/*  */
/**********************************************************************
 * Function: static bnftoken *enlisttokens(int n)
 *
 * Modifications:
 *      <list mods with name and date>
 */
  static bnftoken *enlisttokens(n)
  int n;
{
  bnftoken *p;

  p = Poptoken;
  while (n-- > 1) {
    Toptoken->nextp = p;
    p = Poptoken;
  }
  return (p);
}

/*  */
/**********************************************************************
 * Function: static int definerule()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(definerule)
{
  int n;
  word *defw;

  n = popint();
  defw = symbol(popdata(), 1);
  defw->syntax = Poprule;
  while (n-- > 1) {
    Toprule->nextp = defw->syntax;
    defw->syntax = Poprule;
  }
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: static int makealt()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(makealt)
{
  word *act;
  /* int n;*/

  act = symbol(popdata(), 1);
  Pushrule(makerule((bnfrule *) NULL, act, enlisttokens(popint())));
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: static int nontermelem()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(nontermelem)
{
  Pushtoken(maketoken((bnftoken *) NULL, NONTERM, symbol(popdata(), 1)));
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: static int termelem()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(termelem)
{
  Pushtoken(maketoken((bnftoken *) NULL, TERMTOK, symbol(popdata(), 1)));
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: static int makeending()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(makeending)
{
  bnftoken *p;

  p = maketoken((bnftoken *) NULL, ENDING, (word *) NULL);
  p->extra = enlisttokens(popint());
  Pushtoken(p);
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: static int makewhile()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(makewhile)
{
  Toptoken->termtoken = WTOKEN;
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: static int makeuntil()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(makeuntil)
{
  Toptoken->termtoken = UTOKEN;
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: static int repeatelem()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(repeatelem)
{
  bnftoken *p1, *p2;

  p1 = Poptoken;
  p1->nextp = enlisttokens(popint());
  p2 = maketoken((bnftoken *) NULL, p1->termtoken, (word *) NULL);
  p2->extra = p1;
  Pushtoken(p2);
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: static int primaryrule()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(primaryrule)
{
  word *defw, *actw;
  actw = symbol(popdata(), 1);
  defw = symbol(popdata(), 1);
  defw->primary = actw;
  END_ACTION;
}

/*** Initiation function ***/

static actionentry bnftbl[] =
{
  {"definerule", definerule},
  {"makealt", makealt},
  {"nontermelem", nontermelem},
  {"termelem", termelem},
  {"makeending", makeending},
  {"makewhile", makewhile},
  {"makeuntil", makeuntil},
  {"repeatelem", repeatelem},
  {"primaryrule", primaryrule},
  {0, 0}
};

#define cpypushdata(s)	pushdata(strduplicate(s))

/*  */
/**********************************************************************
 * Function: static void bootgrammar()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void bootgrammar()
{
  defineactions(bnftbl);

  /* <bnf> ::= < <identifier> > ::= [ <bnfalt> while ! ] => definerule
     ! <identifier> is function <identifier> => primaryrule
     */
  cpypushdata("bnf");
  cpypushdata("<");
  (void)termelem();
  cpypushdata("identifier");
  (void)nontermelem();
  cpypushdata(">");
  (void)termelem();
  cpypushdata("::=");
  (void)termelem();
  cpypushdata("bnfalt");
  (void)nontermelem();
  cpypushdata("!");
  (void)termelem();
  cpypushdata("1");
  (void)makeending();
  (void)makewhile();
  cpypushdata("1");
  (void)repeatelem();
  cpypushdata("5");
  cpypushdata("definerule");
  (void)makealt();
  cpypushdata("identifier");
  (void)nontermelem();
  cpypushdata("is");
  (void)termelem();
  cpypushdata("function");
  (void)termelem();
  cpypushdata("identifier");
  (void)nontermelem();
  cpypushdata("4");
  cpypushdata("primaryrule");
  (void)makealt();
  cpypushdata("2");
  (void)definerule();

/*   <bnfalt> ::= [ <token> until => ] <identifier> => makealt */

  cpypushdata("bnfalt");
  cpypushdata("token");
  (void)nontermelem();
  cpypushdata("=>");
  (void)termelem();
  cpypushdata("1");
  (void)makeending();
  (void)makeuntil();
  cpypushdata("1");
  (void)repeatelem();
  cpypushdata("identifier");
  (void)nontermelem();
  cpypushdata("2");
  cpypushdata("makealt");
  (void)makealt();
  cpypushdata("1");
  (void)definerule();

/*   <token> ::= '[ <bnfrepeat> => noop
              ! < <identifier> > => nontermelem
              ! ' <blword> => noop
              ! <blword> => termelem
*/

  cpypushdata("token");
  cpypushdata("[");
  (void)termelem();
  cpypushdata("bnfrepeat");
  (void)nontermelem();
  cpypushdata("2");
  cpypushdata("noop");
  (void)makealt();
  cpypushdata("<");
  (void)termelem();
  cpypushdata("identifier");
  (void)nontermelem();
  cpypushdata(">");
  (void)termelem();
  cpypushdata("3");
  cpypushdata("nontermelem");
  (void)makealt();
  cpypushdata("'");
  (void)termelem();
  cpypushdata("blword");
  (void)nontermelem();
  cpypushdata("2");
  cpypushdata("termelem");
  (void)makealt();
  cpypushdata("blword");
  (void)nontermelem();
  cpypushdata("1");
  cpypushdata("termelem");
  (void)makealt();
  cpypushdata("4");
  (void)definerule();

/* <bnfrepeat> ::= [ <token> until <endclause> ] => repeatelem */

  cpypushdata("bnfrepeat");
  cpypushdata("token");
  (void)nontermelem();
  cpypushdata("endclause");
  (void)nontermelem();
  cpypushdata("1");
  (void)makeending();
  (void)makeuntil();
  cpypushdata("1");
  (void)repeatelem();
  cpypushdata("1");
  cpypushdata("repeatelem");
  (void)makealt();
  cpypushdata("1");
  (void)definerule();

/*   <endclause> ::= while <ending> => makewhile
                  ! until <ending> => makeuntil
*/

  cpypushdata("endclause");
  cpypushdata("while");
  (void)termelem();
  cpypushdata("ending");
  (void)nontermelem();
  cpypushdata("2");
  cpypushdata("makewhile");
  (void)makealt();
  cpypushdata("until");
  (void)termelem();
  cpypushdata("ending");
  (void)nontermelem();
  cpypushdata("2");
  cpypushdata("makeuntil");
  (void)makealt();
  cpypushdata("2");
  (void)definerule();

/*   <ending> ::= [ <token> until '] ] => makeending */

  cpypushdata("ending");
  cpypushdata("token");
  (void)nontermelem();
  cpypushdata("]");
  (void)termelem();
  cpypushdata("1");
  (void)makeending();
  (void)makeuntil();
  cpypushdata("1");
  (void)repeatelem();
  cpypushdata("1");
  cpypushdata("makeending");
  (void)makealt();
  cpypushdata("1");
  (void)definerule();

  bnfgoal = findsymbol("bnf");
}

/* ADDITIONAL MODULE INTERFACING */

/*  */
/**********************************************************************
 * Function: word *setbnf(char *s)
 *
 * Modifications:
 *      <list mods with name and date>
 */
word *setbnf(s)			/*EXPORTED*/
  char *s;
{
  word *p, *g;

  g = bnfgoal;

  if ((s != 0) && ((p = findsymbol(s)) != 0))
    bnfgoal = p;
  return (g);
}

/*  */
/**********************************************************************
 * Function: void bnfdef(char *s)
 *
 * Modifications:
 *      <list mods with name and date>
 */
void bnfdef(s)			/*EXPORTED*/
  char *s;
{
  int code;
  funcell *fp;
  argcell *fa;

  fa = funargs;
  funargs = (argcell *) NULL;

  if ((code = parse(bnfgoal, &s)) == ACCEPT || code == ACCEPTX) {
    fp = funexpr();
    execute(fp);
    freefuncell(fp);
  } else {
    (void)fprintf(stderr, "bnfdef ERROR: %s\n", s);
  }

  funargs = fa;
}

/*  */
/**********************************************************************
 * Function: void definegrammar(char **syntax)
 *
 * Modifications:
 *      <list mods with name and date>
 */
void definegrammar(syntax)
  char **syntax;
{
  while (*syntax != (char *)NULL)
    bnfdef(*(syntax++));
}

/*### Initialisation of the xparser module ###*/

static actionentry initactions[] =
{
  {"noop", noop},
  {"exit", quitinteract},
  {0, 0}
};

static actionentry initprimaries[] =
{
  {"eoln", eoln},
  {"identifier", identifier},
  {"blword", blword},
  {"integer", integer},
  {"empty", empty},
  {"string", quotedstring},
  {"catchline", catchline},
  {0, 0}
};

/*  */
/**********************************************************************
 * Function: void init_parser()
 *
 * Initialises the parser system
 *
 * Modifications:
 *      <list mods with name and date>
 */
void init_parser()
{				/*EXPORTED*/

  memset((void *)dstack, 0, (int)sizeof(dstack));
  defineactions(initactions);
  defineprimaries(initprimaries);

  predefinedwords = setvocabulary((word *) NULL);

  bootgrammar();

  if ((lastfirst = last = inbuf = (char *)malloc((ALLOC_T) (INBUFLIMIT + 10))) == NULL) {
    report_malloc_fail("init_parser");
  }
  limit = inbuf + INBUFLIMIT;
  funargs = (argcell *) NULL;
}

#define freedata(s)	if (s != (FREEPTR *) NULL) free((FREEPTR *)s)

/* Printing the vocabulary */

/*  */
/**********************************************************************
 * Function: static void indent(int i)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void indent(i)		/* Write i spaces */
  int i;
{
  while (i-- > 0)
    (void)fprintf(stderr, " ");
}

/*  */
/**********************************************************************
 * Function: static void printrelems(bnftoken *re)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void printrelems(re)	/* Print the production part of a rule */
  bnftoken *re;
{
  while (re != (bnftoken *) NULL) {
    switch (re->termtoken) {
    case TERMTOK:
      (void)fprintf(stderr, " %s", re->token->pname);
      break;
    case NONTERM:
      (void)fprintf(stderr, " <%s>", re->token->pname);
      break;
    case UTOKEN:
      (void)fprintf(stderr, " [");
      printrelems(re->extra->nextp);
      (void)fprintf(stderr, " until");
      printrelems(re->extra->extra);
      (void)fprintf(stderr, " ]");
      break;
    case WTOKEN:
      (void)fprintf(stderr, " [");
      printrelems(re->extra->nextp);
      (void)fprintf(stderr, " while");
      printrelems(re->extra->extra);
      (void)fprintf(stderr, " ]");
      break;
    default:
      (void)fprintf(stderr, " #%d", re->termtoken);

    }
    re = re->nextp;
  }
}

/*  */
/**********************************************************************
 * Function: static void printrules(bnfrule *r, int ind)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void printrules(r, ind)	/* Print a list of rules */
  bnfrule *r;
  int ind;
{
  (void)fprintf(stderr, " ::=");
  while (r != (bnfrule *) NULL) {
    printrelems(r->elements);
    (void)fprintf(stderr, " => %s\n", r->semantic->pname);
    if ((r = r->nextp) != (bnfrule *) NULL) {
      indent(ind);
      (void)fprintf(stderr, "!");
    }
  }
}

/*  */
/**********************************************************************
 * Function: static void printsyntax(word *r)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void printsyntax(r)	/* Print the syntax associated to word r */
  word *r;
{
  if (r == (word *) NULL) {
    (void)fprintf(stderr, "Undefined symbol.\n");
    return;
  }
  (void)fprintf(stderr, "%s", r->pname);
  if (r->syntax) {
    printrules(r->syntax, strlen(r->pname));
  } else {
    if (r->primary) {
      (void)fprintf(stderr, " is function %s\n", r->primary->pname);
    } else {
      (void)fprintf(stderr, " has no syntax defined\n");
    }
  }
}

static word *gtbl[100];
static int gtblp, gtblg;

/*  */
/**********************************************************************
 * Function: static void pcatch(word *w)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void pcatch(w)
  word *w;
{
  int i;
  for (i = 0; i < gtblp; i++)
    if (w == gtbl[i])
      return;
  gtbl[gtblp++] = w;
}

/*  */
/**********************************************************************
 * Function:  static void catchnonterms(bnftoken *re)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void catchnonterms(re)
  bnftoken *re;
{
  while (re != (bnftoken *) NULL) {
    switch (re->termtoken) {
    case NONTERM:
      pcatch(re->token);
      break;
    case UTOKEN:
    case WTOKEN:
      catchnonterms(re->extra->nextp);
      catchnonterms(re->extra->extra);
      break;
    default:
      ;
    }
    re = re->nextp;
  }
}

/*  */
/**********************************************************************
 * Function: static void catchprint(word *w)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void catchprint(w)
  word *w;
{
  bnfrule *r;
  /* bnftoken *re;*/

  for (r = w->syntax; r != (bnfrule *) NULL; r = r->nextp)
    catchnonterms(r->elements);
  printsyntax(w);
  (void)fprintf(stderr, "\n");
}

/*  */
/**********************************************************************
 * Function: void printall(word *rh)
 *
 * Modifications:
 *      <list mods with name and date>
 */
void printall(rh)		/* Print the grammar from rh */
  word *rh;
{
  if (rh == (word *) NULL) {
    (void)fprintf(stderr, "Undefined symbol.\n");
    return;
  }
  gtblp = gtblg = 0;
  pcatch(rh);

  while (gtblg < gtblp)
    catchprint(gtbl[gtblg++]);

}

/* meta actions */

/*  */
/**********************************************************************
 * Function: static int printcurrentsyntax()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(printcurrentsyntax)
{
  printsyntax(setgoal((char *)NULL));
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: static int printcurrentgrammar()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(printcurrentgrammar)
{
  printall(setgoal((char *)NULL));
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: static int printgrammar()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(printgrammar)
{
  char *s;

  s = popdata();
  printall(findsymbol(s));
  freedata(s);
  s = NULL;
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: static int printvoc()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(printvoc)
{
  word *w;

  for (w = setvocabulary((word *) NULL); w != 0; w = w->nextp)
    printsyntax(w);
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: static int callsystem()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(callsystem)
{
  char *s;

  s = popdata();
  (void)system(s);
  freedata(s);
  s = NULL;
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: static int changegoal()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(changegoal)
{
  char *s;
  word *g;

  s = popdata();
  g = setgoal(s);

  if (g == setgoal((char *)NULL))
    (void)fprintf(stderr, "Goal is not changed!\n");
  freedata(s);
  s = NULL;
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: static int clearscreen()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(clearscreen)
{
  (void)fprintf(stderr, "\33[0X");
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: static int fliptrace()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(fliptrace)
{
  exectrace = !exectrace;
  (void)fprintf(stderr, "Ok\n");
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: static int verbose()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(verbose)
{
  traceinput(1);
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: static int quiet();
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(quiet)
{
  traceinput(0);
  exectrace = 0;
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: static int takeinputfile()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(takeinputfile)
{
  char *fn, *g;

  g = popdata();
  fn = popdata();
  takefile(fn, g);
  freedata(g);
  g = NULL;
  freedata(fn);
  fn = NULL;
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: static int currentgoal()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(currentgoal)
{
  word *g;
  char *gn;

  if ((g = setgoal((char *)NULL)) == (word *) NULL)
    gn = (char *)NULL;
  else {
    if ((gn = (char *)malloc((ALLOC_T)strlen(g->pname) + 1)) == NULL) {
      report_malloc_fail("currentgoal");
    }
    (void)strcpy(gn, g->pname);
  }

  pushdata(gn);
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function:  static int namedgoal()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(namedgoal)
{
  /* Could check the stacked identifier against the current vocabulary */
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function:  static int drop1()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(drop1)
{
  free(popdata());
  END_ACTION;
}

static actionentry metaactions[] =
{
  {"printcurrentsyntax", printcurrentsyntax},
  {"printcurrentgrammar", printcurrentgrammar},
  {"printgrammar", printgrammar},
  {"printvoc", printvoc},
  {"callsystem", callsystem},
  {"changegoal", changegoal},
  {"clearscreen", clearscreen},
  {"fliptrace", fliptrace},
  {"verbose", verbose},
  {"quiet", quiet},
  {"takeinputfile", takeinputfile},
  {"currentgoal", currentgoal},
  {"namedgoal", namedgoal},
  {"drop1", drop1},
  {0, 0}
};

static char *(metasyntax[]) = {
  "<metacmd> ::= ? <goal> => printgrammar\
	 ! words => printvoc\
	 ! bl => clearscreen\
	 ! flip trace => fliptrace\
         ! verbose => verbose\
         ! quiet => quiet\
         ! take <blword> <goal> => takeinputfile\
	 ! ! <catchline> => callsystem\
	 ! > <blword> => changegoal\
	 ! exit => exit\
	 ! <eoln> => noop",

  "<goal> ::= <blword> => namedgoal\
 	 ! <empty> => currentgoal",

  "<meta> ::= ! <metacmd> <eoln> => noop\
         ! # <catchline> => drop1\
	 ! ? <eoln> => printcurrentsyntax",

  "<unix> ::= <meta> => noop ! <catchline> => callsystem",

  0
};

/*  */
/**********************************************************************
 * Function: void metagrammar()
 *
 * Modifications:
 *      <list mods with name and date>
 */
void metagrammar()
{
  defineactions(metaactions);
  definegrammar(metasyntax);
  (void)setmeta("meta");
}

/*  */
/**********************************************************************
 * Function: static void report_malloc_fail(char *Pfunc_name)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void report_malloc_fail(Pfunc_name)
  char *Pfunc_name;
{
  (void)printf("===> malloc failure in libparser routine %s\n",
	       Pfunc_name);

/* this old way depends on liblincks, which makes this module
 * unusable by rechts, etc.
 *    LogMess(LL_uid,
 *            "libparser: Couldn't allocate memory in skipbl, %s",
 *            sys_errlist[(errno > sys_nerr) ? 0 : errno]);
 */
}

/*  */
/**********************************************************************
 * Function: static char *strduplicate( char *input );
 *
 * mallocs and copies the input and returns a pointer to the copy.  this
 * is just a little safer since it checks for null input.  copied from
 * libshared, but i don't want the dependency of this library on that
 * one.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static char *strduplicate(input)
  char *input;
{
  char *output = NULL;

  if (input == NULL)
    return NULL;

  if ((output = (char *)malloc((ALLOC_T)strlen(input) + 1)) == NULL)
    return NULL;

  strcpy(output, input);
  return output;
}
