/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* 
 * MODULE NAME: 	sysfuns.c
 *
 * SCCSINFO:		@(#)sysfuns.c	1.7 5/3/94
 *
 * ORIGINAL AUTHOR(S):  Lin Padgham 1987-09-18
 *
 * MODIFICATIONS:
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 * This file contains the system level functions used by the user
 * level functions
 */
/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */
#include "sunlincks.h"
#include "dbcodes.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_sysfuns.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_dbcomm.h"
#include "f_wssupport.h"
#include "f_logws.h"

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern molecule *LL_moltbl;
extern int LL_uid;
extern char *sys_errlist[];
extern int sys_nerr;
extern int errno;

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
#define LINKMATCH(x,y)	(x->link.vs==y->vs && \
			 (y->inst==INST_UNBOUND || x->link.inst==y->inst))
#define LOG_ERRSTRING   sys_errlist[(errno >= sys_nerr) ? 0 : errno]
#define LOG_MESS(b)     LogMess(LL_uid,"%s, %s", b, LOG_ERRSTRING); \
			perror(b)

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * INTERNAL (STATIC) DATA: 
 *********************************************************************/
/* none */

/*  */
/*
 *  SYSTEM LEVEL FUNCTIONS: Used by the user level functions
 */

/*  */
/**********************************************************************
 * Function: errcode LL_sysgetattrgroup(int mol,char *tag,attrgroup **grouptag)
 * 
 * SYSGETATTRGROUP
 * Retrieve an attribute group for a molecule. Returns (via grouptag)
 * a pointer to the attrgroup record that matched, or NULL. 
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_sysgetattrgroup(mol,tag,grouptag)
  int mol;
  char *tag;
  attrgroup **grouptag;
{
  attrgroup *prevgr;
  int ERRCODE;

  if (LL_moltbl[mol].inWS != 1) 
    if ((ERRCODE = LL_DBRetry(mol, UNKNOWN_TYPE, LL_DBRetrieve)))
      return(ERRCODE);
  if ((ERRCODE = LL_findgrptag((cell *)LL_moltbl[mol].attributes, tag,
                           (cell **)&prevgr, (cell **)grouptag)))
    return(ERRCODE);
  return(SUCCESS);
}

/*  */
/**********************************************************************
 * Function: errcode LL_sysgetattr(int mol,char *group,*tag,attribute **attptr)
 * 
 * SYSGETATTR
 * Retrieve an attribute field for a molecule. Returns a pointer (via attptr) 
 * to the attribute record that matched, or NULL.
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_sysgetattr(mol,group,tag,attptr)
  int mol;
  char *group,*tag;
  attribute **attptr;
{
  attribute *prev;
  int ERRCODE;
  attrgroup *grpptr;

  if (LL_moltbl[mol].inWS != 1) 
    if ((ERRCODE = LL_DBRetry(mol, UNKNOWN_TYPE, LL_DBRetrieve)))
      return(ERRCODE);
  grpptr = LL_moltbl[mol].attributes;
  if ((ERRCODE = LL_findfieldtag((cell **)&grpptr, group, tag,
                             (cell **)&prev, (cell **)attptr)))
    return(ERRCODE);
  return(SUCCESS);
}

/*  */
/**********************************************************************
 * Function: errcode LL_sysgetlinkitem(FIVE PARAMETERS)
 *
 * Parameters:
 *	int mol
 *	char *group
 *	char *tag
 *	int pos
 *	linkitem **itemptr
 * 
 * SYSGETLINKITEM
 * Retrieve a linkitem for a given molecule, group, fieldtag and position. 
 * Returns a pointer (via itemptr) to the linkitem record that matched.
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_sysgetlinkitem(mol,group,tag,pos,itemptr)
  int mol;
  char *group,*tag;
  int pos;
  linkitem **itemptr;
{
  linkgroup *grpptr;
  linktag *prev,*fieldptr;
  int ERRCODE, i = 1;

  if (LL_moltbl[mol].inWS != 1) 
    if ((ERRCODE = LL_DBRetry(mol, UNKNOWN_TYPE, LL_DBRetrieve)))
      return(ERRCODE);
  grpptr=LL_moltbl[mol].links;
  if ((ERRCODE = LL_findfieldtag((cell **)&grpptr, group, tag,
                             (cell **)&prev, (cell **)&fieldptr)))
    return(ERRCODE);
  *itemptr = fieldptr->value;
  while (*itemptr) {
    if (i == pos) 
      return(SUCCESS);
    *itemptr = (*itemptr)->next_item;
    ++i;
  }
  return(ERR_POS);
}

/*  */
/**********************************************************************
 * Function: errcode LL_sysgetlink(FOUR PARAMETERS)
 *
 * SYSGETLINK
 * Retrieve a link field for a molecule. Returns a pointer (via linkptr) 
 * to the linktag record that matched, or NULL.
 *
 * Parameters:
 *	int mol
 *	char *group
 *	char *tag
 *	linktag **linkptr
 * 
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_sysgetlink(mol,group,tag,linkptr)
  int mol;
  char *group;
  char *tag;
  linktag **linkptr;
{
  linktag *prev;
  int ERRCODE;
  linkgroup  *grpptr;

  if (LL_moltbl[mol].inWS != 1) 
    if ((ERRCODE = LL_DBRetry(mol, UNKNOWN_TYPE, LL_DBRetrieve)))
      return(ERRCODE);
  grpptr = LL_moltbl[mol].links;
  if ((ERRCODE = LL_findfieldtag((cell **)&grpptr, group, tag,
                             (cell **)&prev, (cell **)linkptr)))
    return(ERRCODE);
  return(SUCCESS);
}

/*  */
/**********************************************************************
 * Function: errcode LL_sysgetlinkgroup(int mol,char *tag,linkgroup **grouptag)
 * 
 * SYSGETLINKGROUP
 * Retrieve a link group for a bound label. Returns (via grouptag)
 * a pointer to the linkgroup record that matched, or NULL. 
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_sysgetlinkgroup(mol,tag,grouptag)
  int mol;
  char *tag;
  linkgroup **grouptag;
{
  linkgroup *prevgr;
  int ERRCODE;

  if (LL_moltbl[mol].inWS != 1) 
    if ((ERRCODE = LL_DBRetry(mol, UNKNOWN_TYPE, LL_DBRetrieve))) 
      return(ERRCODE);
  if ((ERRCODE = LL_findgrptag((cell *)LL_moltbl[mol].links, tag,
                           (cell **)&prevgr, (cell **)grouptag)))
    return(ERRCODE);
  return(SUCCESS);
}

/*  */
/**********************************************************************
 * Function: errcode LL_sysputlinkitem(FIVE PARAMETERS)
 * 
 * Parameters:
 *	int mol
 *	char *group
 *	char *field
 *	int pos
 *	label *lbl
 * 
 * SYSPUTLINKITEM
 * Puts a linkitem at the specified position, under a given group and field
 * tag for a molecule.
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_sysputlinkitem(mol,group,field,pos,lbl)
  int mol;
  char *group;
  char *field;
  int pos;
  label *lbl;
{
  int ERRCODE,i=1;
  linkgroup *linkgrp, *templinkgroup;
  linktag *linkptr, *templinktag;
  linkitem *temp,*itemptr;

  if ((ERRCODE = LL_sysgetlink(mol,group,field,&linkptr))) {
    if (ERRCODE == ERR_GROUP) {  /* add group and field tags */
      if (!(linkgrp = (linkgroup *)malloc(sizeof(linkgroup)))) {
	LOG_MESS("LL_sysputlinkitem: malloc failed for 'linkgrp'");
      }
      linkgrp->tag = LL_copystring(group);
      if (!(linkgrp->value=linkptr=(linktag *)malloc(sizeof(linktag)))) {
	LOG_MESS("LL_sysputlinkitem: malloc failed for 'linkptr'");
      }
      linkptr->tag = LL_copystring(field);
      linkptr->value = (linkitem *)NULL;
      linkptr->next = (linktag *)NULL;
      templinkgroup = LL_moltbl[mol].links;
      LL_moltbl[mol].links = linkgrp;
      linkgrp->next = templinkgroup;
    }
    else if (ERRCODE == ERR_FIELD) {  /* get group then add field tag */
      if ((ERRCODE = LL_sysgetlinkgroup(mol,group,&linkgrp)))
	return(ERRCODE);
      templinktag = linkgrp->value;
      linkgrp->value = linkptr = (linktag *)malloc(sizeof(linktag));
      if (!linkptr) {
	  LOG_MESS("LL_sysputlinkitem:  malloc failed for 'linkptr'");
	}
      linkptr->tag = LL_copystring(field);
      linkptr->next = templinktag;
      linkptr->value = (linkitem *)NULL;
    }
    else 
      return(ERRCODE);
  }

  if (pos == 1) {
    temp = linkptr->value;
    linkptr->value = itemptr = (linkitem *)malloc(sizeof(linkitem));
    if (!itemptr) {
	  LOG_MESS("LL_sysputlinkitem:  malloc failed for 'itemptr'");
	}
    itemptr->link.vs = lbl->vs;
    itemptr->link.inst = lbl->inst;
    itemptr->next_item = temp;
    return(SUCCESS);
  }
  else {
    itemptr = linkptr->value;
    while (itemptr) {
      if (i == (pos - 1)) {
	temp = itemptr->next_item;
	itemptr->next_item = (linkitem *)malloc(sizeof(linkitem));
	if (!itemptr->next_item) {
	    LOG_MESS("LL_sysputlinkitem:  malloc failed for '->next_item'");
	  }
	itemptr->next_item->link.vs = lbl->vs;
	itemptr->next_item->link.inst = lbl->inst;
	itemptr->next_item->next_item = temp;
	return(SUCCESS);
      }
      else  {
	itemptr = itemptr->next_item;
	++i;
      }
    }
    return(ERR_POS);
  }
}

/*  */
/**********************************************************************
 * Function: errcode LL_sysputattr(int mol,char *group,char *field,attrval *val)
 * 
 * SYSPUTATTR
 * puts a linkitem at the specified position, under a given group and field
 * tag for a molecule.
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_sysputattr(mol,group,field,val)
  int mol;
  char *group;
  char *field;
  attrval *val;
{
  int ERRCODE,i;
  attrgroup *attgrp,*temp;
  attribute *attptr, *tempattribute;

  if ((ERRCODE = LL_sysgetattr(mol,group,field,&attptr))) {
    if (ERRCODE == ERR_GROUP) {  /* add group and field tags */
      if (!(attgrp = (attrgroup *)malloc(sizeof(attrgroup)))) {
	  LOG_MESS("LL_sysputattr: malloc failed for 'attgrp'");
	}
      attgrp->tag = LL_copystring(group); 
      attgrp->value = attptr = (attribute *)malloc(sizeof(attribute));
      if (!attptr) {
	  LOG_MESS("LL_sysputattr: malloc failed for 'attptr'");
	}
      attptr->tag = LL_copystring(field);
      attptr->value = (char *)NULL;
      attptr->size = 0;
      attptr->next = (attribute *)NULL;
      temp = LL_moltbl[mol].attributes;
      LL_moltbl[mol].attributes = attgrp;
      attgrp->next = temp;
    }
    else if (ERRCODE == ERR_FIELD) {  /* get group then add field tag */
      if ((ERRCODE = LL_sysgetattrgroup(mol,group,&attgrp)))
	return(ERRCODE);
      tempattribute = attgrp->value;
      attgrp->value = attptr = (attribute *)malloc(sizeof(attribute));
      if (!attptr) {
	  LOG_MESS("LL_sysputattr: malloc failed for 'attptr'");
	}
      attptr->tag = LL_copystring(field);
      attptr->next = tempattribute;
      attptr->value = (char *)NULL;
      attptr->size = 0;
    }
    else return(ERRCODE);
  }
  if (attptr->size) free(attptr->value);
  if (!(attptr->value = (char *)malloc((ALLOC_T)val->attsize))) {
      LOG_MESS("LL_sysputattr: malloc failed for 'attptr->value'");
    }
  attptr->size = val->attsize;
  for (i = 0; i < val->attsize; ++i)
    attptr->value[i] = val->attvalue[i];
  return(SUCCESS);
}

/*  */
/**********************************************************************
 * Function: void LL_sysdellinkitem(linkitem **p, label *to)
 * 
 * SYSDELLINKITEM
 * Deletes all ocurrences of "to" in the link pointed to by "p".
 *
 * Modifications:
 *      <list mods with name and date>
 */
void LL_sysdellinkitem(p,to)
  linkitem **p;
  label *to;
{
  linkitem *px,*py;

  px = *p;
  while ((px) && (LINKMATCH(px,to))) {
    *p = px->next_item;
    free((FREEPTR *)px);
    px = *p;
  };

  if (!px)
    py=(linkitem *)NULL;
  else
    py=px->next_item;

  while (py) {
    if (LINKMATCH(py,to)) {
      px->next_item=py->next_item;
      free((FREEPTR *)py);
    } else 	px=py;
    py=px->next_item;
  };
}
