/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME: 	rdwrmol.c
 *
 * SCCSINFO:		@(#)rdwrmol.c	1.7 5/3/94
 *
 * ORIGINAL AUTHOR(S):  Peter ]berg, 1987-09-18
 *
 * MODIFICATIONS:
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 * This file contains the functions for sending and receiving a molecule and
 * its parts  over the socket connection
 *
 *	Data format for link section is:
 *	        <group name length>
 *		"group name string"
 *		<tag name length>
 *		"tag name string"
 *		<label1> <label2> <label3>
 *		<0>
 *		...
 *		<group name length>
 *		"group name string"
 *		<0>
 *		...
 */

/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */
#include <netinet/in.h>

#include "sunlincks.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_rdwrmol.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_attrpack.h"
#include "f_dbcomm.h"
#include "f_logws.h"
#include "f_structsupport.h"
#include "f_linkpack.h"

/* libshared.a */
extern char *strdup();

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern int LL_uid;
extern char *sys_errlist[];
extern int sys_nerr;
extern int errno;
extern int LL_socketno;

/* Buffer definitions */
extern char *LL_packbuff;
extern char *LL_packbuffp;
extern int LL_packbuffsize;
extern int LL_packbuffspace;
extern int LL_packbufflen;

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
#define PACKBUFFSIZE 32768	/* Default size of pack buffer */
#define NAMESTRING 0		/* Data type keys */
#define ITEMLABEL 1
#define TERMINATE 2
#define COUNT 3

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static int ReceiveAttributes P_(( attrgroup **attrroot, int size, int s ));
static int ReceiveImage P_(( image_desc **image, int size, int s ));
static int ReceiveLinks P_(( linkgroup **linkroot, int size, int s ));
static int SendAttributes P_(( attrgroup *attrroot, int s ));
static int SendImage P_(( image_desc *image, int s ));
static int SendLinks P_(( linkgroup *linkroot, int s ));

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
/* none */

/*  */
/**********************************************************************
 * Function: errcode LL_SendMolecule(molecule *mol)
 *
 * Sends a molecule to the socket connection
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_SendMolecule(mol)
  molecule *mol;
{
  /* Pack and send links section*/
  if ((SendLinks(mol->links, LL_socketno) < 0) ||
      (SendAttributes(mol->attributes, LL_socketno) < 0) ||
      (SendImage(mol->image, LL_socketno) < 0))
    return (FAIL);
  return (SUCCESS);
}


/*  */
/**********************************************************************
 * Function: static int SendLinks(linkgroup *linkroot, int s)
 *
 * Sends the link structure linkroot to socket s
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int SendLinks(linkroot, s)
  linkgroup *linkroot;
  int s;
{
  register int count, size;

  if (!linkroot) {
    if (LL_SendCount(0, s) < 0)
      return (FAIL);
    else
      return (SUCCESS);
  }
  if (!(LL_packbuff = malloc((ALLOC_T)PACKBUFFSIZE))) {
    LogMess(LL_uid,
	    "Liblincks: No memory for PACKBUFFSIZE in SendLinks, %s",
	    sys_errlist[(errno > sys_nerr) ? 0 : errno]);
    perror("WARNING: No Memory For \"PACKBUFFSIZE\" In \"SendLinks\"\n");
    return (FAIL);
  }

  LL_packbuffp = LL_packbuff;
  LL_packbuffspace = LL_packbuffsize = PACKBUFFSIZE;
  size = 17;	/* Initalisation to something non-zero */

  if (LL_PackGroup(linkroot) >= 0) {
    size = LL_packbuffsize - LL_packbuffspace;
    LL_packbuffp = LL_packbuff;

    if (LL_SendCount(size, s) >= 0) {
      while (size > 0) {
	if ((count = write(s, LL_packbuffp, (IOSIZE_T)size)) < 0) {
	  perror("SendLinks");
	  break;
	}
	if (!count && size) {
	  perror("SendLinks data not accepted at socket");
	  break;
	}

	LL_packbuffp += count;
	size -= count;
      }
    }
  }

  free(LL_packbuff);

  if (size)
    return (FAIL);
  return (SUCCESS);
}


/*  */
/**********************************************************************
 * Function: static int SendAttributes(attrgroup *attrroot, int s)
 *
 * Sends the link structure linkroot to socket s
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int SendAttributes(attrroot, s)
  attrgroup *attrroot;
  int s;
{
  register int count, size;

  if (!attrroot) {
    if (LL_SendCount(0, s) < 0)
      return (FAIL);
    else
      return (SUCCESS);
  }
  if (!(LL_packbuff = malloc((ALLOC_T)PACKBUFFSIZE))) {
    LogMess(LL_uid,
	    "Liblincks: No memory for PACKBUFFSIZE in SendAttributes, %s",
	    sys_errlist[(errno > sys_nerr) ? 0 : errno]);
    perror("WARNING: No Memory For \"PACKBUFFSIZE\" In \"SendAttributes\"\n");
    return (FAIL);
  }

  LL_packbuffp = LL_packbuff;
  LL_packbuffspace = LL_packbuffsize = PACKBUFFSIZE;
  size = 17;	/* Initalisation to something non-zero */

  if (LL_PackAttrGroup(attrroot) >= 0) {
    size = LL_packbuffsize - LL_packbuffspace;
    LL_packbuffp = LL_packbuff;
    if (LL_SendCount(size, s) >= 0) {
      while (size > 0) {
	if ((count = write(s, LL_packbuffp, (IOSIZE_T)size)) < 0) {
	  perror("SendAttributes");
	  break;
	}
	if (!count && size) {
	  perror("SendAttributes data not accepted at socket");
	  break;
	}
	/* Adjust byte count and position pointer */
	LL_packbuffp += count;
	size -= count;
      }
    }
  }

  free(LL_packbuff);

  if (size)
    return (FAIL);
  return (SUCCESS);
}


/*  */
/**********************************************************************
 * Function: static int SendImage(image_desc *image, int s)
 *
 * Sends a buffer structure to the socket connection
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int SendImage(image, s)
  image_desc *image;
  int s;
{
  register int count, size;
  register char *valuep;

  if (!image) {
    if (LL_SendCount(0, s) < 0)
      return (FAIL);
    else
      return (SUCCESS);
  }

  size = image->size;
  valuep = image->value;

  if (LL_SendCount(size, s) >= 0) {
    while (size > 0) {
      if ((count = write(s, valuep, (IOSIZE_T)size)) < 0) {
	perror("SendImage");
	break;
      }
      if (!count && size) {
	perror("SendImage data not accepted at socket");
	break;
      }
      /* Adjust byte count and position pointer */
      valuep += count;
      size -= count;
    }
  }

  if (size)
    return (FAIL);
  return (SUCCESS);
}


/*  */
/**********************************************************************
 * Function: errcode LL_SendCount(int val, int s)
 *
 * Sends an integer to the socket connection
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_SendCount(val, s)
  int val;
  int s;
{
  register int size, count;
  register char *buffp;
  t_u32bits netinet = htonl(val);

  size = sizeof(netinet);
  buffp = (char *)&netinet;

  while (size > 0) {
    if ((count = write(s, buffp, (IOSIZE_T)size)) < 0) {
      perror("LL_SendCount");
      break;
    }
    if (!count && size) {
      perror("LL_SendCount data not accepted at socket");
      break;
    }
    /* Adjust byte count and position pointer */
    buffp += count;
    size -= count;
  }

  if (size)
    return (FAIL);
  return (SUCCESS);
}


/*  */
/**********************************************************************
 * Function: molecule *LL_ReceiveMolecule()
 *
 * Reads a molecule from the socket connection
 *
 * Modifications:
 *      <list mods with name and date>
 */
molecule *LL_ReceiveMolecule()
{
  static molecule mol;
  int count;

  mol.links = (linkgroup *)NULL;
  mol.attributes = (attrgroup *)NULL;
  mol.image = (image_desc *)NULL;
  mol.ITix = 0;
  mol.inWS = mol.edited = mol.mark = (unsigned) 0;

  if ((LL_ReceiveCount(&count, LL_socketno) < 0) ||
      (ReceiveLinks(&mol.links, count, LL_socketno) < 0) ||
      (LL_ReceiveCount(&count, LL_socketno) < 0) ||
      (ReceiveAttributes(&mol.attributes, count, LL_socketno) < 0) ||
      (LL_ReceiveCount(&count, LL_socketno) < 0) ||
      (ReceiveImage(&mol.image, count, LL_socketno) < 0)) {
    LL_releasemol(&mol);
    return (molecule *)NULL;
  }
  return (&mol);
}

/*  */
/**********************************************************************
 * Function: static int ReceiveLinks(linkgroup **linkroot, int size, int s)
 *
 * Reads from the specified socket for the
 * molecule link structure with size size
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int ReceiveLinks(linkroot, size, s)
  linkgroup **linkroot;
  register int size;
  int s;
{
  register int count;

  if (!size) {
    *linkroot = (linkgroup *)NULL;
    return (SUCCESS);
  }
  if (!(LL_packbuff = malloc((ALLOC_T)size))) {
    LogMess(LL_uid,
	    "Liblincks: No memory for buffer in ReceiveLinks, %s",
	    sys_errlist[(errno > sys_nerr) ? 0 : errno]);
    perror("WARNING: No Memory For \"buffer\" In \"ReceiveLinks\"\n");
    return (FAIL);
  }

  LL_packbufflen = size;
  LL_packbuffp = LL_packbuff;

  while (size > 0) {
    if ((count = read(s, LL_packbuffp, (IOSIZE_T)size)) < 0) {
      perror("ReceiveLinks");
      break;
    }
    if (!count && size) {
      perror("ReceiveLinks unexpected EOF on socket");
      break;
    }
    size -= count;
    LL_packbuffp += count;
  }

  LL_packbuffp = LL_packbuff;

  if (size == 0) {
    size = (LL_UnpackGroup(linkroot) < 0);
    /*
     * what happens to linkroot on error?
     */
  }

  free(LL_packbuff);

  if (size)
    return (FAIL);
  return (SUCCESS);

}

/*  */
/**********************************************************************
 * Function: static int ReceiveAttributes(attrgroup **attrroot, int size, s)
 *
 * Reads from the specified socket for the molecule link structure with
 * size size
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int ReceiveAttributes(attrroot, size, s)
  attrgroup **attrroot;
  register int size;
  int s;
{
  register int count;

  if (!size) {
    *attrroot = (attrgroup *)NULL;
    return (SUCCESS);
  }
  if (!(LL_packbuff = malloc((ALLOC_T)size))) {
    LogMess(LL_uid,
	    "Liblincks: No memory for buffer in ReceiveAttributes, %s",
	    sys_errlist[(errno > sys_nerr) ? 0 : errno]);
    perror("WARNING: No Memory For \"buffer\" In \"ReceiveAttributes\"\n");
    return (FAIL);
  }

  LL_packbufflen = size;
  LL_packbuffp = LL_packbuff;

  while (size > 0) {
    if ((count = read(s, LL_packbuffp, (IOSIZE_T)size)) < 0) {
      perror("ReceiveAttributes");
      break;
    }
    if (!count && size) {
      perror("ReceiveAttributes unexpected EOF on socket");
      break;
    }
    size -= count;
    LL_packbuffp += count;
  }

  LL_packbuffp = LL_packbuff;

  if (size == 0) {
    size = (LL_UnpackAttrGroup(attrroot) < 0);
    /*
     * What happens th attrroot on error ?
     */
  }

  free(LL_packbuff);

  if (size)
    return (FAIL);
  return (SUCCESS);
}


/*  */
/**********************************************************************
 * Function: static int ReceiveImage(image_desc **image, int size, int s)
 *
 * Reads a buffer of specified size from the socket
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int ReceiveImage(image, size, s)
  image_desc **image;
  register int size;
  int s;
{
  register int count;
  register char *buffp;

  if (size == 0) {
    *image = (image_desc *)NULL;
    return (SUCCESS);
  }
  if (!(*image = (image_desc *) malloc(sizeof(image_desc)))) {
    LogMess(LL_uid,
	    "Liblincks: no memory for image_desc in ReceiveImage, %s",
	    sys_errlist[(errno > sys_nerr) ? 0 : errno]);
    perror("WARNING: No Memory For \"image_desc\" In \"ReceiveImage\"\n");
    return (FAIL);
  }
  if (!((*image)->value = malloc((ALLOC_T)size))) {
    LogMess(LL_uid,
     "Liblincks: No memory for buffer in ReceiveImage, %s",
	    sys_errlist[(errno > sys_nerr) ? 0 : errno]);
    perror("WARNING: No Memory For \"buffer\" In \"ReceiveImage\"\n");
    free((FREEPTR *)*image);
    return (FAIL);
  }
  buffp = (*image)->value;
  (*image)->size = size;

  while (size > 0) {
    if ((count = read(s, buffp, (IOSIZE_T)size)) < 0) {
      perror("ReceiveImage");
      break;
    }
    if (!count && size) {
      perror("ReceiveImage unexpected EOF on socket");
      break;
    }
    size -= count;
    buffp += count;
  }

  if (size) {
    free((FREEPTR *)(*image)->value);
    free((FREEPTR *)*image);
    return (FAIL);
  }

  return (SUCCESS);
}


/*  */
/**********************************************************************
 * Function: errcode LL_ReceiveCount(int *val, int s)
 *
 * Reads an integer from the socket connection
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_ReceiveCount(val, s)
  int *val;
  int s;
{
  register int size, count;
  register char *buffp;
  t_u32bits netint;

  size = sizeof(netint);  
  buffp = (char *)&netint;

  while (size > 0) {
    if ((count = read(s, buffp, (IOSIZE_T)size)) < 0) {
      perror("LL_ReceiveCount");
      break;
    }
    if (!count && size) {
      perror("LL_ReceiveCount unexpected EOF on socket");
      break;
    }
    size -= count;
    buffp += count;
  }

  /* and convert value ... */
  *val = ntohl(netint);

  if (size)
    return (FAIL);
  return (SUCCESS);
}


/*  */
/**********************************************************************
 * Function: errcode LL_ReceivePenMessage(int *index, int *type, char **msg)
 *
 * Reads a PEN message from the socket connection
 *
 * Modifications:
 *      1993-08-12 Martin Sjlin - Rewrote interface when we
 *                 retrieve a set of pen messag
 */
errcode LL_ReceivePenMessage( index, type, msg)
  int  *index;
  int  *type;
  char **msg;
{
  image_desc *imagep = (image_desc *)NULL;
  int size, itindex;
  int status;

  if ((status = LL_ReceiveCount(&itindex, LL_socketno)  < 0)  ||
      (status = LL_ReceiveCount(type, LL_socketno)   < 0)  ||
      (status = LL_ReceiveCount(&size, LL_socketno) < 0)  ||
      (status = ReceiveImage(&imagep, size, LL_socketno) < 0)) {
    return (status);
  }

  (void)LL_ITtoMol(itindex, index);

  if ( imagep ) {
    *msg = imagep->value;
    free((FREEPTR *)imagep);
  } else
    *msg = (char *)NULL;

  return(SUCCESS);
}
