/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME: 	lowcom.c
 *
 * SCCSINFO:		@(#)lowcom.c	1.9 6/6/94
 *
 * ORIGINAL AUTHOR(S):  Peter ]berg, 1987-09-18
 *
 * MODIFICATIONS:
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 * Low level communication support functions used by LL_DBStore, LL_DBRetrieve
 * and LL_DBRelease
 *
 *********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************
 * errcode LL_SendIndex(int val)
 * errcode LL_SendMolType(int val)
 * errcode LL_SendCommand(byte command)
 * errcode LL_ReceiveStatus(byte *code)
 * errcode LL_SyncTest()
 */

/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */

#ifdef  HAVE_USLEEP			/* included in config.h ifndef */
#include <sys/time.h>
#endif /* n HAVE_USLEEP */
#include <sys/ioctl.h>
#include <sys/socket.h>

#include "sunlincks.h"
#include "dbcodes.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
char LL_errorstring[256];

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_rdwrmol.h"

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern int    LL_socketno;
extern int    LL_selectno;
extern fd_set LL_selectmask;

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
/* none */

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static int LL_ReceiveError P_(( void ));

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
/* none */

/*  */
/**********************************************************************
 * Function: errcode LL_SendIndex(int val)
 *
 * Packs and sends the index over the socket connection
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_SendIndex(val)
  int val;
{
  return (LL_SendCount(val, LL_socketno));
}

/*  */
/**********************************************************************
 * Function: errcode LL_SendMolType(int val)
 *
 * Packs and sends the molecule type over the socket connection.
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_SendMolType(val)
  int val;
{
  return (LL_SendCount(val, LL_socketno));
}

/*  */
/**********************************************************************
 * Function: errcode LL_SendCommand(byte command)
 *
 * Packs and sends the given command over the socket
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_SendCommand(command)
  byte command;
{
  static byte buff[2] = {SPECBYTE, 0};

  buff[1] = command;

  if (write(LL_socketno, (char *)buff, sizeof(buff)) != sizeof(buff)) {
    perror("LL_SendCommand");
    return (FAIL);
  }
#ifdef DEBUGCMD
  LogMess( 5, "ws: send cmd %d\n", (int)command);
#endif
  return (SUCCESS);
}

/*  */
/**********************************************************************
 * Function: errcode LL_ReceiveStatus(byte *code)
 *
 * Waits for and receives status from socket
 *
 * Modifications:
 *      1993-08-26 Martin Sjlin. Added a fix to re-read byte if the 
 *                 last read byte is SPECBYTE + RCOOB in case we are not
 *                 positioned in the stream to read the extra bytes.
 *                 NOTICE - this is only valid for those having 
 *                 OOBSELECTBUG enabled upon compilation.
 *      <list mods with name and date>
 */
errcode LL_ReceiveStatus(code)
  byte *code;
{
  int sc;
  fd_set smask;
  fd_set emask;
  struct timeval timev;

  /* reset the return value to avoid funny values */
  *code = 0;

  /* Perform select call using the portable fd_set */
#ifdef OOBSELECTBUG
 readagain:
#endif /* OOBSELECTBUG */

  for(;;) {
    smask = LL_selectmask;
    emask = LL_selectmask;
    timev.tv_sec = 60; timev.tv_usec = 0; errno = 0;
    while ((sc = select( LL_selectno, &smask, (fd_set *) NULL,
			&emask, &timev)) < 0 && errno == EINTR) {
      timev.tv_sec = 60; timev.tv_usec = 0; errno = 0;
      emask = LL_selectmask;
      smask = LL_selectmask;
    }

    /* Check that all went well */
#if defined(_AIX)			
    if (sc < 0 && errno == 0) {	/* strange ... 931103 msj */
      perror("FYI: LL_ReceiveStatus(0)");
      continue;
    } else
#endif /* AIX */
    if ( sc < 0) {
      perror("LL_ReceiveStatus(0)");
      return (FAIL);
    } else if ( sc == 0 ) {
#ifdef PENDEBUG
      if ( FD_ISSET( LL_socketno, &smask) ) 
	(void) fprintf( stderr, "Select mask 'smask'\n");
      if ( FD_ISSET( LL_socketno, &emask) ) 
	(void) fprintf( stderr, "Select mask 'emask'\n");
#endif
      (void)fprintf(stderr, 
		    "LL_ReceiveStatus: connection time out (errno: %d)\n",
		    errno);
      return(FAIL);
    } else if ( FD_ISSET( LL_socketno, &smask) ) {
      break; /* we have data and just to read it ... */
    } else if ( FD_ISSET( LL_socketno, &emask) ) {
      /* read again, do not care, oob already handled! */
      continue;
    } else
#ifdef OOBSELECTBUG
    /* here we have exited the select, but there was no real data,
     * we probably got here after reading a OOB, but under SunOS 
     * you need to do a read of "in-band" data to clear the exception.
     * I might add it later - but i'm not sure how it would affect
     * the buffering at receiving end. [msj]
     */
    continue;
#else
    break;
#endif
  } /* for */

  /* Read byte from stream */
  if ((sc = read(LL_socketno, (char *)code, (IOSIZE_T)1)) != 1) {
    if ((sc ==  0) ||  /* eof */
	(sc == -1  && (errno == ECONNRESET || errno == ETIMEDOUT))) 
      (void)fprintf(stderr, "LL_ReceiveStatus: connection broken.\n");
    else 
       perror("LL_ReceiveStatus(2)");
    return (FAIL);
  }

  /* Should be sync byte */
  if (*code != SPECBYTE)
    return (FAIL);

  /* Read byte from stream */
  if (read(LL_socketno, (char *)code, (IOSIZE_T)1) != 1) {
    perror("LL_ReceiveStatus(3)");
    return (FAIL);
  }
  /* Check what we have */
  switch (*code) {
#if defined(OOBSELECTBUG) 
  case RCOOB:
#ifdef PENDEBUG
    (void) fprintf( stderr,"LL_ReceiveStatus: Discarded OOB extra bytes.\n");
#endif /* PENDEBUG */
    goto readagain;
#endif /* OOBSELECTBUG */
  case RCOK:
  case RCBUFFER:
  case RCMOL:
  case RCMESSAGE:
    return (SUCCESS);
  case RCLOGO:
    (void)fprintf(stderr, "The database has sent the LOGOUT code.\n");
    return (-RCLOGO);
  case RCERROR:
    *code = LL_ReceiveError();
    return (FAIL);
  case RCUUP:
    return (FAIL);
  case RCBAD:
  case RCNYI:
  default:
    return (FAIL);
  }
}

/*  */
/**********************************************************************
 * Function: static int LL_ReceiveError()
 *
 * Receives an error code and string from the socket
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int LL_ReceiveError()
{
  byte status, len;
  char *message;
  short count;

  if (read(LL_socketno, (char *)&status, sizeof(status)) != sizeof(status)) {
    perror("LL_ReceiveError(1)");
    return (FAIL);
  }
  if (read(LL_socketno, (char *)&len, sizeof(len)) != sizeof(len)) {
    perror("LL_ReceiveError(2)");
    return (FAIL);
  }
  message = LL_errorstring;
  while (len > 0) {
    if ((count = read(LL_socketno, message, (IOSIZE_T)len)) < 0) {
      perror("LL_ReceiveError(3)");
      return (FAIL);
    }
    if (!count && len) {
      perror("LL_ReceiveError unexpected EOF on socket");
      return (FAIL);
    }
    /* Adjust byte count and position */
    len -= count;
    message += count;
  }

  /* Add NULL termination */
  *message = '\0';

  /* Return status */
  return (status);
}

/*  */
/**********************************************************************
 * Function: errcode LL_SyncTest()
 *
 * Performs a sync test and tries to restore sync if failed
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_SyncTest()
{
  int sc;
  fd_set smask;
  byte dump, status, count;
  struct timeval timev;

  for (count = 0; count < 100; ++count) {
    /* Send sync command */
    if (LL_SendCommand(ALIVE_CODE) < 0)
      return (FAIL);

    smask = LL_selectmask;
    if (LL_ReceiveStatus(&status) < 0) {
      timev.tv_sec = timev.tv_usec = 0;
      while ((sc = select( LL_selectno, &smask, (fd_set *) NULL,
			  (fd_set *) NULL, &timev)) > 0) {
	if (read(LL_socketno, (char *)&dump, sizeof(dump)) < 0)
	  return (FAIL);
	smask = LL_selectmask;
	timev.tv_sec = timev.tv_usec = 0;
      }

      if (sc < 0)
	return (FAIL);
    } else if (status == RCOK) {
      timev.tv_sec = timev.tv_usec = 0;
      while ((sc = select(LL_selectno, &smask, (fd_set *) NULL,
			  (fd_set *) NULL, &timev)) > 0) {
	if (read(LL_socketno, (char *)&dump, sizeof(dump)) < 0)
	  return (FAIL);
	smask = LL_selectmask;
	timev.tv_sec = timev.tv_usec = 0;
      }

      if (sc < 0)
	return (FAIL);
      else
	return (SUCCESS);
    }
  }

  /* Return error */
  return (FAIL);
}
