/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME: 	attrpack.c
 *
 * SCCSINFO:		@(#)attrpack.c	1.8 5/3/94
 *
 * ORIGINAL AUTHOR(S):  Peter ]berg, 1987-09-18
 *
 * MODIFICATIONS:
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 * This file contains the functions that perform attribute packing/unpacking
 *
 *********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************
 */

int LL_PackAttrGroup( /* attrgroup *group */ );
int EnsureBuffer( /* int len */ );
int LL_UnpackAttrGroup( /* attrgroup **group */ );

/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */

#include <netinet/in.h>

#include "sunlincks.h"
#include "xconfig.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_logws.h"

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern int LL_uid;		/* dbcomm.c */
extern char *sys_errlist[];
extern int sys_nerr;
extern int errno;

/* Buffer definitions */
extern char *LL_packbuff;
extern char *LL_packbuffp;
extern int LL_packbuffsize;
extern int LL_packbuffspace;
extern int LL_packbufflen;

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
/* Itemtype keys */
#define NAMESTRING 0
#define BUFFER 1
#define TERMINATE 2
#define COUNT 3
#define LOG_ERRSTRING   sys_errlist[(errno >= sys_nerr) ? 0 : errno]
#define LOG_MESS(b)     LogMess(LL_uid,"%s, %s", b, LOG_ERRSTRING); \
                        perror(b)

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static int PackAttribute P_(( attribute *attrvalue ));
static int PackBuff P_(( char *buff, int size ));
static void PackChars P_(( char *buff, int count ));
static int PackString P_(( char *string, int mode ));
static int UnpackBuff P_(( char **buff ));
static void UnpackChars P_(( char *buff, int count, int mode ));
static int UnpackString P_(( char **string, int mode ));

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
/* none */

/*  */
/**********************************************************************
 * Function: int LL_PackAttrGroup(attrgroup *group)
 *
 * Recursively packs all link groups
 *
 * Modifications:
 *      <list mods with name and date>
 */
int LL_PackAttrGroup(group)
  attrgroup *group;
{
  static short termcount = 0;

  if (group) {
    if (PackString(group->tag, NAMESTRING) < 0)
      return (FAIL);

    if (PackAttribute(group->value) < 0)
      return (FAIL);

    return (LL_PackAttrGroup(group->next));
  }
  /* Insert termination marker */
  if (PackString((char *)&termcount, TERMINATE) < 0)
    return (FAIL);

  return (SUCCESS);
}

/*  */
/**********************************************************************
 * Function: static int PackAttribute(attribute *attrvalue)
 *
 * Recursively packs all attributes

 * Modifications:
 *      <list mods with name and date>
 */
static int PackAttribute(attrvalue)
  attribute *attrvalue;
{
  static short termcount = 0;

  if (attrvalue) {
    if (PackString(attrvalue->tag, NAMESTRING) < 0)
      return (FAIL);

    if (PackBuff(attrvalue->value, attrvalue->size) < 0)
      return (FAIL);

    return (PackAttribute(attrvalue->next));
  }
  /* Insert termination marker */
  if (PackString((char *)&termcount, TERMINATE) < 0)
    return (FAIL);

  return (SUCCESS);
}

/*  */
/**********************************************************************
 * Function: static int PackBuff(char *buff, int size)
 *
 *  Packs a buffer with length as specified into LL_packbuff
 *
 * Modifications:
 *      1993-08-18 Martin Sjolin. Added htonl
 *      <list mods with name and date>
 */
static int PackBuff(buff, size)
  char *buff;
  int size;
{
  unsigned int netint = htonl(size);

  if (EnsureBuffer(size + sizeof(netint)) < 0)
    return (FAIL);

  PackChars((char *)&netint, sizeof(netint));
  PackChars(buff, size);

  /* Return status */
  return (SUCCESS);
}

/*  */
/**********************************************************************
 * Function: static int PackString(char *string, int mode)
 *
 * Packs a character string into the global LL_packbuff
 *
 * Modifications:
 *      1993-08-18 Martin Sjolin. Added htonl/htons
 *      <list mods with name and date>
 */
static int PackString(string, mode)
  char *string;
  int   mode;
{
  unsigned short len, netshort;
 
  /* Check which mode we're in */
  switch (mode) {
  case NAMESTRING:
    /* Get length of string */
    len = strlen(string);
    netshort = htons(len);

    /* Adjust len according to how many characters we ACTUALLY will write */
    /* Make sure there is space for string in buffer */
    if (EnsureBuffer((int)len + sizeof(netshort)) < 0)
      return (FAIL);

    /* Append length of string to follow */
    PackChars((char *)&netshort, sizeof(netshort));
    break;

  case TERMINATE:
    len = sizeof(netshort);

    /* Make sure there is space for string in buffer */
    if (EnsureBuffer((int)len) < 0)
      return (FAIL);
    break;

  default:
    return (FAIL);
  }

  /* Append string */
  PackChars(string, (int)len);

  /* Return status */
  return (SUCCESS);
}

/*  */
/**********************************************************************
 * Function: static void PackChars(register char *buff, register int count)
 *
 * Packs characters into the pack buffer
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void PackChars(buff, count)
  register char *buff;
  register int count;
{
  /* Adjust space left in pack buffer */
  LL_packbuffspace -= count;

  /* Copy chars */
  for (; count > 0; count--)
    *LL_packbuffp++ = *buff++;
}

/*  */
/**********************************************************************
 * Function: int EnsureBuffer(int len)
 *
 * Ensures there is enough space in pack buffer to
 * append string with length as specified
 *
 * Modifications:
 *      <list mods with name and date>
 */
int EnsureBuffer(len)
  int len;
{
  char *newbuff;
  int pos;

  /* Check if space left in buffer */
  while (LL_packbuffspace < len) {
    /* Save buffer position to write to */
    pos = LL_packbuffsize - LL_packbuffspace;

    /* Allocate more buffer space */
    newbuff = (char *)realloc((FREEPTR*)LL_packbuff, (ALLOC_T)(LL_packbuffsize *= 2));
    if (!newbuff) {
      LOG_MESS("EnsureBuffer: malloc failed");
      return (FAIL);
    }
    LL_packbuffspace = LL_packbuffsize - pos;

    /* Set buffer pointer to (possibly new) position */
    LL_packbuff = newbuff;
    LL_packbuffp = &LL_packbuff[pos];
  }
  return (SUCCESS);
}

/*  */
/**********************************************************************
 * Function: int LL_UnpackAttrGroup(attrgroup **group)
 *
 * Unpacks attribute groups from buffer
 *
 * Modifications:
 *      <list mods with name and date>
 */

int LL_UnpackAttrGroup(group)
  attrgroup **group;
{
  attribute **field;
  char *buff;
  int err;

  (*group) = (attrgroup *)NULL;
  if (LL_packbufflen <= 0)
    return (SUCCESS);

  for ( ;
       !(err = UnpackString(&buff, NAMESTRING)) && buff;
       group = &((*group)->next)) {

    if (!buff)
      return (SUCCESS);

    if (!(*group = (attrgroup *) malloc(sizeof(attrgroup)))) {
      LOG_MESS("LL_UnpackAttrGroup: malloc failed");
      return (FAIL);
    }
    (*group)->tag = buff;
    (*group)->value = (attribute *)NULL;
    (*group)->next = (attrgroup *)NULL;

    for (field = &((*group)->value);
	 !(err = UnpackString(&buff, NAMESTRING)) && buff;
	 field = &((*field)->next)) {

      if (!((*field) = (attribute *) malloc(sizeof(attribute)))) {
	LOG_MESS("LL_UnpackAttrGroup: malloc failed for field");
	return (FAIL);
      }
      (*field)->tag = buff;
      (*field)->value = (char *)NULL;
      (*field)->next = (attribute *)NULL;
      (*field)->size = UnpackBuff(&((*field)->value));
      if ((*field)->size < 0) {
	(*field)->size = 0;
	(*field)->value = (char *)NULL;
	return (FAIL);
      }
    }
    if (err)
      return (FAIL);
  }
  return (err);
}

/*  */
/**********************************************************************
 * Function: static int UnpackBuff(char **buff)
 *
 * Unpacks characters into supplied buffer
 *
 * Modifications:
 *      1993-08-18 Martin Sjolin. Added ntohl
 *      <list mods with name and date>
 */
static int UnpackBuff(buff)
  char **buff;
{
  int size;
  unsigned int netint;

  /* Get length of buffer */
  UnpackChars((char *)&netint, sizeof(netint), COUNT);
  size = ntohl(netint);

  if (size) {
    if (!(*buff = malloc((ALLOC_T)size))) {
      LOG_MESS("UnpackBuff: malloc failed");
      return (FAIL);
    }
    UnpackChars(*buff, size, BUFFER);
  } else {
    *buff = (char *)NULL;
    return (0);
  }

  /* Return size */
  return (size);
}

/*  */
/**********************************************************************
 * Function: static int UnpackString(char **string, int mode)
 *
 * Unpacks a string from the pack buffer. Mallocs memory depending on mode

 * Modifications:
 *      1993-08-18 Martin Sjolin. Added ntohs
 *      <list mods with name and date>
 */
static int UnpackString(string, mode)
  char **string;
  int mode;
{
  unsigned short len, netshort;

  /* Check if string is to be read */
  switch (mode) {
  case NAMESTRING:
    /* Get length of entry */
    UnpackChars((char *)&netshort, sizeof(netshort), COUNT);
    len = ntohs(netshort);

    if (len) {
      if (!(*string = malloc((ALLOC_T)len + 1))) {
	LOG_MESS("UnpackString: malloc failed");
	return (FAIL);
      }
    } else {
      *string = (char *)NULL;
      return (SUCCESS);
    }
    break;

  default:
    return (FAIL);
  }

  /* Copy into string */
  UnpackChars(*string, (int)len, mode);

  /* Return status */
  return (SUCCESS);
}

/*  */
/**********************************************************************
 * Function: static void UnpackChars(char *buff, int count, int mode)
 *
 * Unpacks characters from the pack buffer.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void UnpackChars(buff, count, mode)
  register char *buff;
  register int count;
  int mode;
{
  /* Adjust length of pack buffer */
  LL_packbufflen -= count;

  /* Copy */
  for (; count  > 0; count--)
    *buff++ = *LL_packbuffp++;

  /* Add NULL character if mode is NAMESTRING */
  if (mode == NAMESTRING)
    *buff = '\0';
}
