/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME:         xstuff.c
 *
 * SCCSINFO:	@(#)xstuff.c	1.17 6/7/94
 *
 * ORIGINAL AUTHOR(S):  Tim Teitenberg, 1990-03-15
 *
 * DESCRIPTION:
 * This module contains the X-window specific display routines used to
 * build widget trees.
 *
 * MODIFICATIONS:
 *      1993-11-02 Martin Sjlin. Moved setup code for pen from 
 *              pen_notify.c to xstuff.c since it is really uim 
 *              dependent. Added pen_select_func which does the
 *              jobs.
 *      1993-11-02 Martin Sjlin - whichref is moved to aimsubr.c and 
 *              changed external section accordingly.
 *      1993-11-05 Martin Sjlin - moved external interface for do_help
 *              here and kept code that extracts strings in xlincks_help.
 *              and added do_help_aim() for that.
 *            - Replaced shared string constant defined in aimtypes.h 
 *              (moved from xlincks_help.c)
 *            - Moved makeshelltitle to aimbuildref.c since to depend on X
 *      1993-12-27 Martin Sjlin. Moved font references into 
 *              application resource file as well fall back resource
 *              (to ease running on small screens)
 *      1994-05-14 Martin Sjlin.  Added:
 *              - application resources.
 *              - usage of colors or pixmaps for pens/diffs
 *              - mark_invisible_change
 *	<list mods here with name and date>
 */
/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */

#ifdef __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif
#include <X11/IntrinsicP.h>	/* ralro */
#include <X11/CoreP.h>		/* ralro */
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Command.h>	/* for popup_message */
#include <X11/Xaw/Box.h>	/* for popup_message */
#include <X11/Xaw/Label.h>	/* for popup_message */
#include <X11/Xaw/TextP.h>	/* for text->sink in changewidget */
#include <X11/Xmu/Converters.h> /* for added converters */
#include <X11/Xmu/Drawing.h>	/* for XmuConvertPixmapFromBitmap */

#include "ltext.h"
#include "layout.h"
#include "view.h"
#include "aimtypes.h"
#include "resources.h"

/********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_xstuff.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE DATA FOUND IN THIS MODULE:
 *********************************************************************/
Widget toplevel = NULL;		/* The Aim "root" widget */
/* Application Data Structure */
typedef struct {
  /* PEN, Bound and difference operations */
  Boolean         use_colors;	/* if we should use colors for diffs/pen */
  
  XColor          pixel_change;
  XColor          pixel_delete;
  XColor          pixel_insert;
  XColor          pixel_invisible;
  XColor          pixel_pen;

  Pixmap          pixmap_change; 
  int pixmap_change_width, pixmap_change_height;
  Pixmap          pixmap_delete;
  int pixmap_delete_width, pixmap_delete_height;
  Pixmap          pixmap_insert;
  int pixmap_insert_width, pixmap_insert_height;
  Pixmap          pixmap_invisible;
  int pixmap_invisible_width, pixmap_invisible_height;
  Pixmap          pixmap_pen;
  int pixmap_pen_width, pixmap_pen_height;
  Pixmap          pixmap_bound;
  int pixmap_bound_width, pixmap_bound_height;

  /* LaTeX export default settings */
  Boolean         LaTeX_export;
  char*           LaTeX_style;
  Boolean         LaTeX_deep;
  int             LaTeX_fontsize;
  Boolean         LaTeX_placeholders;
  Boolean         LaTeX_escape;

} AppData;
AppData app_data;

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_aimcommand.h"
#include "f_xlincks_help.h"
#include "f_layout.h"
#include "f_aimbuildref.h"
#include "f_aimstredit.h"
#include "f_gpdmenu.h"
#include "f_aimpropagate.h"
#include "f_aimcmdstream.h"
#include "f_aimsubr.h"
#include "f_aimstruct.h"
#include "f_aimeditops.h"
#include "f_pen_notify.h"
#include "f_ltext.h"
#include "f_ltextsink.h"

/* X stuff */
/* extern XFontStruct *XLoadQueryFont();*/

/* libshared.a */
extern char *strdup( /* incoming */ );

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern int LL_uid;		/* liblincks.a */
extern char *sys_errlist[];	/* liblincks.a */
extern int sys_nerr;		/* liblincks.a */
extern int errno;		/* liblincks.a */
extern Widget command_menu;	/* aimcommand.c */
extern int use_development_buttons;	/* -D run-time option */
extern int nobuttons;		/* -N run-time option */
extern int chain;		/* -C run-time option */
extern int suppress;		/* -S */
extern infonode *currentinfonode, *previousinfonode;
extern reference_structure *refstructs; /* aimstart.c */
extern int use_colors;		/* -c */
extern char *default_style;	/* aimexport.c */
extern int   default_usedeep;	/* aimexport.c */
extern int   default_fontsize;	/* aimexport.c */
extern int   default_useph;	/* aimexport.c */
extern int   default_doesc;	/* aimexport.c */
extern int   default_latex;	/* aimexport.c */

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
static XtArgVal CVint P_(( char *vp ));
static XtArgVal CVpixel P_(( char *vp ));
static XtArgVal CVstring P_(( char *vp ));

/* Maximum number of args for a widget.  Used for argstk */
#define MAXARGS		512

#define CHAINOFFSET	25

#define DOUBLE_CLICK_DELAY	500

typedef struct {
  char *name;
    XtArgVal(*fn) ();
} CVmap;

static CVmap CVtbl[] =
{
  {XtNstring, CVstring},
  {XtNfont, CVfont},
  {XtNbackground, CVpixel},
  {XtNforeground, CVpixel},
  {XtNborderColor, CVpixel},
  {XtNborderWidth, CVint},
  {XtNwidth, CVint},
  {XtNleftMargin, CVint},
  {XtNheight, CVint},
  {0, 0}
};

static char *TTBL[] =
{
  "INTERNAL",
  "ATTRGROUPTAG",
  "ATTRFIELDTAG",
  "ATTRVALUE",
  "LINKGROUPTAG",
  "LINKFIELDTAG",
  "LINKITEM",
  "IMAGE",
  "SAMEASPARENT"
};

static struct _window_table {
  char *label;
  Boolean newline;
  XtActionsRec cmd;
} normal_buttons[] = {
  {"Store", False, { WSTOREWINDOW, (XtActionProc)storecmd}},
  {"Close", False, { WCLOSEWINDOW, (XtActionProc)closecmd}},
  {"Refetch", False, { WREFETCH, (XtActionProc)refetchcmd}},
  {"Promote", True, { WPROMOTE, (XtActionProc)promotecmd}},
  {"Export ..",	False, { WEXPORTWINDOW, (XtActionProc)exportcmd}},
  {"Dup ..", False, { WDUPLICATE, (XtActionProc)dupwindowcmd}},
  {"GPD ..", False, { WSHOWGPD, (XtActionProc)showgpdcmd}},
  {"History ..", False, { WSHOWHISTORY, (XtActionProc)histcmd}},
  {(char *)NULL, False, {0, (XtActionProc)0}}
}, development_buttons[] = {
  {"Rfnc Stct", True, { WDUMPREFSTR, (XtActionProc)dumpcmd}},
  {"Bndng Tbl", False, { WSHOWBT, (XtActionProc)showbtcmd}},
  {(char *)NULL, False, {0, (XtActionProc)0}}
};

/* extra width to be thrown on a window to account for possible scrollbar */
#define SCROLL_BAR_WIDTH 40

/* strings used for popup_message */
#define INFO_MESSAGE "Information"
#define ERROR_MESSAGE "Error Message"
#define HELP_MESSAGE "Help Message"
#define PEN_MESSAGE "Parallel Editing Notification"

#if (XT_REVISION < 5) && !defined(X11_XAW_NO_DEFINE)
#  define XawChainLeft XtChainLeft
#  define XawChainRight XtChainRight
#  define XawChainTop XtChainBottom
#  define XawChainBottom XtChainTop
#endif

/* Default Application Data Resources */
#define XtNexportLaTeX       "exportLatex"
#define XtCExportLaTeX       "ExportLatex"
#define XtNlatexStyle        "latexStyle"
#define XtCLatexStyle        "LatexStyle"
#define XtNlatexDeep         "latexDeep"
#define XtCLatexDeep         "LatexDeep"
#define XtNlatexFontSize     "latexFontSize"
#define XtCLatexFontSize     "LatexFontSize"
#define XtNlatexPlaceHolders "latexPlaceHolders"
#define XtCLatexPlaceHolders "LatexPlaceHolders"
#define XtNlatexEscape       "latexEscape"
#define XtCLatexEscape       "LatexEscape"

#define XtNuseColors         "useColors"
#define XtCUseColors         "UseColors"

#define XtNpixelChange       "pixelChange"
#define XtNpixelDelete       "pixelDelete"
#define XtNpixelInsert       "pixelInsert"
#define XtNpixelInvisible    "pixelInvisible"
#define XtNpixelPen          "pixelPen"
#define XtCPixelChange       "PixelChange"
#define XtCPixelDelete       "PixelDelete"
#define XtCPixelInsert       "PixelInsert"
#define XtCPixelInvisible    "PixelInvisible"
#define XtCPixelPen          "PixelPen"

#define XtNpixmapChange      "pixmapChange"
#define XtNpixmapDelete      "pixmapDelete"
#define XtNpixmapInsert      "pixmapInsert"
#define XtNpixmapInvisible   "pixmapInvisible"
#define XtNpixmapPen         "pixmapPen"
#define XtNpixmapBound       "pixmapBound"

#define XtNpixmapChangeWidth     "pixmapChangeWidth"
#define XtCPixmapChangeWidth     "PixmapChangeWidth"
#define XtNpixmapChangeHeight    "pixmapChangeHeight"
#define XtCPixmapChangeHeight    "PixmapChangeHeight"
#define XtNpixmapDeleteWidth     "pixmapDeleteWidth"
#define XtCPixmapDeleteWidth     "PixmapDeleteWidth"
#define XtNpixmapDeleteHeight    "pixmapDeleteHeight"
#define XtCPixmapDeleteHeight    "PixmapDeleteHeight"
#define XtNpixmapInsertWidth     "pixmapInsertWidth"
#define XtCPixmapInsertWidth     "PixmapInsertWidth"
#define XtNpixmapInsertHeight    "pixmapInsertHeight"
#define XtCPixmapInsertHeight    "PixmapInsertHeight"
#define XtNpixmapInvisibleWidth  "pixmapInvisibleWidth"
#define XtCPixmapInvisibleWidth  "PixmapInvisibleWidth"
#define XtNpixmapInvisibleHeight "pixmapInvisibleHeight"
#define XtCPixmapInvisibleHeight "PixmapInvisibleHeight"
#define XtNpixmapPenWidth        "pixmapPenWidth"
#define XtCPixmapPenWidth        "PixmapPenWidth"
#define XtNpixmapPenHeight       "pixmapPenHeight"
#define XtCPixmapPenHeight       "PixmapPenHeight"
#define XtNpixmapBoundWidth      "pixmapBoundWidth"
#define XtCPixmapBoundWidth      "PixmapBoundWidth"
#define XtNpixmapBoundHeight     "pixmapBoundHeight"
#define XtCPixmapBoundHeight     "PixmapBoundHeight"

static XtResource app_resources [] = {
  { XtNuseColors, XtCUseColors,
    XtRBoolean, sizeof(Boolean),
    XtOffsetOf(AppData, use_colors),
    XtRImmediate, (XtPointer) False
  },
  { XtNpixelDelete, XtCPixelDelete,
    XtRPixel, sizeof(Pixel),
    XtOffsetOf(AppData, pixel_delete),
    XtRString, "brown"
  },
  { XtNpixelChange, XtCPixelChange,
    XtRPixel, sizeof(Pixel),
    XtOffsetOf(AppData, pixel_change),
    XtRString, "orange"
  },
  { XtNpixelInsert, XtCPixelInsert,
    XtRPixel, sizeof(Pixel),
    XtOffsetOf(AppData,pixel_insert),
    XtRString, "yellow"
  },
  { XtNpixelInvisible, XtCPixelInvisible,
    XtRPixel, sizeof(Pixel),
    XtOffsetOf(AppData,pixel_invisible),
    XtRString, "blue"
  },
  { XtNpixelPen, XtCPixelPen,
    XtRPixel, sizeof(Pixel),
    XtOffsetOf(AppData,pixel_pen),
    XtRString, "red"
  },
  { XtNpixmapChange, XtCPixmap,
    XtRBitmap, sizeof(Pixmap),
    XtOffsetOf(AppData,pixmap_change),
    XtRString, (XtPointer) "gray"
  },
  { XtNpixmapChangeWidth, XtCPixmapChangeWidth,
    XtRInt, sizeof(int),
    XtOffsetOf(AppData,pixmap_change_width),
    XtRImmediate, (XtPointer) 2		/* see X11/bitmaps/grey */
  },
  { XtNpixmapChangeHeight, XtCPixmapChangeHeight,
    XtRInt, sizeof(int),
    XtOffsetOf(AppData,pixmap_change_height),
    XtRImmediate, (XtPointer) 2		/* see X11/bitmaps/grey */
  },
  { XtNpixmapDelete, XtCPixmap,
    XtRBitmap, sizeof(Pixmap),
    XtOffsetOf(AppData,pixmap_delete),
    XtRString, (XtPointer) "gray"
  },
  { XtNpixmapDeleteWidth, XtCPixmapDeleteWidth,
    XtRInt, sizeof(int),
    XtOffsetOf(AppData,pixmap_delete_width),
    XtRImmediate, (XtPointer) 2		/* see X11/bitmaps/grey */
  },
  { XtNpixmapDeleteHeight, XtCPixmapDeleteHeight,
    XtRInt, sizeof(int),
    XtOffsetOf(AppData,pixmap_delete_height),
    XtRImmediate, (XtPointer) 2		/* see X11/bitmaps/grey */
  },
  { XtNpixmapInsert, XtCPixmap,
    XtRBitmap, sizeof(Pixmap),
    XtOffsetOf(AppData,pixmap_insert),
    XtRString, (XtPointer) "gray3"
  },
  { XtNpixmapInsertWidth, XtCPixmapInsertWidth,
    XtRInt, sizeof(int),
    XtOffsetOf(AppData,pixmap_insert_width),
    XtRImmediate, (XtPointer) 4		/* see X11/bitmaps/grey3 */
  },
  { XtNpixmapInsertHeight, XtCPixmapInsertHeight,
    XtRInt, sizeof(int),
    XtOffsetOf(AppData,pixmap_insert_height),
    XtRImmediate, (XtPointer) 4		/* see X11/bitmaps/grey3 */
  },
  { XtNpixmapInvisible, XtCPixmap,
    XtRBitmap, sizeof(Pixmap),
    XtOffsetOf(AppData,pixmap_invisible),
    XtRString, (XtPointer) "gray1"
  },
  { XtNpixmapInvisibleWidth, XtCPixmapInvisibleWidth,
    XtRInt, sizeof(int),
    XtOffsetOf(AppData,pixmap_invisible_width),
    XtRImmediate, (XtPointer) 2		/* see X11/bitmaps/grey1 */
  },
  { XtNpixmapInvisibleHeight, XtCPixmapInvisibleHeight,
    XtRInt, sizeof(int),
    XtOffsetOf(AppData,pixmap_invisible_height),
    XtRImmediate, (XtPointer) 2		/* see X11/bitmaps/grey1 */
  },
  { XtNpixmapPen, XtCPixmap,
    XtRBitmap, sizeof(Pixmap),
    XtOffsetOf(AppData,pixmap_pen),
    XtRString, (XtPointer) "gray"
  },
  { XtNpixmapPenWidth, XtCPixmapPenWidth,
    XtRInt, sizeof(int),
    XtOffsetOf(AppData,pixmap_pen_width),
    XtRImmediate, (XtPointer) 2		/* see X11/bitmaps/grey */
  },
  { XtNpixmapPenHeight, XtCPixmapPenHeight,
    XtRInt, sizeof(int),
    XtOffsetOf(AppData,pixmap_pen_height),
    XtRImmediate, (XtPointer) 2		/* see X11/bitmaps/grey */
  },
  { XtNpixmapBound, XtCPixmap,
    XtRBitmap, sizeof(Pixmap),
    XtOffsetOf(AppData,pixmap_bound),
    XtRString, (XtPointer) "gray3"	
  },
  { XtNpixmapBoundWidth, XtCPixmapBoundWidth,
    XtRInt, sizeof(int),
    XtOffsetOf(AppData,pixmap_bound_width),
    XtRImmediate, (XtPointer) 4		/* see X11/bitmaps/grey3 */
  },
  { XtNpixmapBoundHeight, XtCPixmapBoundHeight,
    XtRInt, sizeof(int),
    XtOffsetOf(AppData,pixmap_bound_height),
    XtRImmediate, (XtPointer) 4		/* see X11/bitmaps/grey3 */
  },
  { XtNlatexStyle, XtCLatexStyle,
    XtRString, sizeof(char *),
    XtOffsetOf(AppData, LaTeX_style),
    XtRString, (XtPointer) "article"
  },
  { XtNexportLaTeX, XtCExportLaTeX,
    XtRBoolean, sizeof(Boolean),
    XtOffsetOf(AppData, LaTeX_export),
    XtRImmediate, (XtPointer) True,
  },
  { XtNlatexDeep, XtCLatexDeep,
    XtRBoolean, sizeof(Boolean),
    XtOffsetOf(AppData, LaTeX_deep),
    XtRImmediate, (XtPointer) False,
  },
  { XtNlatexFontSize, XtCLatexFontSize,
    XtRInt, sizeof(int),
    XtOffsetOf(AppData,LaTeX_fontsize),
    XtRImmediate, (XtPointer) 11
  },
  { XtNlatexPlaceHolders, XtCLatexPlaceHolders,
    XtRBoolean, sizeof(Boolean),
    XtOffsetOf(AppData,LaTeX_placeholders),
    XtRImmediate, (XtPointer) True
  },
  { XtNlatexEscape, XtCLatexEscape,
    XtRBoolean, sizeof(Boolean),
    XtOffsetOf(AppData,LaTeX_escape),
    XtRImmediate, (XtPointer) True
  },
};

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
static XtAppContext app;
static Arg argstk[MAXARGS];	/* widget creation arguments */
static int stkp;		/* index into argstk */
static Widget shellw;		/* Current shell widget */
static Widget scroller;		/* Current view widget */
static Widget layoutw;		/* Current layout widget */
static reference_structure *ref;
static XrmValue inv, outv;
static Position global_x = 0;
static Position global_y = 0;

static struct {			/* The command menu status record. */
  struct {
    Position x;
    Position y;
  } home;
  struct {
    Position x;
    Position y;
  } wake;
  int map_state;
} command_menu_coord;

typedef struct _fontstuff {	/* for list of fonts */
  char *name;
  XFontStruct *font;
  struct _fontstuff *next;
} fontstuff;

typedef struct popstruct {	/* for list of fonts */
  reference_structure *ref;
  popd_list *pdl;
} pop_data;

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static void beep_at_user P_(( Widget widget, XtPointer client_data,
                               XtPointer call_data ));
static int FatalError P_(( Display *disp ));
static void free_widget_info P_(( Widget widget, XtPointer client_data,
                                   XtPointer call_data ));
static void GetShellCoordinates P_(( Widget w, Position *x, Position *y,
                                      int who_calls ));
static void GetWidgetXYAtCursor P_(( Widget widget, Position *x,
                                      Position *y ));
static void homeCommandMenu P_(( Widget ew, XEvent *ev, String args,
                                  Cardinal n ));
static void install_buttons P_(( Widget form, struct _window_table *tbl,
                                  reference_structure *ref ));
static void popdown_popup P_(( Widget w, XEvent *ev, String *parms,
                                Cardinal *num_parms ));
static void popfn2 P_(( Widget widget, caddr_t client_data, caddr_t
                         call_data ));
static void popstorecmd P_(( Widget widget, caddr_t client_data, caddr_t
                              call_data ));
static void printinode P_(( infonode *inp ));
static void printpath P_(( infonode *inode ));
static void pushXtArg P_(( char *nmp, XtArgVal vp ));
static void quit_popup P_(( Widget widget, XtPointer client_data,
                             XtPointer call_data ));
static void set_win_name P_(( Widget w, char *text ));
static void wakeCommandMenu P_(( Widget ew, XEvent *ev, String args,
                                  Cardinal n ));
static void WindowFocusIn P_(( Widget w, XEvent *event ));
static void WindowFocusOut P_(( Widget w, XEvent *event ));

/*  */
/****************************************************************
 * The following functions handle the widget argument stack.
 * - getarglevel() returns the present stack level;
 * - pusharg(name,value) pushes one text argument if convertable;
 * - popargs(i) restores the argument stack to level i.
 * - pushXtArg(name,value) pushes one converted argument;
 */

/****************************************************************
 * Function: int getarglevel()
 *
 * This function returns the current index into the stack of
 * formatting attributes (argstk)
 *
 * Called by propagation_queue_process().
 *
 * Modifications:
 *	<list mods here with name and date>
 */
int getarglevel()
{
  return stkp;
}

/*  */
/****************************************************************
 * Function: void pusharg(char *nmp, char* valp)
 *
 * Pushes one text argument if convertable.
 *
 * Called by propagation_queue_process().
 *
 * Modifications:
 *	<list mods here with name and date>
 */
void pusharg(nmp, valp)
  char *nmp, *valp;
{
  int i;
  XtArgVal value;

  inv.addr = valp;
  inv.size = strlen(valp) + 1;

  for (i = 0; CVtbl[i].name != 0; i++)
    if (strcmp(CVtbl[i].name, nmp) == 0) {
      value = (XtArgVal) ((*(CVtbl[i].fn)) (valp));
      if ((CVtbl[i].fn != CVfont) || (value != (XtArgVal) 0))
	pushXtArg(nmp, value);
      return;
    }
}

/*  */
/****************************************************************
 * Function: void popargs(int i)
 *
 * Sets the index into the formatting attribute stack to the
 * level of the input parameter.
 *
 * Called by propagation_queue_process().
 *
 * Modifications:
 *	<list mods here with name and date>
 */
void popargs(i)
  int i;
{
  stkp = i;
}

/*  */
/****************************************************************
 * Function: static void pushXtArg(char *nmp, XtArgVal vp)
 *
 * Pushes the X converted strings onto the formatting attribute
 * stack.
 *
 * Called internally by pusharg().
 *
 * Modifications:
 *	<list mods here with name and date>
 */
static void pushXtArg(nmp, vp)
  char *nmp;
  XtArgVal vp;
{
  argstk[stkp].name = nmp;
  argstk[stkp++].value = vp;
}

/*  */
/***********************************************************************
 * Function: void setupdefaultargs(reference_structure *rsp)
 *
 * Sets up the default widget arguments for X-windows.  Also
 * sets the current reference structure value.
 *
 * Modifications:
 *	<list mods here with name and date>
 */
void setupdefaultargs(rsp)
  reference_structure *rsp;
{
  stkp = 0;			/* Resets the widget argument stack */

  /* Default arguments for texts */
  pushXtArg(XtNeditType, XawtextEdit);	/* Editable */
  pushXtArg(XtNresize, XawtextResizeBoth);	/* Resize height+width */
  pushXtArg(XtNwrap, XawtextWrapWord);	/* Do word wrap */
  pushXtArg(XtNbottomMargin, 0);/* No top margins */
  pushXtArg(XtNtopMargin, 0);	/* No bottom margins */
  pushXtArg(XtNborderWidth, 0);	/* No borders */

  ref = rsp;
}

/*  */
/***********************************************************************
 * Function: void setplaceholder(infonode *np, int flag)
 *
 * Sets the value for a place-holder widget.
 *
 * Called by aimpropagate.c
 *
 * Modifications:
 *	<list mods here with name and date>
 */
void setplaceholder(np, flag)
  infonode *np;
  int flag;
{
  np->iflg |= flag;

  np->val.attsize = strlen(np->head->idname) + ((flag) ? 5 : 8);
  np->val.attvalue = (char *)malloc((ALLOC_T)np->val.attsize);
  if (np->val.attvalue != NULL) {
    if (flag)
      (void)sprintf(np->val.attvalue, "<<%s>>", np->head->idname);
    else
      (void)sprintf(np->val.attvalue, "empty: %s", np->head->idname);
  } else {
    LogMess(LL_uid,
	    "xlincks: Couldn't allocate memory for attsize in setplaceholder, %s",
	    sys_errlist[(errno > sys_nerr) ? 0 : errno]);
    np->val.attsize = 0;
  }
}

/*  */
/***********************************************************************
 * Function: static void set_win_name(Widget w, char *text)
 *
 * Modifications:
 *	<list mods here with name and date>
 */
static void set_win_name(w, text)
  Widget w;
  char *text;
{
  XTextProperty props;

  while (!XtIsSubclass(w, shellWidgetClass)) {
    if ((w = XtParent(w)) == (Widget) NULL)
      return;
  }

  XGetWMName(XtDisplay(toplevel), XtWindow(w), &props);
  if (strcmp((char *)props.value,text) != 0) {
    XFree((char *)props.value);
    props.value = (unsigned char *)text;
    props.nitems = (unsigned long)strlen(text);
    XSetWMName(XtDisplay(toplevel), XtWindow(w), &props);
  }
  else {
    XFree(props.value);
  }
}

/*  */
/***********************************************************************
 * Function: static void install_buttons(THREE PARAMETERS)
 * Parameters:
 *	Widget form
 *	struct _window_table *tbl
 *	reference structure *ref
 *
 * Add buttons from the table to the box.
 *
 * Modifications:
 *	<list mods here with name and date>
 */
static void install_buttons(form, tbl, ref)
  Widget form;
  struct _window_table *tbl;
  reference_structure *ref;
{
  static Widget left;
  Widget button = NULL;
  Widget top = NULL;

  if (tbl == normal_buttons)
    left = NULL;
  button = left;
  while (tbl->label != NULL) {
    button = XtVaCreateManagedWidget
      (tbl->cmd.string, commandWidgetClass, form,
       XtNlabel, tbl->label,  
       XtNhighlightThickness, 0,
       XtNfromVert, top,
       XtNfromHoriz, button,
       NULL);

    XtAddCallback(button, XtNcallback,
		  (XtCallbackProc) tbl->cmd.proc, (XtPointer) ref);
    if (tbl->newline) {
      top = button;
      button = left;
    }
    tbl++;
  }
  left = (top) ? top : button;
}


/*  */
/***********************************************************************
 * Function: void create_container(reference_structure *rsp)
 *
 * Creates the surrounding widget structure for a reference structure
 * display.
 *
 * Modifications:
 *	<list mods here with name and date>
 */
void create_container(rsp)
  reference_structure *rsp;
{
  Widget windowform = NULL;
  Widget buttons = NULL;
  ref = rsp;

  /* This is the window carrying widget; the shell. */
  /* the title of the window is created by makeshelltitle */
  shellw = XtVaCreatePopupShell((String)makeshelltitle(rsp->target),
				applicationShellWidgetClass, toplevel,
				XtNallowShellResize, True,
				NULL);

  /* Inside the shell, we put an ordinary form widget: */
  windowform = XtVaCreateManagedWidget
    ("windowform", formWidgetClass, shellw,
     XtNdefaultDistance, (XtArgVal) 0,
     NULL);

  /* Buttons appear unless -nobuttons option was given */
  if (!nobuttons) {
    buttons = XtVaCreateManagedWidget
      ("buttons", formWidgetClass, windowform,
       XtNborderWidth, (Dimension) 1,
       XtNleft, XawChainLeft,
       XtNright, XawChainLeft,
       XtNtop, XawChainTop,
       XtNbottom, XawChainTop,
       NULL);

    install_buttons(buttons, normal_buttons, rsp);

    /* Development buttons appear only if -D option was given */
    if (use_development_buttons)
      install_buttons(buttons, development_buttons, rsp);

   /* We add display accelerators for the buttons to preprocess 
    * translations as if into the command menu. */
    XtInstallAllAccelerators(buttons, command_menu);
  }

  /* The bottom pane is a viewport variation widget */
  scroller = XtVaCreateManagedWidget
    ("scroller", XlviewWidgetClass, windowform,
     XtNborderWidth, (Dimension) 0,
     XtNdefaultDistance, (XtArgVal) 0,
     XtNfromVert, buttons,
     XtNleft, XawChainLeft,
     XtNright, XawChainRight,
     XtNtop, XawChainTop,
     XtNbottom, XawChainBottom,
     XtNresizable, True,
     NULL);

  layoutw = XtVaCreateManagedWidget
    ("contents", XllayoutWidgetClass, scroller, NULL);

  rsp->contw = layoutw;		/* This is where to put actual contents. */

  /*
   * We add display accelerators for the whole layout (though contents
   * items are handled separately), to preprocess translations  as if
   * into the command menu.
   */
  XtInstallAllAccelerators(shellw, command_menu);
}


/*  */
/***********************************************************************
 * Function: void mapwidgets()
 *
 * This function gets called when the entire widget hierarchy is created
 * and should be mapped onto the screen.  It should, however, only be
 * called once.
 *
 * Called in blddisplay() in blddisplay.c.
 *
 * Modifications:
 *	<list mods here with name and date>
 */

void mapwidgets()
{
  static XtWidgetGeometry reply;
  static XtWidgetGeometry request =
  {CWHeight, 0, 0, 0, 0, 0, 0};
  Dimension max_height;
  char geometry[64];

  request.height = XlLayoutHeight(layoutw);
  XtVaGetValues(scroller, XtNmaxWindowHeight, &max_height, NULL);
  if (request.height > max_height) {
    request.height = max_height;
    (void)XtMakeGeometryRequest(scroller, &request, &reply);
  }

  XtAugmentTranslations
    (shellw, XtParseTranslationTable
     ("<FocusIn>: window-focus-in()"));

  if (chain) {
    (void)sprintf(geometry, "+%d+%d", global_x, global_y);
    XtVaSetValues(shellw, XtNgeometry, (String)geometry, NULL);
    global_x += CHAINOFFSET;
    global_y += CHAINOFFSET;
  }
  XtPopup(shellw, XtGrabNone);
}


/*  */
/*******************************************************************
 * Function: static void GetShellCoordinates(Widget w, Position *x, Position *y)
 *
 * This function determines the coordinates of the shell for a widget
 * such that if these coordinates later are assigned to the shell widget
 * by e.g. XtVaSetValues, then the shell widget moves to the very same
 * position as it currently is at.
 *
 * The problem dealt with here is that there might be a titlebar added
 * by the window manager. In that case, the shell window parent is not
 * the root window but the window whose position we are actually
 * interested in.
 *
 * Modifications:
 *	<list mods here with name and date>
 */
static void GetShellCoordinates(w, x, y, who_calls)
  Widget w;
  Position *x;
  Position *y;
  int who_calls;		/* wake (1) or home (0) */
{
  Window root;
  Window parent;
  Window *ctab;
  unsigned int ctabsz;
  XWindowAttributes attrs;

  (*x) = (*y) = 0;
  while (!XtIsSubclass(w, wmShellWidgetClass))
    if ((w = XtParent(w)) == (Widget) NULL)
      return;

  if (!XQueryTree(XtDisplay(w), XtWindow(w), &root, &parent, &ctab, &ctabsz))
    return;

  if (ctabsz > 0)
    XFree((char *)ctab);
  if (parent == root)
    parent = XtWindow(w);

  if (!XGetWindowAttributes(XtDisplay(w), parent, &attrs))
    return;

  (*x) = (Position) (attrs.x + attrs.border_width);
  (*y) = (Position) (attrs.y + attrs.border_width);
  if (who_calls)
    command_menu_coord.map_state = attrs.map_state;
}



/*  */
/* ARGSUSED */
/*******************************************************************
 * Function: static void homeCommandMenu(Widget w, XEvent *ev, String args, Cardinal n)
 *
 * Move the command menu back home.

 * Modifications:
 *	<list mods here with name and date>
 */
static void homeCommandMenu(ew, ev, args, n)
  Widget ew;			/* Translation widget */
  XEvent *ev;			/* Triggering event */
  String args;
  Cardinal n;
{
  Widget w = XtParent(command_menu);
  Position x;
  Position y;

  GetShellCoordinates(w, &x, &y, 0);

#ifdef DEBUG
  (void)printf("current = (%d,%d); home = (%d,%d); wake = (%d,%d)\n",
	       x, y,
	       command_menu_coord.home.x, command_menu_coord.home.y,
	       command_menu_coord.wake.x, command_menu_coord.wake.y);
#endif /* DEBUG */
  if ((x == command_menu_coord.wake.x) && (y == command_menu_coord.wake.y)) {
    x = command_menu_coord.home.x;
    y = command_menu_coord.home.y;
    if (command_menu_coord.map_state != IsUnmapped)
      XtVaSetValues(w, XtNx, x, XtNy, y, NULL);
  }
  command_menu_coord.home.x = x;
  command_menu_coord.home.y = y;
  command_menu_coord.wake.x = -1;
  command_menu_coord.wake.y = -1;

  if (command_menu_coord.map_state == IsUnmapped) {
    /* was iconified before */
    XIconifyWindow(XtDisplay(w), XtWindow(w),
		   XScreenNumberOfScreen(XtScreen(w)));
  }
#ifdef DEBUG
  (void)printf("home = (%d,%d); wake = (%d,%d)\n",
	       command_menu_coord.home.x, command_menu_coord.home.y,
	       command_menu_coord.wake.x, command_menu_coord.wake.y);
#endif /* DEBUG */
}

/*  */
/* ARGSUSED */
/*******************************************************************
 * Function: static void wakeCommandMenu(Widget w, XEvent *ev, String args, Cardinal n)
 *
 * Move and popup command menu at current cursor position
 *
 * Modifications:
 *	<list mods here with name and date>
 */
static void wakeCommandMenu(ew, ev, args, n)
  Widget ew;			/* Translation widget */
  XEvent *ev;			/* Triggering event */
  String args;
  Cardinal n;
{
  Widget w = XtParent(command_menu);
  Position x = 0;
  Position y = 0;

  if (ev->xany.window == XtWindow(command_menu)) {
    (void)printf("You are already in the command menu!\n");
    return;
  }
  GetShellCoordinates(w, &x, &y, 1);
#ifdef DEBUG
  (void)printf("current = (%d,%d); home = (%d,%d); wake = (%d,%d)\n",
	       x, y,
	       command_menu_coord.home.x, command_menu_coord.home.y,
	       command_menu_coord.wake.x, command_menu_coord.wake.y);
#endif /* DEBUG */
  if ((x != command_menu_coord.wake.x) || (y != command_menu_coord.wake.y)) {
    command_menu_coord.home.x = x;
    command_menu_coord.home.y = y;
  }
  GetWidgetXYAtCursor(w, &x, &y);
  XtVaSetValues(w, XtNx, x, XtNy, y, NULL);
  XMapRaised(XtDisplay(w), XtWindow(w));	/* no Xt* function?? */
  command_menu_coord.wake.x = x;
  command_menu_coord.wake.y = y;
#ifdef DEBUG
  (void)printf("home = (%d,%d); wake = (%d,%d)\n",
	       command_menu_coord.home.x, command_menu_coord.home.y,
	       command_menu_coord.wake.x, command_menu_coord.wake.y);
#endif /* DEBUG */
}


/*  */
/*******************************************************************
 * Function: static void GetWidgetXYAtCursor(Widget widget, Position *x, Position *y)
 *
 * Figure out coordinates to center widget at current cursor position.

 * Modifications:
 *	<list mods here with name and date>
 */
static void GetWidgetXYAtCursor(widget, x, y)
  Widget widget;
  Position *x;
  Position *y;
{
  Window root, child;
  Dimension width, height;
  int x_ret, y_ret;
  int foo, fie;
  unsigned int fum;

  XQueryPointer(XtDisplay(widget), XtWindow(widget), &root, &child,
		&x_ret, &y_ret, &foo, &fie, &fum);
  XtVaGetValues(widget, XtNwidth, &width, XtNheight, &height, NULL);

  x_ret -= ((int)width) / 2;
  y_ret -= ((int)height) / 2;
  if (x_ret + (int)width > WidthOfScreen(XtScreen(widget))) {
    x_ret = WidthOfScreen(XtScreen(widget)) - (int)width;
  }
  if (y_ret + (int)height > HeightOfScreen(XtScreen(widget))) {
    y_ret = HeightOfScreen(XtScreen(widget)) - (int)height;
  }
  if (x_ret < 0)
    x_ret = 0;
  if (y_ret < 0)
    y_ret = 0;
  (*x) = (Position) x_ret;
  (*y) = (Position) y_ret;

}

/*  */
/*******************************************************************
 * Function: void centerPopupOverCursor(Widget widget)
 *
 * This function takes a Widget argument, realizes it, and centers it over
 * the current position of the cursor.  It does not pop it up.
 *
 * Added 1991 Oct 9 by David Partain.
 *
 * called in aimcommand.c
 *
 * Modifications:
 *	<list mods here with name and date>
 */
void centerPopupOverCursor(widget)
  Widget widget;
{
  Position x = 0;
  Position y = 0;

  XtRealizeWidget(widget);
  GetWidgetXYAtCursor(widget, &x, &y);
  XtVaSetValues(widget, XtNx, x, XtNy, y, NULL);
}

/*  */
/***********************************************************************
 * Function: void popup_message(va_alist)
 *
 * This function pops up a message in a popup window -- one of two
 * types:  either an error message (POPUP_MESS_ERROR), an
 * informational message (POPUP_MESS_INFO), or a help
 * message(POPUP_MESS_HELP).  The widget is centered
 * over the cursor and is dismissed by the user.
 *
 * Modifications:
 *	<list mods here with name and date>
 */
#ifdef __STDC__
void popup_message(int message_type, char * fmt, ...)
#else
void popup_message(va_alist)
  va_dcl
#endif /* n __STDC__ */
{
  Widget box, popup, message, button, title;
  static char *mess_text[] = {
    ERROR_MESSAGE,
    INFO_MESSAGE,
    HELP_MESSAGE,
    PEN_MESSAGE
  };
  Position width, x, y;
  static char message_str[1024];
  va_list ap;
#ifndef __STDC__
  int message_type;
  char *fmt;
  va_start(ap);
  message_type = va_arg(ap, int);
  fmt = va_arg(ap, char *);
#else
  va_start(ap, fmt);
#endif /* __STDC__ */
  (void)vsprintf(message_str, fmt, ap);
  va_end(ap);

  /* check on things... */
  if (message_str == (char *)NULL) {
    popup_message(POPUP_MESS_ERROR, "popup_message:  null message input");
    return;
  }
  if ((message_type != POPUP_MESS_HELP) &&
      (message_type != POPUP_MESS_ERROR) &&
      (message_type != POPUP_MESS_PEN) &&
      (message_type != POPUP_MESS_INFO)) {
    popup_message(POPUP_MESS_ERROR, "popup_message:  invalid message type");
    return;
  }
  /* the shell widget which holds everything else in it */
  popup = XtVaCreatePopupShell
    ("popup", transientShellWidgetClass, toplevel, NULL);

  /* the box widget which will hold the two command widgets */
  box = XtVaCreateManagedWidget("box", boxWidgetClass, popup, NULL);

  /* A command widget (beeps if you click it) that tells what the
   * window is all about.
   */
  title = XtVaCreateManagedWidget
    ("title", commandWidgetClass, box,
     XtNlabel, mess_text[message_type],
     XtNhighlightThickness, 0,
     NULL);

  XtAddCallback(title, XtNcallback, beep_at_user, NULL);

  /* the dismiss button.  pops the widget down when clicked */
  message = XtVaCreateManagedWidget
    ("message", labelWidgetClass, box,
     XtNlabel, (CVstring(message_str)),
     NULL);

  /* see whether we need to make the label widget wider.  kinda weird */
  if ((int)strlen(message_str) > (int)strlen(mess_text[message_type])) {
    XtVaGetValues(message, XtNwidth, &width, NULL);
    XtVaSetValues(title, XtNwidth, width, 0);
  }

  /* add the button that will dismiss the window when the user's read it */
  button = XtVaCreateManagedWidget
    ("button", commandWidgetClass, box,
     XtNlabel, (XtArgVal) "Dismiss",
     NULL);
  XtAddCallback(button, XtNcallback, quit_popup, NULL);

  GetShellCoordinates(command_menu, &x, &y, 0);
  XtVaSetValues(popup, XtNx, x, XtNy, y, NULL);
  XtPopup(popup, XtGrabNone);
  XBell(XtDisplay(popup), 50);
}

/*  */
/***********************************************************************
 * Function: int issingular(infonode *inode)
 *
 * The following function makes a debugging trace of an infonode.
 *
 * Called in ltextaction.c
 *
 * Modifications:
 *	<list mods here with name and date>
 */
int issingular(inode)
  infonode *inode;
{
  switch (inode->index) {
  case INTERNAL:
  case ATTRVALUE:
  case IMAGE:
  case SAMEASPARENT:
    return TRUE;
  case ATTRGROUPTAG:
  case ATTRFIELDTAG:
  case LINKGROUPTAG:
  case LINKFIELDTAG:
    return FALSE;
  case LINKITEM:
    return (inode->head->tmpl->value->pos != -1);
  default:
    return FALSE;
  }
}

/*  */
/*ARGSUSED*/
/**************************************************************************
 * Function: void popfn(THREE PARAMETERS)
 * Parameters:
 *	Widget this_widget
 *	widget_info *cld
 *	int inplace	- whether to build it in the same window or not
 *
 * Called by libxl to "pop" one window off of the pdl stack.  This 
 * will return to the window from which the current window was created.
 *
 *	<list mods here with name and date>
 */
void popfn(this_widget, cld, inplace)
  Widget this_widget;
  widget_info *cld;
  int inplace;
{
  reference_structure *ref;
  popd_list *pdl = cld->ref->pdl;
  static pop_data pd;

  if (pdl == NULL) {
    short_trace("no previous compositions\n");
    XBell(XtDisplay(this_widget), 50);
    return;
  }

  (void)printf("%s %s.\n", "Creating previous composition",
	       ((inplace == 0) ? "in new window" : "in place"));
  
  /*
   * Whack the widgets associated with the reference structure if we're
   * reusing one.
   */
  if (inplace)
    free_widls(cld->ref);

  if (chain) {
    GetShellCoordinates(this_widget, &global_x, &global_y, 0);
    global_x -= CHAINOFFSET;
    global_y -= CHAINOFFSET;
  }

  if (inplace) {
    pd.ref = cld->ref;
    pd.pdl = pdl;
    checksave(cld->ref, popstorecmd, popfn2, (caddr_t)&pd);
  } else {
    ref = buildrefbt(pdl->tag, &(pdl->bt), pdl->bound, 
		     (reference_structure *)NULL);
    ref->pdl = pdl -> next;
    copy_pdls(&(ref->pdl));
  }
}

/*  */
/**********************************************************************
 * Function: static void popstorecmd(CALLBACK PROTOCOL)
 * Parameters:
 *	Widget widget;
 *	caddr_t client_data;
 *	caddr_t call_data;
 *
 * Callback to checksave()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void popstorecmd(widget, client_data, call_data)
  Widget widget;
  caddr_t client_data;
  caddr_t call_data;
{
  pop_data *pd = (pop_data *)client_data;

  storecmd(widget, (caddr_t) pd->ref, call_data);
}

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static void popfn2(CALLBACK PROTOCOL)
 * Parameters:
 *	Widget widget;
 *	caddr_t client_data;
 *	caddr_t call_data;
 * 
 * Modifications:
 *      <list mods with name and date>
 */
static void popfn2(widget, client_data, call_data)
  Widget widget;
  caddr_t client_data;
  caddr_t call_data;
{
  pop_data *pd = (pop_data *)client_data;
  popd_list *pdl = pd->pdl;
  reference_structure *ref = pd->ref;

  ref = buildrefbt(pdl->tag, &(pdl->bt), pdl->bound, ref);

  ref->pdl = pdl -> next;
  free((FREEPTR *)pdl->tag);
  free((FREEPTR *)pdl);
}

/*  */
/*ARGSUSED*/
/**************************************************************************
 * Function: void expandfn(THREE PARAMETERS)
 * Parameters:
 *	Widget this_widget
 *	widget_info *cld
 *	int inplace 	- left (0) or middle (1)
 *
 * Called by libxl, this will "expand" an infonode.  if the user used
 * the left mouse button, it'll expand in a new window. if the user used
 * the right mouse button, it'll expand in the same window.
 *
 *	<list mods here with name and date>
 */
void expandfn(this_widget, cld, inplace)
  Widget this_widget;
  widget_info *cld;
  int inplace;
{
  previousinfonode = currentinfonode;
  currentinfonode = cld->inode;
  SHOWPOINTS();
  printinode(currentinfonode);

  /*
   * Whack the widgets associated with the reference structure if we're
   * reusing one.
   */
  if (inplace) {
    free_widls(whichref(currentinfonode, (identry *)NULL, 0));
  }

  if (chain) {
    GetShellCoordinates(this_widget, &global_x, &global_y, 0);
    global_x += CHAINOFFSET;
    global_y += CHAINOFFSET;
  }

  (void)printf("%s %s.\n", "Expanding current infonode",
	       ((inplace == 0) ? "in new window" : "in place"));
  (void)expand(currentinfonode,inplace);
}


/*  */
/*ARGSUSED*/
/**************************************************************************
 * Function: void selectfn(Widget this_widget, widget_info *cld, caddr_t cad)
 *
 * Called in ltextaction.c.  Although this looks like a callback,
 * it isn't.
 *
 * Modifications:
 *	<list mods here with name and date>
 */
void selectfn(this_widget, cld, cad)
  Widget this_widget;
  widget_info *cld;
  caddr_t cad;
{
  previousinfonode = currentinfonode;
  currentinfonode = cld->inode;
  SHOWPOINTS();
  printinode(currentinfonode);
}

/*  */
/*******************************************************************
 * Function: void uim_destroy(Widget widget)
 *
 * Destroy a widget.  Currently used to destroy widgets associated
 * with a reference structure being destroyed.
 *
 * Modifications:
 *	<list mods here with name and date>
 */
void uim_destroy(widget)
  Widget widget;
{
  if (widget == NULL)
    return;

  XtDestroyWidget(widget);
}
/*  */
/*******************************************************************
 * Function: void uim_highlight(infonode *inp)
 *
 * mark widget if the widget has been "penned": the subject of
 * a parallel editing notification.
 *
 * Modifications:
 *	<list mods here with name and date>
 */
void uim_highlight(inp)
  infonode *inp;
{


  if ((inp == NULL) || (inp->widp == NULL))
    return;

  if (is_penned((inp->index == LINKITEM) ? inp->lbl.vs : inp->obj.vs)) {
    if (use_colors)
      XtVaSetValues(inp->widp, XtNbackground, app_data.pixel_pen.pixel, NULL);
    else
      XtVaSetValues(inp->widp, XtNbackgroundPixmap, app_data.pixmap_pen, NULL);
  } else {
    if (use_colors) {
      /* get parent background color and use it */
      Pixel background;
      XtVaGetValues(XtParent(inp->widp), XtNbackground, &background, NULL);
      XtVaSetValues(inp->widp, XtNbackground, background, NULL);
    } else
      XtVaSetValues(inp->widp, XtNbackgroundPixmap, XtUnspecifiedPixmap, NULL);
  }
}

/*  */
/*******************************************************************
 * Function: void makewidget(infonode *np, Widget previousw)
 *
 * This function is called to create a new contents widget.
 *
 * Called by propagation_queue_process().
 *
 * Modifications:
 *	<list mods here with name and date>
 */
void makewidget(np, previousw)
  infonode *np;
  Widget previousw;
{
  Widget w;
  widget_info *cld;

  if ((cld = (widget_info *) malloc(sizeof(widget_info))) == NULL)
    fatal_error("%s\n", "malloc failed in makewidget");

  cld->ref = ref;
  cld->inode = np;
  if (np->val.attsize == 0)
    setplaceholder(np, 0);

  pushXtArg(XtNstring, (XtArgVal)np->val.attvalue);
  pushXtArg(XtNreference, (XtArgVal)cld);
  pushXtArg(XtNwidth, 300);

  if (ref->flags.bound)
    pushXtArg(XtNeditType, XawtextRead);	/* Not Editable */

  w = XtCreateWidget
    (np->head->idname, XltextWidgetClass, ref->contw, argstk, stkp);
  XtAddCallback(w, XtNdestroyCallback, free_widget_info, (XtPointer)cld);
  (void)XlInsertChild(ref->contw, w, previousw);

  np->widp = w;

  { /*
     * We add display accelerators for each contents item,
     * to preprocess translations as if into the command menu.
     */
    XtInstallAllAccelerators(w, command_menu);
  }

  if (!ref->flags.bound)
    uim_highlight(np);
}

/*  */
/* ARGSUSED */
/*******************************************************************
 * Function: void refreshlayout(reference_structure *ref,int focus)
 *
 * Makes the layout to be resized etc after changes.
 *
 * Modified:
 */
void refreshlayout(ref,focus)
  reference_structure *ref;
  int focus; /* not used */
{
  XlRefreshLayout((XlLayoutWidget)ref->contw);
  set_win_name(ref->contw, makeshelltitle(ref->target));
}


/*  */
/*******************************************************************
 * Function: void deletewidget(Widget widp)
 *
 * This function deletes a (text) widget from its layout
 *
 * Called by propagation_queue_process().
 *
 * Modified:
 * 	David Partain 1992-02-07:  no longer has an infonode
 *	in-parameter.
 */
void deletewidget(widp)
  Widget widp;
{
  if (widp == (Widget) NULL)
    return;

  XtUnmanageChild(widp);	/* Seems destroy doesn't do this -Ralph */
  XtDestroyWidget(widp);
}


/*  */
/*******************************************************************
 * Function: void changewidget(infonode *np)
 *
 * This function changes the (text) contents of a widget
 *
 * Called by propagation_queue_process().
 *
 * Modified:
 *	David Partain 1992-03-03:  took out call to popargs(0) at
 *	the end, since the formatting attributes stack is handled
 *	by the propagation queue processing function.
 *	R & D, 1992-12-07: added height computation.
 *	<list mods with name and date>
 */
void changewidget(np)
  infonode *np;
{
  TextWidget w;
  Dimension height;

  if (np == (infonode *)NULL ||
      np->subnodes != (identry *)NULL ||
      np->widp == 0)
    return;

  w = (TextWidget) np->widp;

  XtVaSetValues((Widget) w, XtNstring, (XtArgVal) np->val.attvalue, NULL);

  height = XlSinkMaxHeight((TextSinkObject)w->text.sink, XlTextNumLines((Widget) w) + 1);
  XtVaSetValues((Widget) w, XtNheight, height, NULL);
}


/*  */
/* ARGSUSED */
/*******************************************************************
 * Function: static void WindowFocusIn( Widget w, XEvent *event)
 *
 * Modifications:
 *	<list mods here with name and date>
 */
static void WindowFocusIn(w, event)
  Widget w;
  XEvent *event;
{
  reference_structure *rsp;
  Widget lw;

  for (rsp = refstructs; rsp; rsp=rsp->next) {
    for (lw = rsp->contw; lw; lw = XtParent(lw))
      if (XtIsSubclass(lw, shellWidgetClass)) {
	if (lw != w)
	  lw = NULL;
	break;
      }
    if (lw)
      break;
  }

  if (rsp)
    LayoutFocusIn(rsp->contw);
}


/*  */
/* ARGSUSED */
/*******************************************************************
 * Function: static void WindowFocusOut( Widget w, XEvent *event)
 *
 * Modifications:
 *	<list mods here with name and date>
 */
static void WindowFocusOut(w, event)
  Widget w;
  XEvent *event;
{
  XSetInputFocus(XtDisplay(w),None,RevertToNone,CurrentTime);
}


/*  */
/*******************************************************************
 * Function: int initwindows(int *argc, char **argv)
 *
 * This function initializes X.
 *
 * Modifications:
 *	<list mods here with name and date>
 */
int initwindows(argc, argv)
  int *argc;
  char **argv;
{
  Pixmap pixmap;
  static XtActionsRec app_action[] = {
    {"do_help", (XtActionProc)do_help},
    {"translinkcmd", (XtActionProc)translinkcmd},
    {"popdown_popup", (XtActionProc)popdown_popup},
    {"popdown_yesnomenu", (XtActionProc)popdown_yesnomenu},
    {"wake-command-menu", (XtActionProc)wakeCommandMenu},
    {"home-command-menu", (XtActionProc)homeCommandMenu},
    {"window-focus-in",   (XtActionProc)WindowFocusIn},
    {"window-focus-out",  (XtActionProc)WindowFocusOut},
    {0, 0}
  };

  /* Add Xmu converts */
  XtAddConverter(XtRString, XtRJustify, XmuCvtStringToJustify, NULL, 0);
  XtAddConverter(XtRString, XtROrientation, XmuCvtStringToOrientation, NULL,0);
  XtAddConverter(XtRString, XtRBitmap, XmuCvtStringToBitmap, NULL, 0);
  XtAddConverter(XtRString, XtRCursor, XmuCvtStringToCursor, NULL, 0);
#if XtSpecificationRelease > 4		/* hack alert */
  XtAddConverter(XtRString, XtRGravity, XmuCvtStringToGravity, NULL, 0);
#endif
  XtSetTypeConverter(XtRString, XtRShapeStyle,
		     XmuCvtStringToShapeStyle, NULL, 0, XtCacheNone, 0);

  if ((toplevel = XtAppInitialize(&app, "Xlincks", NULL, 0,
#if XtSpecificationRelease > 4
				  argc, argv,
#else
				  (Cardinal *) argc, argv,
#endif
				  fallback_resources,
				  (ArgList) NULL, (Cardinal) 0)) == NULL)
    (void)fprintf(stderr, "Couldn't start Xwindow\n");
  XSetIOErrorHandler(FatalError);

  XtAppAddActions(app, app_action, XtNumber(app_action));
  
  /* get application default settings */
  XtVaGetApplicationResources(toplevel, 
			      &app_data,
			      app_resources, 
			      XtNumber(app_resources),
			      NULL);

  /* fix use_colors setting */
  use_colors |= app_data.use_colors;

  /* fixup of pixmaps  */
  if (XDefaultDepth(XtDisplay(toplevel), 0) > 1) { 
    pixmap = XmuCreatePixmapFromBitmap(XtDisplay(toplevel),
				       XDefaultRootWindow(XtDisplay(toplevel)),
				       app_data.pixmap_bound,
				       app_data.pixmap_bound_width,
				       app_data.pixmap_bound_height,
				       XDefaultDepth(XtDisplay(toplevel), 0),
				       BlackPixel(XtDisplay(toplevel), 0),
				       WhitePixel(XtDisplay(toplevel), 0));
    app_data.pixmap_bound = pixmap;
  }
  if (!use_colors && XDefaultDepth(XtDisplay(toplevel), 0) > 1) { 
    pixmap = XmuCreatePixmapFromBitmap(XtDisplay(toplevel),
				       XDefaultRootWindow(XtDisplay(toplevel)),
				       app_data.pixmap_change,
				       app_data.pixmap_change_width,
				       app_data.pixmap_change_height,
				       XDefaultDepth(XtDisplay(toplevel), 0),
				       BlackPixel(XtDisplay(toplevel), 0),
				       WhitePixel(XtDisplay(toplevel), 0));
    app_data.pixmap_change = pixmap;

    pixmap = XmuCreatePixmapFromBitmap(XtDisplay(toplevel),
				       XDefaultRootWindow(XtDisplay(toplevel)),
				       app_data.pixmap_delete,
				       app_data.pixmap_delete_width,
				       app_data.pixmap_delete_height,
				       XDefaultDepth(XtDisplay(toplevel), 0),
				       BlackPixel(XtDisplay(toplevel), 0),
				       WhitePixel(XtDisplay(toplevel), 0));
    app_data.pixmap_delete = pixmap;

    pixmap = XmuCreatePixmapFromBitmap(XtDisplay(toplevel),
				       XDefaultRootWindow(XtDisplay(toplevel)),
				       app_data.pixmap_insert,
				       app_data.pixmap_insert_width,
				       app_data.pixmap_insert_height,
				       XDefaultDepth(XtDisplay(toplevel), 0),
				       BlackPixel(XtDisplay(toplevel), 0),
				       WhitePixel(XtDisplay(toplevel), 0));
    app_data.pixmap_insert = pixmap;

    pixmap = XmuCreatePixmapFromBitmap(XtDisplay(toplevel),
				       XDefaultRootWindow(XtDisplay(toplevel)),
				       app_data.pixmap_invisible,
				       app_data.pixmap_invisible_width,
				       app_data.pixmap_invisible_height,
				       XDefaultDepth(XtDisplay(toplevel), 0),
				       BlackPixel(XtDisplay(toplevel), 0),
				       WhitePixel(XtDisplay(toplevel), 0));
    app_data.pixmap_invisible = pixmap;

    pixmap = XmuCreatePixmapFromBitmap(XtDisplay(toplevel),
				       XDefaultRootWindow(XtDisplay(toplevel)),
				       app_data.pixmap_pen,
				       app_data.pixmap_pen_width,
				       app_data.pixmap_pen_height,
				       XDefaultDepth(XtDisplay(toplevel), 0),
				       BlackPixel(XtDisplay(toplevel), 0),
				       WhitePixel(XtDisplay(toplevel), 0));
    app_data.pixmap_pen = pixmap;
  }

  /* check and setup export setting */
  default_latex     = app_data.LaTeX_export;
  default_style     = app_data.LaTeX_style;
  default_usedeep   = app_data.LaTeX_deep;
  default_fontsize  = app_data.LaTeX_fontsize;
  default_useph	    = app_data.LaTeX_placeholders;
  default_doesc     = app_data.LaTeX_escape;

  return toplevel == NULL;
}

/*  */
/*******************************************************************
 * Function: void initcommands()
 *
 * Build and map the command menu
 *
 * Modifications:
 *	<list mods here with name and date>
 */
void initcommands()
{
  buildcommandmenu(toplevel);	/* Assigns global command_menu */
  XtRealizeWidget(toplevel);
  create_GPD_menu(toplevel);

  GetShellCoordinates(XtParent(command_menu),
		      &(command_menu_coord.home.x),
		      &(command_menu_coord.home.y), 0);

  command_menu_coord.wake.x = -1;
  command_menu_coord.wake.y = -1;
}

/*  */
/*******************************************************************
 * Function: void mainloop()
 *
 * Start the main X loop.
 *
 * Modifications:
 *	<list mods here with name and date>
 */
void mainloop()
{
  XEvent event;

  for (;;) {
    XtAppNextEvent(app, &event);
    XtDispatchEvent(&event);
    propagatedisplay();
    while (do_commands()) {
      propagatedisplay();
    }
  }
}

/*  */
/****************************************************************
 * The functions below handle the conversion of data from the string
 * representation in the database to the desired representation as
 * widget resource. The conversion is done using a dispatch table with
 * indexing on the argument name.
 */

/*  */
/****************************************************************
 * Function: XtArgVal CVfont(char *vp)
 *
 * Converts from a char * to a X font description.
 *
 * Modifications:
 *	<list mods here with name and date>
 */
XtArgVal CVfont(vp)
  char *vp;
{
  XFontStruct *font_value;
  static fontstuff *head = NULL;
  fontstuff *tmp;

  for (tmp = head; tmp != (fontstuff *) NULL; tmp = tmp->next) {
    if ((strcmp(vp, tmp->name)) == 0)
      return ((XtArgVal) tmp->font);
  }

  font_value = XLoadQueryFont(XtDisplay(toplevel), vp);

  if (!font_value && !suppress) {
      short_error("%s %s %s.\n",
	"Requested font", vp, "does not exist.\nUsing server's default");
  }

  tmp = (fontstuff *) malloc(sizeof(fontstuff));
  tmp->name = strdup(vp);
  tmp->font = font_value;
  tmp->next = head;
  head = tmp;

  return ((XtArgVal) font_value);
}

/*  */
/*ARGSUSED*/
/****************************************************************
 * Function: static XtArgVal CVpixel(char *vp)
 *
 * Converts from a char * to a pixel description.
 *
 * Modifications:
 *	<list mods here with name and date>
 */
static XtArgVal CVpixel(vp)
  char *vp;
{
  XtConvert(toplevel, XtRString, &inv, XtRPixel, &outv);
  if (outv.addr != 0)
    return ((XtArgVal) * ((Pixel *) outv.addr));
  return ((XtArgVal) 0);
}

/*  */
/****************************************************************
 * Function: static XtArgVal CVint(char *vp)
 *
 * Converts from a char * to an X int.
 *
 * Modifications:
 *	<list mods here with name and date>
 */
static XtArgVal CVint(vp)
  char *vp;
{
  int i;

  (void)sscanf(vp, "%d", &i);
  return ((XtArgVal) i);
}

/*  */
/****************************************************************
 * Function: static XtArgVal CVstring(char *vp)
 *
 * Converts from a char * to an X string.
 *
 * Modifications:
 *	<list mods here with name and date>
 */
static XtArgVal CVstring(vp)
  char *vp;
{
  return (XtArgVal) vp;
}

/*  */
/* ARGSUSED */
/***********************************************************************
 * Function: static int FatalError(Display *disp)
 *
 * Modifications:
 *	<list mods here with name and date>
 */
static int FatalError(disp)
  Display *disp;
{
  fatal_error("%s\n", "Somebody whacked a window.  Exiting gracefully.");
  return 0;			/* to keep gcc happy */
}

/*  */
/***********************************************************************
 * Function: char *makeshelltitle(identry *idp)
 *
 * Returns the name that should be used as the title for a window.
 *
 * Modifications:
 *     <list mods here with name and date>
 */
char *makeshelltitle(idp)
  identry *idp;
{
  static char buf[100];

  if (!(idp && idp->value && idp->value->val.attsize)) {
    /* Use GPD name for title */
    return idp->idname;
  }
  /* Take title from contents */
  if (idp->value->val.attsize < 30) {
    (void)strncpy(buf, idp->value->val.attvalue, idp->value->val.attsize);
    *(buf + idp->value->val.attsize) = '\0';
  } else {
    (void)strncpy(buf, idp->value->val.attvalue, 25);
    (void)strcpy((buf + 25), " ...");
  }
  return buf;

}

/*  */
/***********************************************************************
 * Function: static void printinode(infonode *inp)
 *
 *
 * Modifications:
 *	<list mods here with name and date>
 */
static void printinode(inp)
  infonode *inp;
{
  printpath(inp);
  (void)printf("\n");

  (void)printf("          (%4x) ", inp->iflg);
  (void)printf("<%d:%d> ", inp->obj.vs, inp->obj.inst);
  (void)printf("%s ", TTBL[inp->index]);
  switch (inp->index) {
  case ATTRGROUPTAG:
  case LINKGROUPTAG:
    (void)printf("= %s\n", inp->val.attvalue);
    break;
  case ATTRFIELDTAG:
  case LINKFIELDTAG:
    (void)printf("[%s] = %s\n", inp->group, inp->val.attvalue);
    break;
  case ATTRVALUE:
    (void)printf("[%s:%s] = ...\n", inp->group, inp->field);
    break;
  case LINKITEM:
    (void)printf("[%s:%s:%d] = <%d:%d>\n", inp->group, inp->field, inp->pos,
		 inp->lbl.vs, inp->lbl.inst);
    break;
  case IMAGE:
  case SAMEASPARENT:
  case INTERNAL:
  default:
    (void)printf("\n");
    break;
  }
}


/*  */
/***********************************************************************
 * Function: static void printpath(infonode *inode)
 *
 * Modifications:
 *	<list mods here with name and date>
 */
static void printpath(inode)
  infonode *inode;
{
  infonode *ip;
  int ix;

  if (inode == (infonode *) NULL)
    return;
  if (inode->parent == (infonode *) NULL) {
    (void)printf(inode->head->idname);
    return;
  }
  printpath(inode->parent);
  (void)printf(".%s", inode->head->idname);

  for (ip = inode->head->value, ix = 0; ip != inode; ip = ip->next, ix++);
  (void)printf("(%d)", ix);
}

/*  */
/*ARGSUSED*/
/***********************************************************************
 * Function: static void quit_popup(widget, client_data, call_data)
 *
 * This function is used as a callback to popup_message to destroy
 * the popup widget.
 *
 * Modifications:
 *	<list mods here with name and date>
 */
static void quit_popup(widget, client_data, call_data)
  Widget widget;
  XtPointer client_data, call_data;
{
  XtPopdown(XtParent(XtParent(widget)));
}

/*  */
/*ARGSUSED*/
/***********************************************************************
 * Function: static void beep_at_user(widget, client_data, call_data)
 *
 * This function is used as a callback to popup_message().  Could
 * theorically be used elsewhere...
 *
 * Modifications:
 *	<list mods here with name and date>
 */
static void beep_at_user(widget, client_data, call_data)
  Widget widget;
  XtPointer client_data, call_data;
{
  XBell(XtDisplay(widget), 50);
}

/*  */
/*ARGSUSED*/
/***********************************************************************
 * Function: static void free_widget_info(widget, client_data, call_data)
 *
 * free a widget_info struct attached to the text widget.  added as a
 * destroycallback to each text widget
 *
 * Modifications:
 *	<list mods here with name and date>
 */
static void free_widget_info(widget, client_data, call_data)
  Widget widget;
  XtPointer client_data, call_data;
{
  widget_info *reference = (widget_info *)client_data;
  free((FREEPTR *)reference);
}


/*  */
/*ARGSUSED*/
/***********************************************************************
 * Function: static void popdown_popup(FOUR PARAMETERS)
 * Parameters:
 *	Widget w
 *	XEvent *ev
 *	String *parms
 *	Cardinal *num_parms
 *
 * Modifications:
 *	<list mods here with name and date>
 */
static void popdown_popup(w, ev, parms, num_parms)
 Widget w;
 XEvent *ev;
 String *parms;
 Cardinal *num_parms;
{
  WidgetList worf;

  while (!XtIsSubclass(w, boxWidgetClass))
    if(!(w = XtParent(w)))
      return;

  XtVaGetValues(w, XtNchildren, &worf, NULL);
  XtCallActionProc(worf[2], "set", ev, (String *)NULL, (Cardinal)0);
  XSync(XtDisplay(w), False);
#ifdef HAVE_USLEEP
    usleep(500000);
#else
  {
    struct timeval tval;
    tval.tv_sec  = 0;
    tval.tv_usec =  500000;
    select(0, NULL, NULL, NULL, &tval);
  }
#endif /* n HAVE_USLEEP */
  XtCallActionProc(worf[2], "notify", ev, (String *)NULL, (Cardinal)0);
  XtCallActionProc(worf[2], "unset", ev, (String *)NULL, (Cardinal)0);
}

/*  */
/***********************************************************************
 * Function: void markbound(reference_structure *r, int mode)
 *
 * Give some sort of indication that a reference structure is bound
 * and as such cannot be changed.
 *
 * Modifications:
 *	<list mods here with name and date>
 */
void markbound(r, mode)
  reference_structure *r;
  int mode;
{
  if (!r) {
    (void)fprintf(stderr, "markbound: NULL input parameter.\n");
    return;
  }

  if (r->contw == NULL) {
    (void)fprintf(stderr, "markbound: NULL container widget on refstruct.\n");
    return;
  }

#ifdef DEBUG
  (void)printf("markbound called: %s.\n",
  	       (mode == 0) 
		 ? "changing from bound to unbound"
		 : "changing from unbound to bound"
	      );
#endif 

  switch (mode) {
  case 0:
    XtVaSetValues(r->contw, XtNbackgroundPixmap, XtUnspecifiedPixmap, NULL);
    break;
  case 1:
    XtVaSetValues(r->contw, XtNbackgroundPixmap, app_data.pixmap_bound, NULL);
    break;

  default:
    break;
  }

  if (mode == 0)
    ClearNonEdit((Widget)(r->contw));
}

/*  */
/*******************************************************************
 * Function: int pen_select_func (fd,pen_call_back,dopoll,idp)
 *
 * Add the pen_call_back to be called whenever we receive data
 * on the fd file descriptor. Called from pen_notify.c:PEN_oob.
 *
 * Modifications:
 *      1994-04-15 Martin Sjlin. Added idp argument ...
 *	<list mods here with name and date>
 */

int pen_select_func(fd, pen_call_back, dopoll, idp)
  int fd;
  void (* pen_call_back)();
  void *dopoll;
  XtInputId *idp;
{
  XtInputId id;
  if (toplevel == NULL) 
    fprintf( stderr, "PEN_oob: Called before X started\n");
  else {
    id = XtAppAddInput(app,
		       fd, 
		       (XtPointer)XtInputReadMask,
		       pen_call_back,
		       (XtPointer)dopoll);
    /* if idp is non null set its value */
    if (idp != NULL)
      *idp = id;
  }

  /* Currently we always returns true, but later ... */
  return( TRUE );
}

/*  */
/*******************************************************************
 * Function: void do_help(FOUR PARAMETERS)
 * Parameters:
 *	Widget w		- used to get the resource name of widget
 *	XEvent *event		- unused
 *	String *params		- unused
 *	Cardinal *num_params	- unused
 * 
 * Calls popup_message with the right string when a command button
 * is clicked on with the right mouse button.
 * 
 * The string chosen is based upon the name of the resource of the widget
 * that was clicked on to produce this callback.
 *
 * Modifications:
 *	<list mods here with name and date>
 */
void do_help(w, event, params, num_params)
  Widget w;
  XEvent *event;
  String *params;
  Cardinal *num_params;
{
  Widget tmp_ptr = (Widget) w;

  if (!tmp_ptr->core.name) 
    short_error("%s", "incoming widget param to do_help() is messed up");
  else 
    do_help_aim(tmp_ptr->core.name);
}

/*  */
/*******************************************************************
 * Function: void mark_invisible_change(infonode *inp)
 *
 * mark widget if the widget has been changed ...
 *
 * Modifications:
 *	<list mods here with name and date>
 */
void mark_invisible_change(inp)
  infonode *inp;
{
  if ((inp == NULL) || (inp->widp == NULL))  return;
  if (use_colors)
    XtVaSetValues(inp->widp,XtNbackground,app_data.pixel_invisible.pixel,NULL);
  else
    XtVaSetValues(inp->widp,XtNbackgroundPixmap,app_data.pixmap_invisible,NULL);
}

/*  */
/*******************************************************************
 * Function: void mark_changed(infonode *inp)
 *
 * mark widget if the widget has been changed ...
 *
 * Modifications:
 *	<list mods here with name and date>
 */
void mark_changed(inp)
  infonode *inp;
{
  if ((inp == NULL) || (inp->widp == NULL))  return;
  if (use_colors) 
    XtVaSetValues(inp->widp,XtNbackground,app_data.pixel_change.pixel,NULL);
  else
    XtVaSetValues(inp->widp,XtNbackgroundPixmap,app_data.pixmap_change,NULL);
}

/*  */
/*******************************************************************
 * Function: void mark_inserted(infonode *inp)
 *
 * mark widget if the widget has been inserted ...
 *
 * Modifications:
 *	<list mods here with name and date>
 */
void mark_inserted(inp)
  infonode *inp;
{
  if ((inp == NULL) || (inp->widp == NULL))  return;
  if (use_colors)
    XtVaSetValues(inp->widp,XtNbackground,app_data.pixel_insert.pixel,NULL);
  else
    XtVaSetValues(inp->widp,XtNbackgroundPixmap,app_data.pixmap_insert,NULL);
}

/*  */
/*******************************************************************
 * Function: void mark_deleted(infonode *inp)
 *
 * mark widget if the widget has been deleted
 *
 * Modifications:
 *	<list mods here with name and date>
 */
void mark_deleted(inp)
  infonode *inp;
{
  if ((inp == NULL) || (inp->widp == NULL))  return;
  if (use_colors)
    XtVaSetValues(inp->widp,XtNbackground,app_data.pixel_delete.pixel,NULL);
  else
    XtVaSetValues(inp->widp,XtNbackgroundPixmap,app_data.pixmap_delete,NULL);
}

/*  */
/* lint doesn't deal well with varargs */
#ifndef lint
/*******************************************************************
 * Function: void trace(va_alist)
 *
 * The following function acts like printf, except that it
 * don't necessarily have to write on stdout.
 *
 * Print trace or debug info
 *
 * Usage: function(char *fmt, ...)
 *
 * Modifications:
 *	<list mods here with name and date>
 */
#ifdef __STDC__
void trace(char *fmt, ...)
#else
void trace(va_alist)
  va_dcl
#endif
{
  va_list ap;
#ifndef __STDC__
  char *fmt;
  va_start(ap);
  fmt = va_arg(ap, char *);
#else
  va_start(ap, fmt);
#endif
  vprintf(fmt, ap);
  va_end(ap);
  (void)fflush(stdout);
}

/*  */
/*******************************************************************
 * Function: void short_trace(va_alist)
 *
 * The following function acts like printf, except that it
 * don't necessarily have to write on stdout.
 *
 * Same as above, but only one line long. For xlincks, these are identical,
 * but ttyaim also writes one-liners on the bottom line
 *
 * Usage: function(char *fmt, ...)
 *
 * Modifications:
 *	<list mods here with name and date>
 */
#ifdef __STDC__
void short_trace(char *fmt, ...)
#else
void short_trace(va_alist)
  va_dcl
#endif
{
  va_list ap;
#ifndef __STDC__
  char *fmt;
  va_start(ap);
  fmt = va_arg(ap, char *);
#else
  va_start(ap,fmt);
#endif
  vprintf(fmt, ap);
  va_end(ap);
  (void)fflush(stdout);
}

/*  */
/*******************************************************************
 * Function: void error(va_alist)
 *
 * The following function acts like printf, except that it
 * don't necessarily have to write on stdout.
 *
 * Print an error message
 *
 * Usage: function(char *fmt, ...)
 *
 * Modifications:
 *	<list mods here with name and date>
 */
#ifdef __STDC__
void error(char *fmt, ...)
#else
void error(va_alist)
  va_dcl
#endif
{
  va_list ap;
#ifndef __STDC__
  char *fmt;
  va_start(ap);
  fmt = va_arg(ap, char *);
#else
  va_start(ap, fmt);
#endif /* __STDC__ */
  vfprintf(stderr, fmt, ap);
  va_end(ap);
  (void)fflush(stderr);
}

/*  */
/*******************************************************************
 * Function: void short_error(va_alist)
 *
 * The following function acts like printf, except that it
 * don't necessarily have to write on stdout.
 *
 * Print an error message that is no longer than one line. For xlincks, these
 * are identical, but ttyaim also writes one-liners on the bottom line.
 *
 * Usage: function(char *fmt, ...)
 *
 * Modifications:
 *	<list mods here with name and date>
 */
#ifdef __STDC__
void short_error(char *fmt, ...)
#else
void short_error(va_alist)
  va_dcl
#endif
{
  va_list ap;
  static char message[1000];
#ifndef __STDC__
  char *fmt;
  va_start(ap);
  fmt = va_arg(ap, char *);
#else
  va_start(ap, fmt);  
#endif /* __STDC__ */
  vfprintf(stderr, fmt, ap);
  vsprintf(message, fmt, ap);
  va_end(ap);
  (void)fflush(stderr);
  popup_message(POPUP_MESS_ERROR, message);
}

/*  */
/*******************************************************************
 * Function: void fatal_error(va_alist)
 *
 * This function prints out an error message and subsequently exits.
 * If a fatal error occurs, this is the function which should be used.
 *
 * Modifications:
 *	<list mods here with name and date>
 */
#ifdef __STDC__
void fatal_error(char *fmt, ...)
#else
void fatal_error(va_alist)
  va_dcl
#endif /* n __STDC__ */
{
  va_list ap;
#ifndef __STDC__
  char *fmt;
  va_start(ap);
  fmt = va_arg(ap, char *);
#else
  va_start(ap, fmt);
#endif /* __STDC__ */
  vfprintf(stderr, fmt, ap);
  va_end(ap);
  (void)fflush(stderr);
  exit(-1);
}

#else
/*  */
/* this is to keep lint quiet */
/* VARARGS */
void trace(what, fmt)
  char *what;
  char *fmt;
{
  trace(what, fmt);
}

/* VARARGS */
void short_trace(what, fmt)
  char *what;
  char *fmt;
{
  short_trace(what, fmt);
}

/* VARARGS */
void error(what, fmt)
  char *what;
  char *fmt;
{
  error(what, fmt);
}

/* VARARGS */
void short_error(what, fmt)
  char *what;
  char *fmt;
{
  short_error(what, fmt);
}

/* VARARGS */
void fatal_error(what, fmt)
  char *what;
  char *fmt;
{
  fatal_error(what, fmt);
}

#endif
