/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME: 	xdumpref.c
 *
 * SCCSINFO:		@(#)xdumpref.c	1.14 6/7/94
 *
 * ORIGINAL AUTHOR(S):  Ralph R\"onnquist,  Michael Jansson
 *
 * MODIFICATIONS:
 *      1993-12-27 Martin Sjlin. Moved font references into 
 *          application resource file as well fall back resource
 *          (to ease running on small screens)
 *      1994-05-23 Martin Sjlin. Generalized CreateFileShell (I need
 *          it myself for the newexport/xexportcmd command).
 *      <list mods with name and date>
 *
 * DESCRIPTION:
 *	The operational code which produces reference structure dump.
 *
 */

/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */

#include <X11/StringDefs.h>
#include <X11/Intrinsic.h>
#include <X11/Core.h>
#include <X11/Shell.h>
#include <X11/Composite.h>
#include <X11/Xaw/Paned.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Viewport.h>
#include <X11/Xaw/Dialog.h>

#include "aimtypes.h"
#include "layout.h"
#include "graphP.h"
#include "graphutils.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 ********************************************************************/
#include "f_xdumpref.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
extern Widget toplevel;

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_aimbuildref.h"
#include "f_aimcmdstream.h"
#include "f_aimstruct.h"
#include "f_aimsubr.h"
#include "f_showhistory.h"
#include "f_graph.h"
#include "f_graphutils.h"
#include "f_rendering.h"
#include "f_xstuff.h"

/* libshared */
extern char *strdup( /* char *incoming */ );

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern reference_structure *refstructs;    /* aimstart.c */

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/

typedef struct x_indentrec {	/* For the indent level stack */
  struct x_indentrec *next;
  int indent;
} indentrec;

struct GraphCallData {
  GNode *template;
  GNode *target;
};

struct RefWidgets {
  GraphWidget target;
  GraphWidget template;
};
struct DumpCallData {
  GNode *root;
  reference_structure *ref;
};
struct MyValue {
  char *string;
  enum NodeType {
    identry_e, infonode_e
  } type;
};

#define Max(a, b) ((a) > (b) ? (a) : (b))

#define SPACEY 20
#define SPACEX 85
#define MARGIN 2
#define MAXWIDTH 700
#define MAXHEIGHT 500
#define MAXSTRLEN 10

/*
 *
 * This is the strings used when dumping an EPS file.
 * --------------------------------------------------
 */
#define A4HEIGHT 800
#define EPSDRAWNODE "%d %d %d (%ld) Node\n"
#define EPSDRAWLINK "%d %d %d %d Link\n"
#define EPSDRAWINFONODE "%d %d (%s) DrawInfonode\n"
#define EPSDRAWIDENTRY "%d %d (%s) DrawIdentry\n"
#define EPSTRAILER "\nshowpage\n\n"
#define EPSHEADER "%%!PS-Adobe-2.0 EPSF-1.2\n\
%%%%DocumentFonts:\n\
%%%%Pages: 1\n\
%%%%BoundingBox: 0 %d %d 900\n\
%%%%EndComments\n\
\n\
%% Subroutine for drawing a link between two nodes.\n\
/Link {\n\
newpath\n\
moveto\n\
lineto\n\
stroke\n\
} def\n\
\n\
%% Subroutine for drawing a node.\n\
/Node {\n\
/str exch def\n\
/r exch def\n\
/y exch def\n\
/x exch def\n\
newpath\n\
1 setgray\n\
x y r 0 360 arc\n\
fill\n\
0 setgray\n\
x y r 0 360 arc\n\
stroke\n\
/fontdict /Courier findfont 6 scalefont dup setfont\n\
x r 2 div sub y r 5 div sub moveto\n\
str show\n\
fill\n\
} def\n\
%% Subroutine for drawing an infonode.\n\
/DrawInfonode {\n\
/str exch def\n\
/y exch def\n\
/x exch def\n\
x y 2 sub moveto\n\
x 60 add y 2 sub lineto\n\
x 60 add y 8 add lineto\n\
x y 8 add lineto\n\
x y 2 sub lineto\n\
closepath\n\
gsave\n\
1 setgray\n\
fill\n\
grestore\n\
stroke\n\
0 setgray\n\
/fontdict /Courier findfont 9 scalefont dup setfont\n\
x 2 add y moveto\n\
str show\n\
fill\n\
} def\n\
\n\
%% Subroutine for drawing an identry.\n\
/DrawIdentry {\n\
/str exch def\n\
/y exch def\n\
/x exch def\n\
0 setgray\n\
/fontdict /Courier findfont 9 scalefont dup setfont\n\
x 2 add y 2 add moveto\n\
str show\n\
fill\n\
} def\n\
\n\
%%%%EndProlog"

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
/* Makes window title from ref.struct - P is for private  */
static char *Pmakeshelltitle P_(( reference_structure *ref ));

/* Makes access string from infonode */
static void makeaccessline P_(( infonode *inp, char *buf ));

/* Makes contents string from infonode */
static void makevalueline P_(( infonode *inp, char *buf ));

/* Prints the value of identry */
static int stepinfonodes P_(( infonode *inp, int row, int indent,
                               indentrec *lst, GNode *parent ));
/* Prints the subnodes of infonode */
static int stepidentries P_(( identry *idp, int row, int indent,
                               indentrec *lst, GNode *parent ));

static void ack P_(( GraphWidget widget, GNode *node ));
static GNode *BuildHistoryGraph P_(( label *obj, Widget w ));
static void closehist P_(( Widget widget, XtPointer client_data,
                            XtPointer call_data ));
static void closeref P_(( Widget widget, XtPointer client_data,
                           XtPointer call_data ));
static void component_rollback P_(( GraphWidget widget, GNode *node ));
static void composition_rollback P_(( GraphWidget widget, GNode *node ));
static void DoDumpHistory P_(( Widget widget, XtPointer client_data,
                                XtPointer call_data ));
static void DoDumpRef P_(( Widget widget, XtPointer client_data,
                            XtPointer call_data ));
static void drawlink P_(( GraphWidget graph, GC gc, int x1, int y1, int
                           x2, int y2, int size ));
static void DrawRefNode P_(( GraphWidget graph, GC gc, int x, int y, int
                              size, struct MyValue *value ));
static void EPSComment P_(( void *context, char *str ));
static void EPSdumpLink P_(( GraphWidget graph, GC gc, int x1, int y1,
                              int x2, int y2, int size ));
static void EPSdumpNode P_(( GraphWidget graph, GC gc, int x, int y, int
                              size, void *value ));
static void EPSDumpRefNode P_(( GraphWidget graph, GC gc, int x, int y,
                                 int size, struct MyValue *value ));
static void EPSEpilog P_(( void *context ));
static void EPSProlog P_(( void *context, int width, int height ));
static void freeMyValue P_(( GNode *node ));
static void mark_graph P_(( GNode *root ));
static void whack_refs P_(( Widget widget, XtPointer client_data,
                             XtPointer call_data ));
static void WriteHistory P_(( Widget widget, XtPointer client_data,
                               XtPointer call_data ));
static void WriteRef P_(( Widget widget, XtPointer client_data,
                           XtPointer call_data ));

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
static char buf[1000];		/* Scratch pad */

/*  */
/***********************************************************************
 * Function: static char *Pmakeshelltitle(reference_structure *ref)
 *
 * Constructs a descriptive name (into buf) for the given reference
 * structure. Returns buf.
 * NOTE: this is NOT a copy of the function in xstuff.c with same name.
 *
 * The name is formed from both the given reference structure and the
 * GPD used. It will look like the following:
 *	"A target identific] using [View of text"
 * The target identification is formed similar to how window titles are
 * formed (see xstuff.c), and the GPD identification is obtained from
 * the image of the GPD (root) object.
 * Modifications:
 *	<list mods here with name and date>
 */
static char *Pmakeshelltitle(ref)
  reference_structure *ref;
{
  identry *idp = ref->target;
  attrval v;

  /* Make a target identification string */
  if (!(idp && idp->value && idp->value->val.attsize)) {
    (void)strcat(buf, "(no name)");
  } else {
    if (idp->value->val.attsize > 25) {
      (void)strncpy(buf, idp->value->val.attvalue, 25);
      (void)strcpy(&buf[25], " ...");
    } else {
      (void)strcpy(buf, idp->value->val.attvalue);
    }
  }

  /* Make a GPD identification string */
  if (GI_GETIMAGE(&(ref->gpd), &v) != SUCCESS) {
    v.attvalue = strdup("(view not described)");
    v.attsize = strlen(v.attvalue) + 1;
  }
  if ((int)strlen(v.attvalue) > 25)
    v.attvalue[25] = '\0';

  /* Combine target and GPD identifications into shell title string */
  (void)strcat(buf, "] using [");
  (void)strcat(buf, v.attvalue);
  if (v.attvalue)
    free(v.attvalue);
  return buf;
}

/*  */
/**********************************************************************
 * Function: static void makeaccessline(infonode *inp,char *buf);
 *
 * Puts an access string into buf for a template(!!) infonode.
 * Returns nothing.
 *
 * The access string looks in principle the same as it does in the
 * GPD.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void makeaccessline(inp, buf)
  infonode *inp;
  char *buf;
{
  switch (inp->index) {
  case INTERNAL:
    buf[0] = '"';
    (void)strcpy(buf + 1, inp->val.attvalue);
    (void)strcat(buf, "\"");
    break;
  case ATTRVALUE:
  case ATTRFIELDTAG:
  case ATTRGROUPTAG:
    (void)strcpy(buf, "ATTR ");
    if (inp->group) {
      (void)strcat(buf, inp->group);
      if (inp->field) {
	(void)strcat(buf, " ");
	(void)strcat(buf, inp->field);
      }
    }
    break;
  case LINKITEM:
  case LINKFIELDTAG:
  case LINKGROUPTAG:
    (void)strcpy(buf, "LINK ");
    if (inp->group) {
      (void)strcat(buf, inp->group);
      if (inp->field) {
	(void)strcat(buf, " ");
	(void)strcat(buf, inp->field);
	if (inp->pos != -1) {
	  (void)strcat(buf, " 1");
	}
      }
    }
    break;
  case IMAGE:
    (void)strcpy(buf, "IMAGE");
    break;
  case SAMEASPARENT:
    (void)strcpy(buf, "$");
    break;
  default:
    (void)strcpy(buf, "(null)");
    break;
  }
}

/*  */
/**********************************************************************
 * Function: static void makevalueline(infonode *inp,char *buf)
 *
 * Puts an infonode value string into buf.
 * Returns nothing.
 *
 * The infonode value string consists of up to MAXSTRLEN characters from the
 * val.attvalue string, or the constant "(null)" as appropriate.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void makevalueline(inp, buf)
  infonode *inp;
  char *buf;
{
  int i;

  if (inp->val.attvalue == (char *)NULL) {
    (void)strcpy(buf, "(null)");
    return;
  }
  for (i = 0; (buf[i] = (inp->val.attvalue)[i]) != '\0'; i++)
    if (i == MAXSTRLEN)
      break;
  buf[i] = '\0';
  if (i == 0)
    (void)strcpy(buf, "(null)");
}

/*  */
/**********************************************************************
 * Function: static int stepinfonodes(FIVE PARAMETERS)
 * Parameters:
 *	infonode *inp
 *	int row
 *	int indent
 *	indentrec *lst
 *	GNode *parent
 *
 * The sub-tree is printed to the right of 'indent' column, changing row
 * as needed, and using the indentation stack 'lst' for indentation prefix.
 *
 * Infonodes are printed using the infonode value string produced by
 * makeaccessline (for template) or makevalueline (for target), within
 * square brackets, and preceded by the constant string "---".
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int stepinfonodes(inp, row, indent, lst, parent)
  infonode *inp;
  int row;
  int indent;
  indentrec *lst;
  GNode *parent;
{
  indentrec *p;
  struct MyValue *val = NULL;
  int x, y, width;

  if (inp == (infonode *)NULL)
    return (0);

  if (inp->head->tmpl == (identry *)NULL) {
    /* Dumping the template; display is the access sentence */
    makeaccessline(inp, buf);
  } else {
    /* Dumping the target; display portion of value */
    makevalueline(inp, buf);
  }

  if (strlen(buf)) {
    val = (struct MyValue *)malloc(sizeof(struct MyValue));
    if (val == NULL) {
      fatal_error("%s\n", "malloc failed in stepinfonodes");
    }
    val->string = (char *)malloc((ALLOC_T)strlen(buf) + 1);
    if (val->string == NULL) {
      fatal_error("%s\n", "malloc failed in stepinfonodes");
    }
    width = ((int)strlen(buf) + 
	     (int)strlen(((struct MyValue *)(parent->node.value))->string))/2;
    x = parent->node.x + (width * 10);
    y = parent->node.y;
    (void)strcpy(val->string, buf);
    val->type = infonode_e;
    parent = CreateNode(parent, (void *)val);
    parent->node.x = x;
    parent->node.y = y;
  }
  if ((p = (indentrec *) malloc(sizeof(indentrec))) == NULL) {
    fatal_error("%s\n", "malloc failed in stepinfonodes");
  }
  p->indent = indent + 4;
  p->next = lst;
  if (inp->next != (infonode *)NULL) {
    row = stepinfonodes(inp->next, row, indent + strlen(buf) + 4, p, parent);
  }
  row = stepidentries(inp->subnodes, row, indent + 4, p, parent);
  if (inp->next == NULL)
    row += SPACEY / 2;
  free((FREEPTR *)p);
  return row;
}

/*  */
/**********************************************************************
 * Function: static int stepidentries(FIVE PARAMETERS)
 * Parameters:
 *	identry *idp
 * 	int row
 *	int indent
 *	indentrec *lst
 *	GNode *parent
 *
 * Identries are printed using the indentation of makeindent for given
 * 'indent' and 'lst', followed by idp->idname.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int stepidentries(idp, row, indent, lst, parent)
  identry *idp;
  int row;
  int indent;
  indentrec *lst;
  GNode *parent;
{
  struct MyValue *val = NULL;
  int x;

  for (; idp; idp = idp->next) {
    if (idp->idname && strlen(idp->idname) &&
	(val = (struct MyValue *)malloc(sizeof(struct MyValue))) &&
	  (val->string = (char *)malloc((ALLOC_T)strlen(idp->idname) + 1))) {
      (void)strcpy(val->string, idp->idname);
      val->type = identry_e;
      x = parent->node.x;
      row += SPACEY;
      parent = CreateNode(parent, (void *)val);
      parent->node.x = x;
      parent->node.y = row;
    }
    if (idp->value->head == idp) {
      if ((lst == (indentrec *) NULL) || (idp->next != (identry *)NULL)) {
	row = stepinfonodes(idp->value, row, indent + strlen(idp->idname),
			    lst, parent);
      } else {
	row = stepinfonodes(idp->value, row, indent + strlen(idp->idname),
			    lst->next, parent);
      }
    }
  }
  return row;
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void DrawRefNode(SIX PARAMETERS)
 * Parameters:
 *	GraphWidget graph
 *	GC gc
 *	int x
 *	int y
 *	int size
 *	struct MyValue *value
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void DrawRefNode(graph, gc, x, y, size, value)
  GraphWidget graph;
  GC gc;
  int x;
  int y;
  int size;
  struct MyValue *value;
{
  int width  = 0;			/* assume zero */
  int height = 0;			/* since I (msj) do not this code */
  struct {
    int x;
    int y;
    unsigned int width;
    unsigned int height;
 } r;

  /* Calculate the dimensions. */
  if (graph->graph.font) {
    width =
      XTextWidth(graph->graph.font, value->string, strlen(value->string));
    height = graph->graph.font->ascent;
    x -= width / 2;
    y += height / 2;
  }
  r.x = x - MARGIN;
  r.y = y - MARGIN - height;
  r.width = (unsigned int)width + 2 * MARGIN;
  r.height = (unsigned int)height + 2 * MARGIN;

  /* Clear the background of the node. */
  XSetForeground(XtDisplay(graph), gc, WhitePixelOfScreen(XtScreen(graph)));

  XFillRectangle(XtDisplay(graph), XtWindow(graph), gc,
		 r.x, r.y, r.width, r.height);

  XSetForeground(XtDisplay(graph), gc, BlackPixelOfScreen(XtScreen(graph)));

  /* Draw a border. */
  if (value->type == infonode_e)
    XDrawRectangle(XtDisplay(graph), XtWindow(graph), gc,
		   r.x, r.y, r.width, r.height);

  /* Print the value of the node in the middle of the rectangle. */
  XDrawString(XtDisplay(graph), XtWindow(graph), gc,
	      x, y, value->string, strlen(value->string));
}

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static void whack_refs(CALLBACK PROTOCOL)
 * 
 * free a RefWidgets structure 
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void whack_refs(widget, client_data, call_data)
  Widget widget;
  XtPointer client_data, call_data;
{
  struct RefWidgets *refs = (struct RefWidgets *)client_data;
  free((FREEPTR *)refs);
}

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: void whack_widl(CALLBACK PROTOCOL)
 * 
 * when a graph window is closed, its corresponding widl must be removed
 * from the reference structure widl list.
 *
 * Modifications:
 *      <list mods with name and date>
 */
void whack_widl(widget, client_data, call_data)
  Widget widget;
  XtPointer client_data, call_data;
{
  reference_structure *widref = (reference_structure *)client_data;
  reference_structure *ref = NULL;
  
  for (ref = refstructs; ref; ref = ref->next)
    if (ref == widref)
      break;

  if (ref)
    remove_widl(widget, widref);
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void closeref(CALLBACK PROTOCOL)
 *
 * Frees the two graphs and closes the window associated with a
 * references structure dump.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void closeref(widget, client_data, call_data)
  Widget widget;
  XtPointer client_data, call_data;
{
  struct GraphCallData *graph = (struct GraphCallData *)client_data;

  if (!graph) {
    (void)fprintf(stderr, "closeref: (warning) null graph input param.\n");
    return;
  }

  FreeTree(graph->template, freeMyValue);
  FreeTree(graph->target, freeMyValue);
  free((FREEPTR *)(graph));
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: void destroyshell(THREE PARAMETERS)
 * Parameters:
 *	Widget  widget
 *	XtPointer client_data
 *	XtPointer call_data
 *
 * Modifications:
 *      <list mods with name and date>
 */
void destroyshell(widget, client_data, call_data)
  Widget widget;
  XtPointer client_data, call_data;
{
  Widget shell = (Widget)client_data;

  while (shell && (!XtIsShell(shell)))
    shell = XtParent(shell);
  XtDestroyWidget(shell);
}


/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void closehist(THREE PARAMETERS)
 * Parameters:
 *	Widget  widget
 *	XtPointer client_data
 *	XtPointer call_data
 *
 * Free the graph of the (GraphWidget)client_data.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void closehist(widget, client_data, call_data)
  Widget widget;
  XtPointer client_data, call_data;
{
  GraphWidget graph = (GraphWidget)client_data;

  if (!graph) {
    (void)fprintf(stderr, "closeref: (warning) null graph input param.\n");
    return;
  }
  if (!graph->graph.root)
    return;
  mark_graph((GNode *)graph->graph.root);
  FreeTree((GNode *)(graph->graph.root),NULL);
}

/*  */
/**********************************************************************
 * Function: static void mark_graph(GNode *root)
 * 
 * Go through the graph and make it into a tree for freeing purposes.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void mark_graph(root)
  GNode *root;
{
  int i;

  root->num = -936;
  for (i=0; i<root->used; i++)
    if (((GNode *)root->node.children[i])->num == -936)
      root->node.children[i] = NULL;
    else
      mark_graph((GNode *)(root->node.children[i]));
}

/*  */
/*
 * The following functions are callback functions used in the
 * ExtDumpGraph()/render.c function.
 */

/*ARGSUSED*/
/**********************************************************************
 * Function: static void EPSdumpNode(SIX PARAMETERS)
 * Parameters:
 *	GraphWidget graph;
 *	GC gc;
 *	int x, y, size;
 *	void *value;
 * 
 * Modifications:
 *      <list mods with name and date>
 */
static void EPSdumpNode(graph, gc, x, y, size, value)
  GraphWidget graph;
  GC gc;
  int x, y, size;
  void *value;
{
  (void)fprintf((FILE *) gc, EPSDRAWNODE, x, A4HEIGHT - y, size / 2,
		(long)value);
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void EPSdumpLink(SEVEN PARAMETERS)
 * Parameters:
 *	GraphWidget graph
 *	GC gc
 *	int x1, y1, x2, y2, size
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void EPSdumpLink(graph, gc, x1, y1, x2, y2, size)
  GraphWidget graph;
  GC gc;
  int x1, y1, x2, y2, size;
{
  (void)fprintf((FILE *) gc, EPSDRAWLINK, x1, A4HEIGHT - y1, x2, 
		A4HEIGHT - y2);
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void EPSDumpRefNode(SIX PARAMETERS)
 * Parameters:
 *	GraphWidget graph
 *	GC gc
 *	int x, y, size
 *	struct MyValue *value
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void EPSDumpRefNode(graph, gc, x, y, size, value)
  GraphWidget graph;
  GC gc;
  int x, y, size;
  struct MyValue *value;
{
  char string[MAXPATHLEN];
  int i, j;

  /* Quote all "(" and ")" and "\". */
  for (i = 0, j = 0; i < (int)strlen(value->string); i++) {
    if ((value->string[i] == '(')
	|| (value->string[i] == ')')
	|| (value->string[i] == '\\'))
      string[j++] = '\\';
    string[j++] = value->string[i];
  }
  string[j] = '\0';

  if (value->type == infonode_e)
    (void)fprintf((FILE *) gc, EPSDRAWINFONODE, x, A4HEIGHT - y, string);
  else
    (void)fprintf((FILE *) gc, EPSDRAWIDENTRY, x, A4HEIGHT - y, string);
}

/*  */
/**********************************************************************
 * Function: static void EPSProlog(void *context, int width, int height)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void EPSProlog(context, width, height)
  void *context;
  int width;
  int height;
{
  (void)fprintf((FILE *) context, EPSHEADER, width, A4HEIGHT - height);
}

/*  */
/**********************************************************************
 * Function: static void EPSEpilog(void *context)
 *
 * Modifications:
 *      1993-12-25 Martin Sjlin. Define constant and removed
 *                 overflow "end" in prolog code.
 *      <list mods with name and date>
 */
static void EPSEpilog(context)
  void *context;
{
  (void)fputs(EPSTRAILER, (FILE *) context);
}

/*  */
/**********************************************************************
 * Function: static void EPSComment(void *context, char *str)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void EPSComment(context, str)
  void *context;
  char *str;
{
  (void)fprintf((FILE *) context, "\n\n%% %s\n", str);
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void WriteRef(THREE PARAMETERS)
 * Parameters:
 *	Widget  widget
 *	XtPointer client_data, call_data
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void WriteRef(widget, client_data, call_data)
  Widget widget;
  XtPointer client_data, call_data;
{
  FILE *file = NULL;
  char *stemname = NULL;
  char filename[MAXPATHLEN];
  struct RefWidgets *refs = NULL;
  Widget dialog = NULL, shell = widget;

  if (!(dialog = XtParent(widget))) {
    /* widget doesn't have a parent?! */
    (void)fprintf(stderr, "widget doesn't have a parent?!  Strange..\n");
    XBell(XtDisplay(widget), 50);
    return;
  }

  stemname = XawDialogGetValueString(dialog);
  refs = (struct RefWidgets *)client_data;

  /* Dump template! */
  (void)sprintf(filename, "%s%s", stemname, "_tmpl.eps");
  (void)printf("Writing template to %s.\n", filename);

  if (!(file = fopen(filename, "w"))) {
    (void)fprintf(stderr, "Cannot open %s", filename);
    return;
  }
  ExtDumpGraph(refs->template,
	       EPSdumpLink,
	       EPSDumpRefNode,
	       EPSProlog,
	       EPSEpilog,
	       EPSComment,
	       (void *)file);
  if ((fclose(file)) != 0) {
    (void)fprintf(stderr, "fclose failed in WriteRef.\n");
  }
  /* Dump target! */
  (void)sprintf(filename, "%s%s", stemname, "_trgt.eps");
  (void)printf("Writing target in %s.\n", filename);

  if (!(file = fopen(filename, "w"))) {
    (void)fprintf(stderr, "Cannot open %s", filename);
    return;
  }
  ExtDumpGraph(refs->target,
	       EPSdumpLink,
	       EPSDumpRefNode,
	       EPSProlog,
	       EPSEpilog,
	       EPSComment,
	       (void *)file);

  if ((fclose(file)) != 0) {
    (void)fprintf(stderr, "fclose failed in WriteRef.\n");
  }

  destroyshell(shell, (XtPointer)shell, NULL);
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: void CreateFileShell(Widget top, XtCallbackProc func, XtPointer client_data)
 *
 * Modifications:
 *      <list mods with name and date>
 */
void CreateFileShell(top, func, client_data)
  Widget top;
  XtCallbackProc func;
  XtPointer client_data;
{
#define LBLSTRING "Specify the file to write to: "
  Widget fileinput = NULL, inputform = NULL, dialog = NULL, parent = NULL;

  while (top && !(XtIsShell(top)))
    top = XtParent(top);

  fileinput = XtVaCreatePopupShell("fileinput", transientShellWidgetClass,
				    top,
				    XtNtitle, "File Selection",
				    XtNallowShellResize, True,
				    NULL);
  inputform = XtVaCreateManagedWidget("inputform", formWidgetClass,
				       fileinput,
				       XtNresizable, True,
				       XtNresize, True,
				       NULL);
  dialog = XtVaCreateManagedWidget("inputdialog", dialogWidgetClass,
					inputform,
					XtNlabel, (XtArgVal) LBLSTRING,
					XtNresizable, True,
					XtNvalue, LBLSTRING,
					NULL);
  parent = XtParent(XtParent(dialog));
  XawDialogAddButton(dialog,"Execute", func,(XtPointer)client_data);
  XawDialogAddButton(dialog,"Cancel", destroyshell, (XtPointer)parent);
  centerPopupOverCursor(parent);
  XtVaSetValues(dialog, XtNvalue, "", NULL);
  XtPopup((Widget) parent, XtGrabNone);
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void DoDumpHistory(CALLBACK PROTOCOL)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void DoDumpHistory(widget, client_data, call_data)
  Widget widget;
  XtPointer client_data, call_data;
{
  CreateFileShell((Widget)client_data, WriteHistory, client_data);
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void DoDumpRef(CALLBACK PROTOCOL)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void DoDumpRef(widget, client_data, call_data)
  Widget widget;
  XtPointer client_data, call_data;
{
  struct RefWidgets *refs = (struct RefWidgets *)client_data;
  CreateFileShell((Widget)refs->template,WriteRef, client_data);
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void WriteHistory(THREE PARAMETERS)
 * Parameters:
 *	Widget  widget
 *	XtPointer client_data, call_data
 *
 * Load the version structure and generate a postscript file with the graph
 * of the history. Works more or less like the GraphWidget.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void WriteHistory(widget, client_data, call_data)
  Widget widget;
  XtPointer client_data, call_data;
{
  char *filename = NULL;
  GraphWidget history = NULL;
  FILE *file = NULL;
  Widget dialog = NULL;
  Widget shell = widget;

  if (!(dialog = XtParent(widget))) {
    /* widget doesn't have a parent?! */
    (void)fprintf(stderr, "widget doesn't have a parent?!  Strange..\n");
    XBell(XtDisplay(widget), 50);
    return;
  }

  filename = XawDialogGetValueString(dialog);
  history = (GraphWidget)client_data;

  if (!(file = fopen(filename, "w"))) {
    (void)fprintf(stderr, "Cannot open %s.\n", filename);
    XBell(XtDisplay(widget), 50);
    return;
  }

  /* Dump it! */
  ExtDumpGraph(history, EPSdumpLink, EPSdumpNode, EPSProlog,
	       EPSEpilog, EPSComment, (void *)file);

  if ((fclose(file)) != 0)
    (void)fprintf(stderr, "fclose failed in WriteHistory.\n");

  destroyshell(shell, (XtPointer)shell, NULL);
}

/*  */
/**********************************************************************
 * Function: Widget xdumpref(reference_structure *ref)
 *
 * Produces a reference structure dump in a window.
 *
 * Modifications:
 *      1993-12-27 Martin Sjlin. Moved font references into 
 *          application resource file as well fall back resource
 *          (to ease running on small screens!)
 *      <list mods with name and date>
 */
Widget xdumpref(ref)
  reference_structure *ref;
{
  Widget refpane = NULL, box = NULL, shellw = NULL, scroll = NULL, 
	 contents = NULL, close = NULL, dump = NULL, targetw = NULL, 
	 templatew = NULL;
  Dimension maxwidth = 0;
  struct GraphCallData *graphs = NULL;
  Dimension targetheight, targetwidth;
  Dimension templheight, templwidth;
  Dimension viewheight, viewwidth;
  Dimension templtitleheight, targettitleheight;
  Dimension contentsheight, contentswidth;
  XtArgVal targetgraph, templategraph;
  GNode target, template;
  struct RefWidgets *refs = NULL;

  /* initialisations so we don't get garbage */
  targetheight = targetwidth = templheight = templwidth = 0;
  viewheight = viewwidth = templtitleheight = targettitleheight = 0;
  contentsheight = contentswidth = 0;

  /* Check things first... */
  if (ref == (reference_structure *)NULL) {
    (void)fprintf(stderr, "Attempt to dump NULL reference structure.\n");
    return NULL;
  }
  /* Traverse and dump the target. */
  target.node.value = NULL;
  target.node.flags = 0;
  target.node.children = NULL;
  target.node.x = SPACEX / 2;
  target.node.y = -SPACEY;
  target.num = 0;
  target.used = 0;
  targetheight = SPACEY + stepidentries(ref->target, 20, 2, 
		 (indentrec *) NULL, &target);
  targetwidth = (XtArgVal) GraphWidth(target.node.children[0]) + SPACEX;
  targetgraph = (XtArgVal) target.node.children[0];

  /* Traverse and dump the template. */
  template.node.value = NULL;
  template.node.flags = 0;
  template.node.children = NULL;
  template.node.x = SPACEX / 2;
  template.node.y = -SPACEY;
  template.num = 0;
  template.used = 0;
  templheight = SPACEY +
    stepidentries(ref->template, 20, 2, (indentrec *) NULL, &template);
  templwidth = (XtArgVal) GraphWidth(template.node.children[0]) + SPACEX;
  templategraph = (XtArgVal) template.node.children[0];

  maxwidth = Max(targetwidth, templwidth);

  if (templwidth > MAXWIDTH || targetwidth > MAXWIDTH ||
      templheight > MAXHEIGHT || targetheight > MAXHEIGHT) {
    viewwidth = MAXWIDTH;
    viewheight = MAXHEIGHT;
  }
  /* Create the widgets */
  shellw = XtVaCreatePopupShell((String) Pmakeshelltitle(ref),
				wmShellWidgetClass,
				toplevel,
				"title", (XtArgVal) Pmakeshelltitle(ref),
				NULL);

  refpane = XtVaCreateManagedWidget("refpane",
				  panedWidgetClass,
				  shellw,
				  NULL);

  /* Do the command button section. */
  box = XtVaCreateManagedWidget("box", boxWidgetClass, refpane, NULL);

  close = XtVaCreateManagedWidget("close",
				  commandWidgetClass,
				  box,
				  XtNx, MARGIN,
				  XtNy, MARGIN,
				  XtNlabel, (XtArgVal) "Close Window",
				  NULL);

  dump = XtVaCreateManagedWidget("dump",
				  commandWidgetClass,
				  box,
				  XtNx, 210,
				  XtNy, MARGIN,
				  XtNlabel, (XtArgVal) "Dump on File",
				  NULL);

  /* Do the graph section. */
  scroll = XtVaCreateManagedWidget("scroll",
				  viewportWidgetClass,
				  refpane,
				  XtNx, MARGIN,
				  XtNy, 0,
				  XtNwidth, viewwidth,
				  XtNheight, viewheight,
				  XtNallowVert, (Boolean) TRUE,
				  XtNallowHoriz, (Boolean) TRUE,
				  NULL);

  /* assign layout widget */
  contentswidth = maxwidth + 2 * MARGIN;
  contentsheight = MARGIN + 40 + templheight + targetheight + 2 * MARGIN;
  contents = XtVaCreateManagedWidget("contents",
				  compositeWidgetClass,
				  scroll,
				  XtNwidth, contentswidth,
				  XtNheight, contentsheight,
				  XtNresize, FALSE,
				  NULL);

  /* Make a title for the template graph. */
  (void)XtVaCreateManagedWidget("templatetitle",
				labelWidgetClass,
				contents,
				XtNx, MARGIN,
				XtNy, MARGIN,
				XtNwidth, maxwidth,
				XtNheight, templtitleheight,
				XtNlabel, (XtArgVal) "Template",
				XtNborderWidth, 0,
				NULL);

  /* assign graph widget */
  templatew = XtVaCreateManagedWidget("template",
				      graphWidgetClass,
				      contents,
				      XtNx, 
				        (Dimension)(maxwidth-templwidth)/
				        (Dimension)2,
				      XtNy, MARGIN + 20,
				      XtNwidth, templwidth,
				      XtNheight, templheight,
				      XtNgraph, templategraph,
				      XtNlayout, (Boolean) Done,
				      XtNdrawNode, (XtArgVal) DrawRefNode,
				      XtNnodeHit, (XtArgVal) ack,
				      XtNdrawLink, (XtArgVal) drawlink,
				      NULL);

  /* Make a title for the target graph. */
  (void)XtVaCreateManagedWidget("targettitle",
				labelWidgetClass,
				contents,
				XtNx, MARGIN,
				XtNy, MARGIN + 20 + templheight,
				XtNwidth, maxwidth,
				XtNheight, targettitleheight,
				XtNlabel, (XtArgVal) "Target",
				XtNborderWidth, 0,
				NULL);

  /* assign graph widget */
  targetw = XtVaCreateManagedWidget("target",
				    graphWidgetClass,
				    contents,
				    XtNx, 
				      (Dimension)(maxwidth-targetwidth)/
				      (Dimension)2,
				    XtNy, MARGIN + 40 + templheight,
				    XtNwidth, targetwidth,
				    XtNheight, targetheight,
				    XtNgraph, targetgraph,
				    XtNlayout, (Boolean) Done,
				    XtNdrawNode, (XtArgVal) DrawRefNode,
				    XtNdrawLink, (XtArgVal) drawlink,
				    NULL);

  graphs = (struct GraphCallData *)malloc(sizeof(struct GraphCallData));
  if (graphs == (struct GraphCallData *)NULL) {
    fatal_error("%s\n", "malloc failed in xdumpref");
  }
  graphs->target = (GNode *) target.node.children[0];
  graphs->template = (GNode *) template.node.children[0];

  XtAddCallback(shellw, XtNdestroyCallback, whack_widl, (XtPointer)ref);
  XtAddCallback(shellw, XtNdestroyCallback, closeref, (XtPointer) graphs);
  XtAddCallback(close, XtNcallback, destroyshell, (XtPointer)shellw);

  refs = (struct RefWidgets *)malloc(sizeof(struct RefWidgets));
  if (refs == (struct RefWidgets *)NULL) {
    fatal_error("%s\n", "malloc failed in xdumpref");
  }
  refs->target = (GraphWidget) targetw;
  refs->template = (GraphWidget) templatew;
  XtAddCallback(dump, XtNcallback, DoDumpRef, (XtPointer)refs);
  XtAddCallback(dump, XtNdestroyCallback, whack_refs, (XtPointer)refs);

  XtPopup((Widget) shellw, XtGrabNone);

  free((FREEPTR *)target.node.children);
  free((FREEPTR *)template.node.children);

  return shellw;
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static GNode *BuildHistoryGraph(label *obj, Widget w)
 *
 * Modifications:
 * 	<list mods with name and date>
 */
static GNode *BuildHistoryGraph(obj,w)
     label *obj;
     Widget w;
{
  GNode *graph;
  int x = 7;
  int y = 6;
  int flag = 0;

  if (w)
    flag++;

  graph = CreateNode((GNode *) NULL, (void *)(long)obj->vs);
  if (!BuildHistory(graph, obj, flag) || !graph->node.children) {
    popup_message(POPUP_MESS_INFO, "%s %d", 
		  "Couldn't build history graph for ", obj->vs);
    /* Free graph ? */
    return NULL;
  }

  if (w)
    XtVaGetValues(w, XtNstartX, &x, XtNstartY, &y, NULL);
      
  graph->node.x = x;
  graph->node.y = y;

  /* Compute the width and height. */
  ComputeLayout((GraphNode *) graph, LeftRight, 16, 15);

#if 0
  /* Invert the graph so that it grows from right to left. */
  graph->node.x = GraphWidth((GraphNode *)graph) + 15 + 15;
  ComputeLayout((GraphNode *) graph, LeftRight, -16, 15);
#endif

  return graph;
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void component_rollback(GraphWidget widget, GNode *node)
 *
 * Pick a new version from component history object.
 *
 * Modifications:
 * 	<list mods with name and date>
 */
static void component_rollback(widget, node)
  GraphWidget widget;
  GNode *node;
{
  reference_structure *ref;
  label obj;
  label lbl;
  label bt;
  label root;
  GNode *graph;

  XtVaGetValues((Widget)widget,
                XtNgraph,&graph,
		XtNgraphValue,&ref,
		NULL);

  if (GLI_GETLINKITEM(&(ref->bt), BTBL_ROOT, 1, &lbl)) {
    short_error("The binding table has no root\n");
    return;
  }

  obj.vs = (long) graph->node.value;
  obj.inst = (graph == node) ? -1 : (long) node->node.value;

  switch (obj.inst) {
  case 0:
    (void)fprintf(stderr,"Retrieve hidden history.\n");
    obj.inst = -1;
    if ((graph = BuildHistoryGraph(&obj, (Widget)widget))) {
      closehist((Widget)NULL,(XtPointer)widget,(XtPointer)NULL);
      XtVaSetValues((Widget) widget, XtNgraph, (XtArgVal)graph, NULL);
      XClearWindow(XtDisplay(widget),XtWindow(widget));
      widget->graph.layout = NotDone;
      ClearGraphPart((GraphWidget )NULL, widget);
    }
    (*graphClassRec.core_class.expose)((Widget)widget, (XEvent *)NULL, (Region)NULL);
    return;
  case -1:
    (void)GC_GETCURRENT(&obj);
    (void)printf("Putting up the \"current\" composition version.\n");
    (void)UBL_UNBINDLABEL(copylbl(&root,&(ref->root)));
    *((int *)&(widget->graph.current)) = obj.vs;
    (void)buildref(&root, ref->template->idname, ref);
    return;
  default:
    (void)fprintf(stderr,"Retrieving version %d.\n", obj.inst);
    break;
  }

  if (!EDITED(&(ref->bt))) {
    popup_message(POPUP_MESS_INFO, "%s\n%s",
		  "A transient binding table version is",
		  "made for managing component rollback.");
    (void)copylbl(&bt,&(ref->bt));
    (void)CPO_COPYOBJ(&(ref->bt),&bt);
    newlastcmd(-(COPYOP+1),&(ref->bt),0,(char *)NULL,(char *)NULL,0,
	       (attrval *)NULL, &bt);
  }
  if (lbl.vs == obj.vs) {
    if (!UL_UNLINK(&(ref->bt), BTBL_ROOT, &lbl))
      (void)LINK(&(ref->bt), BTBL_ROOT, &obj, 1);
  } else {
    lbl.vs = obj.vs;
    lbl.inst = -1;
    (void)UL_UNLINK(&(ref->bt), BTBL_PARTS, &lbl);
    (void)LINK(&(ref->bt), BTBL_PARTS, &obj,1);
  }

  *((int *)&(widget->graph.current)) = obj.inst;
  (void)buildrefbt(ref->template->idname,&(ref->bt),TRUE,ref);
  (*graphClassRec.core_class.expose)((Widget)widget, (XEvent *)NULL, (Region)NULL);
}


/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void composition_rollback(GraphWidget widget, GNode *node)
 *
 * Pick a new version of the composite object.
 *
 * Modifications:
 * 	<list mods with name and date>
 */
static void composition_rollback(widget, node)
  GraphWidget widget;
  GNode *node;
{
  GNode *graph;
  reference_structure *oldref;
  label bt;
  label root;

  XtVaGetValues((Widget)widget,
		XtNgraph, &graph,
		XtNgraphValue, &oldref,
		NULL);

  bt.vs = (long)graph->node.value;
  bt.inst = (graph == node) ? -1 : (long) node->node.value;

  switch  (bt.inst) {
  case 0:
    (void)fprintf(stderr,"Retrieving hidden history.\n");
    bt.inst = -1;
    if ((graph = BuildHistoryGraph(&bt, (Widget)widget))) {
      closehist((Widget)NULL,(XtPointer)widget,(XtPointer)NULL);
      XtVaSetValues((Widget) widget, XtNgraph, (XtArgVal)graph, NULL);
      XClearWindow(XtDisplay(widget),XtWindow(widget));
      widget->graph.layout = NotDone;
      ClearGraphPart((GraphWidget )NULL, widget);
    }
    break;
  case -1:
    (void)printf("Retrieving the \"current\" version.\n");
    (void)UBL_UNBINDLABEL(copylbl(&root,&(oldref->root)));
    *((int *)&(widget->graph.current)) = oldref->bt.vs;
    (void)buildref(&root, oldref->template->idname, oldref);
    break;
  default:
    (void)printf("Retrieving version %d.\n", bt.inst);
    *((int *)&(widget->graph.current)) = bt.inst;
    (void)buildrefbt(oldref->template->idname, &bt, TRUE, oldref);
    break;
  }
  (*graphClassRec.core_class.expose)((Widget)widget, (XEvent *)NULL, (Region)NULL);
}

/*  */
/**********************************************************************
 * Function: Widget dump_temporal_history(label *obj, reference_structure *ref)
 *
 * Modifications:
 *      1993-12-27 Martin Sjlin. Moved font references into 
 *          application resource file as well fall back resource
 *          (to ease running on small screens)
 *      <list mods with name and date>
 */
Widget dump_temporal_history(obj, ref, comp)
  label *obj;
  reference_structure *ref;
  int comp;	/* 0 = composition, 1 = component. */
{
  Widget historypane = NULL, scroll = NULL, box = NULL, history = NULL,
	 shellw = NULL, close = NULL, dump = NULL;
  GNode *graph = NULL;
  Dimension width = 0;
  Dimension height = 0;
  char *p = NULL;
  void (*nodehit)();

  if (!(graph = BuildHistoryGraph(obj,NULL)))
    return NULL;

  if (GraphWidth((GraphNode *) graph) * 2 > MAXWIDTH)
    width = MAXWIDTH;
  if (GraphHeight((GraphNode *) graph) * 2 > MAXHEIGHT)
    height = MAXHEIGHT;

  if (!comp) {
    p = "Temporal History";
    nodehit = composition_rollback;
  } else {
    p = "Component Temporal History";
    nodehit = component_rollback;
  }

  shellw = XtVaCreatePopupShell((String)p,
				(WidgetClass) wmShellWidgetClass,
				(Widget) toplevel,
				"title", (String)p,
				NULL);

  historypane = XtVaCreateManagedWidget("historypane", panedWidgetClass,shellw,NULL);

  box = XtVaCreateManagedWidget("box", boxWidgetClass, historypane, NULL);

  close = XtVaCreateManagedWidget("close",
				  commandWidgetClass,
				  box,
				  XtNx, MARGIN,
				  XtNy, MARGIN,
				  XtNlabel, (XtArgVal) "Close Window",
				  NULL);

  dump = XtVaCreateManagedWidget("dump",
				  commandWidgetClass,
				  box,
				  XtNx, 210,
				  XtNy, MARGIN,
				  XtNheight, 0,
				  XtNlabel, (XtArgVal) "Dump on File",
				  NULL);

  scroll = XtVaCreateManagedWidget("scroll",
				  viewportWidgetClass,
				  historypane,
				  XtNx, 0,
				  XtNy, 0,
				  XtNwidth, width,
				  XtNheight, height,
				  XtNallowVert, (Boolean) TRUE,
				  XtNallowHoriz, (Boolean) TRUE,
				  XtNlength, 500,
				  XtNborderWidth, (Dimension) 0,
				  NULL);

  history = XtVaCreateManagedWidget("history",
				     graphWidgetClass,
				     scroll,
				     XtNx, MARGIN, 
				     XtNy, 0,
				     XtNwidth, 0,
				     XtNheight, 0,
				     XtNgraph,(XtArgVal)graph,
				     XtNscale, FIX_OFFSET * 2,
				     XtNspaceX, 16,
				     XtNspaceY, 15,
				     XtNnodeHit, (XtArgVal) nodehit,
				     XtNgraphValue, (XtArgVal) ref,
				     XtNnodeSize, 13, 
				     NULL);

  if (ref && ref->flags.bound) {
    *((int *)&(((GraphWidget)history)->graph.current)) = ref->bt.inst;
  } else {
    *((int *)&(((GraphWidget)history)->graph.current)) = obj->vs;
  }

  XtAddCallback(shellw, XtNdestroyCallback, whack_widl, (XtPointer)ref);
  XtAddCallback(shellw, XtNdestroyCallback, closehist, (XtPointer)history);
  XtAddCallback(close, XtNcallback, destroyshell, (XtPointer)shellw);
  XtAddCallback(dump, XtNcallback, DoDumpHistory, (XtPointer)history);

  XtPopup((Widget) shellw, XtGrabNone);
  return shellw;
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void freeMyValue(GNode *node)
 *
 * free's up the string and structure of a MyValue structure.
 * passed in to the FreeTree function.
 *
 * Modifications:
 * 	<list mods with name and date>
 */
static void freeMyValue(node)
  GNode *node;
{
  struct MyValue *to_free = NULL;

  if (!node) {
    (void)fprintf(stderr, "freeMyValue: (warning) null input parameter.\n");
    return;
  }

  to_free = (struct MyValue *)node->node.value;
  free((FREEPTR *)to_free->string);
  free((FREEPTR *)to_free);
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void ack(GraphWidget widget, GNode *node)
 *
 * Pick a new version of the composite object.
 *
 * Modifications:
 * 	<list mods with name and date>
 */
static void ack(widget, node)
  GraphWidget widget;
  GNode *node;
{
  XBell(XtDisplay((Widget)widget), 100);
}

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static void drawlink(SEVEN PARAMETERS)
 * Parameters:
 *      GraphWidget graph;
 *      GC gc;
 *      int x1, y1, x2, y2, size;
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void drawlink(graph, gc, x1, y1, x2, y2, size)
  GraphWidget graph;
  GC gc;
  int x1, y1, x2, y2, size;
{
  XDrawLine(XtDisplay(graph), XtWindow(graph), gc, x1, y1, x2, y2);
}
