/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME:		xaimexport.c
 *
 * SCCSINFO:		@(#)xaimexport.c	1.2 6/6/94
 *
 * ORIGINAL AUTHOR:	David Partain, 1990-06-06
 *
 * MODIFICATIONS:
 *      1994-05-17 Martin Sjlin. Ripped X11 related stuff out of 
 *                      aimexport.c and moved it here.
 *      <list mods here with name and date>
 *
 * DESCRIPTION:
 *	
 *	
 */

/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Dialog.h>
#include <X11/Xaw/Form.h>

#include "aimtypes.h"			/* _textdump  definition */

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_aimexport.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_xstuff.h"
#include "f_xdumpref.h"

#ifdef NEED_STRDUP
extern char *strdup( /* char *incoming */ );
#endif	/* NEED_STRDUP */

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern Widget toplevel;			/* xstuff.c */
extern EscType global_escape;		/* aimexport.c */
extern DumpType global_type;		/* aimexport.c */

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
#ifndef MAX_WINS		/* they might have it on the command line */
#define MAX_WINS 25		
#endif /* MAX_WINS */

typedef struct _open_win_struct {
  reference_structure *ref;
  _textdump *textdump;
  Widget type;
  Widget style;
  Widget font;
  Widget structw;
  Widget escape;
  Widget placeholder;
  Widget shellw;
  Widget status;
  Widget output;
} open_win_struct;

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static Widget setup_dump P_(( reference_structure *ref, long index ));
static int is_open P_((reference_structure *ref, _textdump *textdump));
static int set_outfile P_(( long index ));
static void execute_dump P_(( Widget widget, XtPointer client_data,
                               XtPointer call_data ));
static void toggle_dumptype P_(( Widget widget, XtPointer client_data,
                                  XtPointer call_data ));
static void toggle_style P_(( Widget widget, XtPointer client_data,
                               XtPointer call_data ));
static void toggle_font P_(( Widget widget, XtPointer client_data,
                              XtPointer call_data ));
static void toggle_structure P_(( Widget widget, XtPointer client_data,
                                   XtPointer call_data ));
static void toggle_phold P_(( Widget widget, XtPointer client_data,
                               XtPointer call_data ));
static void toggle_escape P_(( Widget widget, XtPointer client_data,
                                XtPointer call_data ));
static void free_stuff P_(( Widget widget, XtPointer client_data,
                             XtPointer call_data ));
static void close_dump P_(( Widget widget, XtPointer client_data,
                             XtPointer call_data ));

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
static open_win_struct open_win[MAX_WINS];

/*  */
/*******************************************************************
 * Function: Widget build_export(reference_structure *rsp)
 *
 *  This function builds the LaTeX file.  Creates a hierarchical
 *  LaTeX file which can be edited by the user after creation.
 *
 *  Modifications:
 *	<list mods with name and date>
 */
Widget build_export(rsp)
  reference_structure *rsp;	/* ptr to ref structure being displayed */
{
  _textdump *textdump = NULL;
  long   index;
  static int done = 0;

  if (!done) {
    int i;
    for (i = 0; i < MAX_WINS; i++) {
      open_win[i].ref = NULL;
      open_win[i].textdump = NULL;
      open_win[i].type = NULL;
      open_win[i].style = open_win[i].font = NULL;
      open_win[i].escape = open_win[i].structw = open_win[i].shellw = NULL;
      open_win[i].placeholder=open_win[i].status=open_win[i].output = NULL;
    }
    done = 1;
  }

  if (!rsp) {
    (void)fprintf(stderr,
		  "\nNull reference structure:  cannot create LaTeX file.\n");
    return NULL;
  }

  if((textdump = initialize_textdump()))
    textdump->leftmargin = "leftMargin";

  if ((index = is_open(rsp, textdump)) == -1) {
    return NULL;
  }
  return setup_dump(rsp, index);
}

/*  */
/*******************************************************************
 * Function: static Widget setup_dump(reference_structure *ref, long index)
 *
 *  Modifications:
 *      1993-12-27 Martin Sjlin. Moved font references into 
 *          application resource file as well fall back resource
 *          (to ease running on small screens)
 *      1994-05-14 Martin Sjlin. Using current value for textdump
 *          to setup buttons ...
 *	<list mods with name and date>
 */
static Widget setup_dump(ref, index)
  reference_structure *ref;
  long index;
{
  Widget exportform, title, stylelabel, fontlabel, 
	 structlabel, esclabel, phlabel,
  	 labelw, doitw, close, latex;
  XtArgVal font;
  Dimension width;

  open_win[index].shellw = XtVaCreatePopupShell("export",
						applicationShellWidgetClass,
						toplevel,
						XtNtitle, "Export Window",
						XtNallowShellResize, True,
						NULL);

  exportform = XtVaCreateManagedWidget("exportform", formWidgetClass,
				  open_win[index].shellw,
				  NULL);

  title = XtVaCreateManagedWidget("title", labelWidgetClass,
				   exportform,
				   XtNlabel, "Text Dump Options",
				   XtNresizable, True,
				   NULL);

  labelw = XtVaCreateManagedWidget("label",
				   labelWidgetClass, exportform,
				   XtNlabel, "Dump Type:",
				   XtNfromVert, title,
				   NULL);

  open_win[index].type = XtVaCreateManagedWidget("type",
						 commandWidgetClass, exportform,
					    XtNlabel, "nice long string here",
						 XtNfromVert, title,
						 NULL);

  latex = XtVaCreateManagedWidget("latex", labelWidgetClass,
				  exportform,
				  XtNlabel, "LaTeX Document options:",
				  XtNfromVert, labelw,
				  NULL);

  stylelabel = XtVaCreateManagedWidget("stylelabel",
					labelWidgetClass,
					exportform,
					XtNlabel, "Document style:",
					XtNfromVert, latex,
					NULL);

  open_win[index].style = XtVaCreateManagedWidget("style",
						   commandWidgetClass,
						   exportform,
					    XtNlabel, "nice long string here",
						   XtNfromVert, latex,
						   NULL);

  fontlabel = XtVaCreateManagedWidget("fontlabel", labelWidgetClass,
				       exportform,
				       XtNlabel, "Font size:",
				       XtNfromVert, stylelabel,
				       NULL);

  open_win[index].font = XtVaCreateManagedWidget("font",
						  commandWidgetClass,
						  exportform,
					    XtNlabel, "nice long string here",
						  XtNfromVert, stylelabel,
						  NULL);

  structlabel = XtVaCreateManagedWidget("structlabel",
					 labelWidgetClass,
					 exportform,
					 XtNlabel, "Structure:",
					 XtNfromVert, open_win[index].font,
					 NULL);

  open_win[index].structw = XtVaCreateManagedWidget("struct",
						    commandWidgetClass,
						    exportform,
					    XtNlabel, "nice long string here",
						    XtNfromVert, fontlabel,
						    NULL);

  esclabel = XtVaCreateManagedWidget("esclabel",
					 labelWidgetClass,
					 exportform,
					 XtNlabel, "Escape chars?",
					 XtNfromVert, open_win[index].structw,
					 NULL);

  open_win[index].escape = XtVaCreateManagedWidget("escape",
						    commandWidgetClass,
						    exportform,
					    XtNlabel, "nice long string here",
						    XtNfromVert, structlabel,
						    NULL);

  phlabel = XtVaCreateManagedWidget("phlabel",
				     labelWidgetClass,
				     exportform,
				     XtNlabel, "Placeholders?",
				     XtNfromVert, open_win[index].escape,
				     NULL);

  open_win[index].placeholder = XtVaCreateManagedWidget("placeholder",
						    commandWidgetClass,
						    exportform,
					    XtNlabel, "nice long string here",
						    XtNfromVert, esclabel,
						    NULL);

  open_win[index].output = XtVaCreateManagedWidget("output",
						    dialogWidgetClass,
						    exportform,
						    XtNlabel, "Output file:",
						    XtNvalue, "",
						    XtNfromVert, phlabel,
						    NULL);

  open_win[index].status = XtVaCreateManagedWidget("status",
						    labelWidgetClass,
						    exportform,
				  XtNlabel, "Status: No operations initiated",
					 XtNfromVert, open_win[index].output,
						    XtNresizable, True,
						    NULL);

  doitw = XtVaCreateManagedWidget("doit", commandWidgetClass,
				  exportform,
				  XtNlabel, "Execute",
				  XtNfromVert, open_win[index].status,
				  NULL);

  close = XtVaCreateManagedWidget("close", commandWidgetClass,
				   exportform,
				   XtNlabel, "Close Window",
				   XtNfromVert, open_win[index].status,
				   XtNfromHoriz, doitw,
				   NULL);

  XtVaSetValues(open_win[index].type, XtNfromHoriz, stylelabel, NULL);
  XtVaSetValues(open_win[index].style, XtNfromHoriz, stylelabel, NULL);
  XtVaSetValues(open_win[index].font, XtNfromHoriz, stylelabel, NULL);
  XtVaSetValues(open_win[index].structw, XtNfromHoriz, stylelabel, NULL);
  XtVaSetValues(open_win[index].escape, XtNfromHoriz, stylelabel, NULL);
  XtVaSetValues(open_win[index].placeholder, XtNfromHoriz, stylelabel, NULL);

  if (open_win[index].textdump->outfile) {
    XtVaGetValues(open_win[index].font, XtNfont, &font, NULL);
    width = XTextWidth((XFontStruct *) font, open_win[index].textdump->outfile,
		       (strlen(open_win[index].textdump->outfile)));
    XtVaSetValues(open_win[index].output, XtNvalue,
		  open_win[index].textdump->outfile,
		  XtNwidth, width, NULL);
  }
  XtAddCallback(open_win[index].type, XtNcallback, toggle_dumptype,
		(XtPointer) index);
  XtAddCallback(open_win[index].style, XtNcallback, toggle_style,
		(XtPointer) index);
  XtAddCallback(open_win[index].font, XtNcallback, toggle_font,
		(XtPointer) index);
  XtAddCallback(open_win[index].structw, XtNcallback, toggle_structure,
		(XtPointer) index);
  XtAddCallback(open_win[index].escape, XtNcallback, toggle_escape,
		(XtPointer) index);
  XtAddCallback(open_win[index].placeholder, XtNcallback, toggle_phold,
		(XtPointer) index);
  XtAddCallback(doitw, XtNcallback, execute_dump,
		(XtPointer) index);
  XtAddCallback(close, XtNcallback, close_dump,
		(XtPointer) index);

  XtRealizeWidget(open_win[index].shellw);

  XtVaSetValues(open_win[index].type, XtNlabel, 
		global_type == LaTeX ?
		"LaTeX" : "Straight Text", NULL);
  XtVaSetValues(open_win[index].style, XtNlabel, 
		open_win[index].textdump->docstyle == article ? 
		"article" : "letter", NULL);
  XtVaSetValues(open_win[index].structw, XtNlabel, 
		open_win[index].textdump->structure == deep ? 
		"Deep Structure" : "Flat Structure", NULL);
  XtVaSetValues(open_win[index].escape, XtNlabel, 
		global_escape == doit ? 
		"Do Escape" : "Don't escape", NULL);
  XtVaSetValues(open_win[index].placeholder, XtNlabel, 
		open_win[index].textdump->placeholders == yes ? 
		"Include them" : "No placeholder", NULL);
  XtVaSetValues(open_win[index].font, XtNlabel, 
		open_win[index].textdump->fontsize == medium ? "medium" :
		open_win[index].textdump->fontsize == small  ? "small" : 
		"large", NULL);

  XtAddCallback(open_win[index].shellw, XtNdestroyCallback, free_stuff,
		(XtPointer) index);
  XtAddCallback(open_win[index].shellw, XtNdestroyCallback, whack_widl,
		(XtPointer) ref);

  XtVaGetValues(open_win[index].shellw, XtNwidth, &width, NULL);
  XtVaSetValues(title, XtNwidth, width - 10, NULL);
  XtVaSetValues(open_win[index].status, XtNwidth, width - 10, NULL);

  XtPopup(open_win[index].shellw, XtGrabNone);
  return (open_win[index].shellw);
}

/*  */
/**********************************************************************
 * Function: static int is_open(reference_structure *ref, _textdump *textdump)
 *
 * go through the open_win structure and see if we have a text dump
 * window open for the reference structure.  if so, don't open another,
 * and if not find an empty slot and slam in ref and textdump in the
 * right places.
 *
 *  Modifications:
 *	<list mods with name and date>
 */
static int is_open(ref, textdump)
  reference_structure *ref;
  _textdump *textdump;
{
  int i;

  for (i = 0; i < MAX_WINS; i++)
    if (open_win[i].ref == ref) {
      XtVaSetValues(open_win[i].status, XtNlabel,
		  "Sorry.  Only one Dump Window per composite object.", NULL);
      XBell(XtDisplay(open_win[i].status), 50);
      if (textdump->outfile) {
	free((FREEPTR *)textdump->outfile);
	textdump->outfile = NULL;
      }
      if (textdump) {
	free((FREEPTR *)textdump);
	textdump = NULL;
      }
      return -1;
    }
  for (i = 0; i < MAX_WINS; i++) {
    if (!open_win[i].ref) {
      open_win[i].ref = ref;
      open_win[i].textdump = textdump;
      return i;
    }
  }
  popup_message(POPUP_MESS_INFO, "%s %d %s", "You can only have",
		MAX_WINS, "Text Dump Windows open at the same time.");
  if (textdump->outfile) {
    free((FREEPTR *)textdump->outfile);
    textdump->outfile = NULL;
  }
  if (textdump) {
    free((FREEPTR *)textdump);
    textdump = NULL;
  }
  return -1;
}

/*  */
/**********************************************************************
 * Function: static int set_outfile(long index)
 *
 * get the string value from the dialog widget, ensure it's at least
 * more than just a null char at the beginning.  that string is
 * managed by X, so it cannot be free'ed.  i keep my own copy.
 *
 *  Modifications:
 *	<list mods with name and date>
 */
static int set_outfile(index)
  long index;			/* into open_win */
{
  char *filename = NULL;

  filename = XawDialogGetValueString(open_win[index].output);
  if ((!filename) || (filename[0] == '\0'))
    return 0;

  if (open_win[index].textdump->outfile)
    free((FREEPTR *)open_win[index].textdump->outfile);
  open_win[index].textdump->outfile = strdup(filename);
  return 1;
}

/*  */
/* below are callbacks of various sorts for the widgets.  most are
 * for toggling in the command widgets, but there are also a couple
 * of destroy callbacks
 */

/* ARGSUSED */
/*******************************************************************
 * Function: static void execute_dump(CALLBACK PROTOCOL)
 *
 *  Modifications:
 *	<list mods with name and date>
 */
static void execute_dump(widget, client_data, call_data)
  Widget widget;
  XtPointer client_data, call_data;
{
  long index = (long)client_data;
  FILE *outfile;		/* pointer to the LaTeX file */

  if (!set_outfile(index)) {
    (void)fprintf(stderr, "Cannot have NULL output file.\n");
    XtVaSetValues(open_win[index].status, XtNlabel,
		  "Error: Cannot have NULL output file", NULL);
    XBell(XtDisplay(widget), 50);
    return;
  }
  if ((outfile = fopen(open_win[index].textdump->outfile, "w")) == (FILE *) NULL) {
    (void)fprintf(stderr, "Unable to open output file.\n");
    XtVaSetValues(open_win[index].status, XtNlabel,
		  "Error: Unable to open output file", NULL);
    XBell(XtDisplay(widget), 50);
    return;
  }

  export_file(open_win[index].ref, open_win[index].textdump, outfile);

  if ((fclose(outfile)) != 0)
    (void)fprintf(stderr, "fclose failed in execute_dump.\n");

  XtVaSetValues(open_win[index].status,
		XtNlabel, "Status: Output sent to file successfully.", NULL);

  XBell(XtDisplay(widget), 50);
}

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static void toggle_dumptype(CALLBACK PROTOCOL)
 *
 *  Modifications:
 *	<list mods with name and date>
 */
static void toggle_dumptype(widget, client_data, call_data)
  Widget widget;
  XtPointer client_data, call_data;
{
  long index = (long)client_data;
  Widget w = (Widget) open_win[index].type;

  switch (global_type) {
  case straight:
    global_type = LaTeX;
    XtVaSetValues(w, XtNlabel, "LaTeX", NULL);
    break;
  case LaTeX:
    global_type = straight;
    XtVaSetValues(w, XtNlabel, "Straight Text", NULL);
    break;
  default:
    (void)printf("toggle_dumptype: unknown dump type\n");
    break;
  }
}

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static void toggle_style(CALLBACK PROTOCOL)
 *
 *  Modifications:
 *	<list mods with name and date>
 */
static void toggle_style(widget, client_data, call_data)
  Widget widget;
  XtPointer client_data, call_data;
{
  long index = (long)client_data;
  Widget w = (Widget) open_win[index].style;

  switch (open_win[index].textdump->docstyle) {
  case article:
    open_win[index].textdump->docstyle = letter;
    XtVaSetValues(w, XtNlabel, "letter", NULL);
    break;
  case letter:
    open_win[index].textdump->docstyle = article;
    XtVaSetValues(w, XtNlabel, "article", NULL);
    break;
  default:
    (void)printf("toggle_style: unknown document style\n");
    break;
  }
}

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static void toggle_font(CALLBACK PROTOCOL)
 *
 *  Modifications:
 *	<list mods with name and date>
 */
static void toggle_font(widget, client_data, call_data)
  Widget widget;
  XtPointer client_data, call_data;
{
  long index = (long)client_data;
  Widget w = (Widget) open_win[index].font;

  switch (open_win[index].textdump->fontsize) {
  case small:
    open_win[index].textdump->fontsize = medium;
    XtVaSetValues(w, XtNlabel, "medium", NULL);
    break;
  case medium:
    open_win[index].textdump->fontsize = large;
    XtVaSetValues(w, XtNlabel, "large", NULL);
    break;
  case large:
    open_win[index].textdump->fontsize = small;
    XtVaSetValues(w, XtNlabel, "small", NULL);
    break;
  default:
    (void)printf("toggle_font: unknown font size\n");
    break;
  }
}

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static void toggle_structure(CALLBACK PROTOCOL)
 *
 *  Modifications:
 *	<list mods with name and date>
 */
static void toggle_structure(widget, client_data, call_data)
  Widget widget;
  XtPointer client_data, call_data;
{
  long index = (long)client_data;
  Widget w = (Widget) open_win[index].structw;

  switch (open_win[index].textdump->structure) {
  case flat:
    open_win[index].textdump->structure = deep;
    XtVaSetValues(w, XtNlabel, "Deep Structure", NULL);
    break;
  case deep:
    open_win[index].textdump->structure = flat;
    XtVaSetValues(w, XtNlabel, "Flat Structure", NULL);
    break;
  default:
    (void)printf("toggle_structure: unknown structure type\n");
    break;
  }
}

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static void toggle_phold(CALLBACK PROTOCOL)
 *
 *  Modifications:
 *	<list mods with name and date>
 */
static void toggle_phold(widget, client_data, call_data)
  Widget widget;
  XtPointer client_data, call_data;
{
  long index = (long)client_data;
  Widget w = (Widget) open_win[index].placeholder;

  switch (open_win[index].textdump->placeholders) {
  case yes:
    open_win[index].textdump->placeholders = no;
    XtVaSetValues(w, XtNlabel, "No placeholders", NULL);
    break;
  case no:
    open_win[index].textdump->placeholders = yes;
    XtVaSetValues(w, XtNlabel, "Include them", NULL);
    break;
  default:
    (void)printf("toggle_phold: unknown escape type\n");
    break;
  }
}

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static void toggle_escape(CALLBACK PROTOCOL)
 *
 *  Modifications:
 *	<list mods with name and date>
 */
static void toggle_escape(widget, client_data, call_data)
  Widget widget;
  XtPointer client_data, call_data;
{
  long index = (long)client_data;
  Widget w = (Widget) open_win[index].escape;

  switch (global_escape) {
  case dont:
    global_escape = doit;
    XtVaSetValues(w, XtNlabel, "Do Escape", NULL);
    break;
  case doit:
    global_escape = dont;
    XtVaSetValues(w, XtNlabel, "Don't escape", NULL);
    break;
  default:
    (void)printf("toggle_escape: unknown escape type\n");
    break;
  }
}

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static void free_stuff(CALLBACK PROTOCOL)
 *
 *  Modifications:
 *	<list mods with name and date>
 */
static void free_stuff(widget, client_data, call_data)
  Widget widget;
  XtPointer client_data, call_data;
{
  long index = (long)client_data;

  if (open_win[index].textdump->outfile)
    free((FREEPTR *)open_win[index].textdump->outfile);
  if (open_win[index].textdump)
    free((FREEPTR *)open_win[index].textdump);
  open_win[index].ref = NULL;
  open_win[index].textdump = NULL;
  open_win[index].style = open_win[index].type = NULL;
  open_win[index].font = open_win[index].structw = NULL;
  open_win[index].escape = open_win[index].placeholder = NULL;
  open_win[index].shellw = open_win[index].status = NULL;
  open_win[index].output = NULL;
}

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static void close_dump(CALLBACK PROTOCOL)
 *
 *  Modifications:
 *	<list mods with name and date>
 */
static void close_dump(widget, client_data, call_data)
  Widget widget;
  XtPointer client_data, call_data;
{
  long index = (long)client_data;
  XtDestroyWidget(open_win[index].shellw);
}

