/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* 
 * MODULE NAME: 	ltextsink.c
 *
 * SCCSINFO:		@(#)ltextsink.c	1.12 6/9/94
 *
 * ORIGINAL AUTHOR(S):  ??????????
 *
 * MODIFICATIONS:
 *	<list mods here with name and date>
 *
 * DESCRIPTION:
 *
 *  XlTextSink object. (For use with the XlText widget).
 *
 *********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************
 * int CharWidth(Widget w, int x, unsigned char c)
 */

/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/XawInit.h>
#include "ltextsinkP.h"
#include "ltextsrcP.h"
#include <X11/Xaw/TextP.h>	/* I also reach into the text widget. */

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_ltextsink.h"

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern int LL_uid;		/* liblincks.a */
extern char *sys_errlist[];	/* liblincks.a */
extern int sys_nerr;		/* liblincks.a */
extern int errno;		/* liblincks.a */

extern int horrible_global_flag; /* ltextaction.c */
/*
 * Handled only in ltextaction.c:EndAction and inspected in
 * ltextsink.c:ClearToBackground. Have a fit.
 */
/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
/*
 * The following lines are added for R4-R5 compatibility:
 * R4 defines LF, R5 defines XawLF, etc.
 */
#ifndef TAB
#define LF    XawLF
#define TAB   XawTAB
#define SP    XawSP
#endif
/* End of addition */

#ifdef GETLASTPOS
#undef GETLASTPOS		/* We will use our own GETLASTPOS. */
#endif

#define GETLASTPOS XawTextSourceScan(source, (XawTextPosition)0, XawstAll, (XawTextScanDirection)XawsdRight, 1, TRUE)
#define ASCENT(sink)  (sink->text_sink.font->ascent)
#define DESCENT(sink) (sink->text_sink.font->descent)
#define TEXTHEIGHT(sink)      (ASCENT(sink) + DESCENT(sink) + 1)

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static void ClearToBackground P_(( Widget w, XtPosition x, XtPosition y,
                                    XtDimension width, XtDimension height ));
static void Destroy P_(( Widget w ));
static void DisplayText P_(( Widget w, XtPosition x, XtPosition y,
                              XawTextPosition pos1, XawTextPosition pos2,
                              XtBoolean highlight ));
static void FindDistance P_(( Widget w, XawTextPosition fromPos, int
                               fromx, XawTextPosition toPos, int
                               *resWidth, XawTextPosition *resPos, int
                               *resHeight ));
static void FindPosition P_(( Widget w, XawTextPosition fromPos, int
                               fromx, int width, int stopAtWordBreak,
                               XawTextPosition *resPos, int *resWidth,
                               int *resHeight ));
static void GetCursorBounds P_(( Widget w, XRectangle * rect ));
static void GetGC P_(( XlTextSinkObject sink ));
static void Initialize P_(( Widget request, Widget new ));
static void InsertCursor P_(( Widget w, XtPosition x, XtPosition y,
                               XawTextInsertState state ));
static int MaxLines P_(( TextSinkObject w, XtDimension height ));
static XtDimension PaintText P_(( Widget w, GC gc, XtPosition x, XtPosition y,
                                 unsigned char * buf, int len ));
static void Resolve P_(( Widget w, XawTextPosition pos, int fromx, int
                          width, XawTextPosition *leftPos,
                          XawTextPosition *rightPos ));
static XtBoolean SetValues P_(( Widget current, Widget request, Widget new ));

/*********************************************************************
 * INTERNAL (STATIC) DATA: 
 *********************************************************************/
#define offset(field) XtOffset(XlTextSinkObject, xltext_sink.field)
static XtResource resources[] = {
    {XtNecho, XtCOutput, XtRBoolean, sizeof(Boolean),
	offset(echo), XtRImmediate, (caddr_t) True},
    {XtNdisplayNonprinting, XtCOutput, XtRBoolean, sizeof(Boolean),
	offset(display_nonprinting), XtRImmediate, (caddr_t) True},
    {XtNjustify, XtCJustify, XtRJustifyMode, sizeof(XlJustifyType),
       offset(justify), XtRJustifyMode, (caddr_t)XlLeft},
    {XtNindent, XtCIndent, XtRIndentMode, sizeof(int),
       offset(indent), XtRIndentMode, (caddr_t)NULL},
    {XtNfontmap, XtCFontmap, XtRFontMarks, sizeof(XlFontMap), 
       offset(fontinfo), XtRFontMarks, (caddr_t)NULL}, 
};
#undef offset

#define SuperClass		(&textSinkClassRec)
XlTextSinkClassRec XltextSinkClassRec = {
  {
/* core_class fields */	
    /* superclass	  	*/	(WidgetClass) SuperClass,
    /* class_name	  	*/	"XlTextSink",
    /* widget_size	  	*/	sizeof(XlTextSinkRec),
    /* class_initialize   	*/	XawInitializeWidgetSet,
    /* class_part_initialize	*/	NULL,
    /* class_inited       	*/	FALSE,
    /* initialize	  	*/	(XtInitProc)Initialize,
    /* initialize_hook		*/	NULL,
    /* obj1		  	*/	NULL,
    /* obj2		  	*/	NULL,
    /* obj3		  	*/	0,
    /* resources	  	*/	resources,
    /* num_resources	  	*/	XtNumber(resources),
    /* xrm_class	  	*/	NULLQUARK,
    /* obj4		  	*/	FALSE,
    /* obj5		  	*/	FALSE,
    /* obj6			*/	FALSE,
    /* obj7		  	*/	FALSE,
    /* destroy		  	*/	Destroy,
    /* obj8		  	*/	NULL,
    /* obj9		  	*/	NULL,
    /* set_values	  	*/	(XtSetValuesFunc)SetValues,
    /* set_values_hook		*/	NULL,
    /* obj10			*/	NULL,
    /* get_values_hook		*/	NULL,
    /* obj11		 	*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private   	*/	NULL,
    /* obj12		   	*/	NULL,
    /* obj13			*/	NULL,
    /* obj14			*/	NULL,
    /* extension		*/	NULL
  },
/* text_sink_class fields */
  {
    /* DisplayText              */      DisplayText,
    /* InsertCursor             */      InsertCursor,
    /* ClearToBackground        */      ClearToBackground,
    /* FindPosition             */      FindPosition,
    /* FindDistance             */      FindDistance,
    /* Resolve                  */      Resolve,
    /* MaxLines                 */      MaxLines,
    /* MaxHeight                */      XlSinkMaxHeight,
    /* SetTabs                  */      XtInheritSetTabs,
    /* GetCursorBounds          */      GetCursorBounds
  },
/* xltext_sink_class fields. */
  {
    /* Keep Compiler happy.     */      0
  }
};

WidgetClass XltextSinkObjectClass = (WidgetClass)&XltextSinkClassRec;

/*  */
/* Utilities */
/**********************************************************************
 * Function: int CharWidth (Widget w, int x, unsigned int c)
 * 
 * Modifications:
 *      <list mods with name and date>
 */
int CharWidth (w, x, c)
  Widget w;
  int x;
  unsigned int c;
{
    register int i, width, nonPrinting;
    XlTextSinkObject sink = (XlTextSinkObject) w;
    XFontStruct *font = sink->text_sink.font;
    Position *tab;

    if ( c == LF ) return(0);

    if (c == TAB) {
	/* Adjust for Left Margin. */
	x -= ((TextWidget) XtParent(w))->text.margin.left;

	if (x >= (int)(XtParent(w))->core.width)
	  return 0;
	for (i = 0, tab = sink->text_sink.tabs ; 
	     i < sink->text_sink.tab_count ; i++, tab++) {
	    if (x < *tab) {
		if (*tab < ((unsigned)XtParent(w)->core.width))
		    return *tab - x;
		else
		    return 0;
	    }
	}
	return 0;
    }

    if ( (nonPrinting = (c < (unsigned char) SP)) )
	if (sink->xltext_sink.display_nonprinting)
	    c += '@';
	else {
	    c = SP;
	    nonPrinting = False;
	}

    if (font->per_char &&
	    (c >= font->min_char_or_byte2 && c <= font->max_char_or_byte2))
	width = font->per_char[c - font->min_char_or_byte2].width;
    else
	width = font->min_bounds.width;

    if (nonPrinting)
	width += CharWidth((Widget)w, x, (unsigned char) '^');

    return width;
}

/*  */
/**********************************************************************
 * Function: static XtDimension PaintText(SIX PARAMETERS)
 * Parameters:
 *	Widget w		- the text widget
 *	GC gc			- gc to paint text with
 *	XtPosition x		- location to paint the text
 *	XtPosition y		- location to paint the text
 *	unsigned char *buf	- buffer of text to paint
 *	int len			- length of buffer of text to paint
 * 
 * Actually paints the text into the window.
 *
 * Returns: the width of the text painted, or 0.
 *
 * NOTE:  If this string attempts to paint past the end of the window
 *        then this function will return zero.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static XtDimension PaintText(w, gc, x, y, buf, len)
  Widget w;
  GC gc;
  XtPosition x, y;
  unsigned char * buf;
  int len;
{
    XlTextSinkObject sink = (XlTextSinkObject) w;
    TextWidget ctx = (TextWidget) XtParent(w);

    Position max_x;
    Dimension width = XTextWidth(sink->text_sink.font, (char *) buf, len); 
    max_x = (Position) ctx->core.width;

    if ( ((int) width) <= -x)	           /* Don't draw if we can't see it. */
      return(width);

    XDrawImageString(XtDisplay(ctx), XtWindow(ctx), gc, 
		     (int) x, (int) y, (char *) buf, len);

/*
 * The following statement puts a black box at the end of a line when it
 * goes beyond the window.
 */
#if defined(BOX_AT_END) || defined(lint)
    if ( (((Position) width + x) > max_x) && (ctx->text.margin.right != 0) ) {
	x = ctx->core.width - ctx->text.margin.right;
	width = ctx->text.margin.right;
	XFillRectangle(XtDisplay((Widget) ctx), XtWindow( (Widget) ctx),
		       sink->xltext_sink.normgc, (int) x,
		       (int) y - ASCENT(sink), 
		       (unsigned int) width,
		       (unsigned int) TEXTHEIGHT(sink));
	return(0);
    }
#endif /* BOX_AT_END */
    return(width);
}

/*  */
/* Sink Object Functions */
/**********************************************************************
 * Function: static void ClearToBackground(FIVE PARAMETERS)
 * Parameters:
 *	Widget w
 *	XtPosition x
 *	XtPosition y
 *	XtDimension width
 *	XtDimension height;
 *
 * Clear to background ONLY when it is to "end of window".
 *
 * Modifications:
 *      1994-05-27 Martin Sjlin. Replace all code with call
 *                 to XClearArea which seems to fix clearing bug
 *                 with a "ending" black rectangle.
 *      <list mods with name and date>
 */
static void ClearToBackground(w,x,y,width,height)
  Widget w;
  XtPosition x, y;
  XtDimension width, height;
{
  if ((int)(y+height) < (int)(XtParent(w))->core.height)
    return;
  if ((int)(x+width) < (int)(XtParent(w))->core.width)
    return;
  if ((x == 0) && (y == 0))	/* disallow anything for whole window */
    if (horrible_global_flag) /* Handled only in ltextaction.c:EndAction */
      return;

  XClearArea(XtDisplayOfObject(w), XtWindowOfObject(w), 
	     (int) x, (int) y,
	     (unsigned int) width, (unsigned int) height, TRUE);

  return;
}

/**********************************************************************
 * Function: static void DisplayText(SIX PARAMETERS)
 * Parameters:
 *	Widget w
 *	XtPosition x
 *	XtPosition y
 *	XtBoolean highlight
 *	XawTextPosition pos1
 *	XawTextPosition pos2
 * 
 * This function does not know about drawing more than one line of text.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void DisplayText(w, x, y, pos1, pos2, highlight)
  Widget w;
  XtPosition x;
  XtPosition y;
  XtBoolean highlight;
  XawTextPosition pos1;
  XawTextPosition pos2;
{
  XlTextSinkObject sink = (XlTextSinkObject) w;
  Widget source = XawTextGetSource(XtParent(w));
  unsigned char buf[BUFSIZ];
  int j, k;
  XawTextBlock blk;
  Dimension width;
  unsigned char c;
  GC gc = highlight ? sink->xltext_sink.invgc : sink->xltext_sink.normgc;
  GC invgc = highlight ? sink->xltext_sink.normgc : sink->xltext_sink.invgc;

  if (!sink->xltext_sink.echo) return;

  y += ASCENT(sink);
  for ( j = 0 ; pos1 < pos2 ; ) {
    pos1 = XawTextSourceRead(source, pos1, &blk, (int)(pos2 - pos1));
    for (k = 0; k < blk.length; k++) {
      c = blk.ptr[k];
      if ((j >= BUFSIZ) || (c == '\n') || (c == '\t')) {
	/* Paint text on window. */
	if (j > 0) {
	  if ((width = PaintText(w, gc, x, y, buf, j)) == 0)
	    return;
	  x += width;
	}
	j = 0;
      }
      switch (c) {
      case LF:
      case TAB:
	if (c == LF)
	  width = XtParent(w)->core.width;
	else
	  width = CharWidth((Widget)w, x, (unsigned char) TAB);
	XFillRectangle(XtDisplayOfObject(w), XtWindowOfObject(w),
		       invgc, (int) x,
		       (int) y - ASCENT(sink),
		       (unsigned int) width,
		       (unsigned int) TEXTHEIGHT(sink));
	x += width;
	break;
      default:
	if (c < (unsigned char) ' ' ) {
	  if (sink->xltext_sink.display_nonprinting) {
	    buf[j++] = '^';
	    c += '@';
	  } else {
	    c = ' ';
	  }
	}
	buf[j++] = c;
	break;
      }
    }
  }
  if (j > 0)
    (void) PaintText(w, gc, x, y, buf, j);
}

/*  */
/**********************************************************************
 * Function: static void GetCursorBounds(Widget w, XRectangle *rect)
 * 
 * Returns the size and location of the cursor.
 *
 * Arguments: 
 *	w	- the text object.
 *	rect	- an X rectangle to return the cursor bounds in.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void GetCursorBounds(w, rect)
  Widget w;
  XRectangle * rect;
{
  XlTextSinkObject sink = (XlTextSinkObject) w;
  TextWidget tw = (TextWidget) XtParent(w);
  TextSrcObject source = (TextSrcObject) tw->text.source;
  XawTextPosition pos = tw->text.insertPos;
  XawTextBlock txt;
  TextSrcObjectClass obj_class = (TextSrcObjectClass) XtClass((Widget) source);
    
  rect->height = TEXTHEIGHT(sink);
  rect->y = sink->xltext_sink.cursor_y - (short) rect->height;
  rect->x = sink->xltext_sink.cursor_x ;

  /* read one character of the source at the current position */
  (*(obj_class -> textSrc_class.Read))(source,pos,&txt,1);

  rect->width = CharWidth((Widget)sink,
			  rect->x,
			  (txt.length==0)?
			  (unsigned char)' ':
			  (unsigned char)*(txt.ptr));
  if (rect->width <= 0)
    rect->width = 5; /* why not? (Ralph) */
}

/*  */
/**********************************************************************
 * Function: static void InsertCursor(FOUR PARAMETERS)
 * Parameters:
 *	Widget w
 *	XtPosition x
 * 	XtPosition y
 *	XawTextInsertState state
 * 
 * The following procedure manages the "insert" cursor.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void InsertCursor (w, x, y, state)
  Widget w;
  XtPosition x;
  XtPosition y;
  XawTextInsertState state;
{
  XlTextSinkObject sink = (XlTextSinkObject) w;
  Widget text_widget = XtParent(w);
  XRectangle rect;

  sink->xltext_sink.cursor_x = x;
  sink->xltext_sink.cursor_y = y;

  if (state == sink->xltext_sink.laststate) return;

  sink->xltext_sink.laststate = state;

  if (!XtIsRealized(text_widget)) return;

  GetCursorBounds(w, &rect);

  XFillRectangle(XtDisplay(text_widget),
		 XtWindow(text_widget),
		 sink->xltext_sink.xorgc,
		 (int) rect.x,
		 (int) rect.y,
		 (unsigned int) rect.width,
		 (unsigned int) rect.height);
}

/*  */
/**********************************************************************
 * Function: static void FindDistance (SEVEN PARAMETERS)
 * Parameters:
 * 	Widget w
 *	XawTextPosition fromPos	- First position
 *	int fromx		- Horizontal location of first position
 *	XawTextPosition toPos	- Second position
 *	int *resWidth		- Distance between fromPos and resPos
 * 	XawTextPosition *resPos	- Actual second position used
 *	int *resHeight		- Height required
 * 
 * Given two positions, find the distance between them.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void FindDistance (w, fromPos, fromx, toPos, resWidth, resPos, resHeight)
  Widget w;
  XawTextPosition fromPos;	/* First position. */
  int fromx;			/* Horizontal location of first position. */
  XawTextPosition toPos;	/* Second position. */
  int *resWidth;		/* Distance between fromPos and resPos. */
  XawTextPosition *resPos;	/* Actual second position used. */
  int *resHeight;		/* Height required. */
{
    XlTextSinkObject sink = (XlTextSinkObject) w;
    Widget source = XawTextGetSource(XtParent(w));

    register XawTextPosition index, lastPos;
    register unsigned char c;
    XawTextBlock blk;

    /* we may not need this */
    lastPos = GETLASTPOS;
    XawTextSourceRead(source, fromPos, &blk, (int)(toPos - fromPos));
    *resWidth = 0;
    for (index = fromPos; index != toPos && index < lastPos; index++) {
	if (index - blk.firstPos >= blk.length)
	    XawTextSourceRead(source, index, &blk, (int)(toPos - fromPos));
	c = blk.ptr[index - blk.firstPos];
	*resWidth += CharWidth((Widget)w, fromx + *resWidth, c);
	if (c == LF) {
	    index++;
	    break;
	}
    }
    *resPos = index;
    *resHeight = TEXTHEIGHT(sink);
}

/*  */
/**********************************************************************
 * Function: static void FindPosition(EIGHT PARAMETERS)
 * Parameters:
 * Widget w
 * XawTextPosition fromPos	- Starting position
 * int fromx;			- Horizontal location of starting position
 * int width;			- Desired width
 * int stopAtWordBreak;		- Whether the resulting position should
 *				- be at a word break
 * XawTextPosition *resPos 	- Resulting position
 * int *resWidth;		- Actual width used
 * 
 * Modifications:
 *      <list mods with name and date>
 */
static void FindPosition(w, fromPos, fromx, width, stopAtWordBreak, 
			 resPos, resWidth, resHeight)
  Widget w;
  XawTextPosition fromPos; 	/* Starting position. */
  int fromx;			/* Horizontal location of starting position.*/
  int width;			/* Desired width. */
  int stopAtWordBreak;		/* Whether the resulting position should be at
				   a word break. */
  XawTextPosition *resPos;	/* Resulting position. */
  int *resWidth;		/* Actual width used. */
  int *resHeight;		/* Height required. */
{
    XlTextSinkObject sink = (XlTextSinkObject) w;
    Widget source = XawTextGetSource(XtParent(w));

    XawTextPosition lastPos;
    XawTextPosition index;
    XawTextPosition whiteSpacePosition = 0; /* keep gcc satisfied */
    int     lastWidth = 0;		/* better zero this ... */
    int     whiteSpaceWidth = 0;	/* keep gcc satisfied */
    Boolean whiteSpaceSeen;
    unsigned char c;
    XawTextBlock blk;

    lastPos = GETLASTPOS;

    XawTextSourceRead(source, fromPos, &blk, BUFSIZ);
    *resWidth = 0;
    whiteSpaceSeen = FALSE;
    c = 0;
    for (index = fromPos; *resWidth <= width && index < lastPos; index++) {
	lastWidth = *resWidth;
	if (index - blk.firstPos >= blk.length)
	    XawTextSourceRead(source, index, &blk, BUFSIZ);
	c = blk.ptr[index - blk.firstPos];
	*resWidth += CharWidth((Widget)w, fromx + *resWidth, c);

	if ((c == SP || c == TAB) && *resWidth <= width) {
	    whiteSpaceSeen = TRUE;
	    whiteSpacePosition = index;
	    whiteSpaceWidth = *resWidth;
	}
	if (c == LF) {
	    index++;
	    break;
	}
    }
    if ((*resWidth > width) && (index > fromPos)) {
	*resWidth = lastWidth;
	index--;
	if ((stopAtWordBreak) && (whiteSpaceSeen)) {
	    index = whiteSpacePosition + 1;
	    *resWidth = whiteSpaceWidth;
	}
    }
    if ((index == lastPos) && (c != LF))
      index = lastPos + 1;
    *resPos = index;
    *resHeight = TEXTHEIGHT(sink);
}

/*  */
/**********************************************************************
 * Function: static void Resolve (SIX PARAMETERS)
 * Parameters:
 *	Widget w
 *	XawTextPosition pos
 *	int fromx
 *	int width
 *	XawTextPosition *leftPos
 *	XawTextPosition *rightPos
 * 
 * Modifications:
 *      <list mods with name and date>
 */
static void Resolve (w, pos, fromx, width, leftPos, rightPos)
  Widget w;
  XawTextPosition pos;
  int fromx;
  int width;
  XawTextPosition *leftPos;
  XawTextPosition *rightPos;
{
    int resWidth, resHeight;
    Widget source = XawTextGetSource(XtParent(w));

    FindPosition(w, pos, fromx, width, FALSE, leftPos, &resWidth, &resHeight);
    if (*leftPos > GETLASTPOS)
      *leftPos = GETLASTPOS;
    *rightPos = *leftPos;
}

/*  */
/**********************************************************************
 * Function: static int MaxLines(TextSinkObject w, XtDimension height)
 * 
 * Modifications:
 *      <list mods with name and date>
 */
static int MaxLines(w, height)
  TextSinkObject w;
  XtDimension height;
{
  int fontheight, numlines;
  fontheight = (w->text_sink.font->ascent) + (w->text_sink.font->descent) + 1;
  numlines = (int) height / fontheight;

  return numlines;
}

/*  */
/**********************************************************************
 * Function: int XlSinkMaxHeight(TextSinkObject w, int lines)
 * 
 * Modifications:
 *      <list mods with name and date>
 */
int XlSinkMaxHeight(w, lines)
  TextSinkObject w;
  int lines;
{
  int fontheight;

  fontheight = (w->text_sink.font->ascent) + (w->text_sink.font->descent) + 1;
  return lines * fontheight;
}

/*  */
/**********************************************************************
 * Function: static void GetGC(XlTextSinkObject sink)
 * 
 * Modifications:
 *      <list mods with name and date>
 */
static void GetGC(sink)
  XlTextSinkObject sink;
{
    XtGCMask valuemask = (GCFont | GCGraphicsExposures 
			  | GCForeground | GCBackground );
    XGCValues values;

    values.font = sink->text_sink.font->fid;
    values.graphics_exposures = (Bool) FALSE;
    values.foreground = sink->text_sink.foreground;
    values.background = sink->text_sink.background;

    sink->xltext_sink.normgc = XtGetGC((Widget)sink, valuemask, &values);
    
    values.foreground = sink->text_sink.background;
    values.background = sink->text_sink.foreground;

    sink->xltext_sink.invgc = XtGetGC((Widget)sink, valuemask, &values);
    
    values.function = GXxor;
    values.background = (unsigned long) 0L;	/* (pix ^ 0) = pix */
    values.foreground = (sink->text_sink.background ^ 
			 sink->text_sink.foreground);

    /* change the valuemask so that the graphics exposure events don't
     * cause massive flickering and eventual crash of the system.  added
     * GCGraphicsExposures to the mask - dlp 
     */
    valuemask = GCGraphicsExposures | GCFunction | GCForeground | GCBackground;
    
    sink->xltext_sink.xorgc = XtGetGC((Widget)sink, valuemask, &values);
}


/*  */
/* Public routines */
/* ARGSUSED */
/**********************************************************************
 * Function: static void Initialize(Widget request, Widget new)
 * 
 * Initializes the TextSink Object.
 *
 * Arguments: the requested and new values for the object instance.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void Initialize(request, new)
  Widget request, new;
{
    XlTextSinkObject sink = (XlTextSinkObject) new;

    GetGC(sink);
    
    sink->xltext_sink.laststate = XawisOff;
    sink->xltext_sink.cursor_x = sink->xltext_sink.cursor_y = 0;
}

/*  */
/**********************************************************************
 * Function: static void Destroy(Widget w)
 * 
 * This function cleans up when the object is destroyed.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void Destroy(w)
  Widget w;
{
   XlTextSinkObject sink = (XlTextSinkObject) w;

   XtReleaseGC(w, sink->xltext_sink.normgc);
   XtReleaseGC(w, sink->xltext_sink.invgc);
   XtReleaseGC(w, sink->xltext_sink.xorgc);
}

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static XtBoolean SetValues(Widget current,Widget request,Widget new)
 * 
 * Sets the values for the XlTextSink
 * Arguments:
 *	current - current state of the object.
 *	request - what was requested.
 *	new - what the object will become.
 * Returns: True if redisplay is needed.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static XtBoolean SetValues(current, request, new)
  Widget current, request, new;
{
    XlTextSinkObject w = (XlTextSinkObject) new;
    XlTextSinkObject old_w = (XlTextSinkObject) current;

    if (w->text_sink.font != old_w->text_sink.font ||
	w->text_sink.foreground != old_w->text_sink.foreground ||
	w->text_sink.background != old_w->text_sink.background) {
	XtReleaseGC((Widget)w, w->xltext_sink.normgc);
	XtReleaseGC((Widget)w, w->xltext_sink.invgc);
	GetGC(w);
	((TextWidget)XtParent(new))->text.redisplay_needed = True;
    } else {
	if ( (w->xltext_sink.echo != old_w->xltext_sink.echo) ||
	     (w->xltext_sink.display_nonprinting != 
                                     old_w->xltext_sink.display_nonprinting) )
	    ((TextWidget)XtParent(new))->text.redisplay_needed = True;
    }
    
    return False;
}
