/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME: 	layout.c
 *
 * SCCSINFO:		@(#)layout.c	1.14 6/9/94
 *
 * ORIGINAL AUTHOR(S):  Nilgun & Michael, 1990-09-11
 *
 * MODIFICATIONS:
 *      <list mods with name and date>
 *
 * DESCRIPTION:
 *	Lincks XlLayout Widget.
 */
/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>

#include "layoutP.h"
#include "aimtypes.h"
#include "ltext.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_layout.h"

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_ltextfocussing.h"
#include "f_ltextpublic.h"
#include "f_ltextsaving.h"

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern int avoid_scrolling;	/* view.c */

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
#define FormChangeManaged (*formClassRec.composite_class.change_managed)
#define FormDeleteChild   (*formClassRec.composite_class.delete_child)
#define CHILD	composite.children
#define N_CHILD	composite.num_children

#ifdef DEBUG
/* since SyncHandle() requires that dpy be in local context */
#define sync(dpy) \
	if (dpy->synchandler) (*dpy->synchandler)(dpy)
#endif /* DEBUG */

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static void ChangeManaged P_(( Widget w ));
static void ClassInitialize P_(( void ));
static void DeleteChild P_(( Widget w ));
static void Initialize P_(( Widget request, Widget neww ));
static void ResizeLayout P_(( Widget w ));
static void verticalmanage P_(( Widget w ));

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
static XtResource resources[] =
{
#define offset(field) XtOffset(XlLayoutWidget, layout.field)
 /* {name, class, type, size, offset, default_type, default_addr}, */
  {XtNlayout, XtClayout, XtRlayout, sizeof(XlLayoutType),
   offset(type), XtRlayout, (caddr_t) XlVertical},
#undef offset
};

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
XlLayoutClassRec XllayoutClassRec =
{
  {				/* core_class fields */
    /* superclass         */ (WidgetClass) & formClassRec,
    /* class_name         */ "Layout",
    /* widget_size        */ sizeof(XlLayoutRec),
    /* class_initialize   */ (XtProc)ClassInitialize,
    /* class_part_init    */ NULL,
    /* class_inited       */ FALSE,
    /* initialize         */ (XtInitProc)Initialize,
    /* initialize_hook    */ NULL,
    /* realize            */ XtInheritRealize,
    /* actions            */ NULL,
    /* num_actions        */ 0,
    /* resources          */ resources,
    /* num_resources      */ XtNumber(resources),
    /* xrm_class          */ NULLQUARK,
    /* compress_motion    */ TRUE,
    /* compress_exposure  */ XtExposeCompressSeries,
    /* compress_enterleave*/ TRUE,
    /* visible_interest   */ FALSE,
    /* destroy            */ NULL,
    /* resize             */ ResizeLayout,
    /* expose             */ XtInheritExpose,
    /* set_values         */ NULL,
    /* set_values_hook    */ NULL,
    /* set_values_almost  */ XtInheritSetValuesAlmost,
    /* get_values_hook    */ NULL,
    /* accept_focus       */ NULL,
    /* version            */ XtVersion,
    /* callback_private   */ NULL,
    /* tm_table           */ NULL,
    /* query_geometry	  */ XtInheritQueryGeometry,
    /* display_accelerator*/ XtInheritDisplayAccelerator,
    /* extension          */ NULL
  },
  {				/* composite_class fields */
    /* geometry_manager   */ XtInheritGeometryManager,
    /* change_managed     */ ChangeManaged,
    /* insert_child       */ XtInheritInsertChild,
    /* delete_child       */ DeleteChild,
    /* extension          */ NULL
  },
  {				/* constraint_class fields */
    /* subresourses       */ NULL,
    /* subresource_count  */ 0,
    /* constraint_size    */ sizeof(XlLayoutConstraintsRec),
    /* initialize         */ NULL,
    /* destroy            */ NULL,
    /* set_values         */ NULL,
    /* extension          */ NULL
  },
  {				/* form_class fields */
    /* layout             */ XtInheritLayout
  },
  {				/* layout_class fields */
    /* emtpy               */ 0
  }
};


WidgetClass XllayoutWidgetClass = (WidgetClass) & XllayoutClassRec;

/*  */
/**********************************************************************
 * Function: static void ClassInitialize()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void ClassInitialize()
{
/* WidgetClass cl = XllayoutWidgetClass;*/

 /*
  * we know this doesn't do anything.  we have it in case we need
  * an initialization function.
  */
}

/*  */
/**********************************************************************
 * Function: void LayoutFocusIn(Widget w)
 *
 * Modifications:
 *      <list mods with name and date>
 */
void LayoutFocusIn(w)
  Widget w;
{
  XlLayoutWidget lw = (XlLayoutWidget) w;
  int flag = lw->layout.goal_column.flag;
  Widget child = lw->layout.focus.widget;
  int i;

  if (lw->N_CHILD == 0)
    return;

  for (i = 0; !child && i < lw->N_CHILD; i++) {
    if ((lw->N_CHILD == i + 1) ||
	((i == 0) && (lw->layout.focus.y < lw->CHILD[i + 1]->core.y)) ||
	((lw->layout.focus.y >= lw->CHILD[i]->core.y) &&
	 (lw->layout.focus.y < lw->CHILD[i + 1]->core.y))) {
      child = lw->CHILD[i];
    }
  }

  if (!child)
    return;

  avoid_scrolling = 1;
  XltextSetFocus(child, (XtPosition) lw->layout.focus.y);
  lw->layout.goal_column.flag = flag;
  avoid_scrolling = 0;

#ifdef DEBUG
  (void)fprintf(stderr, "leaving LayoutFocusIn\n");
  sync(XtDisplay(lw));
#endif /* DEBUG */

}


/*  */
/**********************************************************************
 * Function: void XlLayoutFocusDetails(Widget w, int y)
 *
 * Modifications:
 *      <list mods with name and date>
 */
void XlLayoutFocusDetails(w, y)
  Widget w;
  int y;
{
  XlLayoutWidget lw = (XlLayoutWidget) XtParent(w);

  lw->layout.focus.widget = w;
  lw->layout.focus.y = y;
}


/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static void Initialize(Widget request, Widget neww)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void Initialize(request, neww)
  Widget request, neww;
{
  XlLayoutWidget lw = (XlLayoutWidget) neww;

  /* Initialize goal column details. */
  lw->layout.goal_column.x = 0;
  lw->layout.goal_column.flag = 0;

  /* Initialise focussing details. */
  lw->layout.focus.widget = NULL;
  lw->layout.focus.y = 0;

  /* Initialise incremental search details. */
  lw->layout.search.text = NULL;
  lw->layout.search.old_text = NULL;
  lw->layout.search.flag = 0;
  lw->layout.search.wrap = 0;
  lw->layout.search.match = 0;
}

/*  */
/* ARGUSED */
/**********************************************************************
 * Function: static void ResizeLayout(Widget w)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void ResizeLayout(w)
  Widget w;
{
  XlLayoutWidget lw = (XlLayoutWidget) w;

  lw->form.old_width = lw->core.width;
  lw->form.old_height = lw->core.height;
}

/*  */
/**********************************************************************
 * Function: void ClearLayout(Widget w)
 *
 * Go through all of the children and set the background back to the
 * default.  
 *
 * Modifications:
 *      <list mods with name and date>
 */
void ClearLayout(w)
  Widget w;
{
  XlLayoutWidget lw = (XlLayoutWidget) w;
  int i;

  for (i = 0; i < lw->N_CHILD; i++) 
    XltextSetPenned((TextWidget)lw->CHILD[i]);
}

/*  */
/**********************************************************************
 * Function: static void verticalmanage(Widget w)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void verticalmanage(w)
  Widget w;
{
  XlLayoutWidget lw = (XlLayoutWidget) w;
  Widget predecessor = (Widget) NULL;
  Widget child;
  int i;

  for (i = 0; i < lw->N_CHILD; i++) {
    if (XtIsManaged((child = lw->CHILD[i]))) {
      XtVaSetValues
	(child,
	 XtNresizable, True,
	 XtNleft, XtChainLeft,
	 XtNright, XtChainRight,
	 XtNtop, XtChainTop,
	 XtNbottom, XtChainTop,
	 XtNfromVert, predecessor,
	 NULL);
      predecessor = child;
    }
  }
}


/*  */
/* ARGUSED */
/**********************************************************************
 * Function: static void ChangeManaged(Widget w)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void ChangeManaged(w)
  Widget w;
{
  XlLayoutWidget lw = (XlLayoutWidget) w;

  if (XtIsRealized(w))
    return;

  switch (lw->layout.type) {
  case XlVertical:
    verticalmanage(w);
    break;

  default:
    break;
  }

  FormChangeManaged((Widget) lw);

}


/*  */
/* ARGUSED */
/**********************************************************************
 * Function: static void DeleteChild(Widget w)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void DeleteChild(w)
  Widget w;
{
  XlLayoutWidget lw = (XlLayoutWidget) XtParent(w);

  if (lw->layout.focus.widget == w) {
    XlLayoutLostFocus((Widget) lw, XltextYForCurrentPosition((TextWidget) w));
  }
  FormDeleteChild(w);
}


/*  */
/**********************************************************************
 *
 * Public routines
 *
 */
/**********************************************************************
 * Function: Widget XlFirstChild(Widget w)
 *
 * Returns first widget.
 *
 * Modifications:
 *      <list mods with name and date>
 */
Widget XlFirstChild(w)
  Widget w;
{
  XlLayoutWidget lw = (XlLayoutWidget) XtParent(w);

  if (!XtIsSubclass((Widget) lw, XllayoutWidgetClass))
    return (Widget) NULL;

  if (lw->N_CHILD <= 0)
    return (Widget) NULL;

  return lw->CHILD[0];
}


/*  */
/**********************************************************************
 * Function: Widget XlLastChild(Widget w)
 *
 * Returns last widget.
 *
 * Modifications:
 *      <list mods with name and date>
 */
Widget XlLastChild(w)
  Widget w;
{
  XlLayoutWidget lw = (XlLayoutWidget) XtParent(w);

  if (!XtIsSubclass((Widget) lw, XllayoutWidgetClass))
    return (Widget) NULL;

  if (lw->N_CHILD <= 0)
    return (Widget) NULL;

  return lw->CHILD[lw->N_CHILD - 1];
}


/*  */
/**********************************************************************
 * Function: int XlHomeChildY(Widget w)
 *
 * Returns a pixel row coordinate for the "home child", i.e. the first
 * child whose upper left corner is within the layout parent's 0.
 *
 * Modifications:
 *      <list mods with name and date>
 */
int XlHomeChildY(w)
  Widget w;
{
  XlLayoutWidget lw = (XlLayoutWidget)w;
  Position home = -(lw->core.y);
  Position last = 0;
  int i;

  if (!XtIsSubclass((Widget) lw, XllayoutWidgetClass))
    return 0;

  for (i = 0; (i < lw->N_CHILD); i++) {
    last = lw->CHILD[i]->core.y;
    if (lw->CHILD[i]->core.y >= home)
      return last;
  }

  return last;
}

/*  */
/**********************************************************************
 * Function: Widget XlChildIndex(Widget w,int offset)
 *
 * Returns preceding widget.
 *
 * Modifications:
 *      <list mods with name and date>
 */
Widget XlChildIndex(w, offset)
  Widget w;
  int offset;
{
  XlLayoutWidget lw = (XlLayoutWidget) XtParent(w);
  int i;

  if (!XtIsSubclass((Widget) lw, XllayoutWidgetClass))
    return (Widget) NULL;

  for (i = 0; (i < lw->N_CHILD); i++) {
    if (w == lw->CHILD[i]) {
      i = i + offset;
      if ((i < 0) || (i >= lw->N_CHILD))
	return (Widget) NULL;
      return lw->CHILD[i];
    }
  }
  return (Widget) NULL;
}

/*  */
/***********************************************************************
 * Function: XtDimension XlLayoutHeight(XlLayoutWidget w)
 *
 * returns the assumed height that a layout widget has.  used by
 * xlincks to decide whether to put a scrollbar on a window as well as
 * to make it the maximum height
 *
 * Modifications:
 *	<list mods with name and date>
 */
XtDimension XlLayoutHeight(w)
  Widget w;
{
  XlLayoutWidget lw = (XlLayoutWidget) w;
  Widget child;
  int i, h = 0;
  int defaultdistance = 0;
  Dimension height, borderwidth;

  if (lw->N_CHILD == 0)
    return 0;

  height = borderwidth = (Dimension) 0;

  XtVaGetValues((Widget) lw, XtNdefaultDistance, &defaultdistance, NULL);

  for (i = 0; i < lw->N_CHILD; i++) {
    if (XtIsManaged((child = lw->CHILD[i]))) {
      XtVaGetValues(child,
		    XtNheight, &height,
		    XtNborderWidth, &borderwidth,
		    NULL);
      h += height + (2 * borderwidth) + defaultdistance;
    }
  }

  /*
   * the 4 is the "magic 4" which makes it look right... we don't know
   * where it comes from...
   */
  return (h + 4);
}

/*  */
/***********************************************************************
 * Function: XtDimension XlLayoutWidth(XlLayoutWidget w)
 *
 * returns the assumed width that a layout widget has.  used by
 * xlincks to decide how wide a window should be.
 *
 * Modifications:
 *	<list mods with name and date>
 */
XtDimension XlLayoutWidth(lw)
  XlLayoutWidget lw;
{
  Widget w;
  int i = 0;
  Dimension width = 0;

  for (i = 0; i < lw->N_CHILD; i++)
    if (XtIsManaged((w = lw->CHILD[i])))
      if (width < w->core.width)
	width = w->core.width;

  return(width);
}

/*  */
/***********************************************************************
 * Function: void XlRefreshLayout(XlLayoutWidget lw,int focus)
 *
 * Manages its children and tells the superclass (form).
 *
 * Modifications:
 *	<list mods with name and date>
 */
void XlRefreshLayout(lw)
  XlLayoutWidget lw;
{
  Widget w;
  Window focusw;
  int unused;
  int i;
  Dimension width = 0;

#ifdef DEBUG
  (void)fprintf(stderr, "entering XlRefreshLayout\n");
  sync(XtDisplay(lw));
#endif /* DEBUG */

  verticalmanage((Widget) lw);
  FormChangeManaged((Widget) lw);

  if (lw->N_CHILD == 0)
    return;

  for (i = 0; i < lw->N_CHILD; i++)
    XtSetMappedWhenManaged(lw->CHILD[i], True);

  width = XlLayoutWidth(lw);

  for (i = 0; i < lw->N_CHILD; i++)
    XtVaSetValues(lw->CHILD[i], XtNwidth, width, NULL);

  XGetInputFocus(XtDisplay((Widget) lw), &focusw, &unused);

  for (w = (Widget) lw; w; w = XtParent(w))
    if (XtIsSubclass(w, shellWidgetClass))
      break;

  if (w && (focusw == XtWindow(w))) { 
    LayoutFocusIn((Widget)lw);
  }

#ifdef DEBUG
  (void)fprintf(stderr, "leaving XlRefreshLayout\n");
  sync(XtDisplay(lw));
#endif /* DEBUG */
}


/*  */
/**********************************************************************
 * Function: void XlLayoutLostFocus(Widget w, int y)
 *
 * Report the focus to be lost at layout row y.
 *
 * Modifications:
 *      <list mods with name and date>
 */
void XlLayoutLostFocus(w, y)
  Widget w;
  int y;
{
  XlLayoutWidget lw = (XlLayoutWidget) w;

  lw->layout.focus.widget = NULL;
  lw->layout.focus.y = y;

#ifdef DEBUG
  (void)fprintf(stderr, "leaving XlLayoutLostFocus\n");
  sync(XtDisplay(lw));
#endif /* DEBUG */
}

/*  */
/**********************************************************************
 * Function: void XlInsertChild(Widget parent, Widget child, Widget prevchild)
 *
 * Insert a child widget after the specified child in the
 * layout widget.
 *
 * Modifications:
 *      <list mods with name and date>
 */
void XlInsertChild(parent, child, prevchild)
  Widget parent, child, prevchild;
{
  XlLayoutWidget lw = (XlLayoutWidget) parent;
  int pid;

  for (pid = lw->N_CHILD - 1; pid > 0; pid--) {
    if (lw->CHILD[pid - 1] == prevchild)
      break;
    lw->CHILD[pid] = lw->CHILD[pid - 1];
  }

  lw->CHILD[pid] = child;

  XtSetMappedWhenManaged(child, False);

  XtManageChild(child);
}

/*  */
/***********************************************************************
 * Function: void XlSetGoalColumn(Widget w, int c)
 *
 * Translate column c from widget w and set as goal_column.
 *
 * Modifications:
 *	<list mods with name and date>
 */
void XlSetGoalColumn(w, c)
  Widget w;
  int c;
{
  int x, y;
  Window ch;			/* this used to be Widget, which is wrong */
  XlLayoutWidget lw = (XlLayoutWidget) XtParent(w);

  if (lw->layout.goal_column.flag)
    return;

  (void)XTranslateCoordinates(XtDisplay(lw), XtWindow(w), XtWindow(lw),
			      c, 0, &x, &y, &ch);
  lw->layout.goal_column.x = x;
}


/*  */
/***********************************************************************
 * Function: int XlGetGoalColumn(Widget w)
 *
 * Translate the goal column into a column for widget w and return it.
 * Modifications:
 *	<list mods with name and date>
 */
int XlGetGoalColumn(w)
  Widget w;
{
  int x, y;
  Window ch;			/* ch is supposed to be Window, it used to be Widget */
  XlLayoutWidget lw = (XlLayoutWidget) XtParent(w);

  XTranslateCoordinates(XtDisplay(lw), XtWindow(lw), XtWindow(w),
			lw->layout.goal_column.x, 0, &x, &y, &ch);

  return x;
}


/*  */
/***********************************************************************
 * Function: void XlSkipGoalSet(Widget w, int flg)
 *
 * Set/Reset the flag marking that the sub-sequent goal column setting
 * should not take effect.
 *
 * Modifications:
 *	<list mods with name and date>
 */
void XlSkipGoalSet(w, flg)
  Widget w;
  int flg;
{
  XlLayoutWidget lw = (XlLayoutWidget) XtParent(w);

  lw->layout.goal_column.flag = flg;
}


/*  */
/***********************************************************************
 * Function: void XlSearchForward(TextWidget w)
 *
 * Search forward for next occurance of the search string.
 *
 * Modifications:
 *	<list mods with name and date>
 */
void XlSearchForward(w)
  Widget w;
{
  XlLayoutWidget lw = (XlLayoutWidget) XtParent(w);
  Widget ctx = w;
  TextWidget tw = (TextWidget) w;
  int pos = XltextPoint(tw) - lw->layout.search.match;

  if ((!lw->layout.search.text) || (strlen(lw->layout.search.text) == 0))
    return;

  if (lw->layout.search.match == strlen(lw->layout.search.text))
    pos += 1;

  while ((pos = XltextSearchForward((TextWidget) ctx,
				    (XawTextPosition) pos,
				    lw->layout.search.text)) < 0) {
    if ((ctx = XlChildIndex(ctx, 1)) == (Widget) NULL) {
      if (!lw->layout.search.wrap) {
	lw->layout.search.wrap = 1;
	XBell(XtDisplay(w), 50);
	return;
      }
      lw->layout.search.wrap = 0;
      ctx = XlFirstChild(w);
    }
    pos = 0;
  }

  lw->layout.search.wrap = 0;
  lw->layout.search.match = strlen(lw->layout.search.text);

  if (w != ctx) {
    TextLeave(w, NULL);
    XlLayoutLostFocus((Widget)lw,XltextYForCurrentPosition((TextWidget)ctx));
    LayoutFocusIn((Widget)lw);
  }
}

/*  */
/***********************************************************************
 * Function: void XlSearchBackward(TextWidget w)
 *
 * Search backward for next occurance of the search string.
 *
 * Modifications:
 *	<list mods with name and date>
 */
void XlSearchBackward(w)
  Widget w;
{
  XlLayoutWidget lw = (XlLayoutWidget) XtParent(w);
  Widget ctx = w;
  int pos = XltextPoint((TextWidget) w) + lw->layout.search.match;

  if (!lw->layout.search.text)
    return;
  if (strlen(lw->layout.search.text) == 0)
    return;

  if (lw->layout.search.match == strlen(lw->layout.search.text))
    pos -= 1;

  while ((pos = XltextSearchBackward((TextWidget) ctx,
				     (XawTextPosition) pos,
				     lw->layout.search.text)) < 0) {
    if ((ctx = XlChildIndex(ctx, -1)) == (Widget) NULL) {
      if (!lw->layout.search.wrap) {
	lw->layout.search.wrap = 1;
	XBell(XtDisplay(w), 50);
	return;
      }
      lw->layout.search.wrap = 0;
      ctx = XlLastChild(w);
    }
    pos = -1;
  }

  lw->layout.search.wrap = 0;
  lw->layout.search.match = strlen(lw->layout.search.text);

  if (w != ctx) {
    TextLeave(w, NULL);
    XlLayoutLostFocus((Widget)lw,XltextYForCurrentPosition((TextWidget)ctx));
    LayoutFocusIn((Widget)lw);
  }
}


/*  */
/***********************************************************************
 * Function: int XlSearching(Widget w)
 *
 * Modifications:
 *	<list mods with name and date>
 */
int XlSearching(w)
  Widget w;
{
  XlLayoutWidget lw = (XlLayoutWidget) XtParent(w);

  if (!XtIsSubclass((Widget) lw, XllayoutWidgetClass))
    return 0;

  return (lw->layout.search.flag != 0);
}

/*  */
/***********************************************************************
 * Function: void XlBeginSearch(Widget w, int flag)
 *
 * Modifications:
 *	<list mods with name and date>
 */
void XlBeginSearch(w, flag)
  Widget w;
  int flag;
{
  XlLayoutWidget lw = (XlLayoutWidget) XtParent(w);

  if (!XtIsSubclass((Widget) lw, XllayoutWidgetClass))
    return;

  if (lw->layout.search.flag != 0) {
    if (lw->layout.search.flag != flag) {
      /* Change of search direction. */
      lw->layout.search.match = 0;
    } else {
      /* Repeat search */
      if (lw->layout.search.old_text) {
	if (lw->layout.search.text)
	  free(lw->layout.search.text);
	lw->layout.search.text = lw->layout.search.old_text;
	lw->layout.search.old_text = NULL;
      }
    }
  } else {
    /* Fresh start of search. */
    if (lw->layout.search.old_text)
      free(lw->layout.search.old_text);
    lw->layout.search.old_text = lw->layout.search.text;
    lw->layout.search.text = NULL;
    lw->layout.search.wrap = 0;
    lw->layout.search.match = 0;
  }

  lw->layout.search.flag = flag;
  if (flag > 0)
    XlSearchForward(w);
  else
    XlSearchBackward(w);
}


/*  */
/***********************************************************************
 * Function: void XlEndSearch(Widget w, int flag)
 *
 * Modifications:
 *	<list mods with name and date>
 */
void XlEndSearch(w, flag)
  Widget w;
  int flag;
{
  XlLayoutWidget lw = (XlLayoutWidget) XtParent(w);

  if (!XtIsSubclass((Widget) lw, XllayoutWidgetClass))
    return;

  lw->layout.search.flag = flag;
}


/*  */
/***********************************************************************
 * Function: void XlAddSearchChar(Widget w, char c)
 *
 * Modifications:
 *	<list mods with name and date>
 */
void XlAddSearchChar(w, c)
  Widget w;
  int c;
{
  XlLayoutWidget lw = (XlLayoutWidget) XtParent(w);
  int p;

  if (!XtIsSubclass((Widget) lw, XllayoutWidgetClass))
    return;

  if (lw->layout.search.old_text) {
    free(lw->layout.search.old_text);
    lw->layout.search.old_text = NULL;
  }
  if (!lw->layout.search.text) {
    lw->layout.search.text = (char *)malloc((ALLOC_T)100);
    (void)memset((char *)lw->layout.search.text, 0, 100);
  }
  if ((p = strlen(lw->layout.search.text)) >= 99)
    return;

  *(lw->layout.search.text + (p++)) = (char) c;
  *(lw->layout.search.text + p) = '\0';

  if (lw->layout.search.flag > 0)
    XlSearchForward(w);
  else if (lw->layout.search.flag < 0)
    XlSearchBackward(w);
}


/*  */
/***********************************************************************
 * Function: void XlDeleteSearchChar(Widget w)
 *
 * Modifications:
 *	<list mods with name and date>
 */
void XlDeleteSearchChar(w)
  Widget w;
{
  XlLayoutWidget lw = (XlLayoutWidget) XtParent(w);

  if (!XtIsSubclass((Widget) lw, XllayoutWidgetClass))
    return;

  if (lw->layout.search.text == 0)
    return;

  if (strlen(lw->layout.search.text) == 0)
    return;

  *(lw->layout.search.text + strlen(lw->layout.search.text) - 1) = '\0';

  if (lw->layout.search.flag > 0)
    XlSearchForward(w);
  else if (lw->layout.search.flag < 0)
    XlSearchBackward(w);
}

/*  */
/**********************************************************************
 * Function: void SaveChildren(Widget w)
 *
 * Modifications:
 *      <list mods with name and date>
 */
void XlSaveChildren(w)
  Widget w;
{
  XlLayoutWidget lw = (XlLayoutWidget) XtParent(w);
  int i;

  for (i = 0; i < lw->N_CHILD; i++)
    XlSavetoDB((XlTextWidget)lw->CHILD[i]);
}

/*  */
/**********************************************************************
 * Function: void ClearAllSelections(Widget w)
 *
 * Modifications:
 *      <list mods with name and date>
 */
void ClearAllSelections(w)
  Widget w;
{
  XlLayoutWidget lw = (XlLayoutWidget) XtParent(w);
  int i;

  for (i = 0; i < lw->N_CHILD; i++)
    if ((TextWidget)lw->CHILD[i] != (TextWidget)w)
      XawTextUnsetSelection((Widget)lw->CHILD[i]);
}

/*  */
/**********************************************************************
 * Function: void ClearNonEdit(Widget w)
 *
 * This is called from markbound, which has an unbound reference
 * structure, so everything should be editable.
 *
 * Modifications:
 *      <list mods with name and date>
 */
void ClearNonEdit(w)
  Widget w;
{
  XlLayoutWidget lw = (XlLayoutWidget)w;
  int i;

  for (i = 0; i < lw->N_CHILD; i++)
    XtVaSetValues((Widget)lw->CHILD[i], XtNeditType, XawtextEdit, NULL);
}
