#define DEBUG
/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME:         chng_prop.c
 *
 * SCCSINFO:            @(#)chng_prop.c	1.15 6/6/94
 *
 * ORIGINAL AUTHOR(S):  Michael Jansson, 1992-12-02
 *
 * MODIFICATIONS:
 *      1993-11-02 Martin Sjlin - whichref is moved to aimsubr.c and 
 *                 changed external section accordingly.
 *      1994-05-02 Martin Sjlin - uncomment objdiff function and 
 *                 related routine to enable basic diff.
 *      <list mods with name and date>
 *
 * DESCRIPTION:
 * 	This file contains the functions that handle explicit
 * 	propagation of changes into reference structures.
 *	Note the difference between this type of propagation
 *	and that which happens because working versions
 *	shares informations.
 *
 *********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************
 */

/******************************************************************************
 * INCLUDES:
 ******************************************************************************/
#include "config.h"	/* includes system dependent includes */
#include "liblincks.h" 
#include "aimtypes.h"
#include "aimstredit.h"

/*****************************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *****************************************************************************/
#include "f_chng_prop.h"

/*****************************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *****************************************************************************/
/* None */

/*****************************************************************************
 * EXTERNAL DATA USED IN THIS MODULE:
 *****************************************************************************/
extern infonode *currentinfonode;
extern infonode *previousinfonode;
extern int suppress;		/* whether certain output is supressed */

/*****************************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *****************************************************************************/
#include "f_aimbuildref.h"
#include "f_aimcmdstream.h"
#include "f_aimcursor.h"
#include "f_aimstruct.h"
#include "f_aimsubr.h"
#include "f_xstuff.h"

/* libshared */
extern char *strdup( /* char *incoming */ );

/*****************************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *****************************************************************************/

/* -CONSTANTS- */
#define REPLACE_OP	0
#define INSERT_OP	1
#define DELETE_OP	2
#define MOVE_OP		3
#define NUM_OP		4

#define ERR_NOMEM	-30

#define STRUCTURECHANGE 1
#define CONTENTCHANGE 	2
#define HIDDENCHANGE 	3

#define NOVERSIONS	"Select the source version first!"
#define NOMEMORY	"FATAL ERROR: Out of memory!"
#define CMPFAILED	"FATAL ERROR: Failed to compute change between versions."
#define PROPFAILED	"FATAL ERROR: Propagation failed.\n"

/* -TYPES- */
struct StructureChange {
  struct StructureChange *next;
  int op;
  infonode *comp;
  infonode *nextcomp;
  identry  *newhead;
};

struct ContentChange {
  struct ContentChange *next;
  infonode *oldval;
  infonode *newval;
};

struct HContentChange {
  struct HContentChange *next;
  label oldval;
  label newval;
  attrval val;
  char *fld;
  char *grp;
  int index;
  int op;
};

struct DiffPart {
  struct HContentChange *hcontents;
  struct ContentChange *contents;
  struct StructureChange *structures;
};

typedef struct DiffPart ObjDiffRec;

union Change {
  struct HContentChange *hidden;
  struct ContentChange *content;
  struct StructureChange *structure;
};

struct Conflicts {
  struct Conflicts *next;
  int imptype;
  int fixtype;
  union Change imp;
  union Change fix;
};

struct transients {
  label bound;
  label trans;
};

struct trans_tbl {
  int maxtrans;
  int cnt;
  struct transients *trans;
};

struct GrpArg {
  ObjDiffRec *diff;
  label *src;
  label *tgt;
};

struct AttrArg {
  struct GrpArg *grp;
  char *name;
};



/* -MACROS- */
#define lblcmp(l1, l2)	(((l1)->vs != (l2)->vs) ? ((l1)->vs - (l2)->vs) : ((l1)->inst - (l2)->inst))
#define nullstrcmp(s1, s2)	!((s1==NULL && s2==NULL) || !strcmp(s1, s2))

/*****************************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *****************************************************************************/
#ifdef EXPERIMENT
static void ApplyChange P_(( ObjDiffRec *diff, reference_structure *tgt ));
static ObjDiffRec *HandleConflicts P_(( ObjDiffRec *fix, ObjDiffRec *imp ));
static errcode PropagateChange P_(( ObjDiffRec *fixchg, reference_structure
                              *fix, reference_structure *src,
                              reference_structure *tgt ));
static void FreeConflicts P_(( struct Conflicts *croot ));
static infonode *GetCorrComp P_(( reference_structure *tgt, infonode
                                   *comp ));
static struct Conflicts *MarkConflicts P_(( ObjDiffRec *fix, ObjDiffRec
                                             *imp ));
static ObjDiffRec *ResolveConflicts P_(( struct Conflicts *conf,
                                          ObjDiffRec *fix, ObjDiffRec
                                          *imp ));

static label *ResolveTransientVersion P_(( label *obj, struct trans_tbl
                                            *transtbl ));
static boolean SameComp P_(( infonode *p1, infonode *p2 ));
#endif /* EXPERIMENT */
static errcode CheckChange P_(( ObjDiffRec *diff, infonode *p1, infonode
                                 *p2 ));
static errcode CheckDelete P_(( ObjDiffRec *diff, infonode *p1, infonode
                                 *p2 ));
static errcode CheckInsert P_(( ObjDiffRec *diff, infonode *p1, infonode
                                 *p2, identry *src, identry *tgt ));
static errcode CheckMove P_(( ObjDiffRec *diff, infonode *p1, infonode
                               *p2 ));
static errcode CheckReplace P_(( ObjDiffRec *diff, infonode *p1,
                                  infonode *p2 ));
static infonode *GetCorrespondingPart P_(( ObjDiffRec *diff, identry
                                            *src, identry *tgt, infonode
                                            *p1 ));
static boolean IsUpdated P_(( ObjDiffRec *diff, infonode *part ));
static errcode MallocDP P_(( ObjDiffRec *diff, int op, infonode *oldval,
                              infonode *newval, identry *tag ));
static errcode MallocHDP P_(( ObjDiffRec *diff, label *oldval, label
                               *newval, char *grp, char *fld, int index,
                               int op ));
static int MatchAttrGroup P_(( struct GrpArg *grp, char *grpname, int
                                cnt, int len ));
static int MatchAttrVal P_(( struct AttrArg *arg, char *attrname, int
                              cnt, int len ));
static errcode MatchOneLevelParts P_(( ObjDiffRec *diff, identry *src,
                                        identry *tgt, int (*callback) () ));
static errcode MatchOneObjectParts P_(( ObjDiffRec *diff, label *src,
                                         label *tgt ));
static errcode NextLevel P_(( ObjDiffRec *diff, identry *src, identry
                               *tgt ));
static errcode NextLevelH P_(( ObjDiffRec *diff, identry *src, identry
                                *tgt ));
static errcode ComputeChange P_(( ObjDiffRec *diff, identry *src, 
				 identry *tgt ));
static infonode *ResolveParent P_(( infonode *p ));
static boolean SamePart P_(( infonode *p1, infonode *p2 ));
static void SwapHiddenInsertDelete P_(( ObjDiffRec *diff ));

#ifdef DEBUG
static void TraceChangeFully P_(( int ind, ObjDiffRec *diff ));
static void TraceContentChange P_(( struct ContentChange *ch ));
static void TraceHContentChange P_(( struct HContentChange *hch ));
static void TraceStructureChange P_(( struct StructureChange *sh ));
#endif /* DEBUG */

#ifdef EXPERIMENT
static void TraceChangeSummary P_(( ObjDiffRec *diff ));
#endif /* EXPERIMENT */

static void MarkChangeFully P_((ObjDiffRec *diff));
static void MarkContentChange P_((struct ContentChange *ch));
static void MarkHContentChange P_((struct HContentChange *hch));
static void MarkStructureChange P_((struct StructureChange *sh));

/*****************************************************************************
 * INTERNAL (STATIC) DATA:
 *****************************************************************************/
#ifdef EXPERIMENT
static char *op_strings[NUM_OP] = {"replaced", "inserted", "deleted", "moved"};
static char *op_strcurr[NUM_OP] = {"replace", "insert", "delete", "move"};
#endif /* EXPERIMENT */

static char *op_strings2[NUM_OP] = {"REPLACE", "INSERT", "DELETE", "MOVE"};

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static boolean IsUpdated(ObjDiffRec *diff, infonode *part)
 *
 * Determine whether a component "part" of an object has been affected
 * by the represented change "diff".
 *
 * Modifications:
 *      <list mods with name and date>
 */
static boolean IsUpdated(diff, part)
  ObjDiffRec *diff;
  infonode *part;
{
  struct StructureChange *sh;

  if (!part)
    return FALSE;

  /* Look for a matching part in a structure sub-op. */
  for (sh=diff->structures; sh; sh=sh->next)
    if (sh->comp==part)
      return TRUE;

  return FALSE;
}

#ifdef EXPERIMENT
/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static infonode *ResolveTransientVersion()
 *
 * Locate the current transient version of an object.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static label *ResolveTransientVersion(obj, transtbl)
  label *obj;
  struct trans_tbl *transtbl;
{
  attrval av;
  label trans;
  int i;

  if (BOUND(obj) && !ISTRANSIENT(obj)) {
    for (i=0; i<transtbl->cnt && 
	 lblcmp(&(transtbl->trans[i].bound), obj); i++);

    if (i==transtbl->cnt) {		/* Add another mapping. */

      /* Store the current version first if it is transient too. */
      (void)GC_GETCURRENT(copylbl(&trans, obj));
      if (ISTRANSIENT(&trans))
	(void)SO_STOREOBJ(&trans);

      /* Create the transient version by faking an update. */
      av.attsize = 0;
      av.attvalue = NULL;
      (void)SA_SETATTR(obj, SYSTEM_OWNER, &av);

      /* Get the transient version. */
      (void)GC_GETCURRENT(copylbl(&trans, obj));

      /* Record the mapping. */
      (void)copylbl(&(transtbl->trans[transtbl->cnt].bound), obj);
      (void)copylbl(&(transtbl->trans[transtbl->cnt].trans), &trans);

      /* Check for overflow. */
      if (++(transtbl->cnt)==transtbl->maxtrans) {
	if (!(transtbl = (struct trans_tbl *)realloc((FREEPTR*) transtbl,(ALLOC_T)
						     transtbl->maxtrans+20))) {
	  (void)fprintf(stderr, NOMEMORY);
	  return NULL;
	}
	transtbl->maxtrans += 20;
      }

      obj = &(transtbl->trans[transtbl->cnt-1].trans);

    } else {			/* Map it to a transient. */
      obj = &(transtbl->trans[i].trans);
    }
  }

  return obj;
}
#endif /* EXPERIMENT */

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static infonode *ResolveParent(infonode *p)
 *
 * Locate the data carrying parent of a "SAMEASPARENT" node.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static infonode *ResolveParent(p)
  infonode *p;
{
  while (p && p->index==SAMEASPARENT)
    p=p->parent;

  return p;
}


/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static errcode MallocHDP
 *
 * Allocate and record a sub-operation of a hidden part.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static errcode MallocHDP(diff, oldval, newval, grp, fld, index, op)
  ObjDiffRec *diff;
  label *oldval;	/* The value before the change */
  label *newval;	/* The value after the change. */
  char *grp;		/* Name of group */
  char *fld;		/* Name of field. */
  int index;		/* Part of a node. */
  int op;		/* Type of operation. */
{
  struct ContentChange *ch;
  struct HContentChange *hch;

  /* Make sure that we have not recorded the change already. */
  for (ch=diff->contents; ch; ch=ch->next)
    if (!lblcmp(&ch->oldval->obj, oldval) && ch->oldval->index==index &&
	!nullstrcmp(ch->oldval->group, grp) &&
	!nullstrcmp(ch->oldval->field, fld))
      return NO_ERROR;
  for (hch=diff->hcontents; hch; hch=hch->next)
    if (hch->index==index && !lblcmp(&(hch->oldval), oldval) &&
	!nullstrcmp(hch->grp, grp) && !nullstrcmp(hch->fld, fld))
      return NO_ERROR;

  if (!(hch = (struct HContentChange *)malloc(sizeof(*hch)))) {
    (void)fputs(NOMEMORY, stderr);
    return ERR_NOMEM;
  }

  hch->op = op;
  hch->index=index;
  hch->grp = (grp ? strdup(grp) : NULL);
  hch->fld = (fld ? strdup(fld) : NULL);
  (void)copylbl(&hch->oldval, oldval);
  (void)copylbl(&hch->newval, newval);
  hch->next = diff->hcontents;
  diff->hcontents = hch;
  return NO_ERROR;
}



/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static errcode MallocDP(diff, op, oldval, newval, tag)
 *
 * Allocate and record another sub-operation of a change.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static errcode MallocDP(diff, op, oldval, newval, tag)
  ObjDiffRec *diff;
  int op;
  infonode *oldval;	/* The value before the change */
  infonode *newval;	/* The value after the change. */
  identry *tag;
{
  struct ContentChange *ch;
  struct StructureChange *sh;

  switch (op) {
  case REPLACE_OP:
    /* Create a new cell. */
    if (!(ch = (struct ContentChange *)malloc(sizeof(*ch))))
      return FALSE;

    /* Initiate and link it in. */
    ch->oldval = oldval;
    ch->newval = newval;
    ch->next = diff->contents;
    diff->contents = ch;
    break;

  case MOVE_OP:
  case INSERT_OP:
  case DELETE_OP:
    /* Create a new cell. */
    if (!(sh = (struct StructureChange *)malloc(sizeof(*sh))))
      return FALSE;

    /* Initiate and link it in. */
    sh->comp =  oldval;
    sh->nextcomp = newval;
    sh->newhead = tag;
    sh->op = op;
    sh->next = diff->structures;
    diff->structures = sh;
    break;
  }
  return NO_ERROR;
}


/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static boolean SamePart(infonode *p1, infonode *p2)
 *
 * Compare two components in an object and determine if they
 * contain the same part. Note the difficult cases when the
 * value of the part is a also its identity, such as for attribute
 * field tags.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static boolean SamePart(p1, p2)
  infonode *p1;
  infonode *p2;
{
  boolean issame = FALSE;

  if (p1==NULL || p2==NULL)
    return FALSE;

  switch (p1->index) {
  case LINKITEM:		/* Match the ->lbl field */
    if ((p2->index == LINKITEM) &&
	(p1->lbl.vs == p2->lbl.vs))
      issame = TRUE;
    break;

  case ATTRVALUE:
    if ((p2->index == ATTRVALUE) &&
	p1->obj.vs == p2->obj.vs &&
	!strcmp(p2->group, p1->group) &&
	!strcmp(p2->field, p1->field))
      issame = TRUE;
    break;

  case INTERNAL:
  case ATTRGROUPTAG:
  case LINKGROUPTAG:
    if ((p2->index == p1->index) &&
	p1->obj.vs == p2->obj.vs &&
	p1->val.attsize == p2->val.attsize &&
	!memcmp(p1->val.attvalue, p2->val.attvalue, p2->val.attsize))
      issame = TRUE;
    break;

  case ATTRFIELDTAG:
  case LINKFIELDTAG:
    if ((p2->index == p1->index) &&
	p1->obj.vs == p2->obj.vs &&
	!strcmp(p1->group, p2->group) &&
	p1->val.attsize == p2->val.attsize &&
	!memcmp(p1->val.attvalue, p2->val.attvalue, p2->val.attsize))
      issame = TRUE;
    break;

  case IMAGE:
    if ((p2->index == p1->index) &&
	p1->obj.vs == p2->obj.vs)
      issame = TRUE;
    break;

  case SAMEASPARENT:
    issame = FALSE;
    break;
  }

  return issame;
}

#ifdef EXPERIMENT
/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static boolean SameComp(infonode *p1, infonode *p2)
 *
 * Compare two components in an object and determine if they
 * are singular components and located at the same position with 
 * respect to the root of two reference structures.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static boolean SameComp(p1, p2)
  infonode *p1;
  infonode *p2;
{
  boolean issame = FALSE;
  char path1[256];
  char path2[256];
  cursor c;

  if (p1==NULL || p2==NULL)
    return FALSE;

  /* Make sure that they are singular components. */
  if (p1->next==NULL && p1->head->value==p1 &&
      p2->next==NULL && p2->head->value==p2) {

      /* Compute the path for the two components */
      c = targetcursor(p1);
      (void)sprintfcursor(path1, c, sizeof(path1));
      (void)freecursor(c);
      c = targetcursor(p2);
      (void)sprintfcursor(path2, c, sizeof(path2));

      /* Compare the paths. */
      if (!strcmp(path1, path2))
	issame = TRUE;
  }

  return issame;
}
#endif /* EXPERIMENT */

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static infonode *GetCorrespondingPart(diff, src, tgt, p1)
 *
 * Locate a component at a level of a reference structure, given
 * a corresponding level in another reference structure (another
 * version of the same object). The components must contain the
 * same part, e.g. same object image (though maybe a different value).
 *
 * Modifications:
 *      <list mods with name and date>
 */
static infonode *GetCorrespondingPart(diff, src, tgt, p1)
  ObjDiffRec *diff;
  identry *src;
  identry *tgt;
  infonode *p1;
{
  infonode *p2 = NULL;
  infonode *parent = NULL;

  p1 = ResolveParent(p1);
  parent = ResolveParent(tgt->value);

  for (p2 = tgt->value; p2; p2=p2->next)
    if ((p2->index==SAMEASPARENT && SamePart(p1, parent)) || SamePart(p1, p2))
      break;

  /* None plural values must match. */
  if (p2==NULL &&
      tgt->value->next==NULL && 
      p1->head->value==p1 && 
      p1->next==NULL)
    p2 = tgt->value;

  return p2;
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static errcode CheckReplace(diff, p1, p2)
 *
 * Compare two versions of a components  decide if it has changed.
 * 
 * Modifications:
 *      <list mods with name and date>
 */
static errcode CheckReplace(diff, p1, p2)
  ObjDiffRec *diff;
  infonode *p1;
  infonode *p2;
{
  errcode s = NO_ERROR;

  /* Log a replace. */
  if (p2 &&
      p1->subnodes == NULL &&
      p1->val.attsize &&
      p2->val.attsize &&
      strcmp(p1->val.attvalue, p2->val.attvalue)) {
    s = MallocDP(diff, REPLACE_OP, p1, p2, (identry*)NULL);
  }
  return s;
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static errcode CheckInsert(diff, p1, p2, src, tgt)
 *
 * Compare two versions of a component and decide if it has been
 * inserted.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static errcode CheckInsert(diff, p1, p2, src, tgt)
  ObjDiffRec *diff;
  infonode *p1;
  infonode *p2;
  identry *src;
  identry *tgt;
{
  errcode s = NO_ERROR;
  infonode *p;

  /* Log an Insert. */
  if (p2 == NULL) {
    for (p=p1->next;
	 p && (p2=GetCorrespondingPart(diff, src, tgt, p))==NULL;
	 p=p->next);
    s = MallocDP(diff, INSERT_OP, p1, p2, tgt);
  }
  return s;
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static errcode CheckDelete(diff, p1, p2)
 *
 * Compare two versions of a part and decide if it has been deleted.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static errcode CheckDelete(diff, p1, p2)
  ObjDiffRec *diff;
  infonode *p1;
  infonode *p2;
{
  errcode s = NO_ERROR;

  /* Log a delete. */
  if (p2 == NULL) {
    s = MallocDP(diff, DELETE_OP, p1, p1, p1->head);
  }
  return s;
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static errcode CheckMove(diff, p1, p2)
 *
 * Compare two versions of a component and decide if it has been
 * moved within one level of the object.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static errcode CheckMove(diff, p1, p2)
  ObjDiffRec *diff;
  infonode *p1;
  infonode *p2;
{
  errcode s = NO_ERROR;
  infonode *in1, *in2;

  /* A part can only be moved, inserted or deleted once. */
  if (p2 != NULL && !IsUpdated(diff, p1)) {

    /* Check the position of the parts. */
    for (in1 = p1->head->value;
	 in1 && IsUpdated(diff, in1);
	 in1 = in1->next);
    for (in2 = p2->head->value;
	 in2 && IsUpdated(diff, in2);
	 in2 = in2->next);

    /* Skipping the deleted and inserted parts. */
    while (in1 && in2 && in1 != p1 && in2 != p2) {

      /* Move "in1" to the next non-deleted part of the object. */
      do {
	in1 = in1->next;
      } while (in1 && IsUpdated(diff, in1));

      /* Move "in2" to the next non-inserted part of the object. */
      do {
	in2 = in2->next;
      } while (in2 && IsUpdated(diff, in2));
    }

    /* Log a move. */
    if (in1 != p1 || in2 != p2) {
      s = MallocDP(diff, MOVE_OP, p1, p2->next, p2->head);
    }
  }
  return s;
}


/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static errcode CheckChange(diff, p1, p2)
 *
 * Compute the difference between the sub-levels of two versions
 * of a component of an object.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static errcode CheckChange(diff, p1, p2)
  ObjDiffRec *diff;
  infonode *p1;
  infonode *p2;
{
  errcode s = NO_ERROR;
  identry *src;
  identry *tgt;

  if (p1->subnodes && p2 && p2->subnodes) {

    /* Traverse the subnodes. */
    src = p1->subnodes;
    tgt = p2->subnodes;
    do {
      s = NextLevel(diff, src, tgt);
      src = src->next;
      tgt = tgt->next;
    } while (src && tgt && s == NO_ERROR);
  }
  return s;
}


/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static int MatchAttrVal(arg, attrname, cnt, len)
 *
 * Match an attribute in on object with the same attribute in another
 * object. Record a DELETE_OP or a REPLACE_OP is the the attributes
 * don't match.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int MatchAttrVal(arg, attrname, cnt, len)
  struct AttrArg *arg;
  char *attrname;
  int cnt;
  int len;
{
  if (cnt) {
    attrval val1;
    attrval val2;
    errcode ec1;
    errcode ec2;

    val1.attsize = 0; val2.attsize = 0;
    ec1 = GA_GETATTR(arg->grp->src, arg->name, attrname, &val1);
    ec2 = GA_GETATTR(arg->grp->tgt, arg->name, attrname, &val2);
    if (ec1==SUCCESS && ec2!=SUCCESS)
      if (MallocHDP(arg->grp->diff, arg->grp->src, arg->grp->tgt,
		    arg->name,  attrname, 
		    ATTRVALUE, DELETE_OP)!=SUCCESS)
	return ERR_NOMEM;
    else if (ec1==SUCCESS && ec2==SUCCESS &&
	     (val1.attsize!=val2.attsize || memcmp(val1.attvalue,
						   val2.attvalue,
						   val1.attsize)))
      if (MallocHDP(arg->grp->diff, arg->grp->src, arg->grp->tgt,
		    arg->name,  attrname, 
		    ATTRVALUE, REPLACE_OP)!=SUCCESS)
	return ERR_NOMEM;

    if (val1.attsize) free((FREEPTR *) val1.attvalue);
    if (val2.attsize) free((FREEPTR *) val2.attvalue);
  }
  return SUCCESS;
}



/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static int MatchAttrGroup(grp, grpname, cnt, len)
 *
 * A (*mapfn)() function that gets called for each attribute group
 * of an object. Calls MatchAttrVal in turn to compare all attributes
 * in the groups.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int MatchAttrGroup(grp, grpname, cnt, len)
  struct GrpArg *grp;
  char *grpname;
  int cnt;
  int len;
{
  struct AttrArg attr;

  if (cnt) {

    attr.name = grpname;
    attr.grp = grp;
    (void)GAN_GETATTRNAMES(grp->src, grpname, MatchAttrVal, (char *)&attr);

  }
  return SUCCESS;
}


/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void SwapHiddenInsertDelete(diff)
 *
 * Swap the recorded Insert/Delete sub-ops. 
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void SwapHiddenInsertDelete(diff)
  ObjDiffRec *diff;
{
  struct HContentChange *hch;
  label lbl;
  
  for (hch=diff->hcontents; hch; hch=hch->next) {
    if (hch->op==INSERT_OP)
      hch->op = DELETE_OP;
    else if (hch->op==DELETE_OP)
      hch->op = INSERT_OP;
    (void)copylbl(&lbl, &hch->oldval);
    (void)copylbl(&hch->oldval, &hch->newval);
    (void)copylbl(&hch->newval, &lbl);
  }
}


/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static errcode MatchOneObjectParts(diff, src, tgt)
 *
 * Compute the difference w.r.t. a specific sub-operation (given
 * as a callback function) of all components of two nodes.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static errcode MatchOneObjectParts(diff, src, tgt)
  ObjDiffRec *diff;
  label *src;
  label *tgt;
{
  attrval i1, i2;
  struct GrpArg arg;

  /* Match Attributes. */
  arg.diff = diff;
  arg.tgt = src;
  arg.src = tgt;
  (void)GAGN_GETATTRGROUPNAMES(tgt, MatchAttrGroup, (char *)&arg);

  /* Match Image */
  i1.attsize = 0; i2.attsize = 0;
  (void)GI_GETIMAGE(src, &i1);
  (void)GI_GETIMAGE(tgt, &i2);
  if (i1.attvalue && i2.attvalue && 
      (i1.attsize!=i2.attsize ||
       memcmp(i1.attvalue, i2.attvalue, i1.attsize))) {
    if (MallocHDP(diff, src, tgt, (char *)NULL, (char *)NULL, IMAGE, REPLACE_OP))
      return ERR_NOMEM;
  }
  if (i1.attsize) free((FREEPTR *) i1.attvalue);
  if (i2.attsize) free((FREEPTR *) i2.attvalue);
  return SUCCESS;
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static errcode MatchOneLevelParts(diff, src, tgt, callback)
 *
 * Compute the difference w.r.t. a specific sub-operation (given
 * as a callback function) of the components at a level of an object.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static errcode MatchOneLevelParts(diff, src, tgt, callback)
  ObjDiffRec *diff;
  identry *src;
  identry *tgt;
  int (*callback) ();
{
  infonode *p1;
  infonode *p2;
  errcode s = NO_ERROR;

  /* Traverse the list of infonodes and compare the values. */
  for (p1 = src->value; p1 && s == NO_ERROR; p1 = p1->next) {

    /* Don't match placeholders. */
    if (p1->iflg&NONEXIST_MSK)
      continue;

    p2 = GetCorrespondingPart(diff, src, tgt, p1);
    s = callback(diff, p1, p2, src, tgt);
  }

  return s;
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static errcode NextLevel(diff, src, tgt)
 *
 * Compare the components at a level of an object w.r.t. the
 * sub-operations REPLACE, INSERT, DELETE and MOVE in that
 * order.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static errcode NextLevel(diff, src, tgt)
  ObjDiffRec *diff;
  identry *src;
  identry *tgt;
{
  errcode s = NO_ERROR;

  /* REPLACE */
  s = MatchOneLevelParts(diff, src, tgt, CheckReplace);

  /* INSERT */
  if (s == NO_ERROR)
    s = MatchOneLevelParts(diff, tgt, src, CheckInsert);

  /* DELETE */
  if (s == NO_ERROR)
    s = MatchOneLevelParts(diff, src, tgt, CheckDelete);

  /* MOVE */
  if (s == NO_ERROR)
    s = MatchOneLevelParts(diff, src, tgt, CheckMove);

  /* Next level */
  if (s == NO_ERROR)
    s = MatchOneLevelParts(diff, src, tgt, CheckChange);

  return s;
}

/**********************************************************************
 * Function: static errcode NextLevelH(diff, src, tgt)
 *
 * Compare the hidden attributes at a level of an object w.r.t. the
 * sub-operations REPLACE and DELETE .
 *
 * Modifications:
 *      <list mods with name and date>
 */
static errcode NextLevelH(diff, src, tgt)
  ObjDiffRec *diff;
  identry *src;
  identry *tgt;
{
  errcode s = NO_ERROR;
  infonode *p1, *p2;
  identry *nsrc;
  identry *ntgt;

  /* HIDDEN REPLACE */
  s = MatchOneObjectParts(diff, &src->value->obj, &tgt->value->obj);

  /* Next level */
  p1 = src->value;

  /* Don't match placeholders. */
  if (p1->iflg&NONEXIST_MSK)
    return s;

  p2 = GetCorrespondingPart(diff, src, tgt, p1);
  if (p1->subnodes && p2 && p2->subnodes) {

    /* Traverse the subnodes. */
    nsrc = p1->subnodes;
    ntgt = p2->subnodes;
    do {
      s = NextLevelH(diff, nsrc, ntgt);
      nsrc = nsrc->next;
      ntgt = ntgt->next;
    } while (nsrc && ntgt && s == NO_ERROR);
  }

  return s;
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static errcode ComputeChange(diff, src, tgt)
 *
 * Compute a representation of the difference between a version "src"
 * of an object and another version "tgt".
 *
 * Modifications:
 *      <list mods with name and date>
 */
static errcode ComputeChange(diff, src, tgt)
  ObjDiffRec *diff;
  identry *src;
  identry *tgt;
{
  errcode s = NO_ERROR;
  infonode *p;
  struct StructureChange *sh, *sh1, *sh2;

  /* Scan through the versions. */
  if ((s=NextLevel(diff, src, tgt)) != NO_ERROR) {
    return s;
  }

  /* Scan through the whole object and look for hidden attributes. */
  if ((s=NextLevelH(diff, tgt, src)) != NO_ERROR) {
    return s;
  }
  SwapHiddenInsertDelete(diff);
  if ((s=NextLevelH(diff, src, tgt)) != NO_ERROR) {
    return s;
  }
  SwapHiddenInsertDelete(diff);

  /* Remove fake moves. */
  for (sh=diff->structures; sh; sh=sh->next) {
    if (sh->op==MOVE_OP) {
      for (p=sh->comp;
	   p && IsUpdated(diff, p) && p!=sh->nextcomp;
	   p=p->next);
      if (p==sh->nextcomp)
	sh->comp = NULL;
      for (p=sh->nextcomp;
	   p && IsUpdated(diff, p) && p!=sh->comp;
	   p=p->next);
      if (p==sh->comp)
	sh->comp = NULL;
    }
  }

  /* Mapping pairs of Insert/Delete to Move/Replace */
  for (sh1 = diff->structures; sh1; sh1 = sh1->next) {
    if (sh1->comp && sh1->op==DELETE_OP) {
      for (sh2 = diff->structures; sh2; sh2 = sh2->next) {
	if (sh2->comp &&
	    sh2->op==INSERT_OP &&
	    SamePart(sh1->comp, sh2->comp)) {

	  s = MallocDP(diff, MOVE_OP, sh1->comp, sh2->nextcomp, sh2->newhead);

	  if (sh1->comp->val.attsize &&
	      sh2->comp->val.attsize &&
	      strcmp(sh1->comp->val.attvalue, sh2->comp->val.attvalue)) {
	    s = MallocDP(diff, REPLACE_OP, sh1->comp, sh2->comp, (identry*)NULL);
	  }
	  
	  sh1->comp = sh2->comp = NULL;
	  break;
	}
      }
    }
  }

  return s;
}

#ifdef EXPERIMENT
/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static errcode PropagateChange(FOUR PARAMETERS)
 * Parameters:
 *	ObjDiffRec *fixchg
 *	reference_structure *fix
 *	reference_structure *src
 *	reference_structure *tgt
 *
 * Modifications:
 *      <list mods with name and date>
 */
static errcode PropagateChange(fixchg, fix, src, tgt)
  ObjDiffRec *fixchg;
  reference_structure *fix;
  reference_structure *src;
  reference_structure *tgt;
{
  errcode s = NO_ERROR;
  char answer[256];
  ObjDiffRec *cng, imp;
  label dummy;
  int pos, rc;

  /* Compute the prior change to the object */
  imp.contents = NULL; imp.structures = NULL; imp.hcontents = NULL;
  if ((s=ComputeChange(&imp, src->target, tgt->target))!=NO_ERROR)
    return s;

  (void)printf("[Matching against ");
  TraceChangeSummary(&imp);
  (void)printf("]\n");

  (void)printf("\nDo you want to view the improvement and the fix? ");
  if (scanf("%s", answer)==1 && toupper(answer[0])=='Y') {
    (void)printf("*** PRIOR CHANGE (IMPROVEMENT) ***\n");
    TraceChangeFully(4, &imp);
    (void)printf("\n*** PENDING CHANGE (FIX) ***\n");
    TraceChangeFully(4, fixchg);
    (void)printf("\n\n");
  }

  /* Check and resolve conflicts. */
  cng = HandleConflicts(fixchg, &imp);

  /* Apply the changes. */
  ApplyChange(cng, tgt);

  /* Fix the parent history for the object. */
  pos = 1;
  (void)GC_GETCURRENT(&tgt->bt);
  while (GLI_GETLINKITEM(&tgt->bt, SYSTEM_PARENT, pos, &dummy)==NO_ERROR)
    pos++;
  rc = LINK(&tgt->bt, SYSTEM_PARENT, &fix->bt, pos);
  (void)printf("Parent history: %d->%d (%d): %d\n",
	       tgt->bt.inst,  fix->bt.inst, pos, rc);

  return s;
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static ObjDiffRec *HandleConflicts(fix, imp)
 *
 * Compute and resolve the conflicts of two changes.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ObjDiffRec *HandleConflicts(fix, imp)
  ObjDiffRec *fix;
  ObjDiffRec *imp;
{
  ObjDiffRec *cng;
  struct Conflicts *conf;

  if ((conf = MarkConflicts(fix, imp))!=NULL)
    cng = ResolveConflicts(conf, fix, imp);
  else
    cng = fix;

  /* Clean up */
  FreeConflicts(conf);

  return cng;
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static ObjDiffRec *ResolveConflicts(conf, fix, imp)
 *
 * Resolve conflicts.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ObjDiffRec *ResolveConflicts(conf, fix, imp)
  struct Conflicts *conf;
  ObjDiffRec *fix;
  ObjDiffRec *imp;
{
  cursor c;
  char answer[256];
  char buf[256];
  struct Conflicts *cp;
  struct ContentChange *ch;
  struct HContentChange *hch;
  struct StructureChange *sh;

  (void)printf("*CONFLICTS*\n");
  for (cp=conf; cp; cp=cp->next) {
    if (cp->imptype==CONTENTCHANGE) {
      c = targetcursor(cp->imp.content->oldval);
      (void)sprintfcursor(buf, c, 256);
      (void)printf("\tPart already updated:\n\t\t\"%5.5s\" at [%s]",
	     cp->imp.content->oldval->val.attvalue,
	     buf);
      (void)freecursor(c);

      /* Use this sub-op in the change? */
      (void)printf(" [y/N] ? ");
      while (scanf("%s", answer)!=1);
      switch (toupper(answer[0])) {
      case 'y':
      case 'Y':
	break;
      default:
	cp->fix.content->oldval = NULL;
	break;
      }

    } else if (cp->imptype==STRUCTURECHANGE) {
      c = targetcursor(cp->imp.structure->comp);
      (void)sprintfcursor(buf, c, 256);
      (void)printf("\tA %s-op is pending for a part which is already %s:\n\t\t\"%5.5s\" at [%s]",
	     (cp->fixtype==STRUCTURECHANGE) ? 
		   op_strcurr[cp->fix.structure->op] : op_strcurr[REPLACE_OP], 
	     op_strings[cp->imp.structure->op], 
	     cp->imp.structure->comp->val.attvalue, 
	     buf);
      (void)freecursor(c);

      /* Use improvement, fix or none? */
      (void)printf(" [y/N] ? ");
      while(scanf("%s", answer)!=1);
      switch (toupper(answer[0])) {
      case 'y':
      case 'Y':
	break;
      default:
	if (cp->fixtype==STRUCTURECHANGE)
	  cp->fix.structure->comp = NULL;
	else
	  cp->fix.content->oldval = NULL;
	break;
      }
    } else if (cp->fixtype==HIDDENCHANGE) {

      if (cp->fix.hidden->index==IMAGE) {
	(void)printf("\tHidden image in object [%d:%d] is already %s",
		     cp->imp.hidden->oldval.vs, cp->imp.hidden->oldval.inst,
		     op_strings[cp->imp.hidden->op]);
      } else {
	(void)printf("\tHidden attribute %s.%s in object [%d:%d] is already %s",
		     cp->imp.hidden->grp, cp->imp.hidden->fld, 
		     cp->imp.hidden->oldval.vs, cp->imp.hidden->oldval.inst,
		     op_strings[cp->imp.hidden->op]);
      }

      /* Use improvement, fix or none? */
      (void)printf(" [y/N] ? ");
      while (scanf("%s", answer)!=1);
      switch (toupper(answer[0])) {
      case 'y':
      case 'Y':
	break;
      default:
	cp->fix.hidden->index = -1;
	break;
      }
      
    }
  }

  (void)printf("New change is:\n");
  for (ch=fix->contents; ch; ch=ch->next)
    if (ch->oldval)
      TraceContentChange(ch);
  for (sh=fix->structures; sh; sh=sh->next)
    if (sh->comp)
      TraceStructureChange(sh);
  for (hch=fix->hcontents; hch; hch=hch->next)
    if (hch->index>=0)
      TraceHContentChange(hch);

  (void)printf("Use new change? ");
  while(scanf("%s", answer)!=1);
  if (answer[0]!='y' && answer[0]!='Y')
    return ResolveConflicts(conf, fix, imp);

  return fix;
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void FreeConflicts(croot)
 *
 * Free memory used to represent a conflict.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void FreeConflicts(croot)
  struct Conflicts *croot;
{
  struct Conflicts *c;

  while (croot) {
    c = croot;
    croot = c->next;
    free((FREEPTR *)c);
  }
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static struct Conflicts *MarkConflicts(fix, imp)
 *
 * Compute the conflicts between two changes.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static struct Conflicts *MarkConflicts(fix, imp)
  ObjDiffRec *fix;
  ObjDiffRec *imp;
{
  struct ContentChange *ch1;
  struct ContentChange *ch2;
  struct HContentChange *hch1;
  struct HContentChange *hch2;
  struct StructureChange *sh1;
  struct StructureChange *sh2;
  struct Conflicts *root = NULL;
  struct Conflicts *c = NULL;

  /* Compare the contents changes. */
  for (ch1=fix->contents; ch1; ch1=ch1->next) {
    /* Comapre with the other contents changes */
    for (ch2=imp->contents; ch2; ch2=ch2->next) {
      if (SamePart(ch1->oldval, ch2->oldval) ||
	  SameComp(ch1->oldval,  ch2->oldval)) {
	if (!(c = (struct Conflicts *)malloc(sizeof(struct Conflicts)))) {
	  (void)fputs(NOMEMORY, stderr);
	  return NULL;
	}
	c->fixtype = CONTENTCHANGE;
	c->imptype = CONTENTCHANGE;
	c->fix.content = ch1;
	c->imp.content = ch2;
	c->next = root;
	root = c;
	continue;
      }
    }
    /* Comapre with the other structure changes */
    for (sh2=imp->structures; sh2; sh2=sh2->next) {
      if (SamePart(ch1->oldval, sh2->comp) ||
	  SameComp(ch1->oldval,  sh2->comp)) {
	if (!(c = (struct Conflicts *)malloc(sizeof(struct Conflicts)))) {
	  (void)fputs("FATAL ERROR: Out of memory in MarkConflicts()", stderr);
	  return NULL;
	}
	c->fixtype = CONTENTCHANGE;
	c->imptype = STRUCTURECHANGE;
	c->fix.content = ch1;
	c->imp.structure = sh2;
	c->next = root;
	root = c;
	continue;
      }
    }
  }

  /* Compare the structure changes. */
  for (sh1=fix->structures; sh1; sh1=sh1->next) {
    for (sh2=imp->structures; sh2; sh2=sh2->next) {
      if (SamePart(sh1->comp, sh2->comp)) {
	if (!(c = (struct Conflicts *)malloc(sizeof(struct Conflicts)))) {
	  (void)fputs("FATAL ERROR: Out of memory in MarkConflicts()", stderr);
	  return NULL;
	}
	c->fixtype = STRUCTURECHANGE;
	c->imptype = STRUCTURECHANGE;
	c->fix.structure = sh1;
	c->imp.structure = sh2;
	c->next = root;
	root = c;
      }
    }
  }

  /* Compare the hidden changes. */
  for (hch1=fix->hcontents; hch1; hch1=hch1->next) {
    for (hch2=imp->hcontents; hch2; hch2=hch2->next) {
      if (hch1->index == hch2->index && !lblcmp(&hch1->oldval, &hch2->oldval) &&
	  !nullstrcmp(hch1->grp, hch2->grp) && !nullstrcmp(hch1->fld, hch2->fld)) {
	if (!(c = (struct Conflicts *)malloc(sizeof(struct Conflicts)))) {
	  (void)fputs(NOMEMORY, stderr);
	  return NULL;
	}
	c->fixtype = HIDDENCHANGE;
	c->fix.hidden = hch1;
	c->imp.hidden = hch2;
	c->next = root;
	root = c;
      }
    }
  }

  return root;
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void TraceChangeSummary(diff)
 *
 * Make a summary trace of a change.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void TraceChangeSummary(diff)
  ObjDiffRec *diff;
{
  struct ContentChange *ch;
  struct HContentChange *hch;
  struct StructureChange *sh;
  int cnt;

  for (ch=diff->contents, cnt=0; ch; ch=ch->next, cnt++);
  (void)printf("%d Updates, ", cnt);

  for (sh=diff->structures, cnt=0; sh; sh=sh->next)
    if (sh->op==INSERT_OP)
      cnt++;
  (void)printf("%d Inserts, ", cnt);

  for (sh=diff->structures, cnt=0; sh; sh=sh->next)
    if (sh->op==DELETE_OP)
      cnt++;
  (void)printf("%d Deletes ", cnt);

  for (sh=diff->structures, cnt=0; sh; sh=sh->next)
    if (sh->op==MOVE_OP)
      cnt++;
  (void)printf("%d Moves", cnt);

  for (hch=diff->hcontents, cnt=0; hch; hch=hch->next, cnt++);
  (void)printf(" and %d hidden changes.", cnt);
}
#endif /* EXPERIMENT */

#ifdef DEBUG
static void TraceHContentChange(hch)
  struct HContentChange *hch;
{
  if (hch->index==IMAGE)
    (void)printf("%s image in object [%d:%d]\n",
		 op_strings2[hch->op], 
		 hch->oldval.vs, hch->oldval.inst);
  else
    (void)printf("%s attribute %s.%s in object [%d:%d]\n",
		 op_strings2[hch->op], hch->grp, hch->fld, 
		 hch->oldval.vs, hch->oldval.inst);
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void TraceChangeFully(ind, diff)
 *
 * Make a summary trace of a change.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void TraceChangeFully(ind, diff)
  int ind;
  ObjDiffRec *diff;
{
  struct ContentChange *ch;
  struct HContentChange *hch;
  struct StructureChange *sh;
  int i, once = 0;

  for (ch = diff->contents; ch; ch = ch->next) {
    for (i=0; i<ind; i++)
      (void)printf(" ");
    TraceContentChange(ch);
  }

  for (sh=diff->structures; sh; sh=sh->next) {
    if (sh->op==INSERT_OP) {
      for (i=0; i<ind; i++)
	(void)printf(" ");
      TraceStructureChange(sh);
    }
  }

  for (sh=diff->structures; sh; sh=sh->next) {
    if (sh->op==DELETE_OP) {
      for (i=0; i<ind; i++)
	(void)printf(" ");
      TraceStructureChange(sh);
    }
  }

  for (sh=diff->structures; sh; sh=sh->next) {
    if (sh->op==MOVE_OP) {
      for (i=0; i<ind; i++)
	(void)printf(" ");
      TraceStructureChange(sh);
    }
  }

  (void)printf("\n");
  for (i=0; i<ind; i++)
    (void)printf(" ");

  for (hch = diff->hcontents; hch; hch = hch->next) {
    if (once++ == 0)
      (void)printf("-- Changes done to non-displayed attributes --\n");
    for (i=0; i<ind; i++)
      (void)printf(" ");
    TraceHContentChange(hch);
  }
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void TraceContentChange(ch)
 *
 * Make a trace of the sub-ops that are contents changes.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void TraceContentChange(ch)
  struct ContentChange *ch;
{
  cursor c = NULL;
  char buf[256];

  c = targetcursor(ch->oldval);
  (void)sprintfcursor(buf, c, 256);
  (void)freecursor(c);
  (void)printf("REPLACE (%10.10s) at [%s]\n\tWITH (%10.10s)\n",
	 ch->oldval->val.attvalue,
	 buf, 
	 ch->newval->val.attvalue);
}


/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void TraceStructureChange(sh)
 *
 * Make a trace of sub-ops that are structure changes.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void TraceStructureChange(sh)
  struct StructureChange *sh;
{
  cursor c = NULL;
  char buf1[256], buf2[256];

  if (!sh->comp)
    return;

  switch (sh->op) {
  case INSERT_OP:
    if (sh->nextcomp) {
      c = targetcursor(sh->nextcomp);
      (void)sprintfcursor(buf1, c, 256);
      (void)freecursor(c);
      (void)printf("INSERT (%10.10s) BEFORE [%s]\n", 
	     sh->comp->val.attvalue,
	     buf1);
    } else {
      c = targetcursor(sh->newhead->value);
      (void)sprintfcursor(buf1, c, 256);
      (void)freecursor(c);
      (void)printf("INSERT (%10.10s) LAST AT [%s]\n", 
	     sh->comp->val.attvalue,
	     buf1);
    }
    break;
  case DELETE_OP:
    c = targetcursor(sh->nextcomp);
    (void)sprintfcursor(buf1, c, 256);
    (void)freecursor(c);
    (void)printf("DELETE (%10.10s) AT [%s]\n", 
	   sh->comp->val.attvalue,
	   buf1);
    break;
  case MOVE_OP:
    c = targetcursor(sh->comp);
    (void)sprintfcursor(buf2, c, 256);
    (void)freecursor(c);
    if (sh->nextcomp) {
      c = targetcursor(sh->nextcomp);
      (void)sprintfcursor(buf1, c, 256);
      (void)freecursor(c);
      (void)printf("MOVE (%10.10s) AT [%s]\n\tBEFORE [%s]\n", 
	     sh->comp->val.attvalue,
	     buf2,
	     buf1);
    } else {
      c = targetcursor(sh->newhead->value);
      (void)sprintfcursor(buf1, c, 256);
      (void)freecursor(c);
      (void)printf("MOVE (%10.10s) AT [%s]\n\tLAST AT [%s]\n", 
	     sh->comp->val.attvalue,
	     buf2, 
	     buf1);
    }
    break;
  }
}
#endif /* DEBUG */
#ifdef EXPERIMENT
/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static infonode *GetCorrComp(tgt, comp)
 *
 * Get corresponding components in order of appearence: 
 *	(1) referencing to the same part, (2) located at 
 *	the same place in the target reference
 *	structure (3) the same component.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static infonode *GetCorrComp(tgt, comp)
  reference_structure *tgt;
  infonode *comp;
{
  cursor part;
  identry *head;
  infonode *inode, *i;

  /* Get the corresponding infonode in the target. */
  part = targetcursor(comp);
  if ((i = findinfonode(tgt->target->value, part))!=NULL) {
    head = i->head;
    for (i=head->value; i && !SamePart(comp, i); i=i->next);
    if (((inode = i)==NULL) &&
	((inode = findinfonode(tgt->target->value, part))==NULL))
      inode = comp;
  } else {
    inode = comp;
  }
  (void)freecursor(part);

  return inode;
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void ApplyChange(diff, tgt)
 *
 * Incorporate a change into a target object.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void ApplyChange(diff, tgt)
  ObjDiffRec *diff;
  reference_structure *tgt;
{
  struct ContentChange *ch;
  struct HContentChange *hch;
  struct StructureChange *sh;
  attrval v;
  infonode *last;
  label *obj;
  label *lbl;
  int index;
  char *group;
  char *field;
  int pos;
  struct trans_tbl transtbl;
  int op;

  /* Handle transient binding tables. */
  if (!(transtbl.trans =
	(struct transients *)malloc((ALLOC_T)100*sizeof(struct transients)))) {
    (void)fputs(NOMEMORY, stderr);
    return;
  }
  transtbl.cnt = 0;
  transtbl.maxtrans = 100;

  /* Hidden attributes */
  (void)printf("[Executing the data base changes]\n");
  while (diff->hcontents) {
    hch = diff->hcontents;
    diff->hcontents = hch->next;

    if (hch->index>=0) {
      obj = &hch->oldval;
      index = hch->index;
      if (hch->op==DELETE_OP)
	op = REMOVEOP;
      else
	op = CHANGEOP;

      v.attsize = 0;
      if (hch->index==IMAGE)
	(void)GI_GETIMAGE(&hch->newval, &v);
      else
	(void)GA_GETATTR(&hch->newval, hch->grp, hch->fld, &v);

      /* Resolve bound un-transient label. */
      if ((obj = ResolveTransientVersion(obj, &transtbl))) {

	(void)printf("[%s]\n", op_strings[hch->op]);
	newlastcmd(op, 
		   obj,
		   index,
		   hch->grp,
		   hch->fld,
		   0, 
		   &v, 
		   (label *)NULL);
      }

      if (v.attsize)
	free((FREEPTR *)v.attvalue);
    }
    /* Flush commands. */
    (void)do_commands();
    free((FREEPTR *)hch);
  }

  /* Content */
  while (diff->contents) {
    ch = diff->contents;
    diff->contents = ch->next;
    if (ch->oldval) {
      last = ResolveParent(GetCorrComp(tgt, ch->oldval));
      obj = &last->obj;
      index = last->index;
      if (index==LINKITEM) {
	obj = &last->lbl;
	index = IMAGE;
      }

      /* Resolve bound un-transient label. */
      if ((obj = ResolveTransientVersion(obj, &transtbl))) {

	(void)printf("\treplace\n");
	newlastcmd(CHANGEOP,
		   obj,
		   index,
		   ch->oldval->group,
		   ch->oldval->field,
		   ch->newval->pos,
		   &ch->newval->val,
		   (label *)NULL);
      }
      /* Flush commands. */
      (void)do_commands();
    }
    free((FREEPTR *)ch);
  }

  /* Structure */
  while (diff->structures) {
    sh = diff->structures;
    diff->structures = sh->next;
    (void)printf("\t%s\n", op_strings[sh->op]);
    if (sh->comp) {

      /* Do operations that results in information being added to the object. */
      if (sh->op==INSERT_OP || sh->op==MOVE_OP) {

	/* Insert part last at a level. */
	if (sh->nextcomp==NULL) {
	  infonode *inode = GetCorrComp(tgt, sh->newhead->value);
	  for (last=inode; last && last->next; last=last->next);
	  if (last->iflg & NONEXIST_MSK) {
	    pos = 1;
	  } else {
	    pos = last->pos+1;
	  }
	  obj = &inode->obj;
	  group = last->group;
	  field = last->field;

	/* Insert part before a given part. */
	} else {
	  infonode *inode = GetCorrComp(tgt, sh->nextcomp);
	  pos = inode->pos;
	  obj = &inode->obj;
	  group = inode->group;
	  field = inode->field;
	}
	lbl = &sh->comp->lbl;
	index = sh->comp->index;

	/* Resolve bound un-transient label. */
	if ((obj = ResolveTransientVersion(obj, &transtbl))) {

	  if (sh->comp->index == LINKITEM) {
	    newlastcmd(LINKOP, obj, index, group,
		       field, pos, (attrval*)NULL, lbl);
	  } else {
	    newlastcmd(CREATEOP, obj, index, group,
		       field, pos, (attrval*)NULL, lbl);
	  }
	}
      }

      /* Resolve bound un-transient label. */
      last = GetCorrComp(tgt, sh->comp);
      obj = ResolveTransientVersion(&last->obj, &transtbl);

      /* Do operations that results in information being removed from the object. */
      if (obj && (sh->op==DELETE_OP || sh->op==MOVE_OP))
	newlastcmd(REMOVEOP,
		   obj,
		   sh->comp->index,
		   sh->comp->group,
		   sh->comp->field,
		   sh->comp->pos, 
		   (attrval*)NULL, 
		   (label*)NULL);
    }
    /* Flush commands. */
    (void)do_commands();

    free((FREEPTR *)sh);
  }

  free((FREEPTR *)transtbl.trans);
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: void propagatecmd(w, client_data, call_data)
 *
 * Execute an explicit propagation.
 *
 * Modifications:
 *      <list mods with name and date>
 */
void propagatecmd(w, client_data, call_data)
  Widget w;
  caddr_t client_data;
  caddr_t call_data;
{
  reference_structure *srcref, *tgtref, *fixref;
  label srcobj, tgtobj, fixobj;
  ObjDiffRec fix;

  if (currentinfonode==NULL) {
    (void)fputs(NOVERSIONS, stderr);
    return;
  }

  if ((srcref = whichref(currentinfonode, (identry *)NULL, 0))==NULL) {
    (void)fputs("[Internal error. Could not locate reference structure.]\n",
		stderr);
    return;
  }

  srcobj.vs = srcref->bt.vs;
  srcobj.inst = srcref->bt.inst;

  (void)printf("\n\n[Entering debug interface for explicit propagation]\n\n");

  do {
    int cnt;
    char buf[256];

    (void)printf("Give id of fix and target version (e.g. 107:140 230:455): ");
    (void)gets(buf);
    cnt = sscanf(buf,  "%d %d %d %d\n", 
		 &fixobj.vs, &fixobj.inst, 
		 &tgtobj.vs, &tgtobj.inst);
    if (cnt==4)
      break;
    if (cnt==0) {
      (void)fprintf(stderr, "[Propagation aborted]\n");
      return;
    }
  } while (TRUE);

  tgtref = buildrefbt(srcref->template->idname, &tgtobj, FALSE, 
		      (reference_structure *)NULL);
  fixref = buildrefbt(srcref->template->idname, &fixobj, FALSE, 
		      (reference_structure *)NULL);

  fix.hcontents = NULL;
  fix.contents = NULL;
  fix.structures = NULL;
  if (ComputeChange(&fix, srcref->target, fixref->target)!=NO_ERROR) {
    (void)fputs(CMPFAILED, stderr);
    return;
  }
  (void)printf("\n[Going to propagate ");
  TraceChangeSummary(&fix);
  (void)printf("]\n");

  if (PropagateChange(&fix, fixref, srcref, tgtref)!=NO_ERROR)
    (void)fprintf(stderr, PROPFAILED);

  /* Clean up */
  free_ref(tgtref);
  free_ref(fixref);

  (void)printf("[Propagation done!]\n");
}
#endif /* EXPERIMENT */


/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: void objdiffcmd(w, client_data, call_data)
 *
 * Compare two versions of the same composite object and
 * mark the differences ...
 *
 * Modifications:
 *      <list mods with name and date>
 */
void objdiffcmd(w, client_data, call_data)
  Widget w;
  caddr_t client_data;
  caddr_t call_data;
{
  ObjDiffRec diff;
  reference_structure *src, *tgt;

  diff.structures = NULL;
  diff.contents = NULL;
  diff.hcontents = NULL;

  if (currentinfonode == NULL || previousinfonode == NULL) {
    if (!suppress)
      popup_message(POPUP_MESS_INFO, "%s\n%s",
                    "You need to select both source and destination compositions",
                    "(windows) before performing a compare operation.");

  } else if (currentinfonode == previousinfonode) {
    if (!suppress)
      popup_message(POPUP_MESS_INFO, "%s\n%s",
		    "Why would you like to compare something with itself!",
                    "You need to select different source and destinations.");

  } else if (!(tgt = whichref(currentinfonode, (identry *)NULL, 0)) ||
	     !(src = whichref(previousinfonode, (identry *)NULL, 0))) {
    popup_message(POPUP_MESS_ERROR, "%s\n%s",
		  "Failed to find either the source or destination",
		  "reference structure for select compositions.");

  } else if (ComputeChange(&diff, src->target, tgt->target)!=NO_ERROR) {
    popup_message(POPUP_MESS_ERROR, "%s","Compare operation failed!");

  } else {
    MarkChangeFully(&diff);
#ifdef DEBUG
    (void)fprintf(stderr,
		  "\n*****Differences between [%d:%d] and [%d:%d]*****\n\n",
		  src->bt.vs, src->bt.inst, tgt->bt.vs, tgt->bt.inst);
    TraceChangeFully(0, &diff);
    (void)printf("Done!\n\n");
#endif /* DEBUG */
  }  /* ComputeChange == NO_ERROR */
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void MarkChangeFully(diff)
 *
 * Marks the changes in the visisble widgets ...
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void MarkChangeFully(diff)
  ObjDiffRec *diff;
{
  struct ContentChange *ch;
  struct HContentChange *hch;
  struct StructureChange *sh;

  /* context changes, visisble */
  while(diff->contents) {
    ch = diff->contents;
    diff->contents = ch->next;
    MarkContentChange(ch);
    free((FREEPTR *) ch);
  }

  /* structural changes */
  while (diff->structures) {
    sh = diff->structures; 
    diff->structures = sh->next;
    MarkStructureChange(sh);
    free((FREEPTR *) sh);
  }  

  /* internal content changes */
  while(diff->hcontents) {
    hch = diff->hcontents; 
    diff->hcontents = hch->next;
    MarkHContentChange(hch);
    if(hch->grp) free((FREEPTR *) hch->grp);
    if(hch->fld) free((FREEPTR *) hch->fld);
    free((FREEPTR *) hch);
  }
}

/*   */
/*ARGSUSED*/
/**********************************************************************
 * Function: void mark_identry(identry *idp, void (*mark)())
 *
 * call mark function on all entities in the tree during
 * traversal ...
 *
 * Modifications:
 *      <list mods with name and date>
 */

void mark_reference(ref, mark)
  reference_structure *ref;
  void (*mark)P_((infonode *np));
{
  for (; ref != NULL; ref = ref -> next) 
    mark_identry(ref->target, mark);
}

void mark_identry(idp, mark)
  identry *idp;
  void (*mark)P_((infonode *np));
{
  for(;idp != NULL; idp = idp->next) 
    mark_infonode(idp->value,mark);
}

void mark_infonode(inp, mark)
  infonode *inp;
  void (*mark)P_((infonode *np));
{
  for(;inp != NULL; inp = inp->next)
    if (inp->subnodes)
      mark_identry(inp->subnodes,mark);
    else
      (*mark)(inp);
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void MarkContentChange(ch)
 *
 * Mark of the sub-ops that are contents changes.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void MarkContentChange(ch)
  struct ContentChange *ch;
{
  /* let us just mark the new value and keep old */
  mark_changed(ch->newval);
}

static void MarkHContentChange(hch)
  struct HContentChange *hch;
{
  if (hch->index==IMAGE)
    (void)printf("%s image in object [%d:%d]\n",
		 op_strings2[hch->op], 
		 hch->oldval.vs, hch->oldval.inst);
  else
    (void)printf("%s attribute %s.%s in object [%d:%d]\n",
		 op_strings2[hch->op], hch->grp, hch->fld, 
		 hch->oldval.vs, hch->oldval.inst);
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void MarkStructureChange(sh)
 *
 * Mark sub-ops that are structure changes.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void MarkStructureChange(sh)
  struct StructureChange *sh;
{
  if (!sh->comp)
    return;

  switch (sh->op) {
  case INSERT_OP:
    mark_inserted(sh->comp);
    if (sh->comp->index == LINKITEM) 
      mark_identry(sh->comp->subnodes,mark_inserted);
    break;
  case DELETE_OP:
    mark_deleted(sh->comp);		/* mark in other object ... */
    if (sh->comp->index == LINKITEM)
      mark_identry(sh->comp->subnodes, mark_deleted);
    break;
  case MOVE_OP:
    mark_deleted(sh->comp);		/* mark old deleted value */
    if (sh->comp->index == LINKITEM)
      mark_identry(sh->comp->subnodes, mark_deleted);

    if (sh->nextcomp) {			/* before ... */
      identry *child = sh->nextcomp->parent->subnodes;
      infonode *inp = NULL;

      /* aargh, need to find infonode *before* sh->nextcomp */
      while (child != NULL && (inp = child->value) != NULL) {

	/* traverse the list of siblings */
	while (inp != NULL && inp->next!=sh->nextcomp)
	  inp=inp->next;

	if (inp != NULL) {
	  mark_inserted(inp);	
	  if (inp->index == LINKITEM)
	    mark_identry(inp->subnodes, mark_inserted);
	  break;
	} 

	/* mm, no hit, take next tag */
	child=child->next;
      } /* child != NULL && child->value != NULL */
    }  /* sh->nextcomp == NULL */
    else {
      /* aargh, need to move to *last* item in chain ... */
      infonode *inp = sh->newhead->value;
      while(inp != NULL && inp->next != NULL)
	inp=inp->next;
      mark_inserted(inp);
      if (inp->index == LINKITEM)
	mark_identry(inp->subnodes, mark_inserted);
    }
    break;
  }
}
