/*
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME: 	netserv.c
 *
 * SCCSINFO:		@(#)netserv.c	1.15 6/6/94
 *
 * ORIGINAL AUTHOR(S):
 *
 * MODIFICATIONS:
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 *	This is the Lincks TCP/IP connection server
 */
/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */

#include <sys/socket.h>			/* MAXHOSTNAMELEN is here on SCO */
#include <sys/wait.h>
#ifdef  HAVE_USLEEP			/* included in config.h ifndef */
#include <sys/time.h>
#endif /* n HAVE_USLEEP */
#ifdef HAVE_GETDTABLESIZE		/* included in config.h ifndef */
#include <sys/resource.h>
#endif /* HAVE_GETDTABLESIZE */
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <netdb.h>			/* for MAXHOSTNAMELEN */

#include "xconfig.h"
#include "lincks.h"
#include "dbcodes.h"
#include "libshared.h"

#ifdef __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif /* n __STDC__ */

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
void main P_(( int argc, char *argv[] ));

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
int debug = 0;			/* Debug flag */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
extern UID VerifyUser P_(( char *username, char *password ));
#include "f_tcp.h"
#include "f_misc.h"

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern char *sys_errlist[];	/* errno.h */
extern int sys_nerr;		/* errno.h */
extern int errno;		/* errno.h */

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
#define LOGIN_TIME_OUT_LIMIT 300/* 5 min */
#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 64
#endif /* MAXHOSTNAMELEN */

#define LogSignal(uid, sig) \
    LogMessage(uid, "netserv: could not register signal %s.  errno = %d", sig, errno)

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static int GetRequest P_(( void ));
static RETSIGTYPE Handler P_(( int sig ));
static void InitServer P_(( void ));
static int InitSignals P_(( void ));
static void StartServer P_(( int socketno ));
static void SubProcess P_(( int socketno ));
static int WSChars P_(( int socketno, char *buff, int cnt ));

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
static const char netservinfo[] =
#ifdef CRYPT_PASSWD
#ifdef LABEL_64_BITS
 "@(#)netserv.c	1.15 6/6/94, using encrypted passwd, 64 bit label.";
#else	/* LABEL_64_BITS */
 "@(#)netserv.c	1.15 6/6/94, using encrypted passwd, 32 bit label.";
#endif	/* LABEL_64_BITS */
#else	/* CRYPT_PASSWD */
#ifdef LABEL_64_BITS
 "@(#)netserv.c	1.15 6/6/94, using clear text passwd, 64 bit label.";
#else	/* LABEL_64_BITS */
 "@(#)netserv.c	1.15 6/6/94, using clear text passwd, 32 bit label.";
#endif	/* LABEL_64_BITS */
#endif	/* CRYPT_PASSWD */

static char dbservfn[MAXPATHLEN];	/* Name of database server program */
static char host[MAXHOSTNAMELEN];	/* Name of monitor host */
static u_long lincksport;		/* Database identity number */
static int s;			/* The tcp/ip socket */
static UID uid = UNKNOWN;	/* User identification number */

/*  */
/**********************************************************************
 * Function: void main(int argc, char *argv[])
 *
 *  Lincks Server: This program takes care of connecting a user on the
 *                 work space side to a new database server process
 *                 Arguments are:
 *                 [-d] debug mode
 *                 [-h host] name of machine running monitor program
 *                 dbid  database directory
 *
 * Modifications:
 *      <list mods with name and date>
 */
void main(argc, argv)
  int argc;
  char *argv[];
{
  int socketno;
  int c;
  int errflag = 0;
  extern char *optarg;
  extern int optind;

  /* Default host == this one */
  (void)gethostname(host, MAXHOSTNAMELEN);

  /* Get arguments from argv using getopt function */
  while ((c = getopt(argc, argv, "dh:")) != EOF) {
    switch (c) {
    case 'd':
      debug = !debug;
      break;
    case 'h':
      (void)strcpy(host, optarg);
      break;
    default:
      ++errflag;
    }
  }

  if (argc - optind != 1)
    ++errflag;
  else {
    (void)configuration(argv[optind]);
    lincksport = TCPIPNO;
  }

  /* If an illegal argument was given, signal error and exit */
  if (errflag) {
    LogMessage(uid, "netserv: illegal arguments");
    exit(1);
  }
  /* Create path to Database server */
  (void)sprintf(dbservfn, "%s/%s", BINARIES, DBSERVER);

  /* Initialize signals */
  if (InitSignals() != SUCCESS)
    exit(2);

  InitServer();
  LogMessage(SUPERUSER, 
#ifdef CRYPT_PASSWD
	     "netserv %ld started on %s (encrypted passwd)", 
#else
	     "netserv %ld started on %s", 
#endif /* n CRYPT_PASSWD */
	     lincksport,
	     host ? host : "");

  /* Disconnect standard IO */
  (void)fflush(stderr);
  (void)close(fileno(stderr));
  (void)fflush(stdout);
  (void)close(fileno(stdout));
  (void)close(fileno(stdin));

  /* Go into server loop */
  for (;;) {
    /* Wait for requests */
    socketno = GetRequest();

    /* Ok, fork off server process */
    StartServer(socketno);
  }
}
/*  */
/**********************************************************************
 * Function: static void error(char *format, ....)
 *
 * error routine used for tcp_server ...
 *
 * Modifications:
 *      <list mods with name and date>
 */
#ifdef __STDC__
static void error(char *format, ...)
#else
void error(va_alist)
  va_dcl
#endif /* n __STDC__ */
{
  va_list ap;
#ifndef __STDC__
  char *format;
  va_start(ap);
  format = va_arg(ap, char *);
#else
  va_start(ap,format);
#endif
  LogMessage(uid, format, ap);
  va_end(ap);
}

/*  */
/**********************************************************************
 * Function: static void InitServer()
 *
 * Sets up tcp/ip socket to work space
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void InitServer()
{
  int setopt = 1;

  /* Specify max 15 people waiting */
  if ((s = tcp_server(error, lincksport, 15)) < 0) 
    exit(3);

  /* Set TCP_NODELAY flag */
  setopt = 1;
  if (setsockopt(s,IPPROTO_TCP,TCP_NODELAY,(char *)&setopt,sizeof(setopt)) < 0)
    {
      LogMessage(uid, "netserv: Warning - TCP_NODELAY not supported, %s",
		 sys_errlist[(errno > sys_nerr) ? 0 : errno]);
    }

  /* Add a SO_KEEPALIVE to enable dbs to find out when work space dies,
   * the 'accepted' socket inherited the same settings. */
  setopt = 1;
  if (setsockopt(s,SOL_SOCKET,SO_KEEPALIVE,(char *)&setopt,sizeof(setopt)) < 0)
    {
      LogMessage(uid, "netserv: Set SO_KEEPALIVE for socket failed, %s",
		 sys_errlist[(errno > sys_nerr) ? 0 : errno]);
    }
}

/*  */
/**********************************************************************
 * Function: static int GetRequest()
 *
 * Waits for user to attempt login from work space
 * Returns new socket number connected to user
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int GetRequest()
{
  struct sockaddr addr;
  int l, newsocketno;
  int cnt;
  int setopt = 1;
  l = sizeof(addr);
  cnt = 0;

  /* Wait for connection and get new socket handle */
retry:
  newsocketno = accept(s, &addr, &l);
  if (newsocketno < 0) {
    if ((cnt++ < 3) || (errno == EINTR))
      goto retry;
    else {
      LogMessage(uid, "netserv: Accept call failed, %s",
		 sys_errlist[(errno > sys_nerr) ? 0 : errno]);
      exit(5);
    }
  }
  /* Set TCP_NODELAY flag */
  if (setsockopt(s, IPPROTO_TCP, TCP_NODELAY, 
		 (char *)&setopt, sizeof(setopt)) < 0) {
  LogMessage(uid,
	     "netserv:Warning (Re)Set TCP_NODELAY socket failed, %s",
	     sys_errlist[(errno > sys_nerr) ? 0 : errno]);
  }

#if defined(INETD_HACK)
  /* try to set TCP_NODELAY before fork/exec, but this should be set */
  /* by accept (it copies the setting according to the man */
  setopt = 1;
  if (setsockopt(newsocketno, IPPROTO_TCP,
		 TCP_NODELAY, (char *)&setopt, sizeof(setopt)) < 0) {
    LogMessage(uid,
	       "netserv: Set TCP_NODELAY for new socket failed, %s",
	       sys_errlist[(errno > sys_nerr) ? 0 : errno]);
  }
#endif /* INETD_HACK */

  /* Return new socket handle */
  return (newsocketno);
}

/*  */
/**********************************************************************
 * Function: static void StartServer(int socketno)
 *
 * Spawns off a new process to handle the caller
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void StartServer(socketno)
  int socketno;
{
  pid_t childpid;

  /* Fork off a new process */
  if ((childpid = fork()) == 0)
    SubProcess(socketno);
  else {
    if (childpid < 0)
      LogMessage(uid, "netserv: Could not fork, %s",
		 sys_errlist[(errno > sys_nerr) ? 0 : errno]);
    (void)close(socketno);
  }
}

/*  */
/**********************************************************************
 * Function: static void SubProcess(int socketno)
 *
 * Acquires user name and password from new user and
 * performs a verification check on these. If all is OK
 * it starts a new database server process
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void SubProcess(socketno)
  int socketno;
{
  unsigned char ch;
  char ucnt, pcnt;
  char username[MAXPATHLEN], password[MAXPATHLEN];
  char pbuff[10];
  char ubuff[10], sbuff[10];
  int oobbug = 0;
  extern UID VerifyUser();


  /* Notify work space we're here */
  ch = SPECBYTE;
  if (write(socketno, (char *)&ch, (IOSIZE_T)1) != 1) {
    LogMessage(uid, "netserv: Notification to WS failed, %s",
	       sys_errlist[(errno > sys_nerr) ? 0 : errno]);
    _exit(6);			/* do not flush parent buffers */
  }
  /* Command interpretor loop */
comloop:
  if (WSChars(socketno, (char *)&ch, 1) != SUCCESS)
    _exit(7);			/* do not flush parent buffers */

  switch (ch) {
  case 'S':             /* dbstat - send version string + encryption */
    ch = sizeof(netservinfo);
    
    if(write(socketno,&ch, sizeof(ch)) != sizeof(ch) ||
      (write(socketno, netservinfo, sizeof(netservinfo)) !=  
       sizeof(netservinfo)))
      _exit(20);
    sleep(2);
    close(socketno);			/* close it down nicely */
    LogMessage(UNKNOWN, "netserv: Sent status information.");
    _exit(0);

  case 'l':		/* Login, but work space need extra byte after */
    /* OOB to fix bug in select. Done in dbs */
    oobbug = 1;

  case 'L':			/* Login */
    /* Read length of user name string */
    if (WSChars(socketno, &ucnt, 1) != SUCCESS)
      _exit(8);			/* do not flush parent buffers */

    /* Read user name string from net */
    if (WSChars(socketno, username, ucnt) != SUCCESS)
      _exit(9);			/* do not flush parent buffers */
    username[(int)ucnt] = '\0';

    /* Get length of password string */
    if (WSChars(socketno, &pcnt, 1) != SUCCESS)
      _exit(10);		/* do not flush parent buffers */

    /* Read password string from net */
    if (WSChars(socketno, password, pcnt) != SUCCESS)
      _exit(11);		/* do not flush parent buffers */
    password[(int)pcnt] = '\0';

    /* Check uid */
    if ((uid = VerifyUser(username, password)) == UNKNOWN) {
      LogMessage(UNKNOWN, "netserv: User %s failed to log in", username);

      /* Terminate net communications */
      (void)write(socketno, "\377", (IOSIZE_T)1);
      _exit(12);		/* do not flush parent buffers */
    }
    LogMessage(uid,
	       "netserv: User %s logged in, uid %d, socketno %d%s",
	       username ? username : "",
	       (int)uid,
	       socketno,
	       oobbug ? "(oobbug)" : "");

    /* Send uid to WS */
    pbuff[0] = 0;
    pbuff[1] = uid;
    (void)write(socketno, pbuff, (IOSIZE_T)2);
    (void)close(s);

#if defined(INETD_HACK)
    /* quick hack to use 'inetd' using  stdin(0) and stdout(1) */
    if (dup2(socketno, 0) == -1) {
      LogMessage(uid, "netserv: Failed to dup2 socket on stdin %d", errno);
      _exit(13);		/* do not flush parents buffers */
    }
    if (dup2(socketno, 1) == -1) {
      (void)sprintf(sbuff, "%d", socketno),
      LogMessage(uid, "netserv: Failed to dup2 socket on to stdout %d",errno);
      _exit(18);		/* do not flush parents buffers */
    }
    /* do not close duplicated socket., */
    if (socketno > 1)
      (void)close(socketno);

#endif /* INETD_HACK */

    /* Start db server - build argument list ... */
    {
      int argc = 0;
      char *argv[12];

      argv[argc++] = DBSERVER;
      if (debug)
	argv[argc++] = "-d";
      if (oobbug)
	argv[argc++] = "-o";
      argv[argc++] = "-h";
      argv[argc++] = host;
      argv[argc++] = DBDIR;
      (void)sprintf(ubuff, "%d", debug ? (int)SUPERUSER : (int)uid);
      argv[argc++] = ubuff;
      argv[argc++] = debug ? "SUPERUSER" : username;
      (void)sprintf(sbuff, "%d", socketno);
      argv[argc++] = sbuff;
      argv[argc++] = (char *)NULL;

      if (execv(dbservfn, argv) == -1) 
	LogMessage(uid, "netserv: dbs wouldn't start %d", errno);

    }				/* construct argv */
    _exit(19);			/* do not flush the parents buffers */

  default:
    LogMessage(uid, "netserv: Startup procedure failed");
    _exit(14);
  }				/* switch */
  goto comloop;
}

/*  */
/**********************************************************************
 * Function: static int WSChars(int socketno, char *buff, int cnt)
 *
 * Gets a number of characters from the net stream and
 * puts them into buff
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int WSChars(socketno, buff, cnt)
  int socketno;
  char *buff;
  int cnt;
{
  int sc;
  fd_set descf;
  struct timeval timev;

  /* Shortcut */
  if (cnt == 0)
    return (SUCCESS);

  timev.tv_sec = LOGIN_TIME_OUT_LIMIT;	/* 5 min */
  timev.tv_usec = 0;

  FD_ZERO(&descf);		/* use fd_set for portability */
  FD_SET(socketno, &descf);
  sc = select(socketno + 1, (fd_set *) & descf, (fd_set *) NULL, (fd_set *) NULL, &timev);

  if (sc < 0) {
    LogMessage(uid, "netserv: Select call failed %d", errno);
    return (FAIL);
  } else if (sc == 0) {
    LogMessage(uid, "netserv: Login time-out");
    return (FAIL);
  }
  /* Get character */
  if (read(socketno, buff, (IOSIZE_T)cnt) != cnt) {
    LogMessage(uid, "netserv: failed to read %d from socketno %d",
	       cnt, socketno);
    return (FAIL);
  }
  return (SUCCESS);
}

/*  */
/**********************************************************************
 * Function: static int InitSignals()
 *
 * Sets up monitor signals
 * SIGUSR2 starts dbs
 *
 * Modifications:
 *      Sat Oct  9 19:28:05 1993 Martin Sjlin. Added check for SIGURG
 *                   and if not defined try SIGPOLL instead (SCO system).
 *      Thu Nov 11 10:25:19 1993 Martin Sjlin - SCO uses SIGUSR1 for
 *                   SIGURG (OOB) and SIGPOLL for async I/O.
 *      <list mods with name and date>
 */
static int InitSignals()
{
  /* Register signals to take care of */
  if (Signal(SIGPIPE, Handler) == BADSIG) {
    LogSignal(uid, "SIGPIPE");
    return (FAIL);
  }
#ifdef SIGURG
  if (Signal(SIGURG, Handler) == BADSIG) {
    LogSignal(uid, "SIGURG");
    return (FAIL);
  }
#else
#ifdef SIGUSR1_FOR_OOB
  if (Signal(SIGUSR1, Handler) == BADSIG) {
    LogSignal(uid, "SIGUSR1");
    return (FAIL);
  }
#else
  if (Signal(SIGPOLL, Handler) == BADSIG) {
    LogSignal(uid, "SIGPOLL");
    return (FAIL);
  }
#endif /* n SIGUSR1_FOR_OOB */
#endif /* n SIGURG */
  if (Signal(SIGCHLD, Handler) == BADSIG) {
    LogSignal(uid, "SIGCHLD");
    return (FAIL);
  }
  if (Signal(SIGUSR2, Handler) == BADSIG) {
    LogSignal(uid, "SIGUSR2");
    return (FAIL);
  }
  /* Register "clean-up" signals */
  if (Signal(SIGTERM, Handler) == BADSIG) {
    LogSignal(uid, "SIGTERM");
    return (FAIL);
  }
  if (Signal(SIGHUP, Handler) == BADSIG) {
    LogSignal(uid, "SIGHUP");
    return (FAIL);
  }
  if (Signal(SIGQUIT, Handler) == BADSIG) {
    LogSignal(uid, "SIGQUIT");
    return (FAIL);
  }
#ifdef SIGLOST
  if (Signal(SIGLOST, Handler) == BADSIG) {
    LogSignal(uid, "SIGLOST");
    return (FAIL);
  }
#endif

  /* Register signals to ignore */
  (void)Signal(SIGALRM, SIG_IGN);
  (void)Signal(SIGINT, SIG_IGN);
#ifdef SIGXFSZ
  (void)Signal(SIGXFSZ, SIG_DFL);
#endif /* SIGXFSZ */
#ifndef SIGUSR1_FOR_OOB
  (void)Signal(SIGUSR1, SIG_IGN);
#endif /* n SIGUSR1_FOR_OOB */
  return (SUCCESS);
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static RETSIGTYPE Handler(int sig, int code, struct sigcontext *scp)
 *
 * Takes care of system signals and performs appropriate
 * actions on them
 *
 * Modifications:
 *      Sat Oct  9 19:28:05 1993 Martin Sjlin. Added check for SIGURG
 *                   and if not defined try SIGPOLL instead (SCO system).
 *      Thu Nov 11 10:25:19 1993 Martin Sjlin - SCO uses SIGUSR1 for
 *                   SIGURG (OOB) and SIGPOLL for async I/O.
 *      <list mods with name and date>
 */
static RETSIGTYPE Handler(sig)
  int sig;
{
  int savederrno = errno;
  char ubuff[10], sbuff[10];
  int status;
  pid_t childpid;
  int socketno;

  switch (sig) {
  case SIGCHLD:		/* child status has changed */
    /* wait for ALL childs, as a signal is binary */
#ifdef HAVE_WAITPID
    while (waitpid(0, &status, WNOHANG) > 0)
#else
    while (wait3(&status, WNOHANG, (struct rusage *)NULL) > 0)
#endif
    {
      char ex[16], sig[32];

#if defined(WIFEXITED) && defined(WEXITSTATUS)
      if (WIFEXITED(status))
	(void)sprintf(ex, ", exit status %d", WEXITSTATUS(status));
      else
	ex[0] = '\0';
#else
      (void)sprintf(ex, ", exit status %d", status);
#endif /* n WIFEXITED */

#if defined(WIFSIGNALED) && defined(WTERMSIG)
      if (WIFSIGNALED(status))
	(void)sprintf(sig, ", signal %d", WTERMSIG(status));
      else
	sig[0] = '\0';
#else
      (void)strcpy(sig, ", cannot determine signal");
#endif

      /*  And use the macros for extracting the status and values */
      errno = 0;
      LogMessage(uid, "netserv: dbs died%s%s%s", ex, sig,
#if defined(WIFSIGNALED) && defined(WCOREDUMP)
		 (WIFSIGNALED(status) && WCOREDUMP(status))
		 ? " - core dumped." :
		 "."
#else
		 status != 0 ? ", cannot determine if core dumped." : "."
#endif
	);
    }
#ifndef HAVE_RELIABLE_SIGNALS
    if (Signal(SIGCHLD, Handler) == BADSIG)
      LogSignal(uid, "SIGCHLD");
#endif /* HAVE_RELIABLE_SIGNALS */
    break;

  case SIGPIPE:
    LogMessage(uid, "netserv: Socket connection to work space broken");
    exit(15);
    break;

#ifdef SIGURG
  case SIGURG:
#else
#ifdef SIGUSR1_FOR_OOB
  case SIGUSR1:
#else
  case SIGPOLL:
#endif /* n SIGUSR1_FOR_OOB */
#endif /* n SIGURG */
    LogMessage(uid, "netserv: Urgent condition on work space socket: %s",
	       sys_errlist[(errno > sys_nerr) ? 0 : errno]);
    exit(16);
    break;

  case SIGUSR2:
    /* Start netserv */
    if ((childpid = fork()) == 0) {
      (void)Signal(SIGHUP, SIG_DFL);
      (void)Signal(SIGUSR2, SIG_IGN);

      socketno = 6;
      (void)sprintf(sbuff, "%d", socketno);
      /* Start db server */
      if (debug) {
	(void)sprintf(ubuff, "%d", (int)SUPERUSER);
	if ((execl(dbservfn, DBSERVER, "-d",
		   "-h", host, DBDIR,
		   ubuff,
		   sbuff,
		   (char *)NULL)) == -1) {
	  LogMessage(uid, "netserv: dbs wouldn't start");
	}
      } else {
	(void)sprintf(ubuff, "%d", (int)uid);
	if ((execl(dbservfn, DBSERVER,
		   "-h", host, DBDIR,
		   ubuff,
		   sbuff,
		   (char *)NULL)) == -1) {
	  LogMessage(uid, "netserv: dbs wouldn't start");
	}
      }
      exit(17);
    }
    /* fork() == 0 */ 
    else {
      (void)sleep(1);
      LogMessage(SUPERUSER, "netserv: Started dbs with pid: %d",
		 (int) childpid);
    }				/* fork() != 0 */
#ifndef HAVE_RELIABLE_SIGNALS
    if (Signal(SIGUSR2, Handler) == BADSIG)
      LogSignal(uid, "SIGUSR2");
#endif /* HAVE_RELIABLE_SIGNALS */
    break;

    /* Remove any file locks before exiting */
#ifdef SIGLOST
  case SIGLOST:
#endif
  case SIGTERM:
  case SIGHUP:
  case SIGQUIT:
    LogMessage(uid, "netserv: caught signal %d cleaning up locks", sig);
    {
      int fd;
      struct rlimit rlp;

#ifdef RLIMIT_NOFILE
      if (getrlimit(RLIMIT_NOFILE, &rlp) < 0) {
	LogMessage(uid, "Handler: failed to get file limits.");
#endif /* RLIMIT_NOFILE */
#ifdef NOFILE
	rlp.rlim_cur = NOFILE;
#else /* NOFILE */
#ifdef _NFILE
	rlp.rlim_cur = _NFILE;
#else /* _NFILE */
	rlp.rlim_cur = 256;	/* just take ANY value ... */
#endif /* n _NFILE */
#endif /* NOFILE */
#ifdef RLIMIT_NOFILE
      }
#endif /* RLIMIT_NOFILE */

      /* try to unlock all f(fcntl)/flock locks */
      for (fd = 0; fd < rlp.rlim_cur; fd++)
	unlockit(fd, (off_t)0, (off_t)0);
    }
    exit(0);
    break;
  }
  errno = savederrno;
}
