/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* 
 * MODULE NAME: 	locks.c
 *
 * SCCSINFO:		@(#)locks.c	1.8 5/3/94
 *
 * ORIGINAL AUTHOR(S):  ???, Jan 17, 1986
 *
 * MODIFICATIONS:
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 *	This file contains the functions SetLock, TimeLocks, CheckLock
 *	and ClearLock.
 */
/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */

#include <time.h>

#include "lincks.h"
#include "monitor.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_locks.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_monerrors.h"
#include "f_getentry.h"

/* libc */
#if defined(sun) || defined(__sun__)
extern time_t time P_((time_t *tloc));
#endif

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern LOCK *locklist;		/* monitormain.c */

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
/* none */

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * INTERNAL (STATIC) DATA: 
 *********************************************************************/
/* none */

/*  */
/**********************************************************************
 * Function: int SetLock(UID uid, INDEX *entry)
 * 
 * Sets a lock on the specified entry with owner set to uid
 * Returns FAIL if there is no more memory
 *
 * Modifications:
 *      <list mods with name and date>
 */
int SetLock(uid, entry)
  UID uid;
  INDEX *entry;
{
    LOCK *lockentry, *tmplock;

    /* Begin with checking if molecule already locked */
    /* If so, assume uid is owner of lock (checked elsewhere) */
    if (entry->flags & LOCKED)
      return(SUCCESS);

    /* Malloc new space for this lock */
    if ((lockentry = (LOCK *)malloc((ALLOC_T)LOCKSIZE)) == NULL) {
	Error(ER_MALLOC, "SetLock: No more memory available to lock");
	return(FAIL);
    }

    /* Initialize lock entry data */
    lockentry->molecule = entry->label;
    lockentry->owner = uid;
    lockentry->time = time((time_t *)NULL) + TIMEOUT;
    lockentry->l_next = NULL;

    /* Append new entry to lock list */
    if (locklist == NULL) locklist = lockentry;
    else {
	tmplock = locklist;
	while (tmplock->l_next != NULL)
	  tmplock = tmplock->l_next;
	tmplock->l_next = lockentry;
    }

    /* Set entry flag LOCKED */
    entry->flags |= LOCKED;
    return(SUCCESS);
}

/*  */
/**********************************************************************
 * Function: int TimeLocks(INDEX table[])
 * 
 * Goes through the lock list and clears all locks that
 * have timed out. Always returns SUCCESS
 *
 * Modifications:
 *      <list mods with name and date>
 */
int TimeLocks(table)
  INDEX table[];
{
    TIME nowtime;
    LOCK *lockentry;
    int pos;

    /* Quick short cut */
    if (locklist == NULL) return(SUCCESS);

    /* Get present time */
    nowtime = time((time_t *)NULL);

    /* Scan lock list to find entry with time < present time */
    lockentry = locklist;

    while (lockentry != NULL) {
	if (lockentry->time <= nowtime) {

	    /* Remove entry from lock list */
	    locklist = lockentry->l_next;

	    /* Clear entry LOCKED flag */
	    if ((pos = GetEntry(lockentry->molecule, table, -1)) >= 0)
	      table[pos].flags &= ~LOCKED;

	    /* Free memory used by lock entry */
            free((FREEPTR *)lockentry);
	    lockentry = locklist;
	}
	else break;
    }

    /* If nothing found, OK */
    return(SUCCESS);
}

/*  */
/**********************************************************************
 * Function: int ClearLock(INDEX *entry)
 * 
 * Removes the lock specified and also clears its index
 * table flag entry
 *
 * Modifications:
 *      <list mods with name and date>
 */
int ClearLock(entry)
  INDEX *entry;
{
    LOCK *lockentry, *preventry;

    /* Scan lock list for entry: initialize pointers */
    lockentry = preventry = locklist;
    /* Scan until no further entries */
    while (lockentry) {
	if (lockentry->molecule == entry->label) {
	    /* Special case if entry first list entry */
	    if (lockentry == preventry)
	      locklist = lockentry->l_next;
	    else
	      preventry->l_next = lockentry->l_next;
	    /* Free memory used by entry */
            free((FREEPTR *)lockentry);
	    /* Clear entry LOCKED flag */
	    entry->flags &= ~LOCKED;
	    return(SUCCESS);
	}
	else {
	    /* Prepare for next time through */
	    preventry = lockentry;
	    lockentry = lockentry->l_next;
	}
    }

    /* Did not find entry in lock list */
    Error(ER_NOLOCKFOUND, "ClearLock: lock entry not found");
    return(FAIL);
}

/*  */
/**********************************************************************
 * Function: int CheckLock(UID uid, INDEX *entry, int mode)
 * 
 * Checks if a lock is set for entry and if so, that it is
 * set by uid. There are two modes: MUST means that a lock
 * must be set. MAY that a lock may be set (by uid)
 *
 * Modifications:
 *      <list mods with name and date>
 */
int CheckLock(uid, entry, mode)
  UID uid;
  INDEX *entry;
  int mode;
{
    LOCK *lockentry;

    /* Check if any lock set at all. Return mode if not */
    /* MUST -> FAIL, MAY -> SUCCESS */
    if (!(entry->flags & LOCKED))
      return(mode);

    /* Scan lock list to find entry. Check that it belongs to uid */
    lockentry = locklist;
    while (lockentry != NULL) {
	if (lockentry->molecule == entry->label) {
	    if (lockentry->owner == uid)
	      return(SUCCESS);
	    else
	      return(FAIL);
	}
	else lockentry = lockentry->l_next;
    }

    /* Did not find in lock list, error condition */
    Error(ER_NOLOCKFOUND, "Checklock: lock entry not found in lock list");
    return(FAIL);
}
