/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME: 	dbwrmol.c
 *
 * SCCSINFO:		@(#)dbwrmol.c	1.10 6/6/94
 *
 * ORIGINAL AUTHOR(S):  ???, Created: 17 Feb, 1987
 *
 * MODIFICATIONS:
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 *	This file contains WriteMolecule, WriteLinks and SendMolecule
 *	Data file format for link section is:
 *	        <group name length>
 *		"group name string"
 *		<tag name length>
 *		"tag name string"
 *		<label1> <label2> <label3>
 *		<0>
 *		...
 *		<group name length>
 *		"group name string"
 *		<0>
 *		...
 *
 */

/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */

#include <netinet/in.h>

#include "lincks.h"
#include "dbserver.h"
#include "libshared.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_dbwrmol.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_molfile.h"
#include "f_dbrdmol.h"
#include "f_dbserrors.h"
#include "f_dbserv.h"
#include "f_dbwrmol.h"
#include "f_packunpack.h"

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern TABENTRY *filetable;	/* dbserv.c */
#if defined(INETD_HACK)
extern int socketwr;
extern int socketrd;
#else
#define socketrd socketno
#define socketwr socketno
extern int socketno;		/* dbserv.c */
#endif
extern PACKEDBUFF linkbuff;	/* dbserv.c */

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
/* none */

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static int DoWriteMolecule P_(( MOLECULE *newmol, LABEL label, FILEMOL
                                 *molfileinfo, int index ));
static int SendChars P_(( char *str, int len, int sock ));
static int SendLinks P_(( LINKGROUP *linkroot, int s ));
static int WriteBuff P_(( WRITEINFO *info, PACKEDBUFF *pbuff, int fd ));

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
/* none */

/*  */
/**********************************************************************
 * Function: int WriteMolecule(MOLECULE *newmol, LABEL label)
 *
 * Modifications:
 *      <list mods with name and date>
 */
int WriteMolecule(newmol, label)
  MOLECULE *newmol;
  LABEL label;
{
  A_ENTRY *access;
  FILEMOL *molfileinfo;
  int return_code = SUCCESS;
  int index;

  if ((access = MolAccess(label, ACCESS)) == NULL)
    return (FAIL);

  if (((index = GetMolfd(access->a_filename)) < 0) ||
      ((molfileinfo = ReadMolFile(access, index)) == NULL) ||
      (DoWriteMolecule(newmol, label, molfileinfo, index) == FAIL))
    return_code = FAIL;

  return (return_code);
}


/*  */
/**********************************************************************
 * Function: static int DoWriteMolecule(MOLECULE *newmol, LABEL label, FILEMOL *molfileinfo, int index)
 *
 * Writes the molecule newmol to file taking care of
 * blink updating on other molecules
 *
 * Uses MolAccess, GetMolfd, ReadMolFile, WriteLinks
 * WriteBuff, UpdateMolFile, ReadLinks,
 * UpdateBlinks and Error
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int DoWriteMolecule(newmol, label, molfileinfo, index)
  MOLECULE *newmol;
  LABEL label;
  FILEMOL *molfileinfo;
  int index;
{
  WRITEINFO wrinfo;
  int fd;
#ifdef BLINKS_TO_WS
  LINKGROUP *oldmolflinks;
#endif

  fd = filetable[index].datfile.fd;

#ifdef BLINKS_TO_WS
  oldmolflinks = NULL;

  /* Get old link structure from data file */
  if (ReadLinks(&oldmolflinks, (off_t) molfileinfo->flink_filepos,
		molfileinfo->flink_size, fd) < 0)
    return (FAIL);

  if (WriteLinks(&wrinfo, newmol->blinks, fd) == FAIL)
    return (FAIL);
  molfileinfo->blink_filepos = wrinfo->wrpos;
  molfileinfo->blink_size = wrinfo->wrsize;
#endif /* BLINKS_TO_WS */

  /* Put new flink structure on data file */
  if (WriteLinks(&wrinfo, newmol->flinks, fd) == FAIL)
    return (FAIL);
  molfileinfo->flink_filepos = wrinfo.wrpos;
  molfileinfo->flink_size = wrinfo.wrsize;

  /* Put attributes on data file */
  if (WriteBuff(&wrinfo, newmol->attributes, fd) == FAIL)
    return (FAIL);
  molfileinfo->attr_filepos = wrinfo.wrpos;
  molfileinfo->attr_size = wrinfo.wrsize;

  /* Put image on data file */
  if (WriteBuff(&wrinfo, newmol->image, fd) == FAIL)
    return (FAIL);
  molfileinfo->image_filepos = wrinfo.wrpos;
  molfileinfo->image_size = wrinfo.wrsize;

  /* Update molecule file entry */
  if (UpdateMolFile(molfileinfo, label, index) < 0)
    return (FAIL);

#ifdef BLINKS_TO_WS
  /* Now start handling blink updating */
  /* Update other molecule's blink structures */
  if (UpdateBlinks(newmol->flinks, oldmolflinks, label) < 0)
    return (FAIL);
#endif /* BLINKS_TO_WS */

  return (SUCCESS);
}

/*  */
/**********************************************************************
 * Function: int WriteLinks(WRITEINFO *info,LINKGROUP *linkroot, int fd)
 *
 * Writes the link structure linkroot to file fd
 * Uses PackGroup and Error
 *
 * Modifications:
 *      <list mods with name and date>
 */
int WriteLinks(info, linkroot, fd)
  WRITEINFO *info;
  LINKGROUP *linkroot;
  int fd;
{
  info->wrsize = 0;
  info->wrpos = 0;

  /* Short cut if root is NULL */
  if (!linkroot)
    return (SUCCESS);

  /* Pack all groups into buffer */
  if (PackGroup(linkroot, &linkbuff) == FAIL) {
    return (FAIL);
  }
  return WriteBuff(info, &linkbuff, fd);
}

/*  */
/**********************************************************************
 * Function: static int WriteBuff(WRITEINFO *info, PACKEDBUFF *pbuff, int fd)
 *
 * Writes a buffer structure to file fd
 * Uses Error
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int WriteBuff(info, pbuff, fd)
  WRITEINFO *info;
  PACKEDBUFF *pbuff;
  int fd;
{
  info->wrpos = -1;
  info->wrsize = 0;

  /* Short cut if nothing to write */
  if (pbuff == (PACKEDBUFF *) NULL)
    return (SUCCESS);

  /* Try to gain lock on data file */
  if ((writelockit(fd)) == -1) {
    Error(ER_FLOCK, "WriteBuff: could not lock data file");
    return (FAIL);
  }

  /* Seek to end of data file */
  if ((info->wrpos = lseek(fd, (off_t) 0, L_XTND)) < 0) {
    Error(ER_LSEEK, "WriteBuff: could not seek to end of file (%d)", errno);
    unlockit(fd, (off_t)0, (off_t)0);
    return (FAIL);
  }
  info->wrsize = pbuff->size;

  /* Write buffer to file */
  if (write(fd, pbuff->buff, (IOSIZE_T)info->wrsize) != info->wrsize) {
    Error(ER_WRITE, "WriteBuff: could not write to file (%d)", errno);
    unlockit(fd, (off_t) info->wrpos, (off_t)0);
    return (FAIL);
  }
  /* Unlock and release buffer memory */
  unlockit(fd, (off_t)info->wrpos, (off_t)0);
  return (SUCCESS);
}


/*  */
/**********************************************************************
 * Function: int SendMolecule(MOLECULE *mol)
 *
 * Sends a molecule to the socket connection
 * Uses SendLinks and SendBuff
 *
 * Modifications:
 *      <list mods with name and date>
 */
int SendMolecule(mol)
  MOLECULE *mol;
{
  if ((SendLinks(mol->flinks, socketwr) == FAIL) ||
      (SendBuff(mol->attributes, socketwr) == FAIL) ||
      (SendBuff(mol->image, socketwr) == FAIL))
    return (FAIL);

  return (SUCCESS);
}


/*  */
/**********************************************************************
 * Function: static int SendLinks(LINKGROUP *linkroot, int s)
 *
 * Sends the link structure linkroot to socket s
 *
 * Uses PackWSGroup and Error
 *
 * Modifications:
 *      1994-03-14 Martin Sj|lin. Used PackWSGroup since we
 *       are working against the workspace ..
 *      <list mods with name and date>
 */
static int SendLinks(linkroot, s)
  LINKGROUP *linkroot;
  int s;
{
  if (linkroot == (LINKGROUP *) NULL)
    return (SendCount(0, s));

  if (PackWSGroup(linkroot, &linkbuff) == FAIL)
    return (FAIL);

  return (SendBuff(&linkbuff, s));
}


/*  */
/**********************************************************************
 * Function: int SendBuff(PACKEDBUFF *pbuff, int sock)
 *
 * Sends a packed buffer to the socket connection.
 *
 * Modifications:
 *      <list mods with name and date>
 */
int SendBuff(pbuff, sock)
  PACKEDBUFF *pbuff;
  int sock;
{
  if (SendCount(pbuff->size, sock) == FAIL)
    return (FAIL);

  return (SendChars(pbuff->buff, pbuff->size, sock));
}


/*  */
/**********************************************************************
 * Function: static int SendChars(char *str, int len)
 *
 * Sends characters to the socket connection.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int SendChars(str, len, sock)
  char *str;
  int len;
  int sock;
{
  int count;

  while (len > 0) {
    if ((count = write(sock, str, (IOSIZE_T)len)) < 0) {
      Error(ER_WRITE, "SendChars: could not write to socket (%d)", errno);
      return (FAIL);
    }
    if (count == 0) {
      Error(ER_WRITE, "SendChars: data not accepted at socket");
      return (FAIL);
    }
    /* Adjust byte count and position pointer */
    str += count;
    len -= count;
  }
  return (SUCCESS);
}


/*  */
/**********************************************************************
 * Function: int SendCount(t_32bits val, int sock)
 *
 * Sends an integer to the socket connection
 *
 * Modifications:
 *      <list mods with name and date>
 */
int SendCount(val, sock)
  t_32bits val;
  int sock;
{
  t_u32bits netint = htonl(val);

  return SendChars((char *)&netint, sizeof(netint), sock);
}
