/*
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME: 	dbserv.c
 *
 * SCCSINFO:		@(#)dbserv.c	1.15 6/6/94
 *
 * ORIGINAL AUTHOR(S):  ???, Feb, 1987
 *
 * MODIFICATIONS:
 *      1993-08-12 Martin Sjlin - Changed the MESSAGE handling
 *                      to send count of requests in the queue.
 *      1994-01-06 Martin Sjlin - Added case for re-/dis-connect
 *                      in main server code.
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 *  This is the database server. It is spawned by the net server process
 *  when a new user calls. A known stream is supplied to receive data
 *  from the user.
 */
/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */

#ifdef HAVE_USLEEP		/* included in config.h ifndef */
#include <sys/time.h>
#endif /* HAVE_USLEEP */
#include <sys/ioctl.h>
#include <sys/socket.h>			/* socket + MAXHOSTNAME on SCO */
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <netdb.h>			/* for MAXHOSTNAMELEN */
#include "lincks.h"
#include "dbserver.h"
#include "libshared.h"
#include "dbcodes.h"
#include "xconfig.h"
#include "config.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_dbserv.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
u_long lincksport;		/* Database identity number */
UID uid;			/* Global user identification number */
char username[MAXPATHLEN];	/* Global user identification name */
#if defined(INETD_HACK)
int socketrd;			/* when using inetd style fork/exec */
int socketwr;			/* when using inetd style fork/exec */
#else
#define socketrd socketno
#define socketwr socketno
int socketno;			/* Socket number to Lisp machine */
#endif
char host[MAXHOSTNAMELEN];	/* Host we're running on */
int debug = 0;			/* Debug flag */
int filetablesize;		/* Maximum number of open mol/data files */
IT_ENTRY *InterfaceTable;	/* The interface table */
int ITblsize;			/* Size of interface table */
TABENTRY *filetable;		/* The file table */
int sigusr2count = 0;		/* Signal counter */
int synchronous = 0;		/* Command processing mode */
int oobbug = 0;			/* if dbs need to send an extra byte to ws */

MOLECULE the_molecule;
PACKEDBUFF attrbuff;
PACKEDBUFF imagebuff;
PACKEDBUFF linkbuff;

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_dbrdmol.h"
#include "f_dbserfs.h"
#include "f_dbserrors.h"
#include "f_dbshandler.h"
#include "f_dbwrmol.h"
#include "f_query.h"
#include "f_queue.h"
#include "f_misc.h"

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 64
#endif /* MAXHOSTNAMELEN */

#define ErrSignal(uid, sig) \
    Error(uid, "InitSignals: could not register signal %s.  errno = %d", sig, errno)

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static void free_molecule P_(( void ));
static void initialise P_(( void ));
static int InitSignals P_(( void ));
static int ispenned P_(( PEN_MESSAGE *msg ));
static void Logout P_(( int status ));
static int ReceiveCommand P_(( unsigned char *command ));
static int ReceiveIndex P_(( WS_LABEL *val ));
static int ReceiveMolType P_(( int *val ));
static int ReplyPack P_(( unsigned mess ));
static int ReplyString P_(( char *str ));
static int SendIndex P_(( WS_LABEL val ));

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
/* Internals */
static int selectmaskno;	/* fd limit used in select */
static fd_set selectmask;	/* Mask to use in select operation */
static Queue *messagequeuep;	/* Queue for PEN messages */
static WS_LABEL global_index;

/*  */
/**********************************************************************
 * Function: int main(int argc, char **argv)
 *
 * This is the main module of the DBserver
 *        Accepted arguments:
 *            [-d]	debug mode
 *            [-o]      if we need to send an extra byte to WS after OOB
 *            [-h host] monitor host name
 *            dbid	database directory
 *            uid	user identity number
 *            username	user identity name
 *            fd	socket number to use to application
 *
 * Modifications:
 *      <list mods with name and date>
 */
int main(argc, argv)
  int argc;
  char **argv;
{
  int c;
  int errflag = 0;
  int setopt = 1;
  extern char *optarg;
  extern int optind;

  /* Default host == this one */
  (void)gethostname(host, MAXHOSTNAMELEN);

  /* Get arguments from argv using getopt function */
  while ((c = getopt(argc, argv, "dh:o")) != EOF) {
    switch (c) {
    case 'o':
      oobbug = 1;
      break;
    case 'd':
      debug = !debug;
      break;
    case 'h':
      (void)strcpy(host, optarg);
      break;
    default:
      ++errflag;
    }
  }

  if (argc - optind != 4)
    ++errflag;
  else {
    char *eos = (char *)NULL;

    /* Database identity */
    if (!configuration(argv[optind + 0])) {
      /*
       * we have nowhere to "write" an error,
       * so this will have to do
       */
      exit(-1);
    }
    lincksport = TCPIPNO;

    /* User identity number */
    uid = strtol(argv[optind + 1], &eos, 0);
    if (*eos != '\0')
      ++errflag;

    /* User identity name */
    (void)strcpy(username, argv[optind + 2]);

    /* Reset the select mask before start setting relevant bits */
    FD_ZERO(&selectmask);
#if defined(INETD_HACK)
    socketrd = 0;
    socketwr = 1;
    FD_SET(socketrd, &selectmask);
    FD_SET(socketwr, &selectmask);
    selectmaskno = socketwr > socketrd ? socketwr : socketrd;
    selectmaskno++;
#else
    /* Application socket file descriptor */
    socketno = strtol(argv[optind + 3], &eos, 0);
    if (*eos != '\0')
      ++errflag;

    FD_SET(socketno, &selectmask);
    selectmaskno = socketno + 1;/* counts number entries in fd_set */

    /* Set TCP_NODELAY option */
    if (setsockopt(socketno, IPPROTO_TCP,
		   TCP_NODELAY, (char *)&setopt, sizeof(setopt)) < 0) {
      Error(ER_SYSERR,
	    "dbs: Set TCP_NODELAY for socket failed (%d)", errno);
    }
#endif /* n INETD_HACK */
  }

  /* If an illegal argument was given, signal error and exit */
  if (errflag)
    exit(INITFAIL);

  /* Initialize signals */
  if (InitSignals() != SUCCESS)
    exit(INITFAIL);

  /* Log user session start */
  Error(0, "dbs %ld started.", lincksport);

  /* Initialize the interface and file tables */
  TableInit();

  /* Create message queue */
  if ((messagequeuep = (Queue *) CreateQueue()) == NULL) {
    Error(0, "Couldn't create message queue");
    exit(INITFAIL);
  }
  /* Go into command loop */
  Server();
  return 0;
}


/*  */
/**********************************************************************
 * Function: static void initialise()
 *
 * Initialises the molecule structure.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void initialise()
{
  attrbuff.buff = (char *)NULL;
  attrbuff.size = 0;

  imagebuff.buff = (char *)NULL;
  imagebuff.size = 0;

  linkbuff.buff = (char *)NULL;
  linkbuff.size = 0;

  the_molecule.attributes = &attrbuff;
  the_molecule.image = &imagebuff;
  the_molecule.flinks = (LINKGROUP *) NULL;
  the_molecule.blinks = (LINKGROUP *) NULL;
}


/*  */
/**********************************************************************
 * Function: static void free_molecule()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void free_molecule()
{
  if (attrbuff.buff != (char *)NULL)
    free(attrbuff.buff);
  attrbuff.buff = (char *)NULL;
  attrbuff.size = 0;

  if (imagebuff.buff != (char *)NULL)
    free(imagebuff.buff);
  imagebuff.buff = (char *)NULL;
  imagebuff.size = 0;

  if (linkbuff.buff != (char *)NULL)
    free(linkbuff.buff);
  linkbuff.buff = (char *)NULL;
  linkbuff.size = 0;

  if (the_molecule.flinks != (LINKGROUP *) NULL)
    FreeLinks(the_molecule.flinks);
  the_molecule.flinks = (LINKGROUP *) NULL;

#ifdef BLINKS_TO_WS
  if (the_molecule.blinks != (LINKGROUP *) NULL)
    FreeLinks(the_molecule.blinks);
  the_molecule.blinks = (LINKGROUP *) NULL;
#endif /* BLINKS_TO_WS */
}


/*  */
/**********************************************************************
 * Function: static int ispenned(PEN_MESSAGE *msg)
 *
 * Return 1 if msg is a type 0 message for global_index.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int ispenned(msg)
  PEN_MESSAGE *msg;
{
  return ((msg) && (msg->type == 0) && (msg->index == global_index));
}


/*  */
/**********************************************************************
 * Function: void Server()
 *
 * This is the main server command interpreter loop
 *
 * Modifications:
 *      <list mods with name and date>
 */
void Server()
{
  unsigned char ch;
  WS_LABEL index;
  int status, moltype;
  char count;
  PACKEDBUFF buff;
  PEN_MESSAGE *messagep;


  /* Initialise the transmission molecule memory */
  initialise();

  while (5 != 7) {
    /*
     * Assume that the workspace does not await ack/error.
     * This blocks transmission of error messages to the workspace.
     */
    synchronous = 0;
    for (count = 0; count <= 2; count++)
      if (ReceiveCommand(&ch) >= 0)
	break;

    if (count > 2)
      Logout(COMBROKEN);

    /*
     * Most commands are synchronous.
     */
    synchronous = ((ch != STORE_CODE) && (ch != EDIT_CODE));
    /* Execute command */
    switch (ch) {
    case ALIVE_CODE:
      (void)ReplyPack((unsigned) RCOK);
      break;

    case LOGO_CODE:
      if (messagequeuep && messagequeuep->nextp) {
	LogMessage(uid, "dbs: logout failed due to pending PEN");
	(void)ReplyPack((unsigned) RCBAD);
      } else {
	Logout(SUCCESS);
      }
      break;

    case EDIT_CODE:
      if (ReceiveIndex(&index) != SUCCESS)
	break;
      /* Check and insert edit information */
      (void)Check(index, 1);
      break;

    case STORE_CODE:
      if (ReceiveIndex(&index) != SUCCESS)
	break;
      if (ReceiveMolType(&moltype) != SUCCESS)
	break;
      /* moltype = OTHER or VS */
      if ((ReceiveMolecule()) == FAIL) 
	break;
      /*
       * Abort command if the node has a type 0 OOB message unread.
       */
      global_index = index;
      if ((moltype == VS) && Mapqueue(messagequeuep, ispenned))
	break;
      if (Store(index, &the_molecule, moltype) != SUCCESS)
	break;
      if (moltype == VS) 
	/* Remove edit information */
	(void)Remove(index);
      break;

    case RETRIEVE_CODE:
      if (ReceiveIndex(&index) != SUCCESS)
	break;
      if (ReceiveMolType(&moltype) != SUCCESS)
	break;
      /* moltype = UNKNOWN_TYPE, OTHER, VS or VSH */
      if ((status = Retrieve(index)) == FAIL)
	break;
      switch (status) {
      case RCBUFFER:
	(void)ReplyPack((unsigned) RCBUFFER);
	break;
      case RCMOL:
	if (ReplyPack((unsigned) RCMOL) != SUCCESS)
	  break;
	if (SendMolecule(&the_molecule) != SUCCESS)
	  break;
      case RCOK:
	(void)ReplyPack((unsigned) RCOK);
	break;
      default:
	break;
      }
      /* Check edit information */
      if ((moltype == VS) || (moltype == UNKNOWN_TYPE))
	(void)Check(index, 0);
      break;

    case RELEASE_CODE:
      if (ReceiveIndex(&index) != SUCCESS)
	break;
      if (ReceiveMolType(&moltype) != SUCCESS)
	break;
      /* moltype = OTHER or VS */
      if (Release(index) != SUCCESS)
	break;
      (void)ReplyPack((unsigned)RCOK);
      if (moltype == VS)
	(void)Remove(index);
      break;

    case QUERY_CODE:
      Query();
      (void)ReplyPack((unsigned)RCOK);
      break;

    case DISCONNECT_CODE:
    case RECONNECT_CODE:
      (void) ReplyPack((unsigned)RCNYI);
      break;

    case MESSAGE_CODE:
      count = QueueCount(messagequeuep);

      if (count <= 0) {
	(void)ReplyPack((unsigned)RCOK);
	break;
      }
      if ((messagep = (PEN_MESSAGE *) Dequeue(messagequeuep)) == NULL) {
	Error(0, "dbs: No more messages in queue, count = %d", count);
	(void)ReplyPack((unsigned)RCOK);
	break;
      }
      if ((buff.buff = messagep->msg))
	buff.size = strlen(buff.buff) + 1;
      else
	buff.size = 0;

      /* Send ack, counter and data */
      if (ReplyPack((unsigned)RCMESSAGE) != SUCCESS)
	break;
      if (SendCount(count, socketwr) != SUCCESS)
	break;
      if (SendIndex(messagep->index) != SUCCESS)
	break;
      if (SendCount(messagep->type, socketwr) != SUCCESS)
	break;
      if (SendBuff(&buff, socketwr) != SUCCESS)
	break;

      /* Free delivered message */
      if (messagep->msg)
	free(messagep->msg);
      free((FREEPTR *)messagep);
      break;

    default:
      break;
    }				/* switch */

    free_molecule();
  }				/* while */
}

/*  */
/**********************************************************************
 * Function: static void Logout(int status)
 *
 * Handles communications termination
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void Logout(status)
  int status;
{
  int st;

  /* Disable SIGPIPE (write on pipe/socket without reader)
   * when we have a (possible) broken communications channel  */
  if (status == COMBROKEN)
    (void)Signal(SIGPIPE, SIG_IGN);

  if((st = ReplyPack((unsigned)RCLOGO)) != SUCCESS)
    LogMessage(uid,"dbs: ack of logout failed, %d", st);

  /* Remove all edit information in monitor */
  if (RemoveAll() != SUCCESS) {
    LogMessage(uid, "dbs: failed to RemoveAll");
  }
  Error(0, "dbs %ld stopped", lincksport);
  
  /* and close socket descriptor to send off logout message, but  */
  if (close (socketwr) < 0)
    LogMessage(uid, "dbs: Error closing socket (%d)", errno);

  if (status != SUCCESS)
    LogMessage(uid, "dbs: Terminating due to FATAL error (%d)!", status);
  else
    LogMessage(uid, "dbs: User logged out");
  exit(status);
}

/*  */
/**********************************************************************
 * Function: void SendOOB(int type, WS_LABEL index, char *msg)
 *
 * Sends ``out-of-band'' data on socket
 *
 * Modifications:
 *      <list mods with name and date>
 */
void SendOOB(type, index, msg)
  int type;
  WS_LABEL index;
  char *msg;
{
  char oob = OOB_MESSAGE_CODE;
  PEN_MESSAGE *messagep = NULL;

  if ((messagep = (PEN_MESSAGE *) malloc(sizeof(PEN_MESSAGE))) == NULL) {
    Error(0, "SendOOB: malloc failed for PEN_MESSAGE");
    return;
  }
  messagep->index = index;
  messagep->msg = NULL;
  messagep->type = type;

  if (msg && !(messagep->msg = strdup(msg))) {
    Error(0, "SendOOB: malloc failed for pen message text");
    (void)free((FREEPTR *)messagep);
    return;
  }
  Enqueue(messagequeuep, (void *)messagep);

#ifdef PENDEBUG
  Error(0, "Send OOB %d (%ld) %s, pid = %d.",
	type,
	index,
	msg == NULL ? "" : msg,	/* stupid Solaris */
	getpid());
#endif

  if (send(socketwr, &oob, 1, MSG_OOB) != 1)
    Error(0, "SendOOB: Can't send OOB byte, %d, pid = %d", errno, (int)getpid());
  /* See comment in linckshandler.c:Handler (liblincks.c) */
  else if (oobbug) {
#ifdef PENDEBUG
    Error(0, "SendOOB: WS needs extra bytes, try to send it ...");
#endif
    if (ReplyPack((unsigned)RCOOB) != SUCCESS)
      Error(0,
	    "SendOOB: Can't send byte(s) after OOB, %d, pid = %d",
	    errno,
	    (int)getpid());
  }				/* oobbug */
}

/*  */
/**********************************************************************
 * Function: static int ReceiveCommand(unsigned char *command)
 *
 * Reads a command pack from the socket
 *
 * Modifications:
 *      1993-11-03 Martin Sjlin. Ran into interrupted system call
 *                 in the first read after select, restart call
 *                 in this case (appears on SunOS 4.1.x when NOT
 *                 using OOBSELECTBUG - see comment in lowcom.c)
 *      <list mods with name and date>
 */
static int ReceiveCommand(command)
  unsigned char *command;
{
  static unsigned char buff, count;
  int sc;
  fd_set smask;
  struct timeval timev;

  /* Preparations */
  count = 0;

  /* Loop until command has been received */
  while (1) {
    do {
      /* Respond to SIGUSR2 signals from monitor */
      while (sigusr2count > 0) {

#ifdef PENDEBUG
  Error(0, "Calling Poll pid: %d Count: %d.",
	(int) getpid(), 
	sigusr2count);
#endif
	/* Get message from monitor */
	if (Poll() != SUCCESS)
	  LogMessage(uid, "dbs: failed in Poll");
	sigusr2count--;
      }
      /* Perform select call */
      errno = 0; timev.tv_sec = TIME_OUT_LIMIT; timev.tv_usec = 0;
      smask = selectmask;
    } while (((sc = select(selectmaskno, &smask,
			   (fd_set *) NULL, (fd_set *) NULL, &timev)) < 0)
	     && (errno == EINTR));

    /* Check that all went well */
    if (sc == 0) {
      Error(ER_SYSERR, "dbs: Connection timed out (%d)", errno);
      Logout(COMTIMEOUT);
    } else if (sc < 0) {
      Error(ER_SYSERR, "dbs: Select call failed (%d)", errno);
      return (FAIL);
    } else if (!FD_ISSET(socketrd, &smask)) {
      Error(ER_SYSERR, "dbs: select exited, but no data on socket");
      return (FAIL);
      /* possible use do a continue to the top of the loop, since this
       * can signal an oob or exception that was read, but we cannot
       * clear it until we read some "in-band" data (under SunOS). But
       * since we do send oob from dbs to ws (yet).
      continue;
       */
    }
    /* Read byte from stream */
    for(;;) {
      if ((sc = read(socketrd, (char *)&buff, (IOSIZE_T)1)) == 1) 
	break;
      else {
	if (sc == -1 && errno == EINTR) /* 931103. msj */
	  continue;
	if ((sc == 0) ||		/* eof */
	    (sc == -1 && (errno == ECONNRESET || errno == ETIMEDOUT))) {
	  Error(ER_SYSERR, "dbs: Connection broken (%d)", errno);
	  Logout(COMBROKEN);
	}
	Error(ER_READ, "dbs: Could not read(1) = %d (%d)", sc, errno);
	return (FAIL);
      }
    }

    /* Check first byte was SPECBYTE */
    if (buff == SPECBYTE)
      break;
    else {
      if (!count++) {
	LogMessage(uid, "dbs: Command stream out of order");
	continue;
      }
      if (count++ > 100) {
	LogMessage(uid, "dbs: FATAL command stream disorder");
	Logout(COMSTREAM);
      }
    }
  }

  /* Read byte from stream */
  if ((sc = read(socketrd, (char *)command, (IOSIZE_T)1)) != 1) {
    Error(ER_READ, "dbs: Could not read(2)=%d  (%d)", sc, errno);
    return (FAIL);
  }
#ifdef DEBUGCMD
  else
    LogMessage(uid, "dbs command %d", (unsigned int)*command);
#endif /* DEBUG */

  /* All went well */
  return (SUCCESS);
}

/*  */
/**********************************************************************
 * Function: static int ReceiveIndex(WS_LABEL *val)
 *
 * Gets an integer index from the socket connection
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int ReceiveIndex(val)
  WS_LABEL *val;
{
#ifndef LABEL_64_BITS
  return (ReceiveCount((t_32bits *)val, socketrd));
#else
  int      st;
  t_32bits index;
  st = ReceiveCount(&index, socketrd);
  *val = index;
  return(st);
#endif /* n LABEL_64_BITS */
}

/*  */
/**********************************************************************
 * Function: static int SendIndex(WS_LABEL val)
 *
 * Sends an integer index to the socket connection
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int SendIndex(val)
  WS_LABEL val;
{
  return(SendCount((t_32bits) val, socketwr));
}

/*  */
/**********************************************************************
 * Function: static int ReceiveMolType(int *val)
 *
 * Gets the molecule type from the socket connection
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int ReceiveMolType(val)
  int *val;
{
  /* Assume that sizeof(int) == sizeof(t_32bits) */
  return(ReceiveCount((t_32bits *)val, socketrd));
}

/*  */
/**********************************************************************
 * Function: static int ReplyPack(unsigned mess)
 *
 * Sends an attention byte followed by the byte supplied
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int ReplyPack(mess)
  unsigned mess;
{
  static byte buff[2] =
  {SPECBYTE, 0};

  /* Copy byte into string */
  buff[1] = (byte) mess;

  /* Send buffer */
  if (write(socketwr, (char *)buff, (IOSIZE_T) 2) != 2) {
    Error(ER_WRITE, "ReplyPack: could not write (%d)", errno);
    return (FAIL);
  }
  /* All went well */
  return (SUCCESS);
}

/*  */
/**********************************************************************
 * Function: static int ReplyString(char *str)
 *
 * Sends a string to the Lisp machine
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int ReplyString(str)
  char *str;
{
  byte len;

  /* Get length of string */
  len = strlen(str);

  /* Send string */
  if (write(socketwr, (char *)&len, sizeof(len)) != sizeof(len)) {
    Error(ER_WRITE, "ReplyString: could not write (%d)", errno);
    return (FAIL);
  }
  if (write(socketwr, str, (IOSIZE_T)len) != len) {
    Error(ER_WRITE, "ReplyString: could not write (%d)", errno);
    return (FAIL);
  }
  /* All went well */
  return (SUCCESS);
}

/*  */
/**********************************************************************
 * Function: void UserError(int d, char *s)
 *
 * Notifies the user of an error condition
 *
 * Modifications:
 *      <list mods with name and date>
 */
void UserError(d, s)
  int d;
  char *s;
{
  static unsigned char buff[3] =
  {SPECBYTE, RCERROR, 0};

  /*
   * Error messages are sent to workspace only in synchronous mode.
   */
  if (!synchronous)
    return;

  /* Copy byte into buffer */
  buff[2] = d;

  /* Send buffer */
  if (write(socketwr, (char *)buff, (IOSIZE_T)3) != 3) {
    Error(ER_WRITE, "UserError: could not write (%d)", errno);
    return;
  }
  /* Send error string */
  (void)ReplyString(s);
}

/*  */
/**********************************************************************
 * Function: static int InitSignals()
 *
 * Sets up monitor signals
 * SIGUSR1 used to start gobble, but it's been turned off
 *
 * Modifications:
 *      Sat Oct  9 19:28:05 1993 Martin Sjlin. Added check for SIGURG
 *                   and if not defined try SIGPOLL instead (SCO system).
 *      Thu Nov 11 10:25:19 1993 Martin Sjlin - SCO uses SIGUSR1 for
 *                   SIGURG (OOB) and SIGPOLL for async I/O.
 */
static int InitSignals()
{
  /* Register signals to take care of */
  if (Signal(SIGPIPE, Handler) == BADSIG) {
    ErrSignal(uid, "SIGPIPE");
    return (FAIL);
  }
#ifdef SIGURG
  if (Signal(SIGURG, Handler) == BADSIG) {
    ErrSignal(uid, "SIGURG");
    return (FAIL);
  }
#else
#ifdef SIGUSR1_FOR_OOB
   if (Signal(SIGUSR1, Handler) == BADSIG) {
    ErrSignal(uid, "SIGUSR1");
    return (FAIL);
  }
#else
   if (Signal(SIGPOLL, Handler) == BADSIG) {
    ErrSignal(uid, "SIGPOLL");
    return (FAIL);
  }
#endif /* n SIGUSR1_FOR_OOB */
#endif /* n SIGURG */

#ifndef SIGUSR1_FOR_OOB
  if (Signal(SIGUSR1, Handler) == BADSIG) {
    ErrSignal(uid, "SIGUSR1");
    return (FAIL);
  }
#endif /* n SIGUSR1_FOR_OOB */

  if (Signal(SIGUSR2, Handler) == BADSIG) {
    ErrSignal(uid, "SIGUSR2");
    return (FAIL);
  }
  /* Register "clean-up" signals */
  if (Signal(SIGTERM, Handler) == BADSIG) {
    ErrSignal(uid, "SIGTERM");
    return (FAIL);
  }
  if (Signal(SIGHUP, Handler) == BADSIG) {
    ErrSignal(uid, "SIGHUP");
    return (FAIL);
  }
  if (Signal(SIGQUIT, Handler) == BADSIG) {
    ErrSignal(uid, "SIGQUIT");
    return (FAIL);
  }
#ifdef SIGLOST
  if (Signal(SIGLOST, Handler) == BADSIG) {
    ErrSignal(uid, "SIGLOST");
    return (FAIL);
  }
#endif

  /* Register signals to ignore */
  (void)Signal(SIGALRM, SIG_IGN);
  (void)Signal(SIGINT, SIG_IGN);
#ifdef SIGXFSZ
  (void)Signal(SIGXFSZ, SIG_DFL);
#endif /* SIGXFSZ */
  (void)Signal(SIGCHLD, SIG_IGN);

  return (SUCCESS);
}
