/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* 
 * MODULE NAME: 	blinks.c
 *
 * SCCSINFO:		@(#)blinks.c	1.9 6/7/94
 *
 * ORIGINAL AUTHOR(S):  ????, 10 May, 1987
 *
 * MODIFICATIONS:
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 *	This file contains the global function UpdateBlinks.
 *	Back links are only kept on vslabels
 *********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************
 * int UpdateBlinks(LINKGROUP *newblinks, LINKGROUP *oldblinks,
 *		    LABEL tolabel)
 */

/*********************************************************************
 * INCLUDES:
 *********************************************************************/
/* so that lint won't know about this file */
#ifndef lint

#include "config.h"	/* includes system dependent includes */
#include "lincks.h"
#include "dbserver.h"
#include "libshared.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_dbserrors.h"
#include "f_molfile.h"
#include "f_dbrdmol.h"
#include "f_dbwrmol.h"

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
/* External file table definition */
extern TABENTRY *filetable;		/* dbserv.c */

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
/* Local link table definition */
typedef struct
  {
    char *groupname;
    char *linkstag;
    LABEL *vslabel;
    short count;
  } LINKTABLEENTRY;

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static int AddNewBlink P_(( LABEL tolabel, char *name, char *tag, LABEL
                             fromvslabel, int count ));
static int BuildLinkTable P_(( LINKTABLEENTRY *linktableroot[],
                                LINKGROUP *linkroot ));
static int DelOldBlink P_(( LABEL tolabel, char *name, char *tag, LABEL
                             fromvslabel, int count ));
static int FindLinkInTable P_(( LINKTABLEENTRY linktable[], int
                                 tablesize, char *groupname, char
                                 *linkstag, LABEL *vslabel ));
static int HashLinkTable P_(( LINKTABLEENTRY *linktableroot[], int
                               tablesize, char *groupname, char
                               *linkstag, LABEL *vslabel ));
static int HashTableSize P_(( LINKGROUP *linkgroup ));

/*********************************************************************
 * INTERNAL (STATIC) DATA: 
 *********************************************************************/
/* none */

    
/*  */
/**********************************************************************
 * Function: int UpdateBlinks(THREE PARAMETERS)
 * Parameters:
 *	LINKGROUP *newblinks
 *	LINKGROUP *oldblinks
 *	LABEL tolabel;
 *
 * Updates the blink sections of molecules refered
 * to by tolabel by using it's old and new flink
 * structures.
 * Uses BuildLinkTable, FindLinkInTable, AddNewBlink,
 * and DelOldBlink
 * 
 * Modifications:
 *      <list mods with name and date>
 */
int UpdateBlinks(newblinks, oldblinks, tolabel)
  LINKGROUP *newblinks;
  LINKGROUP *oldblinks;
  LABEL tolabel;
{
  LINKTABLEENTRY *newblinktable, *oldblinktable;
  LINKTABLEENTRY *newentry, *oldentry;
  int newtablesize, oldtablesize;
  int pos, atpos;

  /* Build new blink table */
  if ((newtablesize = BuildLinkTable(&newblinktable, newblinks)) < 0)
    {
      newblinktable = oldblinktable = NULL;
      goto fail;
    }

  /* Build old blink table */
  if ((oldtablesize = BuildLinkTable(&oldblinktable, oldblinks)) < 0)
    {
      oldblinktable = NULL;
      goto fail;
    }

  /* Go through new blink table */
  for (pos = 0, newentry = newblinktable;
       pos < newtablesize;
       ++pos, ++newentry)
    {
      /* See if it's used */
      if (newentry->count)
	{
	  /* Find corresponding link in old link table */
	  if ((atpos = FindLinkInTable(oldblinktable, oldtablesize,
				       newentry->groupname,
				       newentry->linkstag,
				       newentry->vslabel)) < 0)
	    {
	      /* If not found, add link(s) */
	      if (AddNewBlink(tolabel,
			      newentry->groupname,
			      newentry->linkstag,
			      *newentry->vslabel,
			      newentry->count) < 0)
		goto fail;
	    }
	  else
	    {
	      /* If found, check that count is same */
	      oldentry = &oldblinktable[atpos];

	      if (oldentry->count == newentry->count)
		{
		  /* If same, mark old table entry as processed */
		  oldentry->count = -1;
		}
	      else
		{
		  /* If not same, add or delete links until it is */
		  if (oldentry->count > newentry->count)
		    {
		      if (DelOldBlink(tolabel,
				      oldentry->groupname,
				      oldentry->linkstag,
				      *oldentry->vslabel,
				      oldentry->count - newentry->count) < 0)
			goto fail;
		    }
		  else
		    {
		      if (AddNewBlink(tolabel,
				      newentry->groupname,
				      newentry->linkstag,
				      *newentry->vslabel,
				      newentry->count - oldentry->count) < 0)
			goto fail;
		    }

		  /* Mark old table entry as processed */
		  oldentry->count = -1;
		}
	    }
	}
    }

  for (pos = 0, oldentry = oldblinktable;
       pos < oldtablesize;
       ++pos, ++oldentry)
    {
      if (oldentry->count > 0)
	{
	  if (DelOldBlink(tolabel,
			  oldentry->groupname,
			  oldentry->linkstag,
			  *oldentry->vslabel,
			  oldentry->count) < 0)
	    goto fail;
	}
    }

  /* Free table memory */
  free((FREEPTR *)oldblinktable);
  free((FREEPTR *)newblinktable);

  /* Return status */
  return(SUCCESS);


 fail:
  /* Failure exit point */

  /* Free table memory */
  free((FREEPTR *)oldblinktable);
  free((FREEPTR *)newblinktable);

  /* Return status */
  return(FAIL);
}

/*  */
/**********************************************************************
 * Function: static int BuildLinkTable(TWO PARAMETERS)
 * Parameters:
 *	LINKTABLEENTRY *linktableroot[]
 *	LINKGROUP *linkroot
 *
 * Builds a link table from the link structure
 * The table is of format:
 *   { groupname, linkstag, vslblp, itmlblp, count }
 * Uses HashLinkTable
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int BuildLinkTable(linktableroot, linkroot)
  LINKTABLEENTRY *linktableroot[];
  LINKGROUP *linkroot;
{
  LINKTABLEENTRY *entry;
  LINKGROUP *group;
  LINKS *links;
  LINKITEM *item;
  int tablesize, pos;

  /* Get proper link table size */
  tablesize = HashTableSize(linkroot);

  /* Create link table */
  if ((*linktableroot = (LINKTABLEENTRY *)
          malloc((ALLOC_T)(tablesize * sizeof(LINKTABLEENTRY)))) == NULL)
    {
      Error(ER_MALLOC, "No more memory");
      return(FAIL);
    }

  /* Set initial table element pointer */
  entry = *linktableroot;

  /* Initialize table */
  for (pos = 0; pos < tablesize; ++pos, ++entry)
    entry->count = 0;
  
  /* Enter links into table */
  group = linkroot;
  while (group)
    {
      links = group->links;
      while (links)
	{
	  item = links->linkitem;
	  while (item)
	    {
	      /* Find proper position in table and insert it there */
	      if (HashLinkTable(linktableroot, tablesize,
				group->name, links->tag,
				&item->vs) < 0)
		{
		  return(FAIL);
		}
	      item = item->nextitem;
	    }
	  links = links->nextlinks;
	}
      group = group->nextgroup;
    }

  return(tablesize);
}

/*  */
/**********************************************************************
 * Function: static int HashLinkTable(FIVE PARAMETERS)
 * Parameters:
 *	LINKTABLEENTRY *linktableroot[];
 *	int tablesize;
 *	char *groupname;
 *	char *linkstag;
 *	LABEL *vslabel;
 * 
 * Finds a place in the table for an entry. If there
 * already is an identical entry in the table then
 * it's count is incremented.
 * Uses Error
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int HashLinkTable(linktableroot, tablesize, groupname, linkstag, vslabel)
  LINKTABLEENTRY *linktableroot[];
  int tablesize;
  char *groupname;
  char *linkstag;
  LABEL *vslabel;
{
  int pos, increment = 1;
  LINKTABLEENTRY *entry;

  /* Get hash position */
  pos = (*vslabel>>1) % tablesize;
  entry = &(*linktableroot)[pos];

  /* Check if position is occupied */
  /* Check if occupant is identical to this link */
  /* Or find another empty slot */
  while (entry->count && (*entry->vslabel != *vslabel ||
			  strcmp(entry->groupname, groupname) ||
			  strcmp(entry->linkstag, linkstag)))
    {
      /* Go to next position */
      pos += increment;
      entry += increment;
      increment += 2;
      
      /* Check if we've run over the table edge */
      if (pos >= tablesize)
	{
	  pos = pos - tablesize;
	  entry = &(*linktableroot)[pos];
	}
      
      /* Check if table is full (should not happen) */
      if (increment > tablesize)
	{
	  Error(ER_SYSERR, "HashLinkTable: Hash table full");
	  free((FREEPTR *)*linktableroot);
	  return(FAIL);
	}
    }

  /* Check if an identical entry was found */
  if (entry->count)
    ++entry->count;
  else
    /* If not, add new entry */
    {
      entry->groupname = groupname;
      entry->linkstag = linkstag;
      entry->vslabel = vslabel;
      entry->count = 1;
    }

  /* Return status */
  return(SUCCESS);
}

/*  */
/**********************************************************************
 * Function: static int FindLinkInTable(FIVE PARAMETERS)
 * Parameters:
 *	LINKTABLEENTRY linktable[]
 *	int tablesize
 *	void *groupname
 *	char *linkstag
 *	LABEL *vslabel
 * 
 *  Finds a link in the link table.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int FindLinkInTable(linktable, tablesize, groupname, linkstag, vslabel)
  LINKTABLEENTRY linktable[];
  int tablesize;
  char *groupname;
  char *linkstag;
  LABEL *vslabel;
{
  int pos, increment = 1;
  LINKTABLEENTRY *entry;

  /* Get hash position */
  pos = (*vslabel>>1) % tablesize;
  entry = &linktable[pos];

  /* Check if this position is it */
  /* Scan forward if it isn't */
  while (entry->count &&
	 (*entry->vslabel != *vslabel ||
	  strcmp(entry->groupname, groupname) ||
	  strcmp(entry->linkstag, linkstag)))
    {
      /* Go to next position */
      pos += increment;
      entry += increment;
      increment += 2;

      /* Check if we've run over the table edge */
      if (pos >= tablesize)
        {
	  pos = pos - tablesize;
	  entry = &linktable[pos];
        }
	    
        /* Check if it's time to quit */
        if (increment > tablesize)
        {
	  return(FAIL);
        }
    }

  /* Check if we found an entry or not */
  if (entry->count)
    return(pos);
  else
    return(FAIL);
}

/*  */
/**********************************************************************
 * Function: static int DelOldBlink(FIVE PARAMETERS)
 * Parameters:
 *	LABEL tolabel
 *	char *name
 *	char *tag
 *	LABEL fromvslabel
 *	int count
 *
 * Removes the old back link from molecule fromvslabel
 * to molecule tolabel. The updated molecule is saved and updated
 *
 * Uses MolAccess, ReadMolFile, GetMolfd, ReadLinks,
 * DelItem, WriteLinks, UpdateMolFile and Error
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int DelOldBlink(tolabel, name, tag, fromvslabel, count)
  LABEL tolabel;
  char *name, *tag;
  LABEL fromvslabel;
  int count;
{
  A_ENTRY *access;
  FILEMOL *molentry;
  LINKGROUP *linkroot;
  WRITEINFO *wrinfo = NULL;
  int index;

  /* Get molecule access data */
  if ((access = MolAccess(fromvslabel, LOCKACCESS)) == NULL)
    return(FAIL);

  /* Get file descriptor corresponding to molecule file name */
  if ((index = GetMolfd(access->a_filename)) < 0)
    return(FAIL);

  /* Get molecule file entry */
  if ((molentry = ReadMolFile(access, index)) == NULL)
    return(FAIL);

  /* Get link structure from data file */
  if (ReadLinks(&linkroot, (off_t) molentry->blink_filepos,
		molentry->blink_size,
		filetable[index].datfile.fd) < 0)
    {
      return(FAIL);
    }

  /* Remove link item(s) specified */
  while (count--)
    linkroot = DelLink(linkroot, name, tag, tolabel, 0L);

  /* Write new link structure to data file */
  if (WriteLinks(wrinfo, linkroot, filetable[index].datfile.fd) != SUCCESS)
    {
      return(FAIL);
    }

  /* Set new molecule file info */
  molentry->blink_filepos = wrinfo->wrpos;
  molentry->blink_size = wrinfo->wrsize;

  /* Update molecule file entry */
  if (UpdateMolFile(molentry, fromvslabel, index) < 0)
    return(FAIL);

  /* Remove lock on molecule */
  (void)Unlock(fromvslabel);

  /* Free memory */
  FreeLinks(linkroot);

  /* All went well */
  return(SUCCESS);
}

/*  */
/**********************************************************************
 * Function: static int AddNewBlink(FIVE PARAMETNERS)
 * Parameters:
 *	LABEL tolabel
 *	char *name
 *	char *tag
 *	LABEL fromvslabel
 *	int count
 * 
 * Adds a new back link from fromvslabel to the molecule
 * tolabel and saves the updated molecule
 * Uses LockAccess, ReadMolFile, GetMolfd, ReadLinks,
 * FindGroup, FindLinks, FindItem, AddGroup,
 * AddLinks, AddItem, WriteLinks, UpdateMolFile and Error
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int AddNewBlink(tolabel, name, tag, fromvslabel, count)
  LABEL tolabel;
  char *name, *tag;
  LABEL fromvslabel;
  int count;
{
  A_ENTRY *access;
  FILEMOL *molentry;
  WRITEINFO *wrinfo = NULL;
  LINKGROUP *linkroot, *newroot;
  int index;

  /* Get molecule access data */
  if ((access = MolAccess(fromvslabel, LOCKACCESS)) == NULL)
    return(FAIL);

  /* Get file descriptor corresponding to molecule file name */
  if ((index = GetMolfd(access->a_filename)) < 0)
    return(FAIL);

  /* Get molecule file entry */
  if ((molentry = ReadMolFile(access, index)) == NULL)
    return(FAIL);

  /* Get link structure from data file */
  if (ReadLinks(&linkroot, (off_t) molentry->blink_filepos,
		molentry->blink_size,
		filetable[index].datfile.fd) < 0)
    {
      return(FAIL);
    }

  /* Add new link(s) to link structure */
  while (count--)
    {
      if ((newroot = AddLink(linkroot, name, tag, tolabel, 0L)) == NULL)
	{
          FreeLinks(linkroot);
	  return(FAIL);
	}
      linkroot = newroot;
    }
	
  /* Write new link structure to data file */
  if (WriteLinks(wrinfo, linkroot, filetable[index].datfile.fd) != SUCCESS)
    {
      return(FAIL);
    }

  /* Set new molecule file info */
  molentry->blink_filepos = wrinfo->wrpos;
  molentry->blink_size = wrinfo->wrsize;

  /* Update molecule file entry */
  if (UpdateMolFile(molentry, fromvslabel, index) < 0)
    return(FAIL);

  /* Remove lock on molecule */
  (void)Unlock(fromvslabel);

  /* Free memory */
  FreeLinks(linkroot);

  /* All went well */
  return(SUCCESS);
}

/*  */
/**********************************************************************
 * Function: static int HashTableSize(LINKGROUP *linkgroup)
 * 
 * Counts the number of links and calculates how large
 * the hash table should be.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int HashTableSize(linkgroup)
  LINKGROUP *linkgroup;
{
  static int primes[] = {23, 43, 83, 163, 331, 641, 1283, 2579, 5147, 10243};
  int i, count;
  LINKS *links;
  LINKITEM *linkitem;

  count = 0;

  while (linkgroup) {
      links = linkgroup->links;
      while (links) {
	  linkitem = links->linkitem;
	  while (linkitem) {
	      ++count;
	      linkitem = linkitem->nextitem;
	  }
	  links = links->nextlinks;
      }
      linkgroup = linkgroup->nextgroup;
  }

  /* Calculate table size */
  count *= 2;
  for (i = 0; i < sizeof(primes) && count > primes[i]; ++i);

  /* If no prime available: return count plus one (to make it odd) */
  if (i == sizeof(primes))
    return(count + 1);
  
  /* Return table size */
  return(primes[i]);
}
#endif
