--[[
   Script to update outfits and ships from a saved game in the case they don't exist.
--]]
local fmt = require "format"
local changes_done = {}
local save_updated

--[[
   Run before anything is done, so it can be used to store what was changed.
--]]
function start ()
   changes_done = {}
   save_updated = false
   naev.cache().save_updater = {}
end

--[[
   Run after finished processing. Allows doing additional changes as necessary.
--]]
function finish ()
   if not save_updated then
      return
   end
   local cache = naev.cache().save_updater
   cache.split_list = {}

   -- Old cores that now use the primary / secondary system (0.13.0)
   local split = {
      ["Krain Patagium Engine"]                 = true,
      ["Melendez Mammoth XL Engine"]            = true,
      ["Nexus Bolt 6500 Engine"]                = true,
      ["Tricon Typhoon II Engine"]              = true,
      ["Unicorp Eagle 6500 Engine"]             = true,
      ["Melendez Buffalo XL Engine"]            = true,
      ["Nexus Arrow 1400 Engine"]               = true,
      ["Tricon Cyclone II Engine"]              = true,
      ["Unicorp Falcon 1400 Engine"]            = true,
      ["Melendez Ox XL Engine"]                 = true,
      ["Nexus Dart 360 Engine"]                 = true,
      ["Tricon Zephyr II Engine"]               = true,
      ["Unicorp Hawk 360 Engine"]               = true,
      ["Milspec Orion 9901 Core System"]        = true,
      ["Milspec Thalos 9802 Core System"]       = true,
      ["Unicorp PT-1750 Core System"]           = true,
      ["Milspec Orion 5501 Core System"]        = true,
      ["Milspec Thalos 5402 Core System"]       = true,
      ["Unicorp PT-310 Core System"]            = true,
      ["Milspec Orion 3701 Core System"]        = true,
      ["Milspec Thalos 3602 Core System"]       = true,
      ["Unicorp PT-68 Core System"]             = true,
      ["Unicorp D-72 Heavy Plating"]            = true,
      ["S&K Superheavy Combat Plating"]         = true,
      ["Unicorp D-38 Medium Plating"]           = true,
      ["S&K Medium-Heavy Combat Plating"]       = true,
      ["Unicorp D-9 Light Plating"]             = true,
      ["S&K Light Combat Plating"]              = true,
      ["Nexus Light Stealth Plating"]           = true,
      -- Updates for even older cores from 0.11.0 to 0.12.0
      ["Unicorp PT-2200 Core System"]           = true,
      ["Unicorp PT-500 Core System"]            = true,
      -- Updates for even older cores from 0.10.0 to 0.11.0
      ["Unicorp D-68 Heavy Plating"]            = true,
      ["Unicorp D-24 Medium Plating"]           = true,
      ["Unicorp D-4 Light Plating"]             = true,
      -- Updates for even older cores from 0.8.2 to 0.9.0
      ["Milspec Prometheus 9803 Core System"]   = true,
      ["Milspec Prometheus 5403 Core System"]   = true,
      ["Milspec Prometheus 3603 Core System"]   = true,
      ["Milspec Aegis 9801 Core System"]        = true,
      ["Milspec Aegis 5401 Core System"]        = true,
      ["Milspec Aegis 3601 Core System"]        = true,
   }

   for original,value in pairs(changes_done) do
      if split[original] then
         player.outfitAdd( value.new, value.q ) -- Have to add an additional core to compensate
         print( fmt.f("   {original} => {new} x2 [{q}]", {original=original, new=value.new, q=value.q} ) )

         -- Used to inform the updater event that cores probably need some checking
         cache.split_list[value.new] = true
         cache.split_cores = true
      elseif value.new ~= nil then
         print( fmt.f("   {original} => {new} [{q}]", {original=original, new=value.new, q=value.q} ) )
      else
         print( fmt.f("   {original} => nil [{q}]", {original=original, q=value.q} ) )
      end
   end
end

local function apply_change( original, new, q )
   local value = changes_done[original]
   if value then
      value.q = value.q + q
   else
      changes_done[original] = { new=new, q=q }
   end
   save_updated = true
   return new
end

--[[
   The format is ["oldname"] = newvalue where newvalue can either take a string
   for the new name of the ship (if there is a direct equivalent) or a number
   value indicating the amount of credits to refund the player.
--]]
local ship_list = {
   ["Proteron Derivative"] = "Proteron Dalton",
   ["Proteron Kahan"]      = "Proteron Gauss",
}

--[[--
   Takes an ship name and should return either a new ship name or the amount of credits to give back to the player.
--]]
function ship( name, q )
   return apply_change( name, ship_list[name], q )
end

--[[
   The format is ["oldname"] = newvalue where newvalue can either take a string
   for the new name of the outfit (if there is a direct equivalent) or a number
   value indicating the amount of credits to refund the player.
--]]
local outfit_list = {
   -- Below is a list of changes from 0.12.0 to 0.13.0
   ["Ratchet Point Defense"]              = "Ratchet Point Defence",
   ["Racing Trophy"]                      = "Racing Trophy (Silver)", -- getting the gold is now much harder
   ["Avatar of Sirichana"]                = "Avatar of the Sirichana",
   ["Auxiliary Proccessing Unit I"]       = "Auxiliary Processing Unit I", -- codespell:ignore Proccessing
   ["Auxiliary Proccessing Unit II"]      = "Auxiliary Processing Unit II", -- codespell:ignore Proccessing
   ["Auxiliary Proccessing Unit III"]     = "Auxiliary Processing Unit III", -- codespell:ignore Proccessing
   ["Auxiliary Proccessing Unit IV"]      = "Auxiliary Processing Unit IV", -- codespell:ignore Proccessing
   ["Adaptive Stealth Plating"]           = "Adaptive Camouflage Plating",
   ["Nexus Stealth Coating"]              = "Nexus Concealment Coating",
   ["Camouflage Burster"]                 = "Stealth Burster",

   -- Multicore transformation (engines).
   -- Each core in the left is split in 2 cores the right, one in the main core slot, the other one on the secondary core slot.
   -- See finish() above.
   ["Krain Patagium Engine"]              = "Krain Patagium Twin Engine",
   ["Melendez Mammoth XL Engine"]         = "Melendez Mammoth Engine",
   ["Nexus Bolt 6500 Engine"]             = "Nexus Bolt 3000 Engine",
   ["Tricon Typhoon II Engine"]           = "Tricon Typhoon Engine",
   ["Unicorp Eagle 6500 Engine"]          = "Unicorp Eagle 3000 Engine",
   ["Melendez Buffalo XL Engine"]         = "Melendez Buffalo Engine",
   ["Nexus Arrow 1400 Engine"]            = "Nexus Arrow 700 Engine",
   ["Tricon Cyclone II Engine"]           = "Tricon Cyclone Engine",
   ["Unicorp Falcon 1400 Engine"]         = "Unicorp Falcon 700 Engine",
   ["Melendez Ox XL Engine"]              = "Melendez Ox Engine",
   ["Nexus Dart 360 Engine"]              = "Nexus Dart 160 Engine",
   ["Tricon Zephyr II Engine"]            = "Tricon Zephyr Engine",
   ["Unicorp Hawk 360 Engine"]            = "Unicorp Hawk 160 Engine",

   -- Just renamings
   ["Nexus Ultralight Stealth Plating"]   = "Nexus Shadow Weave",
   ["Nexus Medium Stealth Plating"]       = "Nexus Ghost Weave",
   ["Nexus Heavy Stealth Plating"]        = "Nexus Phantasm Weave",
   ["S&K Ultralight Combat Plating"]      = "S&K Skirmish Plating",
   ["S&K Medium Combat Plating"]          = "S&K Battle Plating",
   ["S&K Heavy Combat Plating"]           = "S&K War Plating",

   -- Multicore transformation (hulls).
   -- Each core in the left is split in 2 cores the right, one in the main core slot, the other one on the secondary core slot.
   -- See finish() above.
   ["Nexus Light Stealth Plating"]        = "Nexus Shadow Weave",
   ["Unicorp D-72 Heavy Plating"]         = "Unicorp D-58 Heavy Plating",
   ["S&K Superheavy Combat Plating"]      = "S&K War Plating",
   ["Unicorp D-38 Medium Plating"]        = "Unicorp D-23 Medium Plating",
   ["S&K Medium-Heavy Combat Plating"]    = "S&K Battle Plating",
   ["Unicorp D-9 Light Plating"]          = "Unicorp D-2 Light Plating",
   ["S&K Light Combat Plating"]           = "S&K Skirmish Plating",
   ["Milspec Orion 9901 Core System"]     = "Milspec Orion 8601 Core System",
   ["Milspec Thalos 9802 Core System"]    = "Milspec Thalos 8502 Core System",
   ["Unicorp PT-1750 Core System"]        = "Unicorp PT-440 Core System",
   ["Milspec Orion 5501 Core System"]     = "Milspec Orion 4801 Core System",
   ["Milspec Thalos 5402 Core System"]    = "Milspec Thalos 4702 Core System",
   ["Unicorp PT-310 Core System"]         = "Unicorp PT-200 Core System",
   ["Milspec Orion 3701 Core System"]     = "Milspec Orion 2301 Core System",
   ["Milspec Thalos 3602 Core System"]    = "Milspec Thalos 2202 Core System",
   ["Unicorp PT-68 Core System"]          = "Unicorp PT-16 Core System",

   -- Below is a list of changes from 0.11.0 to 0.12.0
   ["Unicorp PT-2200 Core System"]        = "Unicorp PT-440 Core System",
   ["Unicorp PT-500 Core System"]         = "Unicorp PT-440 Core System",

   -- Below is a list of changes from 0.10.0 to 0.11.0
   ["Unicorp D-68 Heavy Plating"]         = "Unicorp D-72 Heavy Plating",
   ["Unicorp D-48 Heavy Plating"]         = "Unicorp D-58 Heavy Plating",
   ["Unicorp D-24 Medium Plating"]        = "Unicorp D-38 Medium Plating",
   ["Unicorp D-12 Medium Plating"]        = "Unicorp D-23 Medium Plating",
   ["Unicorp D-4 Light Plating"]          = "Unicorp D-9 Light Plating",
   ["Unicorp Eagle 7000 Engine"]          = "Unicorp Eagle 6500 Engine",
   ["Nexus Bolt 3500 Engine"]             = "Nexus Bolt 3000 Engine",
   ["Unicorp Falcon 1300 Engine"]         = "Unicorp Falcon 1400 Engine",
   ["Unicorp Hawk 350 Engine"]            = "Unicorp Hawk 360 Engine",
   ["Nexus Dart 150 Engine"]              = "Nexus Dart 160 Engine",
   ["Razor MK1"]                          = "Razor Artillery S1",
   ["Razor MK2"]                          = "Razor Artillery S2",
   ["Slicer"]                             = "Razor Artillery S3",
   ["Razor Turret MK1"]                   = 50e3,
   ["Razor Turret MK2"]                   = "Razor Battery S2",
   ["Tesla Cannon"]                       = 780e3,
   ["Heavy Razor Turret"]                 = "Disruptor Battery S2",
   ["Fidelity Bay"]                       = "Sirius Fidelity Bay",

   -- Below is a list of changes from 0.9.0 to 0.10.0
   ["Drone Fighter Bay"]                  = "Drone Bay",
   ["Empire Lancelot Fighter Bay"]        = "Empire Lancelot Bay",
   --["Fidelity Fighter Bay"]             = "Fidelity Bay",
   ["Hyena Fighter Dock"]                 = "Hyena Bay",
   ["Hyena Fighter Bay"]                  = "Hyena Dock",
   ["Lancelot Fighter Bay"]               = "Lancelot Bay",
   ["Pirate Hyena Fighter Dock"]          = "Pirate Hyena Bay",
   ["Pirate Hyena Fighter Bay"]           = "Pirate Hyena Dock",
   ["Proteron Derivative Bay"]            = "Proteron Dalton Bay",
   ["Proteron Derivative Fighter Bay"]    = "Proteron Dalton Bay",
   ["Soromid Brigand Fighter Bay"]        = "Soromid Brigand Bay",
   ["Thurion Perspicacity Fighter Dock"]  = "Thurion Perspicacity Bay",
   ["Thurion Perspicacity Fighter Bay"]   = "Thurion Perspicacity Dock",
   ["Za'lek Bomber Drone Fighter Dock"]   = "Za'lek Bomber Drone Bay",
   ["Za'lek Bomber Drone Fighter Bay"]    = "Za'lek Bomber Drone Mini Bay",
   ["Za'lek Heavy Drone Fighter Dock"]    = "Za'lek Heavy Drone Bay",
   ["Za'lek Heavy Drone Fighter Bay"]     = "Za'lek Heavy Drone Mini Bay",
   ["Za'lek Light Drone Fighter Dock"]    = "Za'lek Light Drone Bay",
   ["Za'lek Light Drone Fighter Bay"]     = "Za'lek Light Drone Mini Bay",
   ["Unicorp Banshee Launcher"]           = 35e3,
   ["Unicorp Mace Launcher"]              = 19e3,
   ["Unicorp Fury Launcher"]              = 65e3,
   ["Unicorp Headhunter Launcher"]        = 70e3,
   ["Unicorp Medusa Launcher"]            = 73e3,
   ["Unicorp Vengeance Launcher"]         = 78e3,

   -- Below are the 0.9.x naming scheme and prices
   ["BioPlasma Claw Stage 1"] = 8500,
   ["BioPlasma Claw Stage 2"] = 10600,
   ["BioPlasma Claw Stage 3"] = 12700,
   ["BioPlasma Claw Stage 4"] = 14800,
   ["BioPlasma Claw Stage X"] = 17000,
   ["BioPlasma Fang Stage 1"] = 21000,
   ["BioPlasma Fang Stage 2"] = 25200,
   ["BioPlasma Fang Stage 3"] = 29400,
   ["BioPlasma Fang Stage 4"] = 33600,
   ["BioPlasma Fang Stage 5"] = 37800,
   ["BioPlasma Fang Stage X"] = 42000,
   ["BioPlasma Stinger Stage 1"] = 4500,
   ["BioPlasma Stinger Stage 2"] = 6000,
   ["BioPlasma Stinger Stage 3"] = 7500,
   ["BioPlasma Stinger Stage X"] = 9000,
   ["BioPlasma Talon Stage 1"] = 36000,
   ["BioPlasma Talon Stage 2"] = 42000,
   ["BioPlasma Talon Stage 3"] = 48000,
   ["BioPlasma Talon Stage 4"] = 54000,
   ["BioPlasma Talon Stage 5"] = 60000,
   ["BioPlasma Talon Stage 6"] = 66000,
   ["BioPlasma Talon Stage X"] = 72000,
   ["BioPlasma Tentacle Stage 1"] = 62500,
   ["BioPlasma Tentacle Stage 2"] = 71000,
   ["BioPlasma Tentacle Stage 3"] = 80000,
   ["BioPlasma Tentacle Stage 4"] = 89000,
   ["BioPlasma Tentacle Stage 5"] = 98000,
   ["BioPlasma Tentacle Stage 6"] = 107000,
   ["BioPlasma Tentacle Stage 7"] = 116000,
   ["BioPlasma Tentacle Stage X"] = 125000,
   ["Heavy Brain Stage 1"] = 1500000,
   ["Heavy Brain Stage 2"] = 1750000,
   ["Heavy Brain Stage 3"] = 2000000,
   ["Heavy Brain Stage 4"] = 2250000,
   ["Heavy Brain Stage 5"] = 2500000,
   ["Heavy Brain Stage 6"] = 2750000,
   ["Heavy Brain Stage X"] = 3000000,
   ["Light Brain Stage 1"] = 105000,
   ["Light Brain Stage 2"] = 105000,
   ["Light Brain Stage 3"] = 105000,
   ["Light Brain Stage X"] = 210000,
   ["Medium Brain Stage 1"] = 165000,
   ["Medium Brain Stage 2"] = 206250,
   ["Medium Brain Stage 3"] = 247500,
   ["Medium Brain Stage 4"] = 288750,
   ["Medium Brain Stage X"] = 330000,
   ["Medium-Heavy Brain Stage 1"] = 300000,
   ["Medium-Heavy Brain Stage 2"] = 360000,
   ["Medium-Heavy Brain Stage 3"] = 420000,
   ["Medium-Heavy Brain Stage 4"] = 480000,
   ["Medium-Heavy Brain Stage 5"] = 540000,
   ["Medium-Heavy Brain Stage X"] = 600000,
   ["Superheavy Brain Stage 1"] = 2000000,
   ["Superheavy Brain Stage 2"] = 2250000,
   ["Superheavy Brain Stage 3"] = 2500000,
   ["Superheavy Brain Stage 4"] = 2750000,
   ["Superheavy Brain Stage 5"] = 3000000,
   ["Superheavy Brain Stage 6"] = 3250000,
   ["Superheavy Brain Stage 7"] = 3500000,
   ["Superheavy Brain Stage X"] = 4000000,
   ["Ultralight Brain Stage 1"] = 60000,
   ["Ultralight Brain Stage 2"] = 100000,
   ["Ultralight Brain Stage X"] = 120000,
   ["Heavy Fast Gene Drive Stage 1"] = 1350000,
   ["Heavy Fast Gene Drive Stage 2"] = 1575000,
   ["Heavy Fast Gene Drive Stage 3"] = 1800000,
   ["Heavy Fast Gene Drive Stage 4"] = 2025000,
   ["Heavy Fast Gene Drive Stage 5"] = 2250000,
   ["Heavy Fast Gene Drive Stage 6"] = 2475000,
   ["Heavy Fast Gene Drive Stage X"] = 2700000,
   ["Heavy Strong Gene Drive Stage 1"] = 550000,
   ["Heavy Strong Gene Drive Stage 2"] = 640000,
   ["Heavy Strong Gene Drive Stage 3"] = 730000,
   ["Heavy Strong Gene Drive Stage 4"] = 820000,
   ["Heavy Strong Gene Drive Stage 5"] = 910000,
   ["Heavy Strong Gene Drive Stage 6"] = 1000000,
   ["Heavy Strong Gene Drive Stage X"] = 1100000,
   ["Light Fast Gene Drive Stage 1"] = 110000,
   ["Light Fast Gene Drive Stage 2"] = 145000,
   ["Light Fast Gene Drive Stage 3"] = 180000,
   ["Light Fast Gene Drive Stage X"] = 220000,
   ["Light Strong Gene Drive Stage 1"] = 45000,
   ["Light Strong Gene Drive Stage 2"] = 60000,
   ["Light Strong Gene Drive Stage 3"] = 75000,
   ["Light Strong Gene Drive Stage X"] = 90000,
   ["Medium Fast Gene Drive Stage 1"] = 180000,
   ["Medium Fast Gene Drive Stage 2"] = 225000,
   ["Medium Fast Gene Drive Stage 3"] = 270000,
   ["Medium Fast Gene Drive Stage 4"] = 315000,
   ["Medium Fast Gene Drive Stage X"] = 360000,
   ["Medium Strong Gene Drive Stage 1"] = 75000,
   ["Medium Strong Gene Drive Stage 2"] = 93750,
   ["Medium Strong Gene Drive Stage 3"] = 112500,
   ["Medium Strong Gene Drive Stage 4"] = 131250,
   ["Medium Strong Gene Drive Stage X"] = 150000,
   ["Medium-Heavy Fast Gene Drive Stage 1"] = 325000,
   ["Medium-Heavy Fast Gene Drive Stage 2"] = 390000,
   ["Medium-Heavy Fast Gene Drive Stage 3"] = 455000,
   ["Medium-Heavy Fast Gene Drive Stage 4"] = 520000,
   ["Medium-Heavy Fast Gene Drive Stage 5"] = 585000,
   ["Medium-Heavy Fast Gene Drive Stage X"] = 650000,
   ["Medium-Heavy Strong Gene Drive Stage 1"] = 140000,
   ["Medium-Heavy Strong Gene Drive Stage 2"] = 168000,
   ["Medium-Heavy Strong Gene Drive Stage 3"] = 196000,
   ["Medium-Heavy Strong Gene Drive Stage 4"] = 224000,
   ["Medium-Heavy Strong Gene Drive Stage 5"] = 252000,
   ["Medium-Heavy Strong Gene Drive Stage X"] = 280000,
   ["Superheavy Fast Gene Drive Stage 1"] = 1800000,
   ["Superheavy Fast Gene Drive Stage 2"] = 2050000,
   ["Superheavy Fast Gene Drive Stage 3"] = 2300000,
   ["Superheavy Fast Gene Drive Stage 4"] = 2550000,
   ["Superheavy Fast Gene Drive Stage 5"] = 2800000,
   ["Superheavy Fast Gene Drive Stage 6"] = 3050000,
   ["Superheavy Fast Gene Drive Stage 7"] = 3300000,
   ["Superheavy Fast Gene Drive Stage X"] = 3600000,
   ["Superheavy Strong Gene Drive Stage 1"] = 750000,
   ["Superheavy Strong Gene Drive Stage 2"] = 850000,
   ["Superheavy Strong Gene Drive Stage 3"] = 950000,
   ["Superheavy Strong Gene Drive Stage 4"] = 1050000,
   ["Superheavy Strong Gene Drive Stage 5"] = 1150000,
   ["Superheavy Strong Gene Drive Stage 6"] = 1250000,
   ["Superheavy Strong Gene Drive Stage 7"] = 1350000,
   ["Superheavy Strong Gene Drive Stage X"] = 1500000,
   ["Ultralight Fast Gene Drive Stage 1"] = 70000,
   ["Ultralight Fast Gene Drive Stage 2"] = 105000,
   ["Ultralight Fast Gene Drive Stage X"] = 140000,
   ["Ultralight Strong Gene Drive Stage 1"] = 25000,
   ["Ultralight Strong Gene Drive Stage 2"] = 37500,
   ["Ultralight Strong Gene Drive Stage X"] = 50000,
   ["Heavy Shell Stage 1"] = 1100000,
   ["Heavy Shell Stage 2"] = 1450000,
   ["Heavy Shell Stage 3"] = 1275000,
   ["Heavy Shell Stage 4"] = 1625000,
   ["Heavy Shell Stage 5"] = 1800000,
   ["Heavy Shell Stage 6"] = 1975000,
   ["Heavy Shell Stage X"] = 2200000,
   ["Light Shell Stage 1"] = 120000,
   ["Light Shell Stage 2"] = 160000,
   ["Light Shell Stage 3"] = 200000,
   ["Light Shell Stage X"] = 240000,
   ["Medium Shell Stage 1"] = 180000,
   ["Medium Shell Stage 2"] = 225000,
   ["Medium Shell Stage 3"] = 270000,
   ["Medium Shell Stage 4"] = 315000,
   ["Medium Shell Stage X"] = 360000,
   ["Medium-Heavy Shell Stage 1"] = 320000,
   ["Medium-Heavy Shell Stage 2"] = 384000,
   ["Medium-Heavy Shell Stage 3"] = 448000,
   ["Medium-Heavy Shell Stage 4"] = 512000,
   ["Medium-Heavy Shell Stage 5"] = 576000,
   ["Medium-Heavy Shell Stage X"] = 640000,
   ["Superheavy Shell Stage 1"] = 1450000,
   ["Superheavy Shell Stage 2"] = 1650000,
   ["Superheavy Shell Stage 3"] = 1850000,
   ["Superheavy Shell Stage 4"] = 2050000,
   ["Superheavy Shell Stage 5"] = 2250000,
   ["Superheavy Shell Stage 6"] = 2450000,
   ["Superheavy Shell Stage 7"] = 2650000,
   ["Superheavy Shell Stage X"] = 2900000,
   ["Ultralight Shell Stage 1"] = 65000,
   ["Ultralight Shell Stage 2"] = 97500,
   ["Ultralight Shell Stage X"] = 130000,

   -- Below are the 0.8.x naming scheme and prices
   ["BioPlasma Organ Stage 1"] = 18000,
   ["BioPlasma Organ Stage 2"] = 18000,
   ["BioPlasma Organ Stage 3"] = 18000,
   ["BioPlasma Organ Stage X"] = 18000,
   ["Heavy Bioship Brain Stage 1"] = 1500000,
   ["Heavy Bioship Brain Stage 2"] = 1500000,
   ["Heavy Bioship Brain Stage 3"] = 1500000,
   ["Heavy Bioship Brain Stage 4"] = 1500000,
   ["Heavy Bioship Brain Stage 5"] = 1500000,
   ["Heavy Bioship Brain Stage 6"] = 1500000,
   ["Heavy Bioship Brain Stage X"] = 1500000,
   ["Light Bioship Brain Stage 1"] = 105000,
   ["Light Bioship Brain Stage 2"] = 105000,
   ["Light Bioship Brain Stage 3"] = 105000,
   ["Light Bioship Brain Stage X"] = 105000,
   ["Medium Bioship Brain Stage 1"] = 165000,
   ["Medium Bioship Brain Stage 2"] = 165000,
   ["Medium Bioship Brain Stage 3"] = 165000,
   ["Medium Bioship Brain Stage 4"] = 165000,
   ["Medium Bioship Brain Stage X"] = 165000,
   ["Medium-Heavy Bioship Brain Stage 1"] = 300000,
   ["Medium-Heavy Bioship Brain Stage 2"] = 300000,
   ["Medium-Heavy Bioship Brain Stage 3"] = 300000,
   ["Medium-Heavy Bioship Brain Stage 4"] = 300000,
   ["Medium-Heavy Bioship Brain Stage 5"] = 300000,
   ["Medium-Heavy Bioship Brain Stage X"] = 300000,
   ["Superheavy Bioship Brain Stage 1"] = 2000000,
   ["Superheavy Bioship Brain Stage 2"] = 2000000,
   ["Superheavy Bioship Brain Stage 3"] = 2000000,
   ["Superheavy Bioship Brain Stage 4"] = 2000000,
   ["Superheavy Bioship Brain Stage 5"] = 2000000,
   ["Superheavy Bioship Brain Stage 6"] = 2000000,
   ["Superheavy Bioship Brain Stage 7"] = 2000000,
   ["Superheavy Bioship Brain Stage X"] = 2000000,
   ["Ultralight Bioship Brain Stage 1"] = 60000,
   ["Ultralight Bioship Brain Stage 2"] = 60000,
   ["Ultralight Bioship Brain Stage X"] = 60000,
   ["Heavy Bioship Fast Fin Stage 1"] = 1350000,
   ["Heavy Bioship Fast Fin Stage 2"] = 1350000,
   ["Heavy Bioship Fast Fin Stage 3"] = 1350000,
   ["Heavy Bioship Fast Fin Stage 4"] = 1350000,
   ["Heavy Bioship Fast Fin Stage 5"] = 1350000,
   ["Heavy Bioship Fast Fin Stage 6"] = 1350000,
   ["Heavy Bioship Fast Fin Stage X"] = 1350000,
   ["Heavy Bioship Strong Fin Stage 1"] = 550000,
   ["Heavy Bioship Strong Fin Stage 2"] = 550000,
   ["Heavy Bioship Strong Fin Stage 3"] = 550000,
   ["Heavy Bioship Strong Fin Stage 4"] = 550000,
   ["Heavy Bioship Strong Fin Stage 5"] = 550000,
   ["Heavy Bioship Strong Fin Stage 6"] = 550000,
   ["Heavy Bioship Strong Fin Stage X"] = 550000,
   ["Light Bioship Fast Fin Stage 1"] = 110000,
   ["Light Bioship Fast Fin Stage 2"] = 110000,
   ["Light Bioship Fast Fin Stage 3"] = 110000,
   ["Light Bioship Fast Fin Stage X"] = 110000,
   ["Light Bioship Strong Fin Stage 1"] = 45000,
   ["Light Bioship Strong Fin Stage 2"] = 45000,
   ["Light Bioship Strong Fin Stage 3"] = 45000,
   ["Light Bioship Strong Fin Stage X"] = 45000,
   ["Medium Bioship Fast Fin Stage 1"] = 180000,
   ["Medium Bioship Fast Fin Stage 2"] = 180000,
   ["Medium Bioship Fast Fin Stage 3"] = 180000,
   ["Medium Bioship Fast Fin Stage 4"] = 180000,
   ["Medium Bioship Fast Fin Stage X"] = 180000,
   ["Medium Bioship Strong Fin Stage 1"] = 75000,
   ["Medium Bioship Strong Fin Stage 2"] = 75000,
   ["Medium Bioship Strong Fin Stage 3"] = 75000,
   ["Medium Bioship Strong Fin Stage 4"] = 75000,
   ["Medium Bioship Strong Fin Stage X"] = 75000,
   ["Medium-Heavy Bioship Fast Fin Stage 1"] = 325000,
   ["Medium-Heavy Bioship Fast Fin Stage 2"] = 325000,
   ["Medium-Heavy Bioship Fast Fin Stage 3"] = 325000,
   ["Medium-Heavy Bioship Fast Fin Stage 4"] = 325000,
   ["Medium-Heavy Bioship Fast Fin Stage 5"] = 325000,
   ["Medium-Heavy Bioship Fast Fin Stage X"] = 325000,
   ["Medium-Heavy Bioship Strong Fin Stage 1"] = 140000,
   ["Medium-Heavy Bioship Strong Fin Stage 2"] = 140000,
   ["Medium-Heavy Bioship Strong Fin Stage 3"] = 140000,
   ["Medium-Heavy Bioship Strong Fin Stage 4"] = 140000,
   ["Medium-Heavy Bioship Strong Fin Stage 5"] = 140000,
   ["Medium-Heavy Bioship Strong Fin Stage X"] = 140000,
   ["Superheavy Bioship Fast Fin Stage 1"] = 1800000,
   ["Superheavy Bioship Fast Fin Stage 2"] = 1800000,
   ["Superheavy Bioship Fast Fin Stage 3"] = 1800000,
   ["Superheavy Bioship Fast Fin Stage 4"] = 1800000,
   ["Superheavy Bioship Fast Fin Stage 5"] = 1800000,
   ["Superheavy Bioship Fast Fin Stage 6"] = 1800000,
   ["Superheavy Bioship Fast Fin Stage 7"] = 1800000,
   ["Superheavy Bioship Fast Fin Stage X"] = 1800000,
   ["Superheavy Bioship Strong Fin Stage 1"] = 750000,
   ["Superheavy Bioship Strong Fin Stage 2"] = 750000,
   ["Superheavy Bioship Strong Fin Stage 3"] = 750000,
   ["Superheavy Bioship Strong Fin Stage 4"] = 750000,
   ["Superheavy Bioship Strong Fin Stage 5"] = 750000,
   ["Superheavy Bioship Strong Fin Stage 6"] = 750000,
   ["Superheavy Bioship Strong Fin Stage 7"] = 750000,
   ["Superheavy Bioship Strong Fin Stage X"] = 750000,
   ["Ultralight Bioship Fast Fin Stage 1"] = 70000,
   ["Ultralight Bioship Fast Fin Stage 2"] = 70000,
   ["Ultralight Bioship Fast Fin Stage X"] = 70000,
   ["Ultralight Bioship Strong Fin Stage 1"] = 25000,
   ["Ultralight Bioship Strong Fin Stage 2"] = 25000,
   ["Ultralight Bioship Strong Fin Stage X"] = 25000,
   ["Heavy Bioship Shell Stage 1"] = 1100000,
   ["Heavy Bioship Shell Stage 2"] = 1100000,
   ["Heavy Bioship Shell Stage 3"] = 1100000,
   ["Heavy Bioship Shell Stage 4"] = 1100000,
   ["Heavy Bioship Shell Stage 5"] = 1100000,
   ["Heavy Bioship Shell Stage 6"] = 1100000,
   ["Heavy Bioship Shell Stage X"] = 1100000,
   ["Light Bioship Shell Stage 1"] = 120000,
   ["Light Bioship Shell Stage 2"] = 120000,
   ["Light Bioship Shell Stage 3"] = 120000,
   ["Light Bioship Shell Stage X"] = 120000,
   ["Medium Bioship Shell Stage 1"] = 180000,
   ["Medium Bioship Shell Stage 2"] = 180000,
   ["Medium Bioship Shell Stage 3"] = 180000,
   ["Medium Bioship Shell Stage 4"] = 180000,
   ["Medium Bioship Shell Stage X"] = 180000,
   ["Medium-Heavy Bioship Shell Stage 1"] = 320000,
   ["Medium-Heavy Bioship Shell Stage 2"] = 320000,
   ["Medium-Heavy Bioship Shell Stage 3"] = 320000,
   ["Medium-Heavy Bioship Shell Stage 4"] = 320000,
   ["Medium-Heavy Bioship Shell Stage 5"] = 320000,
   ["Medium-Heavy Bioship Shell Stage X"] = 320000,
   ["Superheavy Bioship Shell Stage 1"] = 1450000,
   ["Superheavy Bioship Shell Stage 2"] = 1450000,
   ["Superheavy Bioship Shell Stage 3"] = 1450000,
   ["Superheavy Bioship Shell Stage 4"] = 1450000,
   ["Superheavy Bioship Shell Stage 5"] = 1450000,
   ["Superheavy Bioship Shell Stage 6"] = 1450000,
   ["Superheavy Bioship Shell Stage 7"] = 1450000,
   ["Superheavy Bioship Shell Stage X"] = 1450000,
   ["Ultralight Bioship Shell Stage 1"] = 65000,
   ["Ultralight Bioship Shell Stage 2"] = 65000,
   ["Ultralight Bioship Shell Stage X"] = 65000,

   -- Below is a list of changes from 0.8.2 to 0.9.0
   ["Unicorp PT-1000 Core System"] = "Unicorp PT-1750 Core System", -- Modified for 0.12.0
   ["Unicorp PT-900 Core System"] = "Unicorp PT-440 Core System", -- Modified for 0.12.0
   ["Unicorp PT-600 Core System"] = "Unicorp PT-310 Core System",
   ["Unicorp PT-100 Core System"] = "Unicorp PT-16 Core System",
   ["Unicorp Hawk 150 Engine"] = "Unicorp Hawk 160 Engine", -- Modified for 0.11.0
   ["Unicorp Falcon 550 Engine"] = "Unicorp Falcon 700 Engine",
   ["Unicorp Falcon 1200 Engine"] = "Unicorp Falcon 1400 Engine",
   ["Unicorp Eagle 4500 Engine"] = "Unicorp Eagle 3000 Engine",
   ["Unicorp D-8 Medium Plating"] = "Unicorp D-23 Medium Plating",
   ["Unicorp D-20 Heavy Plating"] = "Unicorp D-72 Heavy Plating",
   ["Unicorp D-16 Heavy Plating"] = "Unicorp D-72 Heavy Plating",
   ["Unicorp B-8 Medium Plating"] = "Unicorp D-23 Medium Plating",
   ["Unicorp B-4 Light Plating"] = "Unicorp D-9 Light Plating",
   ["Unicorp B-20 Heavy Plating"] = "Unicorp D-72 Heavy Plating",
   ["Unicorp B-2 Light Plating"] = "Unicorp D-2 Light Plating",
   ["Unicorp B-16 Heavy Plating"] = "Unicorp D-38 Medium Plating",
   ["Unicorp B-12 Medium Plating"] = "Unicorp D-38 Medium Plating",
   ["Thurion Reactor Class III"] = 155e4,
   ["Thurion Reactor Class II"] = 97e4,
   ["Thurion Reactor Class I"] = 53e4,
   ["Thurion Engine Reroute"] = "Engine Reroute",
   ["Thurion Energy Cell III"] = 0,
   ["Thurion Energy Cell II"] = 0,
   ["Thurion Energy Cell I"] = 0,
   ["Steering Thrusters"] = 13e4,
   ["Solar Panel"] = 25e3,
   ["Shield Nullifier"] = 1,
   ["Shield Capacitor"] = "Shield Capacitor I",
   ["Shattershield Lance"] = 28e3,
   ["S&K Ultralight Stealth Plating"] = 95e3,
   ["S&K Medium-Heavy Stealth Plating"] = 48e4,
   ["S&K Medium Stealth Plating"] = 27e4,
   ["S&K Light Stealth Plating"] = 18e4,
   ["Razor Turret MK3"] = 50e3, -- "Razor Turret MK1",
   ["Razor MK3"] = "Razor Artillery S1", -- "Razor MK1",
   ["Pulse Beam"] = 26e3,
   ["Power Regulation Override"] = 55e3,
   ["Plasma Cluster Cannon MK3"] = "Plasma Cluster Cannon",
   ["Plasma Cluster Cannon MK2"] = "Plasma Cluster Cannon",
   ["Plasma Cluster Cannon MK1"] = "Plasma Cluster Cannon",
   ["Plasma Blaster MK3"] = 26e3,
   ["Plasma Blaster MK2"] = 17e3,
   ["Plasma Blaster MK1"] = 9e3,
   ["Nexus Dart 300 Engine"] = "Nexus Dart 360 Engine",
   ["Nexus Bolt 4500 Engine"] = "Nexus Bolt 3000 Engine",
   ["Nexus Arrow 550 Engine"] = "Nexus Arrow 700 Engine",
   ["Nexus Arrow 1200 Engine"] = "Nexus Arrow 1400 Engine",
   ["Milspec Prometheus 9803 Core System"] = "Milspec Prometheus 8503 Core System",
   ["Milspec Prometheus 5403 Core System"] = "Milspec Prometheus 4703 Core System",
   ["Milspec Prometheus 3603 Core System"] = "Milspec Prometheus 2203 Core System",
   ["Milspec Hermes 9802 Core System"] = "Milspec Orion 8601 Core System",
   ["Milspec Hermes 8502 Core System"] = "Milspec Orion 8601 Core System",
   ["Milspec Hermes 5402 Core System"] = "Milspec Orion 4801 Core System",
   ["Milspec Hermes 4702 Core System"] = "Milspec Orion 4801 Core System",
   ["Milspec Hermes 3602 Core System"] = "Milspec Orion 2301 Core System",
   ["Milspec Hermes 2202 Core System"] = "Milspec Orion 2301 Core System",
   ["Milspec Aegis 9801 Core System"] = "Milspec Aegis 8501 Core System",
   ["Milspec Aegis 5401 Core System"] = "Milspec Aegis 4701 Core System",
   ["Milspec Aegis 3601 Core System"] = "Milspec Aegis 2201 Core System",
   ["Mass Driver MK3"] = "Mass Driver",
   ["Mass Driver MK2"] = "Mass Driver",
   ["Mass Driver MK1"] = "Mass Driver",
   ["Laser PD MK3"] = 125e2,
   ["Laser PD MK2"] = 185e2,
   ["Laser PD MK1"] = 125e2,
   ["Laser Cannon MK3"] = "Laser Cannon MK1",
   ["Laser Cannon MK0"] = "Laser Cannon MK1",
   ["Improved Refrigeration Cycle"] = 19e4,
   ["Improved Power Regulator"] = 23e4,
   ["Heavy Laser"] = "Heavy Laser Turret",
   --[[
   ["Superheavy Bioship Strong Fin Stage X"] = "Superheavy Strong Gene Drive Stage X",
   ["Superheavy Bioship Strong Fin Stage 7"] = "Superheavy Strong Gene Drive Stage 7",
   ["Superheavy Bioship Strong Fin Stage 6"] = "Superheavy Strong Gene Drive Stage 6",
   ["Superheavy Bioship Strong Fin Stage 5"] = "Superheavy Strong Gene Drive Stage 5",
   ["Superheavy Bioship Strong Fin Stage 4"] = "Superheavy Strong Gene Drive Stage 4",
   ["Superheavy Bioship Strong Fin Stage 3"] = "Superheavy Strong Gene Drive Stage 3",
   ["Superheavy Bioship Strong Fin Stage 2"] = "Superheavy Strong Gene Drive Stage 2",
   ["Superheavy Bioship Strong Fin Stage 1"] = "Superheavy Strong Gene Drive Stage 1",
   ["Superheavy Bioship Shell Stage X"] = "Superheavy Shell Stage X",
   ["Superheavy Bioship Shell Stage 7"] = "Superheavy Shell Stage 7",
   ["Superheavy Bioship Shell Stage 6"] = "Superheavy Shell Stage 6",
   ["Superheavy Bioship Shell Stage 5"] = "Superheavy Shell Stage 5",
   ["Superheavy Bioship Shell Stage 4"] = "Superheavy Shell Stage 4",
   ["Superheavy Bioship Shell Stage 3"] = "Superheavy Shell Stage 3",
   ["Superheavy Bioship Shell Stage 2"] = "Superheavy Shell Stage 2",
   ["Superheavy Bioship Shell Stage 1"] = "Superheavy Shell Stage 1",
   ["Superheavy Bioship Fast Fin Stage X"] = "Superheavy Fast Gene Drive Stage X",
   ["Superheavy Bioship Fast Fin Stage 7"] = "Superheavy Fast Gene Drive Stage 7",
   ["Superheavy Bioship Fast Fin Stage 6"] = "Superheavy Fast Gene Drive Stage 6",
   ["Superheavy Bioship Fast Fin Stage 5"] = "Superheavy Fast Gene Drive Stage 5",
   ["Superheavy Bioship Fast Fin Stage 4"] = "Superheavy Fast Gene Drive Stage 4",
   ["Superheavy Bioship Fast Fin Stage 3"] = "Superheavy Fast Gene Drive Stage 3",
   ["Superheavy Bioship Fast Fin Stage 2"] = "Superheavy Fast Gene Drive Stage 2",
   ["Superheavy Bioship Fast Fin Stage 1"] = "Superheavy Fast Gene Drive Stage 1",
   ["Superheavy Bioship Brain Stage X"] = "Superheavy Brain Stage X",
   ["Superheavy Bioship Brain Stage 7"] = "Superheavy Brain Stage 7",
   ["Superheavy Bioship Brain Stage 6"] = "Superheavy Brain Stage 6",
   ["Superheavy Bioship Brain Stage 5"] = "Superheavy Brain Stage 5",
   ["Superheavy Bioship Brain Stage 4"] = "Superheavy Brain Stage 4",
   ["Superheavy Bioship Brain Stage 3"] = "Superheavy Brain Stage 3",
   ["Superheavy Bioship Brain Stage 2"] = "Superheavy Brain Stage 2",
   ["Superheavy Bioship Brain Stage 1"] = "Superheavy Brain Stage 1",
   ["Medium-Heavy Bioship Strong Fin Stage X"] = "Medium-Heavy Strong Gene Drive Stage X",
   ["Medium-Heavy Bioship Strong Fin Stage 5"] = "Medium-Heavy Strong Gene Drive Stage 5",
   ["Medium-Heavy Bioship Strong Fin Stage 4"] = "Medium-Heavy Strong Gene Drive Stage 4",
   ["Medium-Heavy Bioship Strong Fin Stage 3"] = "Medium-Heavy Strong Gene Drive Stage 3",
   ["Medium-Heavy Bioship Strong Fin Stage 2"] = "Medium-Heavy Strong Gene Drive Stage 2",
   ["Medium-Heavy Bioship Strong Fin Stage 1"] = "Medium-Heavy Strong Gene Drive Stage 1",
   ["Medium-Heavy Bioship Shell Stage X"] = "Medium-Heavy Shell Stage X",
   ["Medium-Heavy Bioship Shell Stage 5"] = "Medium-Heavy Shell Stage 5",
   ["Medium-Heavy Bioship Shell Stage 4"] = "Medium-Heavy Shell Stage 4",
   ["Medium-Heavy Bioship Shell Stage 3"] = "Medium-Heavy Shell Stage 3",
   ["Medium-Heavy Bioship Shell Stage 2"] = "Medium-Heavy Shell Stage 2",
   ["Medium-Heavy Bioship Shell Stage 1"] = "Medium-Heavy Shell Stage 1",
   ["Medium-Heavy Bioship Fast Fin Stage X"] = "Medium-Heavy Fast Gene Drive Stage X",
   ["Medium-Heavy Bioship Fast Fin Stage 5"] = "Medium-Heavy Fast Gene Drive Stage 5",
   ["Medium-Heavy Bioship Fast Fin Stage 4"] = "Medium-Heavy Fast Gene Drive Stage 4",
   ["Medium-Heavy Bioship Fast Fin Stage 3"] = "Medium-Heavy Fast Gene Drive Stage 3",
   ["Medium-Heavy Bioship Fast Fin Stage 2"] = "Medium-Heavy Fast Gene Drive Stage 2",
   ["Medium-Heavy Bioship Fast Fin Stage 1"] = "Medium-Heavy Fast Gene Drive Stage 1",
   ["Medium-Heavy Bioship Brain Stage X"] = "Medium-Heavy Brain Stage X",
   ["Medium-Heavy Bioship Brain Stage 5"] = "Medium-Heavy Brain Stage 5",
   ["Medium-Heavy Bioship Brain Stage 4"] = "Medium-Heavy Brain Stage 4",
   ["Medium-Heavy Bioship Brain Stage 3"] = "Medium-Heavy Brain Stage 3",
   ["Medium-Heavy Bioship Brain Stage 2"] = "Medium-Heavy Brain Stage 2",
   ["Medium-Heavy Bioship Brain Stage 1"] = "Medium-Heavy Brain Stage 1",
   ["Medium Bioship Strong Fin Stage X"] = "Medium Strong Gene Drive Stage X",
   ["Medium Bioship Strong Fin Stage 4"] = "Medium Strong Gene Drive Stage 4",
   ["Medium Bioship Strong Fin Stage 3"] = "Medium Strong Gene Drive Stage 3",
   ["Medium Bioship Strong Fin Stage 2"] = "Medium Strong Gene Drive Stage 2",
   ["Medium Bioship Strong Fin Stage 1"] = "Medium Strong Gene Drive Stage 1",
   ["Medium Bioship Shell Stage X"] = "Medium Shell Stage X",
   ["Medium Bioship Shell Stage 4"] = "Medium Shell Stage 4",
   ["Medium Bioship Shell Stage 3"] = "Medium Shell Stage 3",
   ["Medium Bioship Shell Stage 2"] = "Medium Shell Stage 2",
   ["Medium Bioship Shell Stage 1"] = "Medium Shell Stage 1",
   ["Medium Bioship Fast Fin Stage X"] = "Medium Fast Gene Drive Stage X",
   ["Medium Bioship Fast Fin Stage 4"] = "Medium Fast Gene Drive Stage 4",
   ["Medium Bioship Fast Fin Stage 3"] = "Medium Fast Gene Drive Stage 3",
   ["Medium Bioship Fast Fin Stage 2"] = "Medium Fast Gene Drive Stage 2",
   ["Medium Bioship Fast Fin Stage 1"] = "Medium Fast Gene Drive Stage 1",
   ["Medium Bioship Brain Stage X"] = "Medium Brain Stage X",
   ["Medium Bioship Brain Stage 4"] = "Medium Brain Stage 4",
   ["Medium Bioship Brain Stage 3"] = "Medium Brain Stage 3",
   ["Medium Bioship Brain Stage 2"] = "Medium Brain Stage 2",
   ["Medium Bioship Brain Stage 1"] = "Medium Brain Stage 1",
   ["Light Bioship Strong Fin Stage X"] = "Light Strong Gene Drive Stage X",
   ["Light Bioship Strong Fin Stage 3"] = "Light Strong Gene Drive Stage 3",
   ["Light Bioship Strong Fin Stage 2"] = "Light Strong Gene Drive Stage 2",
   ["Light Bioship Strong Fin Stage 1"] = "Light Strong Gene Drive Stage 1",
   ["Light Bioship Shell Stage X"] = "Light Shell Stage X",
   ["Light Bioship Shell Stage 3"] = "Light Shell Stage 3",
   ["Light Bioship Shell Stage 2"] = "Light Shell Stage 2",
   ["Light Bioship Shell Stage 1"] = "Light Shell Stage 1",
   ["Light Bioship Fast Fin Stage X"] = "Light Fast Gene Drive Stage X",
   ["Light Bioship Fast Fin Stage 3"] = "Light Fast Gene Drive Stage 3",
   ["Light Bioship Fast Fin Stage 2"] = "Light Fast Gene Drive Stage 2",
   ["Light Bioship Fast Fin Stage 1"] = "Light Fast Gene Drive Stage 1",
   ["Light Bioship Brain Stage X"] = "Light Brain Stage X",
   ["Light Bioship Brain Stage 3"] = "Light Brain Stage 3",
   ["Light Bioship Brain Stage 2"] = "Light Brain Stage 2",
   ["Light Bioship Brain Stage 1"] = "Light Brain Stage 1",
   ["Ultralight Bioship Strong Fin Stage X"] = "Ultralight Strong Gene Drive Stage X",
   ["Ultralight Bioship Strong Fin Stage 2"] = "Ultralight Strong Gene Drive Stage 2",
   ["Ultralight Bioship Strong Fin Stage 1"] = "Ultralight Strong Gene Drive Stage 1",
   ["Ultralight Bioship Shell Stage X"] = "Ultralight Shell Stage X",
   ["Ultralight Bioship Shell Stage 2"] = "Ultralight Shell Stage 2",
   ["Ultralight Bioship Shell Stage 1"] = "Ultralight Shell Stage 1",
   ["Ultralight Bioship Fast Fin Stage X"] = "Ultralight Fast Gene Drive Stage X",
   ["Ultralight Bioship Fast Fin Stage 2"] = "Ultralight Fast Gene Drive Stage 2",
   ["Ultralight Bioship Fast Fin Stage 1"] = "Ultralight Fast Gene Drive Stage 1",
   ["Ultralight Bioship Brain Stage X"] = "Ultralight Brain Stage X",
   ["Ultralight Bioship Brain Stage 2"] = "Ultralight Brain Stage 2",
   ["Ultralight Bioship Brain Stage 1"] = "Ultralight Brain Stage 1",
   ["Heavy Bioship Strong Fin Stage X"] = "Heavy Strong Gene Drive Stage X",
   ["Heavy Bioship Strong Fin Stage 6"] = "Heavy Strong Gene Drive Stage 6",
   ["Heavy Bioship Strong Fin Stage 5"] = "Heavy Strong Gene Drive Stage 5",
   ["Heavy Bioship Strong Fin Stage 4"] = "Heavy Strong Gene Drive Stage 4",
   ["Heavy Bioship Strong Fin Stage 3"] = "Heavy Strong Gene Drive Stage 3",
   ["Heavy Bioship Strong Fin Stage 2"] = "Heavy Strong Gene Drive Stage 2",
   ["Heavy Bioship Strong Fin Stage 1"] = "Heavy Strong Gene Drive Stage 1",
   ["Heavy Bioship Shell Stage X"] = "Heavy Shell Stage X",
   ["Heavy Bioship Shell Stage 6"] = "Heavy Shell Stage 6",
   ["Heavy Bioship Shell Stage 5"] = "Heavy Shell Stage 5",
   ["Heavy Bioship Shell Stage 4"] = "Heavy Shell Stage 4",
   ["Heavy Bioship Shell Stage 3"] = "Heavy Shell Stage 3",
   ["Heavy Bioship Shell Stage 2"] = "Heavy Shell Stage 2",
   ["Heavy Bioship Shell Stage 1"] = "Heavy Shell Stage 1",
   ["Heavy Bioship Fast Fin Stage X"] = "Heavy Fast Gene Drive Stage X",
   ["Heavy Bioship Fast Fin Stage 6"] = "Heavy Fast Gene Drive Stage 6",
   ["Heavy Bioship Fast Fin Stage 5"] = "Heavy Fast Gene Drive Stage 5",
   ["Heavy Bioship Fast Fin Stage 4"] = "Heavy Fast Gene Drive Stage 4",
   ["Heavy Bioship Fast Fin Stage 3"] = "Heavy Fast Gene Drive Stage 3",
   ["Heavy Bioship Fast Fin Stage 2"] = "Heavy Fast Gene Drive Stage 2",
   ["Heavy Bioship Fast Fin Stage 1"] = "Heavy Fast Gene Drive Stage 1",
   ["Heavy Bioship Brain Stage X"] = "Heavy Brain Stage X",
   ["Heavy Bioship Brain Stage 6"] = "Heavy Brain Stage 6",
   ["Heavy Bioship Brain Stage 5"] = "Heavy Brain Stage 5",
   ["Heavy Bioship Brain Stage 4"] = "Heavy Brain Stage 4",
   ["Heavy Bioship Brain Stage 3"] = "Heavy Brain Stage 3",
   ["Heavy Bioship Brain Stage 2"] = "Heavy Brain Stage 2",
   ["Heavy Bioship Brain Stage 1"] = "Heavy Brain Stage 1",
   --]]
   ["Generic Afterburner"] = "Unicorp Light Afterburner",
   ["Fuel Pod"] = "Small Fuel Pod",
   ["Forward Shock Absorbers"] = 35e3,
   ["Enygma Systems Turreted Vengeance Launcher"] = "Enygma Systems Turreted Headhunter Launcher",
   ["Energy Torpedo"] = 1e5,
   ["Energy Missile"] = 75e3,
   ["Energy Dart"] = 5e4,
   ["Cargo Pod"] = "Small Cargo Pod",
   --[[
   ["BioPlasma Organ Stage X"] = "BioPlasma Stinger Stage X",
   ["BioPlasma Organ Stage 3"] = "BioPlasma Stinger Stage 3",
   ["BioPlasma Organ Stage 2"] = "BioPlasma Stinger Stage 2",
   ["BioPlasma Organ Stage 1"] = "BioPlasma Stinger Stage 1",
   --]]
   ["Battery"] = "Battery I",
}

--[[--
   Takes an outfit name and should return either a new outfit name or the amount of credits to give back to the player.
--]]
function outfit( name, q )
   return apply_change( name, outfit_list[name], q )
end

local licence_list = {
   -- Below is a list of changes from 0.9.0-alpha to 0.9.0-beta
   ["Heavy Combat Vessel Licence"] = "Heavy Combat Vessel",
   ["Heavy Weapon Licence"] = "Heavy Weapon",
   ["Large Civilian Vessel Licence"] = "Large Civilian Vessel",
   ["Light Combat Vessel Licence"] = "Light Combat Vessel",
   ["Medium Combat Vessel Licence"] = "Medium Combat Vessel",
   ["Medium Weapon Licence"] = "Medium Weapon",
   ["Mercenary Licence"] = "Mercenary",
}
--[[--
   Takes a licence name and should return either a new licence name or the amount of credits to give back to the player.
--]]
function licence( name, q )
   return apply_change( name, licence_list[name], q )
end

local unidiff_list = {
   ["heavy_combat_vessel_license"] = "heavy_combat_vessel_licence",
   ["heavy_weapons_license"] = "heavy_weapons_license",
}
function unidiff( name )
   -- Internals stuff so we don't expose to the player
   return  unidiff_list[name]
end
