/*
 * Decompiled with CFR 0.152.
 */
package utilities.util;

import generic.jar.ResourceFile;
import ghidra.util.MonitoredOutputStream;
import ghidra.util.Msg;
import ghidra.util.SystemUtilities;
import ghidra.util.exception.CancelledException;
import ghidra.util.task.TaskMonitor;
import java.awt.Desktop;
import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.URI;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileSystem;
import java.nio.file.FileSystemNotFoundException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.function.Consumer;
import java.util.stream.Stream;
import utilities.util.FileResolutionResult;

public final class FileUtilities {
    private static final int MAX_FILE_SIZE = 0x10000000;
    public static final int IO_BUFFER_SIZE = 32768;
    private static final ThreadLocal<NumberFormat> SIZE_FORMAT_THREAD_LOCAL = ThreadLocal.withInitial(() -> new DecimalFormat("#,###,###.##"));
    private static final FileFilter ACCEPT_ALL_FILE_FILTER = pathname -> true;

    private FileUtilities() {
    }

    public static boolean directoryExistsAndIsNotEmpty(File directory) {
        if (directory == null) {
            return false;
        }
        if (!directory.exists()) {
            return false;
        }
        if (!directory.isDirectory()) {
            return false;
        }
        return !FileUtilities.directoryIsEmpty(directory);
    }

    public static boolean directoryIsEmpty(File directory) {
        if (directory == null) {
            return true;
        }
        if (!directory.exists()) {
            return true;
        }
        if (!directory.isDirectory()) {
            return false;
        }
        File[] files = directory.listFiles();
        boolean hasFiles = files != null && files.length > 0;
        return !hasFiles;
    }

    public static final byte[] getBytesFromFile(File sourceFile) throws IOException {
        return FileUtilities.getBytesFromFile(new ResourceFile(sourceFile));
    }

    public static final byte[] getBytesFromFile(File sourceFile, long offset, long length) throws IOException {
        return FileUtilities.getBytesFromFile(new ResourceFile(sourceFile), offset, length);
    }

    public static final byte[] getBytesFromFile(ResourceFile sourceFile) throws IOException {
        long fileLen = sourceFile.length();
        return FileUtilities.getBytesFromFile(sourceFile, 0L, fileLen);
    }

    public static void writeBytes(File file, byte[] bytes) throws FileNotFoundException, IOException {
        try (FileOutputStream os = new FileOutputStream(file);){
            ((OutputStream)os).write(bytes);
        }
    }

    public static final byte[] getBytesFromFile(ResourceFile sourceFile, long offset, long length) throws IOException {
        if (length > 0x10000000L) {
            throw new IOException("File is too large: " + sourceFile.getName() + " file must be less than 268435456 bytes");
        }
        if (offset < 0L || length < 0L) {
            throw new IllegalArgumentException("offset[" + offset + "] and length[" + length + "] must be greater than 0");
        }
        byte[] data = new byte[(int)length];
        try (InputStream fis = sourceFile.getInputStream();){
            if (fis.skip(offset) != offset) {
                throw new IOException("Did not skip to the specified offset!");
            }
            int n = fis.read(data);
            if ((long)n != length) {
                throw new IOException("Did not read expected number of bytes! Expected " + length + ", but read " + n);
            }
            byte[] byArray = data;
            return byArray;
        }
    }

    public static byte[] getBytesFromStream(InputStream is) throws IOException {
        int n;
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        byte[] bytes = new byte[4096];
        while ((n = is.read(bytes)) > 0) {
            baos.write(bytes, 0, n);
        }
        return baos.toByteArray();
    }

    public static byte[] getBytesFromStream(InputStream inputStream, int expectedLength) throws IOException {
        byte[] buf = new byte[expectedLength];
        int bytesCopied = 0;
        int bytesRead = 0;
        for (int bufRemaining = expectedLength; bufRemaining > 0 && (bytesRead = inputStream.read(buf, bytesCopied, bufRemaining)) > 0; bufRemaining -= bytesRead) {
            bytesCopied += bytesRead;
        }
        if (bytesCopied != expectedLength) {
            throw new IOException("Could only read " + bytesCopied + " bytes from input stream when trying to read " + expectedLength + " bytes.");
        }
        return buf;
    }

    public static final long copyFile(File fromFile, File toFile, boolean append, TaskMonitor monitor) throws IOException {
        try (FileInputStream fin = new FileInputStream(fromFile);){
            if (monitor != null) {
                monitor.initialize((int)fromFile.length());
            }
            long l = FileUtilities.copyStreamToFile(fin, toFile, append, monitor);
            return l;
        }
    }

    public static final void copyFile(ResourceFile fromFile, File toFile, boolean append, TaskMonitor monitor) throws IOException {
        try (InputStream fin = fromFile.getInputStream();){
            if (monitor != null) {
                monitor.initialize((int)fromFile.length());
            }
            FileUtilities.copyStreamToFile(fin, toFile, append, monitor);
        }
    }

    public static final void copyFile(ResourceFile fromFile, ResourceFile toFile, TaskMonitor monitor) throws IOException {
        try (InputStream fin = fromFile.getInputStream();
             OutputStream out = toFile.getOutputStream();){
            if (monitor != null) {
                monitor.initialize((int)fromFile.length());
            }
            FileUtilities.copyStreamToStream(fin, out, monitor);
        }
    }

    public static boolean createDir(File dir) {
        if (dir.isDirectory()) {
            return true;
        }
        dir.mkdir();
        return dir.isDirectory();
    }

    public static boolean mkdirs(File dir) {
        if (FileUtilities.createDir(dir)) {
            return true;
        }
        File canonFile = null;
        try {
            canonFile = dir.getCanonicalFile();
        }
        catch (IOException e) {
            return false;
        }
        File parent = canonFile.getParentFile();
        return parent != null && FileUtilities.mkdirs(parent) && FileUtilities.createDir(canonFile);
    }

    public static File checkedMkdir(File dir) throws IOException {
        if (!FileUtilities.createDir(dir)) {
            throw new IOException("Failed to create directory " + String.valueOf(dir));
        }
        return dir;
    }

    public static File checkedMkdirs(File dir) throws IOException {
        File canonFile;
        File parent;
        if (!FileUtilities.createDir(dir) && (parent = (canonFile = dir.getCanonicalFile()).getParentFile()) != null) {
            FileUtilities.checkedMkdirs(parent);
            FileUtilities.checkedMkdir(canonFile);
        }
        return dir;
    }

    public static boolean deleteDir(Path dir) {
        return FileUtilities.deleteDir(dir.toFile());
    }

    public static final boolean deleteDir(File dir) {
        try {
            return FileUtilities.deleteDir(dir, TaskMonitor.DUMMY);
        }
        catch (CancelledException cancelledException) {
            return true;
        }
    }

    public static final boolean deleteDir(File dir, TaskMonitor monitor) throws CancelledException {
        File[] files = dir.listFiles();
        if (files == null) {
            return dir.delete();
        }
        monitor.initialize(files.length);
        for (int i = 0; i < files.length; ++i) {
            monitor.checkCancelled();
            if (files[i].isDirectory()) {
                if (!FileUtilities.doDeleteDir(files[i], monitor)) {
                    FileUtilities.printDebug("Unable to delete directory: " + String.valueOf(files[i]));
                    return false;
                }
            } else {
                monitor.setMessage("Deleting file: " + String.valueOf(files[i]));
                if (!files[i].delete()) {
                    FileUtilities.printDebug("Unable to delete file: " + String.valueOf(files[i]));
                    return false;
                }
            }
            monitor.incrementProgress(i);
        }
        return dir.delete();
    }

    private static final boolean doDeleteDir(File dir, TaskMonitor monitor) throws CancelledException {
        File[] files = dir.listFiles();
        if (files == null) {
            return dir.delete();
        }
        for (File file : files) {
            monitor.checkCancelled();
            if (file.isDirectory()) {
                if (FileUtilities.doDeleteDir(file, monitor)) continue;
                FileUtilities.printDebug("Unable to delete directory: " + String.valueOf(file));
                return false;
            }
            monitor.setMessage("Deleting file: " + String.valueOf(file));
            if (file.delete()) continue;
            FileUtilities.printDebug("Unable to delete file: " + String.valueOf(file));
            return false;
        }
        return dir.delete();
    }

    public static final int copyDir(File originalDir, File copyDir, TaskMonitor monitor) throws IOException, CancelledException {
        return FileUtilities.copyDir(originalDir, copyDir, ACCEPT_ALL_FILE_FILTER, monitor);
    }

    public static final int copyDir(File originalDir, File copyDir, FileFilter filter, TaskMonitor monitor) throws IOException, CancelledException {
        if (monitor == null) {
            monitor = TaskMonitor.DUMMY;
        }
        if (!originalDir.exists()) {
            return 0;
        }
        File[] originalDirFiles = originalDir.listFiles(filter);
        if (originalDirFiles == null || originalDirFiles.length == 0) {
            return 0;
        }
        int copiedFilesCount = 0;
        monitor.initialize(originalDirFiles.length);
        for (File file : originalDirFiles) {
            monitor.checkCancelled();
            monitor.setMessage("Copying " + file.getAbsolutePath());
            File destinationFile = new File(copyDir, file.getName());
            if (file.isDirectory()) {
                copiedFilesCount += FileUtilities.doCopyDir(file, destinationFile, filter, monitor);
            } else {
                destinationFile.getParentFile().mkdirs();
                FileUtilities.copyFile(file, destinationFile, false, TaskMonitor.DUMMY);
                ++copiedFilesCount;
            }
            monitor.incrementProgress(1L);
        }
        return copiedFilesCount;
    }

    private static int doCopyDir(File originalDir, File copyDir, FileFilter filter, TaskMonitor monitor) throws IOException, CancelledException {
        if (!originalDir.exists()) {
            return 0;
        }
        File[] originalDirFiles = originalDir.listFiles(filter);
        if (originalDirFiles == null || originalDirFiles.length == 0) {
            return 0;
        }
        int copiedFilesCount = 0;
        for (File file : originalDirFiles) {
            monitor.checkCancelled();
            monitor.setMessage("Copying " + file.getAbsolutePath());
            File destinationFile = new File(copyDir, file.getName());
            if (file.isDirectory()) {
                copiedFilesCount += FileUtilities.doCopyDir(file, destinationFile, filter, monitor);
                continue;
            }
            destinationFile.getParentFile().mkdirs();
            FileUtilities.copyFile(file, destinationFile, false, monitor);
            ++copiedFilesCount;
        }
        return copiedFilesCount;
    }

    private static void printDebug(String text) {
        boolean isProductionMode;
        boolean bl = isProductionMode = !SystemUtilities.isInTestingMode() && !SystemUtilities.isInDevelopmentMode();
        if (isProductionMode) {
            return;
        }
        Msg.debug(FileUtilities.class, text);
    }

    public static final long copyStreamToFile(InputStream in, File toFile, boolean append, TaskMonitor monitor) throws IOException {
        try (FileOutputStream out = new FileOutputStream(toFile, append);){
            long l = FileUtilities.copyStreamToStream(in, out, monitor);
            return l;
        }
    }

    public static final void copyFileToStream(File fromFile, OutputStream out, TaskMonitor monitor) throws IOException {
        try (FileInputStream fin = new FileInputStream(fromFile);){
            if (monitor != null) {
                monitor.initialize((int)fromFile.length());
            }
            FileUtilities.copyStreamToStream(fin, out, monitor);
        }
    }

    public static long copyStreamToStream(InputStream in, OutputStream out, TaskMonitor monitor) throws IOException {
        long totalBytesCopied = 0L;
        byte[] buffer = new byte[32768];
        if (monitor != null) {
            out = new MonitoredOutputStream(out, monitor);
        }
        int bytesRead = 0;
        while ((bytesRead = in.read(buffer)) >= 0) {
            out.write(buffer, 0, bytesRead);
            totalBytesCopied += (long)bytesRead;
        }
        out.flush();
        return totalBytesCopied;
    }

    public static List<String> getLines(File file) throws IOException {
        return FileUtilities.getLines(new ResourceFile(file));
    }

    public static List<String> getLines(ResourceFile file) throws IOException {
        List<String> list;
        block8: {
            InputStream is = file.getInputStream();
            try {
                list = FileUtilities.getLines(is);
                if (is == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (is != null) {
                        try {
                            is.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (FileNotFoundException exc) {
                    return new ArrayList<String>();
                }
            }
            is.close();
        }
        return list;
    }

    public static List<String> getLinesQuietly(ResourceFile file) {
        try {
            return FileUtilities.getLines(file);
        }
        catch (IOException e) {
            Msg.error(FileUtilities.class, "Error parsing lines in file: " + String.valueOf(file), e);
            return Collections.emptyList();
        }
    }

    public static List<String> getLines(URL url) throws IOException {
        try (InputStream is = url.openStream();){
            List<String> list = FileUtilities.getLines(new BufferedReader(new InputStreamReader(is, StandardCharsets.UTF_8)));
            return list;
        }
    }

    public static List<String> getLines(InputStream is) throws IOException {
        return FileUtilities.getLines(new BufferedReader(new InputStreamReader(is, StandardCharsets.UTF_8)));
    }

    public static String getText(InputStream is) throws IOException {
        StringBuilder buf = new StringBuilder();
        List<String> lines = FileUtilities.getLines(is);
        for (String string : lines) {
            buf.append(string);
            buf.append("\n");
        }
        return buf.toString();
    }

    public static String getText(File f) throws IOException {
        if (f.length() > 0x10000000L) {
            throw new IOException("Text file too large to read: " + String.valueOf(f) + ", length: " + f.length());
        }
        try (FileInputStream is = new FileInputStream(f);){
            String string = FileUtilities.getText(is);
            return string;
        }
    }

    public static List<String> getLines(BufferedReader in) throws IOException {
        String line;
        ArrayList<String> fileLines = new ArrayList<String>();
        while ((line = in.readLine()) != null) {
            fileLines.add(line);
        }
        return fileLines;
    }

    public static void writeLinesToFile(File file, List<String> lines) throws IOException {
        try (FileWriter writer = new FileWriter(file);){
            for (String string : lines) {
                writer.write(string);
                writer.write("\n");
            }
        }
    }

    public static void writeStringToFile(File file, String s) throws IOException {
        try (FileWriter writer = new FileWriter(file);){
            writer.write(s);
        }
    }

    public static boolean isEmpty(File f) {
        if (f == null) {
            return true;
        }
        return f.isFile() && f.length() == 0L;
    }

    public static boolean isPathContainedWithin(File potentialParentFile, File otherFile) {
        try {
            Object parentPath = potentialParentFile.getCanonicalPath().replace('\\', '/');
            String otherPath = otherFile.getCanonicalPath().replace('\\', '/');
            if (((String)parentPath).equals(otherPath)) {
                return true;
            }
            if (!((String)parentPath).endsWith("/")) {
                parentPath = (String)parentPath + "/";
            }
            return otherPath.startsWith((String)parentPath);
        }
        catch (IOException e) {
            return false;
        }
    }

    public static boolean isPathContainedWithin(Collection<ResourceFile> potentialParents, ResourceFile otherFile) {
        return potentialParents.stream().anyMatch(parent -> parent.containsPath(otherFile));
    }

    public static String relativizePath(File f1, File f2) throws IOException {
        String otherPath;
        Object parentPath = f1.getCanonicalPath().replace('\\', '/');
        if (((String)parentPath).equals(otherPath = f2.getCanonicalPath().replace('\\', '/'))) {
            return null;
        }
        if (!((String)parentPath).endsWith("/")) {
            parentPath = (String)parentPath + "/";
        }
        if (!otherPath.startsWith((String)parentPath)) {
            return null;
        }
        String childPath = otherPath.substring(((String)parentPath).length());
        return childPath;
    }

    public static String relativizePath(ResourceFile f1, ResourceFile f2) {
        StringBuilder sb = new StringBuilder(f2.getName());
        for (f2 = f2.getParentFile(); f2 != null; f2 = f2.getParentFile()) {
            if (f1.equals(f2)) {
                return sb.toString();
            }
            sb.insert(0, f2.getName() + File.separator);
        }
        return null;
    }

    public static boolean exists(URI uri) {
        String scheme = uri.getScheme();
        if ("file".equals(scheme)) {
            File file = new File(uri);
            return file.exists();
        }
        if (!"jar".equals(scheme)) {
            return false;
        }
        FileSystem fs = FileUtilities.getOrCreateJarFS(uri);
        if (fs == null) {
            return false;
        }
        Path path = Paths.get(uri);
        return Files.exists(path, new LinkOption[0]);
    }

    private static FileSystem getOrCreateJarFS(URI jarURI) {
        HashMap env = new HashMap();
        try {
            return FileSystems.getFileSystem(jarURI);
        }
        catch (FileSystemNotFoundException e) {
            try {
                return FileSystems.newFileSystem(jarURI, env);
            }
            catch (IOException e1) {
                Msg.debug(FileUtilities.class, "Unexepecedly could not create jar filesystem");
                return null;
            }
        }
    }

    public static FileResolutionResult existsAndIsCaseDependent(File file) {
        return FileUtilities.existsAndIsCaseDependent(new ResourceFile(file));
    }

    public static FileResolutionResult existsAndIsCaseDependent(ResourceFile file) {
        String canonicalPath;
        if (!file.exists()) {
            return FileResolutionResult.doesNotExist(file);
        }
        try {
            canonicalPath = file.getCanonicalPath();
        }
        catch (IOException e) {
            return FileResolutionResult.doesNotExist(file);
        }
        String absolutePath = file.getAbsolutePath();
        FileResolutionResult result = FileUtilities.pathIsCaseDependent(canonicalPath, absolutePath);
        return result;
    }

    static FileResolutionResult pathIsCaseDependent(String canonicalPath, String absolutePath) {
        int aIndex;
        int size;
        List<String> canonical = FileUtilities.pathToParts(canonicalPath);
        List<String> absolute = FileUtilities.pathToParts(absolutePath);
        int cIndex = canonical.size() - 1;
        for (int i = size = (aIndex = absolute.size() - 1); i >= 0; --i) {
            String a;
            String c = canonical.get(cIndex);
            if (c.equalsIgnoreCase(a = absolute.get(aIndex))) {
                if (!c.equals(a)) {
                    return FileResolutionResult.notCaseDependent(canonicalPath, absolutePath);
                }
                --cIndex;
                --aIndex;
                continue;
            }
            --aIndex;
        }
        return FileResolutionResult.ok();
    }

    public static File resolveFileCaseSensitive(File caseSensitiveFile) {
        String canonicalName = null;
        try {
            File canonicalFile = caseSensitiveFile.getCanonicalFile();
            canonicalName = canonicalFile.getName();
        }
        catch (IOException canonicalFile) {
            // empty catch block
        }
        String caseSensitiveName = caseSensitiveFile.getName();
        return canonicalName != null && canonicalName.equalsIgnoreCase(caseSensitiveName) && !canonicalName.equals(caseSensitiveName) ? null : caseSensitiveFile;
    }

    public static File resolveFileCaseInsensitive(File f) {
        String fName;
        if (f.exists()) {
            return f;
        }
        File fParent = f.getParentFile();
        File tmp = new File(fParent, (fName = f.getName()).toLowerCase());
        if (tmp.exists()) {
            return tmp;
        }
        tmp = new File(fParent, fName.toUpperCase());
        if (tmp.exists()) {
            return tmp;
        }
        File[] fileList = fParent.listFiles();
        if (fileList != null) {
            for (File otherFile : fileList) {
                if (!otherFile.getName().equalsIgnoreCase(fName)) continue;
                return otherFile;
            }
        }
        return f;
    }

    public static List<String> pathToParts(String path) {
        String[] parts = path.split("\\\\|/");
        ArrayList<String> list = new ArrayList<String>(parts.length);
        for (String part : parts) {
            list.add(part);
        }
        return list;
    }

    public static String getPrettySize(File file) {
        return FileUtilities.formatLength(file.length());
    }

    public static String formatLength(long length) {
        NumberFormat formatter = SIZE_FORMAT_THREAD_LOCAL.get();
        if (length < 1000L) {
            return length + "B";
        }
        if (length < 1000000L) {
            return formatter.format((float)length / 1000.0f) + "KB";
        }
        return formatter.format((float)length / 1000000.0f) + "MB";
    }

    public static void setOwnerOnlyPermissions(File f) {
        f.setReadable(false, false);
        f.setReadable(true, true);
        f.setWritable(false, false);
        f.setWritable(true, true);
    }

    public static void openNative(File file) throws IOException {
        if (!Desktop.isDesktopSupported()) {
            Msg.showError(FileUtilities.class, null, "Native Desktop Unsupported", "Access to the user's native desktop is not supported in the current environment.\nUnable to open file: " + String.valueOf(file));
            return;
        }
        Desktop.getDesktop().open(file);
    }

    public static void forEachFile(Path path, Consumer<Path> consumer) throws IOException {
        if (!Files.isDirectory(path, new LinkOption[0])) {
            return;
        }
        try (Stream<Path> pathStream = Files.list(path);){
            pathStream.forEach(consumer);
        }
    }

    public static void forEachFile(File resourceFile, Consumer<File> consumer) {
        if (!resourceFile.isDirectory()) {
            return;
        }
        File[] files = resourceFile.listFiles();
        if (files == null) {
            return;
        }
        for (File child : files) {
            consumer.accept(child);
        }
    }

    public static void forEachFile(ResourceFile resourceFile, Consumer<ResourceFile> consumer) {
        if (!resourceFile.isDirectory()) {
            return;
        }
        ResourceFile[] files = resourceFile.listFiles();
        if (files == null) {
            return;
        }
        for (ResourceFile child : files) {
            consumer.accept(child);
        }
    }
}

