﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ec2/EC2Request.h>
#include <aws/ec2/EC2_EXPORTS.h>

#include <utility>

namespace Aws {
namespace EC2 {
namespace Model {

/**
 */
class ModifyInstanceCpuOptionsRequest : public EC2Request {
 public:
  AWS_EC2_API ModifyInstanceCpuOptionsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ModifyInstanceCpuOptions"; }

  AWS_EC2_API Aws::String SerializePayload() const override;

 protected:
  AWS_EC2_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The ID of the instance to update.</p>
   */
  inline const Aws::String& GetInstanceId() const { return m_instanceId; }
  inline bool InstanceIdHasBeenSet() const { return m_instanceIdHasBeenSet; }
  template <typename InstanceIdT = Aws::String>
  void SetInstanceId(InstanceIdT&& value) {
    m_instanceIdHasBeenSet = true;
    m_instanceId = std::forward<InstanceIdT>(value);
  }
  template <typename InstanceIdT = Aws::String>
  ModifyInstanceCpuOptionsRequest& WithInstanceId(InstanceIdT&& value) {
    SetInstanceId(std::forward<InstanceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of CPU cores to activate for the specified instance.</p>
   */
  inline int GetCoreCount() const { return m_coreCount; }
  inline bool CoreCountHasBeenSet() const { return m_coreCountHasBeenSet; }
  inline void SetCoreCount(int value) {
    m_coreCountHasBeenSet = true;
    m_coreCount = value;
  }
  inline ModifyInstanceCpuOptionsRequest& WithCoreCount(int value) {
    SetCoreCount(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of threads to run for each CPU core.</p>
   */
  inline int GetThreadsPerCore() const { return m_threadsPerCore; }
  inline bool ThreadsPerCoreHasBeenSet() const { return m_threadsPerCoreHasBeenSet; }
  inline void SetThreadsPerCore(int value) {
    m_threadsPerCoreHasBeenSet = true;
    m_threadsPerCore = value;
  }
  inline ModifyInstanceCpuOptionsRequest& WithThreadsPerCore(int value) {
    SetThreadsPerCore(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Checks whether you have the required permissions for the operation, without
   * actually making the request, and provides an error response. If you have the
   * required permissions, the error response is <code>DryRunOperation</code>.
   * Otherwise, it is <code>UnauthorizedOperation</code>.</p>
   */
  inline bool GetDryRun() const { return m_dryRun; }
  inline bool DryRunHasBeenSet() const { return m_dryRunHasBeenSet; }
  inline void SetDryRun(bool value) {
    m_dryRunHasBeenSet = true;
    m_dryRun = value;
  }
  inline ModifyInstanceCpuOptionsRequest& WithDryRun(bool value) {
    SetDryRun(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_instanceId;

  int m_coreCount{0};

  int m_threadsPerCore{0};

  bool m_dryRun{false};
  bool m_instanceIdHasBeenSet = false;
  bool m_coreCountHasBeenSet = false;
  bool m_threadsPerCoreHasBeenSet = false;
  bool m_dryRunHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
