﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ec2/EC2Request.h>
#include <aws/ec2/EC2_EXPORTS.h>

#include <utility>

namespace Aws {
namespace EC2 {
namespace Model {

/**
 */
class EnableFastSnapshotRestoresRequest : public EC2Request {
 public:
  AWS_EC2_API EnableFastSnapshotRestoresRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "EnableFastSnapshotRestores"; }

  AWS_EC2_API Aws::String SerializePayload() const override;

 protected:
  AWS_EC2_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>One or more Availability Zones. For example, <code>us-east-2a</code>.</p>
   * <p>Either <code>AvailabilityZone</code> or <code>AvailabilityZoneId</code> must
   * be specified in the request, but not both.</p>
   */
  inline const Aws::Vector<Aws::String>& GetAvailabilityZones() const { return m_availabilityZones; }
  inline bool AvailabilityZonesHasBeenSet() const { return m_availabilityZonesHasBeenSet; }
  template <typename AvailabilityZonesT = Aws::Vector<Aws::String>>
  void SetAvailabilityZones(AvailabilityZonesT&& value) {
    m_availabilityZonesHasBeenSet = true;
    m_availabilityZones = std::forward<AvailabilityZonesT>(value);
  }
  template <typename AvailabilityZonesT = Aws::Vector<Aws::String>>
  EnableFastSnapshotRestoresRequest& WithAvailabilityZones(AvailabilityZonesT&& value) {
    SetAvailabilityZones(std::forward<AvailabilityZonesT>(value));
    return *this;
  }
  template <typename AvailabilityZonesT = Aws::String>
  EnableFastSnapshotRestoresRequest& AddAvailabilityZones(AvailabilityZonesT&& value) {
    m_availabilityZonesHasBeenSet = true;
    m_availabilityZones.emplace_back(std::forward<AvailabilityZonesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>One or more Availability Zone IDs. For example, <code>use2-az1</code>.</p>
   * <p>Either <code>AvailabilityZone</code> or <code>AvailabilityZoneId</code> must
   * be specified in the request, but not both.</p>
   */
  inline const Aws::Vector<Aws::String>& GetAvailabilityZoneIds() const { return m_availabilityZoneIds; }
  inline bool AvailabilityZoneIdsHasBeenSet() const { return m_availabilityZoneIdsHasBeenSet; }
  template <typename AvailabilityZoneIdsT = Aws::Vector<Aws::String>>
  void SetAvailabilityZoneIds(AvailabilityZoneIdsT&& value) {
    m_availabilityZoneIdsHasBeenSet = true;
    m_availabilityZoneIds = std::forward<AvailabilityZoneIdsT>(value);
  }
  template <typename AvailabilityZoneIdsT = Aws::Vector<Aws::String>>
  EnableFastSnapshotRestoresRequest& WithAvailabilityZoneIds(AvailabilityZoneIdsT&& value) {
    SetAvailabilityZoneIds(std::forward<AvailabilityZoneIdsT>(value));
    return *this;
  }
  template <typename AvailabilityZoneIdsT = Aws::String>
  EnableFastSnapshotRestoresRequest& AddAvailabilityZoneIds(AvailabilityZoneIdsT&& value) {
    m_availabilityZoneIdsHasBeenSet = true;
    m_availabilityZoneIds.emplace_back(std::forward<AvailabilityZoneIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The IDs of one or more snapshots. For example,
   * <code>snap-1234567890abcdef0</code>. You can specify a snapshot that was shared
   * with you from another Amazon Web Services account.</p>
   */
  inline const Aws::Vector<Aws::String>& GetSourceSnapshotIds() const { return m_sourceSnapshotIds; }
  inline bool SourceSnapshotIdsHasBeenSet() const { return m_sourceSnapshotIdsHasBeenSet; }
  template <typename SourceSnapshotIdsT = Aws::Vector<Aws::String>>
  void SetSourceSnapshotIds(SourceSnapshotIdsT&& value) {
    m_sourceSnapshotIdsHasBeenSet = true;
    m_sourceSnapshotIds = std::forward<SourceSnapshotIdsT>(value);
  }
  template <typename SourceSnapshotIdsT = Aws::Vector<Aws::String>>
  EnableFastSnapshotRestoresRequest& WithSourceSnapshotIds(SourceSnapshotIdsT&& value) {
    SetSourceSnapshotIds(std::forward<SourceSnapshotIdsT>(value));
    return *this;
  }
  template <typename SourceSnapshotIdsT = Aws::String>
  EnableFastSnapshotRestoresRequest& AddSourceSnapshotIds(SourceSnapshotIdsT&& value) {
    m_sourceSnapshotIdsHasBeenSet = true;
    m_sourceSnapshotIds.emplace_back(std::forward<SourceSnapshotIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Checks whether you have the required permissions for the action, without
   * actually making the request, and provides an error response. If you have the
   * required permissions, the error response is <code>DryRunOperation</code>.
   * Otherwise, it is <code>UnauthorizedOperation</code>.</p>
   */
  inline bool GetDryRun() const { return m_dryRun; }
  inline bool DryRunHasBeenSet() const { return m_dryRunHasBeenSet; }
  inline void SetDryRun(bool value) {
    m_dryRunHasBeenSet = true;
    m_dryRun = value;
  }
  inline EnableFastSnapshotRestoresRequest& WithDryRun(bool value) {
    SetDryRun(value);
    return *this;
  }
  ///@}
 private:
  Aws::Vector<Aws::String> m_availabilityZones;

  Aws::Vector<Aws::String> m_availabilityZoneIds;

  Aws::Vector<Aws::String> m_sourceSnapshotIds;

  bool m_dryRun{false};
  bool m_availabilityZonesHasBeenSet = false;
  bool m_availabilityZoneIdsHasBeenSet = false;
  bool m_sourceSnapshotIdsHasBeenSet = false;
  bool m_dryRunHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
