/* SPDX-License-Identifier: GPL-2.0-only */

#include <console/console.h>
#include <device/device.h>
#include <device/pci.h>
#include <device/azalia_device.h>
#include <device/mmio.h>
#include <delay.h>
#include <timer.h>
#include <types.h>

static enum cb_err azalia_assert_reset(u8 *base, bool reset)
{
	const u32 val = reset ? 0 : HDA_GCTL_CRST; /* active-low CRST# */
	struct stopwatch sw;

	clrsetbits32(base + HDA_GCTL_REG, HDA_GCTL_CRST, val);

	/* Wait for the controller to complete the link-reset sequence */
	stopwatch_init_msecs_expire(&sw, 50);

	do {
		/* Wait 1ms based on BKDG wait time */
		mdelay(1);
		u32 reg32 = read32(base + HDA_GCTL_REG);
		if ((reg32 & HDA_GCTL_CRST) == val)
			return CB_SUCCESS;
	} while (!stopwatch_expired(&sw));

	return CB_ERR;
}

enum cb_err azalia_enter_reset(u8 *base)
{
	return azalia_assert_reset(base, true);
}

enum cb_err azalia_exit_reset(u8 *base)
{
	if (azalia_assert_reset(base, false) != CB_SUCCESS)
		return CB_ERR;

	/* Codecs have up to 25 frames (at 48kHz) to signal an
	   initialization request (HDA Spec 1.0a "4.3 Codec Discovery"). */
	udelay(521);
	return CB_SUCCESS;
}

static u16 codec_detect(u8 *base)
{
	u16 reg16;

	if (azalia_exit_reset(base) != CB_SUCCESS)
		goto no_codec;

	/*
	 * In the HD Audio Specification Rev. 1.0a, every bitfield in the GCAP
	 * register is RO (Read Only). However, it is known that in some Intel
	 * PCHs (e.g 6-series and 7-series, documents 324645 and 326776), some
	 * of the bitfields in the GCAP register are R/WO (Read / Write Once).
	 * GCAP is RO on 5-series PCHs; 8-series and 9-series PCHs have a lock
	 * bit for GCAP elsewhere.
	 *
	 * Lock GCAP by reading GCAP and writing back the same value. This has
	 * no effect on platforms that implement GCAP as a RO register or lock
	 * GCAP through a different mechanism.
	 */
	write16(base + HDA_GCAP_REG, read16(base + HDA_GCAP_REG));

	/* clear STATESTS bits (BAR + 0x0e)[14:0] */
	setbits16(base + HDA_STATESTS_REG, 0x7fff);

	if (azalia_enter_reset(base) != CB_SUCCESS)
		goto no_codec;

	if (azalia_exit_reset(base) != CB_SUCCESS)
		goto no_codec;

	/* Read in Codec location (BAR + 0x0e)[14:0] */
	reg16 = read16(base + HDA_STATESTS_REG);
	reg16 &= 0x7fff;
	if (!reg16)
		goto no_codec;

	return reg16;

no_codec:
	/* Codec Not found */
	azalia_enter_reset(base);
	printk(BIOS_DEBUG, "azalia_audio: no codec!\n");
	return 0;
}

/*
 * Find a specific entry within a verb table
 *
 * @param verb_table:		verb table data
 * @param verb_table_bytes:	verb table size in bytes
 * @param viddid:		vendor/device to search for
 * @param verb:			pointer to entry within table
 *
 * Returns size of the entry within the verb table,
 * Returns 0 if the entry is not found
 *
 * The HDA verb table is composed of dwords. A set of 4 dwords is
 * grouped together to form a "jack" descriptor.
 *   Bits 31:28 - Codec Address
 *   Bits 27:20 - NID
 *   Bits 19:8  - Verb ID
 *   Bits  7:0  - Payload
 *
 * coreboot groups different codec verb tables into a single table
 * and prefixes each with a specific header consisting of 3
 * dword entries:
 *   1 - Codec Vendor/Device ID
 *   2 - Subsystem ID
 *   3 - Number of jacks (groups of 4 dwords) for this codec
 */
u32 azalia_find_verb(const u32 *verb_table, u32 verb_table_bytes, u32 viddid, const u32 **verb)
{
	int idx = 0;

	while (idx < (verb_table_bytes / sizeof(u32))) {
		/* Header contains the number of jacks, aka groups of 4 dwords */
		u32 verb_size = 4 * verb_table[idx + 2];
		if (verb_table[idx] != viddid) {
			idx += verb_size + 3;	// skip verb + header
			continue;
		}
		*verb = &verb_table[idx + 3];
		return verb_size;
	}

	/* Not all codecs need to load another verb */
	return 0;
}

/*
 * Wait 50usec for the controller to indicate it is ready.
 */
static int wait_for_ready(u8 *base)
{
	struct stopwatch sw;
	/* Use a 50 usec timeout - the Linux kernel uses the same duration */
	stopwatch_init_usecs_expire(&sw, 50);

	while (!stopwatch_expired(&sw)) {
		u32 reg32 = read32(base + HDA_ICII_REG);
		if (!(reg32 & HDA_ICII_BUSY))
			return 0;
		udelay(1);
	}

	return -1;
}

/*
 * Wait for the codec to indicate that it accepted the previous command.
 * No response would imply that the codec is non-operative.
 */
static int wait_for_valid(u8 *base)
{
	struct stopwatch sw;
	u32 reg32;

	/* Send the verb to the codec */
	setbits32(base + HDA_ICII_REG, HDA_ICII_BUSY | HDA_ICII_VALID);

	/*
	 * The timeout is never reached when the codec is functioning properly.
	 * Using a small timeout value can result in spurious errors with some
	 * codecs, e.g. a codec that is slow to respond but operates correctly.
	 * When a codec is non-operative, the timeout is only reached once per
	 * verb table, thus the impact on booting time is relatively small. So,
	 * use a reasonably long enough timeout to cover all possible cases.
	 */
	stopwatch_init_msecs_expire(&sw, 1);
	while (!stopwatch_expired(&sw)) {
		reg32 = read32(base + HDA_ICII_REG);
		if ((reg32 & (HDA_ICII_VALID | HDA_ICII_BUSY)) == HDA_ICII_VALID)
			return 0;
		udelay(1);
	}

	/*
	 * HDA spec 1.0a "3.4.3 Offset 68h: Immediate Command Status"
	 * tells us to clear the busy bit explicitly, then poll until
	 * the controller is ready.
	 */
	write32(base + HDA_ICII_REG, 0);
	if (wait_for_ready(base) < 0) {
		printk(BIOS_WARNING, "azalia_audio: controller is unresponsive.\n");
		return -2;
	}
	return -1;
}

static int azalia_write_verb(u8 *base, u32 verb)
{
	if (wait_for_ready(base) < 0)
		return -1;

	write32(base + HDA_IC_REG, verb);

	return wait_for_valid(base);
}

int azalia_program_verb_table(u8 *base, const u32 *verbs, u32 verb_size)
{
	if (!verbs)
		return 0;

	for (u32 i = 0; i < verb_size; i++) {
		if (azalia_write_verb(base, verbs[i]) < 0)
			return -1;
	}
	return 0;
}

__weak void mainboard_azalia_program_runtime_verbs(u8 *base, u32 viddid)
{
}

#if CONFIG(AZALIA_USE_LEGACY_VERB_TABLE)
void azalia_codec_init(u8 *base, int addr, const u32 *verb_table, u32 verb_table_bytes)
{
	if (azalia_write_verb(base, AZALIA_VERB_GET_VENDOR_ID(addr)) < 0)
		return;
	u32 viddid = read32(base + HDA_IR_REG);

	const u32 *verb;
	u32 verb_size;

	printk(BIOS_DEBUG, "azalia_audio: initializing codec #%d...\n", addr);
	printk(BIOS_DEBUG, "azalia_audio:  - vendor/device id: 0x%08x\n", viddid);

	verb_size = azalia_find_verb(verb_table, verb_table_bytes, viddid, &verb);

	if (verb_size == 0) {
		printk(BIOS_DEBUG, "azalia_audio:  - no verb!\n");
		return;
	}
	printk(BIOS_DEBUG, "azalia_audio:  - verb size: %u\n", verb_size);

	if (azalia_program_verb_table(base, verb, verb_size) < 0)
		printk(BIOS_DEBUG, "azalia_audio:  - verb not loaded\n");
	else
		printk(BIOS_DEBUG, "azalia_audio:  - verb loaded\n");

	mainboard_azalia_program_runtime_verbs(base, viddid);
}

void azalia_codecs_init(u8 *base, u16 codec_mask)
{
	for (int i = AZALIA_MAX_CODECS - 1; i >= 0; i--) {
		if (codec_mask & BIT(i))
			azalia_codec_init(base, i, cim_verb_data, cim_verb_data_size);
	}

	azalia_program_verb_table(base, pc_beep_verbs, pc_beep_verbs_size);
}
#else
static struct azalia_codec *find_codec(struct azalia_codec *codecs, u32 vid, u8 addr)
{
	for (struct azalia_codec *codec = codecs; codec->vendor_id; codec++) {
		if (codec->vendor_id == vid && codec->address == addr)
			return codec;
	}
	return NULL;
}

void azalia_codec_init(u8 *base, struct azalia_codec *codec)
{
	/* Give a generic name if nothing was provided */
	const char *name = codec->name ? codec->name : "codec";

	printk(BIOS_DEBUG, "azalia_audio: initializing %s...\n", name);
	printk(BIOS_DEBUG, "azalia_audio:  - vendor ID:    0x%08x\n", codec->vendor_id);
	printk(BIOS_DEBUG, "azalia_audio:  - subsystem ID: 0x%08x\n", codec->subsystem_id);
	printk(BIOS_DEBUG, "azalia_audio:  - address:      %u\n", codec->address);

	if (azalia_program_verb_table(base, codec->verbs, codec->verb_count) < 0) {
		printk(BIOS_WARNING, "azalia_audio: failed to load verbs\n");
		return;
	}

	mainboard_azalia_program_runtime_verbs(base, codec->vendor_id);

	printk(BIOS_DEBUG, "azalia_audio: done\n");
}

void azalia_custom_codecs_init(u8 *base, struct azalia_codec *codecs, u16 codec_mask)
{
	for (u8 i = 0; i < AZALIA_MAX_CODECS; i++) {
		if (!(codec_mask & BIT(i)))
			continue;

		if (azalia_write_verb(base, AZALIA_VERB_GET_VENDOR_ID(i)) < 0)
			continue;
		u32 vid = read32(base + HDA_IR_REG);

		struct azalia_codec *codec = find_codec(codecs, vid, i);
		if (codec == NULL) {
			printk(BIOS_WARNING,
			       "azalia_audio: cannot find verbs for codec with vendor ID: 0x%08x (address %u)\n",
			       vid, i);
			continue;
		}

		azalia_codec_init(base, codec);
	}

	azalia_program_verb_table(base, pc_beep_verbs, pc_beep_verbs_size);
}

void azalia_codecs_init(u8 *base, u16 codec_mask)
{
	azalia_custom_codecs_init(base, mainboard_azalia_codecs, codec_mask);
}
#endif

void azalia_audio_init(struct device *dev)
{
	u8 *base;
	struct resource *res;
	u16 codec_mask;

	res = probe_resource(dev, PCI_BASE_ADDRESS_0);
	if (!res)
		return;

	// NOTE this will break as soon as the azalia_audio gets a bar above 4G.
	// Is there anything we can do about it?
	base = res2mmio(res, 0, 0);
	printk(BIOS_DEBUG, "azalia_audio: base = %p\n", base);
	codec_mask = codec_detect(base);

	if (codec_mask) {
		printk(BIOS_DEBUG, "azalia_audio: codec_mask = 0x%02x\n", codec_mask);
		azalia_codecs_init(base, codec_mask);
	}
}

struct device_operations default_azalia_audio_ops = {
	.read_resources		= pci_dev_read_resources,
	.set_resources		= pci_dev_set_resources,
	.enable_resources	= pci_dev_enable_resources,
	.init			= azalia_audio_init,
	.ops_pci		= &pci_dev_ops_pci,
};
