﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/swf/SWFRequest.h>
#include <aws/swf/SWF_EXPORTS.h>
#include <aws/swf/model/Decision.h>
#include <aws/swf/model/TaskList.h>

#include <utility>

namespace Aws {
namespace SWF {
namespace Model {

/**
 * <p>Input data for a TaskCompleted response to a decision task.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/swf-2012-01-25/RespondDecisionTaskCompletedInput">AWS
 * API Reference</a></p>
 */
class RespondDecisionTaskCompletedRequest : public SWFRequest {
 public:
  AWS_SWF_API RespondDecisionTaskCompletedRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "RespondDecisionTaskCompleted"; }

  AWS_SWF_API Aws::String SerializePayload() const override;

  AWS_SWF_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The <code>taskToken</code> from the <a>DecisionTask</a>.</p>  <p>
   * <code>taskToken</code> is generated by the service and should be treated as an
   * opaque value. If the task is passed to another process, its
   * <code>taskToken</code> must also be passed. This enables it to provide its
   * progress and respond with results.</p>
   */
  inline const Aws::String& GetTaskToken() const { return m_taskToken; }
  inline bool TaskTokenHasBeenSet() const { return m_taskTokenHasBeenSet; }
  template <typename TaskTokenT = Aws::String>
  void SetTaskToken(TaskTokenT&& value) {
    m_taskTokenHasBeenSet = true;
    m_taskToken = std::forward<TaskTokenT>(value);
  }
  template <typename TaskTokenT = Aws::String>
  RespondDecisionTaskCompletedRequest& WithTaskToken(TaskTokenT&& value) {
    SetTaskToken(std::forward<TaskTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The list of decisions (possibly empty) made by the decider while processing
   * this decision task. See the docs for the <a>Decision</a> structure for
   * details.</p>
   */
  inline const Aws::Vector<Decision>& GetDecisions() const { return m_decisions; }
  inline bool DecisionsHasBeenSet() const { return m_decisionsHasBeenSet; }
  template <typename DecisionsT = Aws::Vector<Decision>>
  void SetDecisions(DecisionsT&& value) {
    m_decisionsHasBeenSet = true;
    m_decisions = std::forward<DecisionsT>(value);
  }
  template <typename DecisionsT = Aws::Vector<Decision>>
  RespondDecisionTaskCompletedRequest& WithDecisions(DecisionsT&& value) {
    SetDecisions(std::forward<DecisionsT>(value));
    return *this;
  }
  template <typename DecisionsT = Decision>
  RespondDecisionTaskCompletedRequest& AddDecisions(DecisionsT&& value) {
    m_decisionsHasBeenSet = true;
    m_decisions.emplace_back(std::forward<DecisionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>User defined context to add to workflow execution.</p>
   */
  inline const Aws::String& GetExecutionContext() const { return m_executionContext; }
  inline bool ExecutionContextHasBeenSet() const { return m_executionContextHasBeenSet; }
  template <typename ExecutionContextT = Aws::String>
  void SetExecutionContext(ExecutionContextT&& value) {
    m_executionContextHasBeenSet = true;
    m_executionContext = std::forward<ExecutionContextT>(value);
  }
  template <typename ExecutionContextT = Aws::String>
  RespondDecisionTaskCompletedRequest& WithExecutionContext(ExecutionContextT&& value) {
    SetExecutionContext(std::forward<ExecutionContextT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The task list to use for the future decision tasks of this workflow
   * execution. This list overrides the original task list you specified while
   * starting the workflow execution. </p>
   */
  inline const TaskList& GetTaskList() const { return m_taskList; }
  inline bool TaskListHasBeenSet() const { return m_taskListHasBeenSet; }
  template <typename TaskListT = TaskList>
  void SetTaskList(TaskListT&& value) {
    m_taskListHasBeenSet = true;
    m_taskList = std::forward<TaskListT>(value);
  }
  template <typename TaskListT = TaskList>
  RespondDecisionTaskCompletedRequest& WithTaskList(TaskListT&& value) {
    SetTaskList(std::forward<TaskListT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies a timeout (in seconds) for the task list override. When this
   * parameter is missing, the task list override is permanent. This parameter makes
   * it possible to temporarily override the task list. If a decision task scheduled
   * on the override task list is not started within the timeout, the decision task
   * will time out. Amazon SWF will revert the override and schedule a new decision
   * task to the original task list.</p> <p>If a decision task scheduled on the
   * override task list is started within the timeout, but not completed within the
   * start-to-close timeout, Amazon SWF will also revert the override and schedule a
   * new decision task to the original task list.</p>
   */
  inline const Aws::String& GetTaskListScheduleToStartTimeout() const { return m_taskListScheduleToStartTimeout; }
  inline bool TaskListScheduleToStartTimeoutHasBeenSet() const { return m_taskListScheduleToStartTimeoutHasBeenSet; }
  template <typename TaskListScheduleToStartTimeoutT = Aws::String>
  void SetTaskListScheduleToStartTimeout(TaskListScheduleToStartTimeoutT&& value) {
    m_taskListScheduleToStartTimeoutHasBeenSet = true;
    m_taskListScheduleToStartTimeout = std::forward<TaskListScheduleToStartTimeoutT>(value);
  }
  template <typename TaskListScheduleToStartTimeoutT = Aws::String>
  RespondDecisionTaskCompletedRequest& WithTaskListScheduleToStartTimeout(TaskListScheduleToStartTimeoutT&& value) {
    SetTaskListScheduleToStartTimeout(std::forward<TaskListScheduleToStartTimeoutT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_taskToken;
  bool m_taskTokenHasBeenSet = false;

  Aws::Vector<Decision> m_decisions;
  bool m_decisionsHasBeenSet = false;

  Aws::String m_executionContext;
  bool m_executionContextHasBeenSet = false;

  TaskList m_taskList;
  bool m_taskListHasBeenSet = false;

  Aws::String m_taskListScheduleToStartTimeout;
  bool m_taskListScheduleToStartTimeoutHasBeenSet = false;
};

}  // namespace Model
}  // namespace SWF
}  // namespace Aws
