/*
 * Decompiled with CFR 0.152.
 */
package org.apache.storm;

import java.nio.channels.ClosedByInterruptException;
import java.util.Comparator;
import java.util.Random;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.storm.utils.Time;
import org.apache.storm.utils.Utils;

public class StormTimer
implements AutoCloseable {
    private final StormTimerTask task = new StormTimerTask();

    public StormTimer(String name, Thread.UncaughtExceptionHandler onKill) {
        if (onKill == null) {
            throw new RuntimeException("onKill func is null!");
        }
        if (name == null) {
            this.task.setName("timer");
        } else {
            this.task.setName(name);
        }
        this.task.setOnKillFunc(onKill);
        this.task.setActive(true);
        this.task.setDaemon(true);
        this.task.setPriority(10);
        this.task.start();
    }

    public void schedule(int delaySecs, Runnable func, boolean checkActive, int jitterMs) {
        this.scheduleMs(Time.secsToMillisLong(delaySecs), func, checkActive, jitterMs);
    }

    public void schedule(int delaySecs, Runnable func) {
        this.schedule(delaySecs, func, true, 0);
    }

    public void scheduleMs(long delayMs, Runnable func, boolean checkActive, int jitterMs) {
        if (func == null) {
            throw new RuntimeException("function to schedule is null!");
        }
        if (checkActive) {
            this.checkActive();
        }
        String id = Utils.uuid();
        long endTimeMs = Time.currentTimeMillis() + delayMs;
        if (jitterMs > 0) {
            endTimeMs = (long)this.task.random.nextInt(jitterMs) + endTimeMs;
        }
        this.task.add(new QueueEntry(endTimeMs, func, id));
    }

    public void scheduleMs(long delayMs, Runnable func) {
        this.scheduleMs(delayMs, func, true, 0);
    }

    public void scheduleRecurring(int delaySecs, final int recurSecs, final Runnable func) {
        this.schedule(delaySecs, new Runnable(){

            @Override
            public void run() {
                func.run();
                StormTimer.this.schedule(recurSecs, this, false, 0);
            }
        });
    }

    public void scheduleRecurringMs(long delayMs, final long recurMs, final Runnable func) {
        this.scheduleMs(delayMs, new Runnable(){

            @Override
            public void run() {
                func.run();
                StormTimer.this.scheduleMs(recurMs, this, true, 0);
            }
        });
    }

    public void scheduleRecurringWithJitter(int delaySecs, final int recurSecs, final int jitterMs, final Runnable func) {
        this.schedule(delaySecs, new Runnable(){

            @Override
            public void run() {
                func.run();
                StormTimer.this.schedule(recurSecs, this, false, jitterMs);
            }
        });
    }

    private void checkActive() {
        if (!this.task.isActive()) {
            throw new IllegalStateException("Timer is not active");
        }
    }

    @Override
    public void close() throws InterruptedException {
        if (this.task.isActive()) {
            this.task.setActive(false);
            this.task.interrupt();
            this.task.join();
        }
    }

    public boolean isTimerWaiting() {
        return Time.isThreadWaiting(this.task);
    }

    public static class StormTimerTask
    extends Thread {
        private PriorityBlockingQueue<QueueEntry> queue = new PriorityBlockingQueue<QueueEntry>(11, new Comparator<QueueEntry>(){

            @Override
            public int compare(QueueEntry o1, QueueEntry o2) {
                return o1.endTimeMs.intValue() - o2.endTimeMs.intValue();
            }
        });
        private AtomicBoolean active = new AtomicBoolean(false);
        private Thread.UncaughtExceptionHandler onKill;
        private Random random = new Random();

        @Override
        public void run() {
            while (this.active.get()) {
                QueueEntry queueEntry = null;
                try {
                    queueEntry = this.queue.peek();
                    if (queueEntry != null && Time.currentTimeMillis() >= queueEntry.endTimeMs) {
                        this.queue.remove(queueEntry);
                        queueEntry.func.run();
                    } else if (queueEntry != null) {
                        Time.sleep(Math.min(1000L, queueEntry.endTimeMs - Time.currentTimeMillis()));
                    } else {
                        Time.sleep(1000L);
                    }
                    if (!Thread.interrupted()) continue;
                    this.active.set(false);
                }
                catch (Throwable e) {
                    if (Utils.exceptionCauseIsInstanceOf(InterruptedException.class, e) || Utils.exceptionCauseIsInstanceOf(ClosedByInterruptException.class, e)) continue;
                    this.setActive(false);
                    this.onKill.uncaughtException(this, e);
                }
            }
        }

        public void setOnKillFunc(Thread.UncaughtExceptionHandler onKill) {
            this.onKill = onKill;
        }

        public boolean isActive() {
            return this.active.get();
        }

        public void setActive(boolean flag) {
            this.active.set(flag);
        }

        public void add(QueueEntry queueEntry) {
            this.queue.add(queueEntry);
        }
    }

    public static class QueueEntry {
        public final Long endTimeMs;
        public final Runnable func;
        public final String id;

        public QueueEntry(Long endTimeMs, Runnable func, String id) {
            this.endTimeMs = endTimeMs;
            this.func = func;
            this.id = id;
        }
    }
}

