/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.security;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.protobuf.Message;
import java.io.File;
import java.io.IOException;
import java.math.BigInteger;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.security.KeyPair;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.time.Duration;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.commons.io.FileUtils;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.conf.OzoneConfiguration;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.scm.ha.HASecurityUtils;
import org.apache.hadoop.hdds.scm.ha.SCMContext;
import org.apache.hadoop.hdds.scm.ha.SCMServiceException;
import org.apache.hadoop.hdds.scm.ha.SequenceIdGenerator;
import org.apache.hadoop.hdds.scm.ha.StatefulService;
import org.apache.hadoop.hdds.scm.security.RootCARotationHandler;
import org.apache.hadoop.hdds.scm.security.RootCARotationHandlerImpl;
import org.apache.hadoop.hdds.scm.security.RootCARotationMetrics;
import org.apache.hadoop.hdds.scm.server.SCMStorageConfig;
import org.apache.hadoop.hdds.scm.server.StorageContainerManager;
import org.apache.hadoop.hdds.security.SecurityConfig;
import org.apache.hadoop.hdds.security.x509.certificate.CertInfo;
import org.apache.hadoop.hdds.security.x509.certificate.authority.CertificateServer;
import org.apache.hadoop.hdds.security.x509.certificate.authority.profile.DefaultCAProfile;
import org.apache.hadoop.hdds.security.x509.certificate.authority.profile.PKIProfile;
import org.apache.hadoop.hdds.security.x509.certificate.client.SCMCertificateClient;
import org.apache.hadoop.hdds.security.x509.certificate.utils.CertificateCodec;
import org.apache.hadoop.hdds.security.x509.certificate.utils.CertificateSignRequest;
import org.apache.hadoop.hdds.security.x509.keys.HDDSKeyGenerator;
import org.apache.hadoop.hdds.security.x509.keys.KeyStorage;
import org.apache.hadoop.ozone.OzoneConsts;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RootCARotationManager
extends StatefulService {
    private static final Logger LOG = LoggerFactory.getLogger(RootCARotationManager.class);
    private static final String SERVICE_NAME = RootCARotationManager.class.getSimpleName();
    private final StorageContainerManager scm;
    private final OzoneConfiguration ozoneConf;
    private final SecurityConfig secConf;
    private final SCMContext scmContext;
    private final ScheduledExecutorService executorService;
    private final Duration checkInterval;
    private final Duration renewalGracePeriod;
    private final Date timeOfDay;
    private final Duration ackTimeout;
    private final Duration rootCertPollInterval;
    private final SCMCertificateClient scmCertClient;
    private final AtomicBoolean isRunning = new AtomicBoolean(false);
    private final AtomicBoolean isProcessing = new AtomicBoolean(false);
    private final AtomicReference<Long> processStartTime = new AtomicReference();
    private final AtomicBoolean isPostProcessing = new AtomicBoolean(false);
    private final String threadName;
    private final String newCAComponent = OzoneConsts.SCM_ROOT_CA_COMPONENT_NAME + "-next" + "-progress";
    private RootCARotationHandler handler;
    private final SequenceIdGenerator sequenceIdGen;
    private ScheduledFuture rotationTask;
    private ScheduledFuture waitAckTask;
    private ScheduledFuture waitAckTimeoutTask;
    private final RootCARotationMetrics metrics;
    private ScheduledFuture clearPostProcessingTask;

    public RootCARotationManager(StorageContainerManager scm) {
        super(scm.getStatefulServiceStateManager());
        this.scm = scm;
        this.ozoneConf = scm.getConfiguration();
        this.secConf = new SecurityConfig((ConfigurationSource)this.ozoneConf);
        this.scmContext = scm.getScmContext();
        this.checkInterval = this.secConf.getCaCheckInterval();
        this.ackTimeout = this.secConf.getCaAckTimeout();
        this.renewalGracePeriod = this.secConf.getRenewalGracePeriod();
        this.timeOfDay = Date.from(LocalDateTime.parse(this.secConf.getCaRotationTimeOfDay()).atZone(ZoneId.systemDefault()).toInstant());
        this.rootCertPollInterval = this.secConf.getRootCaCertificatePollingInterval();
        this.threadName = scm.threadNamePrefix() + SERVICE_NAME;
        this.executorService = Executors.newScheduledThreadPool(1, new ThreadFactoryBuilder().setNameFormat(this.threadName).setDaemon(true).build());
        this.scmCertClient = (SCMCertificateClient)scm.getScmCertificateClient();
        this.sequenceIdGen = scm.getSequenceIdGen();
        this.handler = new RootCARotationHandlerImpl.Builder().setRatisServer(scm.getScmHAManager().getRatisServer()).setStorageContainerManager(scm).setRootCARotationManager(this).build();
        scm.getSCMServiceManager().register(this);
        this.metrics = RootCARotationMetrics.create();
    }

    @Override
    public void notifyStatusChanged() {
        if (!this.scmContext.isLeader() || this.scmContext.isInSafeMode()) {
            if (this.isRunning.compareAndSet(true, false)) {
                LOG.info("notifyStatusChanged: disable monitor task.");
                if (this.rotationTask != null) {
                    this.rotationTask.cancel(true);
                }
                if (this.waitAckTask != null) {
                    this.waitAckTask.cancel(true);
                }
                if (this.waitAckTimeoutTask != null) {
                    this.waitAckTimeoutTask.cancel(true);
                }
                if (this.clearPostProcessingTask != null) {
                    this.clearPostProcessingTask.cancel(true);
                }
                this.isProcessing.set(false);
                this.processStartTime.set(null);
                this.isPostProcessing.set(false);
            }
            return;
        }
        if (this.isRunning.compareAndSet(false, true)) {
            LOG.info("notifyStatusChanged: enable monitor task");
            try {
                this.checkAndHandlePostProcessing();
            }
            catch (IOException | CertificateException e) {
                throw new RuntimeException("Error while checking post-processing state.", e);
            }
        }
    }

    @Override
    public boolean shouldRun() {
        return true;
    }

    @Override
    public String getServiceName() {
        return RootCARotationManager.class.getSimpleName();
    }

    @Override
    public void start() throws SCMServiceException {
        this.executorService.scheduleAtFixedRate(new MonitorTask(this.scmCertClient, this.scm.getScmStorageConfig()), 0L, this.checkInterval.toMillis(), TimeUnit.MILLISECONDS);
        LOG.info("Monitor task for root certificate {} is started with interval {}.", (Object)this.scmCertClient.getCACertificate().getSerialNumber(), (Object)this.checkInterval);
        this.executorService.scheduleAtFixedRate(this::removeExpiredCertTask, 0L, this.secConf.getExpiredCertificateCheckInterval().toMillis(), TimeUnit.MILLISECONDS);
        LOG.info("Scheduling expired certificate removal with interval {}s", (Object)this.secConf.getExpiredCertificateCheckInterval().getSeconds());
    }

    private void removeExpiredCertTask() {
        if (!this.isRunning.get()) {
            return;
        }
        if (this.scm.getCertificateStore() != null) {
            try {
                this.scm.getCertificateStore().removeAllExpiredCertificates();
            }
            catch (IOException e) {
                LOG.error("Failed to remove some expired certificates", (Throwable)e);
            }
        }
    }

    public boolean isRunning() {
        return this.isRunning.get();
    }

    public void scheduleSubCaRotationPrepareTask(String rootCertId) {
        this.executorService.schedule(new SubCARotationPrepareTask(rootCertId), 0L, TimeUnit.MILLISECONDS);
    }

    public boolean isRotationInProgress() {
        return this.isProcessing.get();
    }

    public boolean isPostRotationInProgress() {
        return this.isPostProcessing.get();
    }

    private void checkInterruptState() {
        if (Thread.currentThread().isInterrupted()) {
            this.cleanupAndStop(this.getClass().getSimpleName() + " is interrupted");
            return;
        }
    }

    private void cleanupAndStop(String reason) {
        try {
            this.scm.getSecurityProtocolServer().setRootCertificateServer(null);
            FileUtils.deleteDirectory((File)new File(this.scmCertClient.getSecurityConfig().getLocation(this.newCAComponent).toString()));
            LOG.info("In-progress root CA directory {} is deleted for '{}'", (Object)this.scmCertClient.getSecurityConfig().getLocation(this.newCAComponent), (Object)reason);
        }
        catch (IOException ex) {
            LOG.error("Error when deleting in-progress root CA directory {} for {}", new Object[]{this.scmCertClient.getSecurityConfig().getLocation(this.newCAComponent), reason, ex});
        }
        this.isProcessing.set(false);
        this.processStartTime.set(null);
    }

    public Duration timeBefore2ExpiryGracePeriod(X509Certificate certificate) {
        LocalDateTime currentTime;
        LocalDateTime gracePeriodStart = certificate.getNotAfter().toInstant().minus(this.renewalGracePeriod).minus(this.renewalGracePeriod).atZone(ZoneId.systemDefault()).toLocalDateTime();
        if (gracePeriodStart.isBefore(currentTime = LocalDateTime.now())) {
            return Duration.ZERO;
        }
        return Duration.between(currentTime, gracePeriodStart);
    }

    private void sendRotationPrepareAck(String newRootCACertId, String newSubCACertId) {
        try {
            this.handler.setSubCACertId(newSubCACertId);
            this.handler.rotationPrepareAck(newRootCACertId, newSubCACertId, this.scm.getScmId());
            LOG.info("SubCARotationPrepareTask[rootCertId = {}] - rotation prepare ack sent out, new scm certificate {}", (Object)newRootCACertId, (Object)newSubCACertId);
        }
        catch (Exception e) {
            LOG.error("Failed to send ack to rotationPrepare request", (Throwable)e);
            String message = "Terminate SCM, encounter exception(" + e.getMessage() + ") when sending out rotationPrepare ack";
            this.scm.shutDown(message);
        }
    }

    private void enterPostProcessing(long delay) {
        this.isPostProcessing.set(true);
        LOG.info("isPostProcessing is true for {} ms", (Object)delay);
        this.clearPostProcessingTask = this.executorService.schedule(() -> {
            this.isPostProcessing.set(false);
            LOG.info("isPostProcessing is false");
            try {
                this.deleteConfiguration();
                LOG.info("Stateful configuration is deleted");
            }
            catch (IOException e) {
                LOG.error("Failed to delete stateful configuration", (Throwable)e);
            }
        }, delay, TimeUnit.MILLISECONDS);
    }

    @Override
    public void stop() {
        if (this.metrics != null) {
            this.metrics.unRegister();
        }
        if (this.executorService != null) {
            this.executorService.shutdownNow();
        }
    }

    @VisibleForTesting
    public void setRootCARotationHandler(RootCARotationHandler newHandler) {
        this.handler = newHandler;
    }

    public boolean shouldSkipRootCert(String newRootCertId) throws IOException {
        List scmCertChain = this.scmCertClient.getTrustChain();
        Preconditions.checkArgument((scmCertChain.size() > 1 ? 1 : 0) != 0);
        X509Certificate rootCert = (X509Certificate)scmCertChain.get(scmCertChain.size() - 1);
        if (rootCert.getSerialNumber().compareTo(new BigInteger(newRootCertId)) >= 0) {
            LOG.info("Sub CA certificate {} is already signed by root certificate {} or a newer root certificate.", (Object)((X509Certificate)scmCertChain.get(0)).getSerialNumber().toString(), (Object)newRootCertId);
            return true;
        }
        return false;
    }

    private void checkAndHandlePostProcessing() throws IOException, CertificateException {
        HddsProtos.CertInfoProto proto = this.readConfiguration(HddsProtos.CertInfoProto.class);
        if (proto == null) {
            LOG.info("No {} configuration found in stateful storage", (Object)this.getServiceName());
            return;
        }
        X509Certificate cert = CertificateCodec.getX509Certificate((String)proto.getX509Certificate());
        List scmCertChain = this.scmCertClient.getTrustChain();
        Preconditions.checkArgument((scmCertChain.size() > 1 ? 1 : 0) != 0);
        X509Certificate rootCert = (X509Certificate)scmCertChain.get(scmCertChain.size() - 1);
        int result = rootCert.getSerialNumber().compareTo(cert.getSerialNumber());
        if (result > 0) {
            LOG.warn("Root CA certificate ID {} in stateful storage is smaller than current scm's root certificate ID {}", (Object)cert.getSerialNumber(), (Object)rootCert.getSerialNumber());
            this.deleteConfiguration();
            LOG.warn("Stateful configuration is deleted");
            return;
        }
        if (result < 0) {
            throw new RuntimeException("Root CA certificate ID " + cert.getSerialNumber() + " in stateful storage is bigger than current scm's root CA certificate ID " + rootCert.getSerialNumber());
        }
        Date issueTime = rootCert.getNotBefore();
        Date now = Calendar.getInstance().getTime();
        Duration gap = Duration.between(issueTime.toInstant(), now.toInstant());
        if ((gap = gap.minus(this.rootCertPollInterval)).isNegative()) {
            long delay = -gap.toMillis();
            this.enterPostProcessing(delay);
        } else {
            LOG.info("Root CA certificate ID {} in stateful storage has already come out of post-processing state", (Object)cert.getSerialNumber());
            this.deleteConfiguration();
        }
    }

    public class MonitorTask
    implements Runnable {
        private SCMCertificateClient certClient;
        private SCMStorageConfig scmStorageConfig;

        public MonitorTask(SCMCertificateClient client, SCMStorageConfig storageConfig) {
            this.certClient = client;
            this.scmStorageConfig = storageConfig;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            Thread.currentThread().setName(RootCARotationManager.this.threadName + (RootCARotationManager.this.isRunning() ? "-Active" : "-Inactive"));
            if (!RootCARotationManager.this.isRunning.get()) {
                return;
            }
            Class<RootCARotationManager> clazz = RootCARotationManager.class;
            synchronized (RootCARotationManager.class) {
                if (RootCARotationManager.this.isProcessing.get()) {
                    LOG.info("Root certificate rotation task is already running.");
                    // ** MonitorExit[var1_1] (shouldn't be in output)
                    return;
                }
                try {
                    X509Certificate rootCACert = this.certClient.getCACertificate();
                    Duration timeLeft = RootCARotationManager.this.timeBefore2ExpiryGracePeriod(rootCACert);
                    if (timeLeft.isZero()) {
                        LOG.info("Root certificate {} has entered the 2 * expiry grace period({}).", (Object)rootCACert.getSerialNumber().toString(), (Object)RootCARotationManager.this.renewalGracePeriod);
                        LocalDateTime now = LocalDateTime.now();
                        LocalDateTime timeToSchedule = LocalDateTime.of(now.getYear(), now.getMonthValue(), now.getDayOfMonth(), RootCARotationManager.this.timeOfDay.getHours(), RootCARotationManager.this.timeOfDay.getMinutes(), RootCARotationManager.this.timeOfDay.getSeconds());
                        if (timeToSchedule.isBefore(now)) {
                            timeToSchedule = timeToSchedule.plusDays(1L);
                        }
                        long delay = Duration.between(now, timeToSchedule).toMillis();
                        if (timeToSchedule.isAfter(rootCACert.getNotAfter().toInstant().atZone(ZoneId.systemDefault()).toLocalDateTime())) {
                            LOG.info("Configured rotation time {} is after root certificate {} end time {}. Start the rotation immediately.", new Object[]{timeToSchedule, rootCACert.getSerialNumber().toString(), rootCACert.getNotAfter()});
                            delay = 0L;
                        }
                        RootCARotationManager.this.rotationTask = RootCARotationManager.this.executorService.schedule(new RotationTask(this.certClient, this.scmStorageConfig), delay, TimeUnit.MILLISECONDS);
                        RootCARotationManager.this.isProcessing.set(true);
                        RootCARotationManager.this.metrics.incrTotalRotationNum();
                        LOG.info("Root certificate {} rotation task is scheduled with {} ms delay", (Object)rootCACert.getSerialNumber().toString(), (Object)delay);
                    }
                }
                catch (Throwable e) {
                    LOG.error("Error while scheduling root CA rotation task", e);
                    RootCARotationManager.this.scm.shutDown("Error while scheduling root CA rotation task");
                }
                return;
            }
        }
    }

    public class SubCARotationPrepareTask
    implements Runnable {
        private String rootCACertId;

        public SubCARotationPrepareTask(String newRootCertId) {
            this.rootCACertId = newRootCertId;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            Class<RootCARotationManager> clazz = RootCARotationManager.class;
            synchronized (RootCARotationManager.class) {
                try {
                    String message;
                    LOG.info("SubCARotationPrepareTask[rootCertId = {}] - started.", (Object)this.rootCACertId);
                    if (RootCARotationManager.this.shouldSkipRootCert(this.rootCACertId)) {
                        RootCARotationManager.this.sendRotationPrepareAck(this.rootCACertId, RootCARotationManager.this.scmCertClient.getCertificate().getSerialNumber().toString());
                        // ** MonitorExit[var1_1] (shouldn't be in output)
                        return;
                    }
                    SecurityConfig securityConfig = RootCARotationManager.this.scmCertClient.getSecurityConfig();
                    String progressComponent = SCMCertificateClient.COMPONENT_NAME + "-next" + "-progress";
                    String newSubCAProgressPath = securityConfig.getLocation(progressComponent).toString();
                    String newSubCAPath = securityConfig.getLocation(SCMCertificateClient.COMPONENT_NAME + "-next").toString();
                    File newProgressDir = new File(newSubCAProgressPath);
                    File newDir = new File(newSubCAPath);
                    try {
                        FileUtils.deleteDirectory((File)newProgressDir);
                        FileUtils.deleteDirectory((File)newDir);
                        Files.createDirectories(newProgressDir.toPath(), new FileAttribute[0]);
                    }
                    catch (IOException e) {
                        LOG.error("Failed to delete and create {}, or delete {}", new Object[]{newProgressDir, newDir, e});
                        String message2 = "Terminate SCM, encounter IO exception(" + e.getMessage() + ") when deleting and create directory";
                        RootCARotationManager.this.scm.shutDown(message2);
                    }
                    KeyStorage keyStorage = new KeyStorage(securityConfig, progressComponent);
                    KeyPair newKeyPair = null;
                    try {
                        HDDSKeyGenerator keyGenerator = new HDDSKeyGenerator(securityConfig);
                        newKeyPair = keyGenerator.generateKey();
                        keyStorage.storeKeyPair(newKeyPair);
                        LOG.info("SubCARotationPrepareTask[rootCertId = {}] - scm key generated.", (Object)this.rootCACertId);
                    }
                    catch (Exception e) {
                        LOG.error("Failed to generate key under {}", (Object)newProgressDir, (Object)e);
                        String message3 = "Terminate SCM, encounter exception(" + e.getMessage() + ") when generating new key under " + newProgressDir;
                        RootCARotationManager.this.scm.shutDown(message3);
                    }
                    RootCARotationManager.this.checkInterruptState();
                    String newCertSerialId = "";
                    try {
                        CertificateSignRequest.Builder csrBuilder = RootCARotationManager.this.scmCertClient.configureCSRBuilder();
                        csrBuilder.setKey(newKeyPair);
                        newCertSerialId = RootCARotationManager.this.scmCertClient.signAndStoreCertificate(csrBuilder.build(), Paths.get(newSubCAProgressPath, "certs"), true);
                        LOG.info("SubCARotationPrepareTask[rootCertId = {}] - scm certificate {} signed.", (Object)this.rootCACertId, (Object)newCertSerialId);
                    }
                    catch (Exception e) {
                        LOG.error("Failed to generate certificate under {}", (Object)newProgressDir, (Object)e);
                        message = "Terminate SCM, encounter exception(" + e.getMessage() + ") when generating new certificate " + newProgressDir;
                        RootCARotationManager.this.scm.shutDown(message);
                    }
                    try {
                        Files.move(newProgressDir.toPath(), newDir.toPath(), StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING);
                    }
                    catch (IOException e) {
                        LOG.error("Failed to move {} to {}", new Object[]{newSubCAProgressPath, newSubCAPath, e});
                        message = "Terminate SCM, encounter exception(" + e.getMessage() + ") when moving " + newSubCAProgressPath + " to " + newSubCAPath;
                        RootCARotationManager.this.scm.shutDown(message);
                    }
                    RootCARotationManager.this.checkInterruptState();
                    RootCARotationManager.this.sendRotationPrepareAck(this.rootCACertId, newCertSerialId);
                }
                catch (Throwable e) {
                    LOG.error("Unexpected error happen", e);
                    RootCARotationManager.this.scm.shutDown("Unexpected error happen, " + e.getMessage());
                }
                return;
            }
        }
    }

    public class WaitSubCARotationPrepareAckTask
    implements Runnable {
        private String rootCACertId;
        private X509Certificate rootCACertificate;

        public WaitSubCARotationPrepareAckTask(X509Certificate rootCACertificate) {
            this.rootCACertificate = rootCACertificate;
            this.rootCACertId = rootCACertificate.getSerialNumber().toString();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            RootCARotationManager.this.checkInterruptState();
            if (!RootCARotationManager.this.isRunning()) {
                LOG.info("SCM is not leader anymore. Delete the in-progress root CA directory");
                RootCARotationManager.this.cleanupAndStop("SCM is not leader anymore");
                return;
            }
            Class<RootCARotationManager> clazz = RootCARotationManager.class;
            synchronized (RootCARotationManager.class) {
                int numFromHADetails = RootCARotationManager.this.scm.getSCMHANodeDetails().getPeerNodeDetails().size() + 1;
                int numFromRatisServer = RootCARotationManager.this.scm.getScmHAManager().getRatisServer().getDivision().getRaftConf().getCurrentPeers().size();
                LOG.info("numFromHADetails {}, numFromRatisServer {}", (Object)numFromHADetails, (Object)numFromRatisServer);
                if (RootCARotationManager.this.handler.rotationPrepareAcks() == numFromRatisServer) {
                    try {
                        RootCARotationManager.this.waitAckTimeoutTask.cancel(true);
                        RootCARotationManager.this.handler.rotationCommit(this.rootCACertId);
                        RootCARotationManager.this.handler.rotationCommitted(this.rootCACertId);
                        RootCARotationManager.this.metrics.incrSuccessRotationNum();
                        long timeTaken = System.nanoTime() - (Long)RootCARotationManager.this.processStartTime.get();
                        RootCARotationManager.this.metrics.setSuccessTimeInNs(timeTaken);
                        RootCARotationManager.this.processStartTime.set(null);
                        try {
                            if (RootCARotationManager.this.scm.getCertificateStore().getCertificateByID(this.rootCACertificate.getSerialNumber()) == null) {
                                LOG.info("Persist root certificate {} to cert store", (Object)this.rootCACertId);
                                RootCARotationManager.this.scm.getCertificateStore().storeValidCertificate(this.rootCACertificate.getSerialNumber(), this.rootCACertificate, HddsProtos.NodeType.SCM);
                            }
                        }
                        catch (IOException e) {
                            LOG.error("Failed to save root certificate {} to cert store", (Object)this.rootCACertId);
                            RootCARotationManager.this.scm.shutDown("Failed to save root certificate to cert store");
                        }
                        RootCARotationManager.this.handler.resetRotationPrepareAcks();
                        String msg = "Root certificate " + this.rootCACertId + " rotation is finished successfully after " + timeTaken + " ns";
                        RootCARotationManager.this.cleanupAndStop(msg);
                        RootCARotationManager.this.enterPostProcessing(RootCARotationManager.this.rootCertPollInterval.toMillis());
                        RootCARotationManager.this.saveConfiguration((Message)new CertInfo.Builder().setX509Certificate(this.rootCACertificate).setTimestamp(this.rootCACertificate.getNotBefore().getTime()).build().getProtobuf());
                    }
                    catch (Throwable e) {
                        LOG.error("Execution error", e);
                        RootCARotationManager.this.handler.resetRotationPrepareAcks();
                        RootCARotationManager.this.cleanupAndStop("Execution error, " + e.getMessage());
                    }
                    finally {
                        RootCARotationManager.this.waitAckTask.cancel(true);
                    }
                }
                // ** MonitorExit[var1_1] (shouldn't be in output)
                return;
            }
        }
    }

    public class RotationTask
    implements Runnable {
        private SCMCertificateClient certClient;
        private SCMStorageConfig scmStorageConfig;

        public RotationTask(SCMCertificateClient client, SCMStorageConfig storageConfig) {
            this.certClient = client;
            this.scmStorageConfig = storageConfig;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            if (!RootCARotationManager.this.isRunning.get()) {
                RootCARotationManager.this.isProcessing.set(false);
                RootCARotationManager.this.processStartTime.set(null);
                return;
            }
            Class<RootCARotationManager> clazz = RootCARotationManager.class;
            synchronized (RootCARotationManager.class) {
                X509Certificate rootCACert = this.certClient.getCACertificate();
                Duration timeLeft = RootCARotationManager.this.timeBefore2ExpiryGracePeriod(rootCACert);
                if (timeLeft.isZero()) {
                    X509Certificate newRootCertificate;
                    LOG.info("Root certificate {} rotation is started.", (Object)rootCACert.getSerialNumber().toString());
                    RootCARotationManager.this.processStartTime.set(System.nanoTime());
                    CertificateServer newRootCAServer = null;
                    BigInteger newId = BigInteger.ONE;
                    try {
                        newId = new BigInteger(String.valueOf(RootCARotationManager.this.sequenceIdGen.getNextId("CertificateId")));
                        newRootCAServer = HASecurityUtils.initializeRootCertificateServer(RootCARotationManager.this.secConf, RootCARotationManager.this.scm.getCertificateStore(), this.scmStorageConfig, newId, (PKIProfile)new DefaultCAProfile(), RootCARotationManager.this.newCAComponent);
                    }
                    catch (Throwable e) {
                        LOG.error("Error while generating new root CA certificate under {}", (Object)RootCARotationManager.this.newCAComponent, (Object)e);
                        String message = "Terminate SCM, encounter exception(" + e.getMessage() + ") when generating new root CA certificate under " + RootCARotationManager.this.newCAComponent;
                        RootCARotationManager.this.cleanupAndStop(message);
                        RootCARotationManager.this.scm.shutDown(message);
                    }
                    String newRootCertId = "";
                    try {
                        if (newRootCAServer == null) {
                            throw new Exception("New root CA server should not be null");
                        }
                        newRootCertificate = newRootCAServer.getCACertificate();
                        newRootCertId = newRootCertificate.getSerialNumber().toString();
                        Preconditions.checkState((boolean)newRootCertId.equals(newId.toString()), (Object)("Root certificate doesn't match, expected:" + newId + ", fetched:" + newRootCertId));
                        RootCARotationManager.this.scm.getSecurityProtocolServer().setRootCertificateServer(newRootCAServer);
                        if (!RootCARotationManager.this.isRunning()) {
                            LOG.info("SCM is not leader anymore. Delete the in-progress root CA directory");
                            RootCARotationManager.this.cleanupAndStop("SCM is not leader anymore");
                            // ** MonitorExit[var1_1] (shouldn't be in output)
                            return;
                        }
                        RootCARotationManager.this.checkInterruptState();
                        RootCARotationManager.this.handler.rotationPrepare(newRootCertId);
                        LOG.info("Send out sub CA rotation prepare request for new root certificate {}", (Object)newRootCertId);
                    }
                    catch (Exception e) {
                        LOG.error("Error while sending rotation prepare request", (Throwable)e);
                        RootCARotationManager.this.cleanupAndStop("Error while sending rotation prepare request");
                        // ** MonitorExit[var1_1] (shouldn't be in output)
                        return;
                    }
                    RootCARotationManager.this.waitAckTask = RootCARotationManager.this.executorService.scheduleAtFixedRate(new WaitSubCARotationPrepareAckTask(newRootCertificate), 1L, 1L, TimeUnit.SECONDS);
                    RootCARotationManager.this.waitAckTimeoutTask = RootCARotationManager.this.executorService.schedule(() -> {
                        RootCARotationManager.this.waitAckTask.cancel(true);
                        String msg = "Failed to receive all acks of rotation prepare after " + RootCARotationManager.this.ackTimeout + ", received " + RootCARotationManager.this.handler.rotationPrepareAcks() + " acks";
                        RootCARotationManager.this.cleanupAndStop(msg);
                    }, RootCARotationManager.this.ackTimeout.toMillis(), TimeUnit.MILLISECONDS);
                } else {
                    LOG.warn("Root certificate {} hasn't entered the 2 * expiry grace period {}. Skip root certificate rotation this time.", (Object)rootCACert.getSerialNumber().toString(), (Object)RootCARotationManager.this.renewalGracePeriod);
                    RootCARotationManager.this.isProcessing.set(false);
                    RootCARotationManager.this.processStartTime.set(null);
                }
                // ** MonitorExit[var1_1] (shouldn't be in output)
                return;
            }
        }
    }
}

