/*
 * Decompiled with CFR 0.152.
 */
package org.apache.arrow.memory;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.Map;
import java.util.Set;
import org.apache.arrow.memory.Accountant;
import org.apache.arrow.memory.AllocationListener;
import org.apache.arrow.memory.AllocationManager;
import org.apache.arrow.memory.AllocationOutcome;
import org.apache.arrow.memory.AllocationReservation;
import org.apache.arrow.memory.ArrowBuf;
import org.apache.arrow.memory.BufferAllocator;
import org.apache.arrow.memory.BufferLedger;
import org.apache.arrow.memory.BufferManager;
import org.apache.arrow.memory.ChildAllocator;
import org.apache.arrow.memory.DefaultAllocationManagerOption;
import org.apache.arrow.memory.ForeignAllocation;
import org.apache.arrow.memory.ForeignAllocationManager;
import org.apache.arrow.memory.ImmutableConfig;
import org.apache.arrow.memory.OutOfMemoryException;
import org.apache.arrow.memory.RootAllocator;
import org.apache.arrow.memory.rounding.DefaultRoundingPolicy;
import org.apache.arrow.memory.rounding.RoundingPolicy;
import org.apache.arrow.memory.util.AssertionUtil;
import org.apache.arrow.memory.util.CommonUtil;
import org.apache.arrow.memory.util.HistoricalLog;
import org.apache.arrow.util.Preconditions;
import org.checkerframework.checker.nullness.qual.KeyFor;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.immutables.value.Value;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

abstract class BaseAllocator
extends Accountant
implements BufferAllocator {
    public static final String DEBUG_ALLOCATOR = "arrow.memory.debug.allocator";
    public static final int DEBUG_LOG_LENGTH = 6;
    public static final boolean DEBUG;
    private static final Logger logger;
    public static final Config DEFAULT_CONFIG;
    private final String name;
    private final RootAllocator root;
    private final Object DEBUG_LOCK = new Object();
    private final AllocationListener listener;
    private final @Nullable BaseAllocator parentAllocator;
    private final Map<BaseAllocator, Object> childAllocators;
    private final ArrowBuf empty;
    private final @Nullable IdentityHashMap<BufferLedger, @Nullable Object> childLedgers;
    private final @Nullable IdentityHashMap<Reservation, Object> reservations;
    private final @Nullable HistoricalLog historicalLog;
    private final RoundingPolicy roundingPolicy;
    private final @NonNull AllocationManager.Factory allocationManagerFactory;
    private volatile boolean isClosed = false;

    protected BaseAllocator(@Nullable BaseAllocator parentAllocator, String name, Config config) throws OutOfMemoryException {
        super(parentAllocator, name, config.getInitReservation(), config.getMaxAllocation());
        this.listener = config.getListener();
        this.allocationManagerFactory = config.getAllocationManagerFactory();
        if (parentAllocator != null) {
            this.root = parentAllocator.root;
            this.empty = parentAllocator.empty;
        } else if (this instanceof RootAllocator) {
            this.root = (RootAllocator)this;
            this.empty = this.createEmpty();
        } else {
            throw new IllegalStateException("An parent allocator must either carry a root or be the root.");
        }
        this.parentAllocator = parentAllocator;
        this.name = name;
        this.childAllocators = Collections.synchronizedMap(new IdentityHashMap());
        if (DEBUG) {
            this.reservations = new IdentityHashMap();
            this.childLedgers = new IdentityHashMap();
            this.historicalLog = new HistoricalLog(6, "allocator[%s]", name);
            this.hist("created by \"%s\", owned = %d", name, this.getAllocatedMemory());
        } else {
            this.reservations = null;
            this.historicalLog = null;
            this.childLedgers = null;
        }
        this.roundingPolicy = config.getRoundingPolicy();
    }

    @Override
    public AllocationListener getListener() {
        return this.listener;
    }

    @Override
    public @Nullable BaseAllocator getParentAllocator() {
        return this.parentAllocator;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Collection<BufferAllocator> getChildAllocators() {
        Map<BaseAllocator, Object> map = this.childAllocators;
        synchronized (map) {
            return new HashSet<BufferAllocator>(this.childAllocators.keySet());
        }
    }

    private static String createErrorMsg(BufferAllocator allocator, long rounded, long requested) {
        if (rounded != requested) {
            return String.format("Unable to allocate buffer of size %d (rounded from %d) due to memory limit. Current allocation: %d", rounded, requested, allocator.getAllocatedMemory());
        }
        return String.format("Unable to allocate buffer of size %d due to memory limit. Current allocation: %d", rounded, allocator.getAllocatedMemory());
    }

    public static boolean isDebug() {
        return DEBUG;
    }

    @Override
    public void assertOpen() {
        if (AssertionUtil.ASSERT_ENABLED && this.isClosed) {
            throw new IllegalStateException("Attempting operation on allocator when allocator is closed.\n" + this.toVerboseString());
        }
    }

    @Override
    public String getName() {
        return this.name;
    }

    @Override
    public ArrowBuf getEmpty() {
        return this.empty;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void associateLedger(BufferLedger ledger) {
        this.assertOpen();
        if (DEBUG) {
            Object object = this.DEBUG_LOCK;
            synchronized (object) {
                if (this.childLedgers != null) {
                    this.childLedgers.put(ledger, null);
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void dissociateLedger(BufferLedger ledger) {
        this.assertOpen();
        if (DEBUG) {
            Object object = this.DEBUG_LOCK;
            synchronized (object) {
                Preconditions.checkState(this.childLedgers != null, "childLedgers must not be null");
                if (!this.childLedgers.containsKey(ledger)) {
                    throw new IllegalStateException("Trying to remove a child ledger that doesn't exist.");
                }
                this.childLedgers.remove(ledger);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void childClosed(BaseAllocator childAllocator) {
        this.assertOpen();
        if (DEBUG) {
            Preconditions.checkArgument(childAllocator != null, "child allocator can't be null");
            Object object = this.DEBUG_LOCK;
            synchronized (object) {
                Object object2 = this.childAllocators.remove(childAllocator);
                if (object2 == null) {
                    if (childAllocator.historicalLog != null) {
                        childAllocator.historicalLog.logHistory(logger);
                    }
                    throw new IllegalStateException("Child allocator[" + childAllocator.name + "] not found in parent allocator[" + this.name + "]'s childAllocators");
                }
            }
        } else {
            this.childAllocators.remove(childAllocator);
        }
        this.listener.onChildRemoved(this, childAllocator);
    }

    @Override
    public ArrowBuf wrapForeignAllocation(ForeignAllocation allocation) {
        this.assertOpen();
        long size = allocation.getSize();
        this.listener.onPreAllocation(size);
        AllocationOutcome outcome = this.allocateBytes(size);
        if (!outcome.isOk()) {
            if (this.listener.onFailedAllocation(size, outcome)) {
                outcome = this.allocateBytes(size);
            }
            if (!outcome.isOk()) {
                throw new OutOfMemoryException(BaseAllocator.createErrorMsg(this, size, size), outcome.getDetails());
            }
        }
        try {
            ForeignAllocationManager manager = new ForeignAllocationManager(this, allocation);
            BufferLedger ledger = manager.associate(this);
            ArrowBuf buf = new ArrowBuf(ledger, null, size, allocation.memoryAddress());
            buf.writerIndex(size);
            this.listener.onAllocation(size);
            return buf;
        }
        catch (Throwable t) {
            try {
                this.releaseBytes(size);
            }
            catch (Throwable e) {
                t.addSuppressed(e);
            }
            try {
                allocation.release0();
            }
            catch (Throwable e) {
                t.addSuppressed(e);
            }
            throw t;
        }
    }

    @Override
    public ArrowBuf buffer(long initialRequestSize) {
        this.assertOpen();
        return this.buffer(initialRequestSize, null);
    }

    private ArrowBuf createEmpty() {
        return this.allocationManagerFactory.empty();
    }

    @Override
    public ArrowBuf buffer(long initialRequestSize, @Nullable BufferManager manager) {
        this.assertOpen();
        Preconditions.checkArgument(initialRequestSize >= 0L, "the requested size must be non-negative");
        if (initialRequestSize == 0L) {
            return this.getEmpty();
        }
        long actualRequestSize = this.roundingPolicy.getRoundedSize(initialRequestSize);
        this.listener.onPreAllocation(actualRequestSize);
        AllocationOutcome outcome = this.allocateBytes(actualRequestSize);
        if (!outcome.isOk()) {
            if (this.listener.onFailedAllocation(actualRequestSize, outcome)) {
                outcome = this.allocateBytes(actualRequestSize);
            }
            if (!outcome.isOk()) {
                throw new OutOfMemoryException(BaseAllocator.createErrorMsg(this, actualRequestSize, initialRequestSize), outcome.getDetails());
            }
        }
        boolean success = false;
        try {
            ArrowBuf buffer = this.bufferWithoutReservation(actualRequestSize, manager);
            success = true;
            this.listener.onAllocation(actualRequestSize);
            ArrowBuf arrowBuf = buffer;
            return arrowBuf;
        }
        catch (OutOfMemoryError e) {
            throw e;
        }
        finally {
            if (!success) {
                this.releaseBytes(actualRequestSize);
            }
        }
    }

    private ArrowBuf bufferWithoutReservation(long size, @Nullable BufferManager bufferManager) throws OutOfMemoryException {
        this.assertOpen();
        AllocationManager manager = this.newAllocationManager(size);
        BufferLedger ledger = manager.associate(this);
        ArrowBuf buffer = ledger.newArrowBuf(size, bufferManager);
        Preconditions.checkArgument(buffer.capacity() == size, "Allocated capacity %d was not equal to requested capacity %d.", buffer.capacity(), size);
        return buffer;
    }

    private AllocationManager newAllocationManager(long size) {
        return this.newAllocationManager(this, size);
    }

    private AllocationManager newAllocationManager(BaseAllocator accountingAllocator, long size) {
        return this.allocationManagerFactory.create(accountingAllocator, size);
    }

    @Override
    public BufferAllocator getRoot() {
        return this.root;
    }

    @Override
    public BufferAllocator newChildAllocator(String name, long initReservation, long maxAllocation) {
        return this.newChildAllocator(name, this.listener, initReservation, maxAllocation);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public BufferAllocator newChildAllocator(String name, AllocationListener listener, long initReservation, long maxAllocation) {
        this.assertOpen();
        ChildAllocator childAllocator = new ChildAllocator(this, name, BaseAllocator.configBuilder().listener(listener).initReservation(initReservation).maxAllocation(maxAllocation).roundingPolicy(this.roundingPolicy).allocationManagerFactory(this.allocationManagerFactory).build());
        if (DEBUG) {
            Object object = this.DEBUG_LOCK;
            synchronized (object) {
                this.childAllocators.put(childAllocator, childAllocator);
                if (this.historicalLog != null) {
                    this.historicalLog.recordEvent("allocator[%s] created new child allocator[%s]", name, childAllocator.getName());
                }
            }
        } else {
            this.childAllocators.put(childAllocator, childAllocator);
        }
        this.listener.onChildAdded(this, childAllocator);
        return childAllocator;
    }

    @Override
    public AllocationReservation newReservation() {
        this.assertOpen();
        return new Reservation();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public synchronized void close() {
        long allocated;
        Map<BaseAllocator, Object> map;
        if (this.isClosed) {
            return;
        }
        this.isClosed = true;
        StringBuilder outstandingChildAllocators = new StringBuilder();
        if (DEBUG) {
            map = this.DEBUG_LOCK;
            synchronized (map) {
                int allocatedCount;
                this.verifyAllocator();
                if (!this.childAllocators.isEmpty()) {
                    for (BaseAllocator childAllocator : this.childAllocators.keySet()) {
                        if (!childAllocator.isClosed) continue;
                        logger.warn(String.format("Closed child allocator[%s] on parent allocator[%s]'s child list.\n%s", childAllocator.name, this.name, this.toString()));
                    }
                    throw new IllegalStateException(String.format("Allocator[%s] closed with outstanding child allocators.\n%s", this.name, this.toString()));
                }
                int n = allocatedCount = this.childLedgers != null ? this.childLedgers.size() : 0;
                if (allocatedCount > 0) {
                    throw new IllegalStateException(String.format("Allocator[%s] closed with outstanding buffers allocated (%d).\n%s", this.name, allocatedCount, this.toString()));
                }
                if (this.reservations != null && this.reservations.size() != 0) {
                    throw new IllegalStateException(String.format("Allocator[%s] closed with outstanding reservations (%d).\n%s", this.name, this.reservations.size(), this.toString()));
                }
            }
        }
        if (!this.childAllocators.isEmpty()) {
            outstandingChildAllocators.append("Outstanding child allocators : \n");
            map = this.childAllocators;
            synchronized (map) {
                for (BaseAllocator childAllocator : this.childAllocators.keySet()) {
                    outstandingChildAllocators.append(String.format("  %s", childAllocator.toString()));
                }
            }
        }
        if ((allocated = this.getAllocatedMemory()) > 0L) {
            if (this.parent != null && this.reservation > allocated) {
                this.parent.releaseBytes(this.reservation - allocated);
            }
            String msg = String.format("Memory was leaked by query. Memory leaked: (%d)\n%s%s", allocated, outstandingChildAllocators.toString(), this.toString());
            logger.error(msg);
            throw new IllegalStateException(msg);
        }
        super.close();
        if (this.parentAllocator != null) {
            this.parentAllocator.childClosed(this);
        }
        if (DEBUG) {
            if (this.historicalLog != null) {
                this.historicalLog.recordEvent("closed", new Object[0]);
            }
            logger.debug(String.format("closed allocator[%s].", this.name));
        }
    }

    public String toString() {
        Verbosity verbosity = logger.isTraceEnabled() ? Verbosity.LOG_WITH_STACKTRACE : Verbosity.BASIC;
        StringBuilder sb = new StringBuilder();
        this.print(sb, 0, verbosity);
        return sb.toString();
    }

    @Override
    public String toVerboseString() {
        StringBuilder sb = new StringBuilder();
        this.print(sb, 0, Verbosity.LOG_WITH_STACKTRACE);
        return sb.toString();
    }

    private void hist(String noteFormat, Object ... args) {
        if (this.historicalLog != null) {
            this.historicalLog.recordEvent(noteFormat, args);
        }
    }

    void verifyAllocator() {
        IdentityHashMap<AllocationManager, BaseAllocator> seen = new IdentityHashMap<AllocationManager, BaseAllocator>();
        this.verifyAllocator(seen);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void verifyAllocator(IdentityHashMap<AllocationManager, BaseAllocator> buffersSeen) {
        if (!DEBUG) {
            return;
        }
        Object object = this.DEBUG_LOCK;
        synchronized (object) {
            Set<BufferLedger> ledgerSet;
            long allocated = this.getAllocatedMemory();
            Set<BaseAllocator> childSet = this.childAllocators.keySet();
            for (BaseAllocator childAllocator : childSet) {
                childAllocator.verifyAllocator(buffersSeen);
            }
            long childTotal = 0L;
            for (BaseAllocator childAllocator : childSet) {
                childTotal += Math.max(childAllocator.getAllocatedMemory(), childAllocator.reservation);
            }
            if (childTotal > this.getAllocatedMemory()) {
                if (this.historicalLog != null) {
                    this.historicalLog.logHistory(logger);
                }
                logger.debug("allocator[" + this.name + "] child event logs BEGIN");
                for (BaseAllocator childAllocator : childSet) {
                    if (childAllocator.historicalLog == null) continue;
                    childAllocator.historicalLog.logHistory(logger);
                }
                logger.debug("allocator[" + this.name + "] child event logs END");
                throw new IllegalStateException("Child allocators own more memory (" + childTotal + ") than their parent (name = " + this.name + " ) has allocated (" + this.getAllocatedMemory() + ')');
            }
            long bufferTotal = 0L;
            Set<BufferLedger> set = ledgerSet = this.childLedgers != null ? this.childLedgers.keySet() : null;
            if (ledgerSet != null) {
                for (BufferLedger ledger : ledgerSet) {
                    if (!ledger.isOwningLedger()) continue;
                    AllocationManager am = ledger.getAllocationManager();
                    BaseAllocator otherOwner = buffersSeen.get(am);
                    if (otherOwner != null) {
                        throw new IllegalStateException("This allocator's ArrowBuf already owned by another allocator");
                    }
                    buffersSeen.put(am, this);
                    bufferTotal += am.getSize();
                }
            }
            Set<Reservation> reservationSet = this.reservations != null ? this.reservations.keySet() : null;
            long reservedTotal = 0L;
            if (reservationSet != null) {
                for (Reservation reservation : reservationSet) {
                    if (reservation.isUsed()) continue;
                    reservedTotal += (long)reservation.getSize();
                }
            }
            if (bufferTotal + reservedTotal + childTotal != this.getAllocatedMemory()) {
                StringBuilder sb = new StringBuilder();
                sb.append("allocator[");
                sb.append(this.name);
                sb.append("]\nallocated: ");
                sb.append(Long.toString(allocated));
                sb.append(" allocated - (bufferTotal + reservedTotal + childTotal): ");
                sb.append(Long.toString(allocated - (bufferTotal + reservedTotal + childTotal)));
                sb.append('\n');
                if (bufferTotal != 0L) {
                    sb.append("buffer total: ");
                    sb.append(Long.toString(bufferTotal));
                    sb.append('\n');
                    this.dumpBuffers(sb, ledgerSet);
                }
                if (childTotal != 0L) {
                    sb.append("child total: ");
                    sb.append(Long.toString(childTotal));
                    sb.append('\n');
                    for (BaseAllocator childAllocator : childSet) {
                        sb.append("child allocator[");
                        sb.append(childAllocator.name);
                        sb.append("] owned ");
                        sb.append(Long.toString(childAllocator.getAllocatedMemory()));
                        sb.append('\n');
                    }
                }
                if (reservedTotal != 0L) {
                    sb.append(String.format("reserved total : %d bytes.", reservedTotal));
                    if (reservationSet != null) {
                        for (Reservation reservation : reservationSet) {
                            if (reservation.historicalLog != null) {
                                reservation.historicalLog.buildHistory(sb, 0, true);
                            }
                            sb.append('\n');
                        }
                    }
                }
                logger.debug(sb.toString());
                long l = this.getAllocatedMemory();
                if (l != allocated) {
                    throw new IllegalStateException(String.format("allocator[%s]: allocated t1 (%d) + allocated t2 (%d). Someone released memory while in verification.", this.name, allocated, l));
                }
                throw new IllegalStateException(String.format("allocator[%s]: buffer space (%d) + prealloc space (%d) + child space (%d) != allocated (%d)", this.name, bufferTotal, reservedTotal, childTotal, allocated));
            }
        }
    }

    void print(StringBuilder sb, int level, Verbosity verbosity) {
        CommonUtil.indent(sb, level).append("Allocator(").append(this.name).append(") ").append(this.reservation).append('/').append(this.getAllocatedMemory()).append('/').append(this.getPeakMemoryAllocation()).append('/').append(this.getLimit()).append(" (res/actual/peak/limit)").append('\n');
        if (DEBUG) {
            CommonUtil.indent(sb, level + 1).append(String.format("child allocators: %d\n", this.childAllocators.size()));
            for (BaseAllocator child : this.childAllocators.keySet()) {
                child.print(sb, level + 2, verbosity);
            }
            CommonUtil.indent(sb, level + 1).append(String.format("ledgers: %d\n", this.childLedgers != null ? this.childLedgers.size() : 0));
            if (this.childLedgers != null) {
                for (BufferLedger ledger : this.childLedgers.keySet()) {
                    ledger.print(sb, level + 2, verbosity);
                }
            }
            Set<Reservation> reservations = this.reservations != null ? this.reservations.keySet() : null;
            CommonUtil.indent(sb, level + 1).append(String.format("reservations: %d\n", reservations != null ? reservations.size() : 0));
            if (reservations != null) {
                for (Reservation reservation : reservations) {
                    if (!verbosity.includeHistoricalLog || reservation.historicalLog == null) continue;
                    reservation.historicalLog.buildHistory(sb, level + 3, true);
                }
            }
        }
    }

    private void dumpBuffers(StringBuilder sb, @Nullable Set<@KeyFor(value={"this.childLedgers"}) BufferLedger> ledgerSet) {
        if (ledgerSet != null) {
            for (BufferLedger ledger : ledgerSet) {
                if (!ledger.isOwningLedger()) continue;
                AllocationManager am = ledger.getAllocationManager();
                sb.append("UnsafeDirectLittleEndian[identityHashCode == ");
                sb.append(Integer.toString(System.identityHashCode(am)));
                sb.append("] size ");
                sb.append(Long.toString(am.getSize()));
                sb.append('\n');
            }
        }
    }

    public static Config defaultConfig() {
        return DEFAULT_CONFIG;
    }

    public static ImmutableConfig.Builder configBuilder() {
        return ImmutableConfig.builder();
    }

    @Override
    public RoundingPolicy getRoundingPolicy() {
        return this.roundingPolicy;
    }

    static {
        logger = LoggerFactory.getLogger(BaseAllocator.class);
        String propValue = System.getProperty(DEBUG_ALLOCATOR);
        DEBUG = propValue != null ? Boolean.parseBoolean(propValue) : false;
        logger.info("Debug mode " + (DEBUG ? "enabled." : "disabled. Enable with the VM option -Darrow.memory.debug.allocator=true."));
        DEFAULT_CONFIG = ImmutableConfig.builder().build();
    }

    public class Reservation
    implements AllocationReservation {
        private final @Nullable HistoricalLog historicalLog;
        private int nBytes = 0;
        private boolean used = false;
        private boolean closed = false;

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public Reservation() {
            if (DEBUG) {
                this.historicalLog = new HistoricalLog("Reservation[allocator[%s], %d]", BaseAllocator.this.name, System.identityHashCode(this));
                this.historicalLog.recordEvent("created", new Object[0]);
                Object object = BaseAllocator.this.DEBUG_LOCK;
                synchronized (object) {
                    if (BaseAllocator.this.reservations != null) {
                        BaseAllocator.this.reservations.put(this, this);
                    }
                }
            } else {
                this.historicalLog = null;
            }
        }

        @Override
        public boolean add(int nBytes) {
            BaseAllocator.this.assertOpen();
            Preconditions.checkArgument(nBytes >= 0, "nBytes(%d) < 0", nBytes);
            Preconditions.checkState(!this.closed, "Attempt to increase reservation after reservation has been closed");
            Preconditions.checkState(!this.used, "Attempt to increase reservation after reservation has been used");
            int nBytesTwo = CommonUtil.nextPowerOfTwo(nBytes);
            if (!this.reserve(nBytesTwo)) {
                return false;
            }
            this.nBytes += nBytesTwo;
            return true;
        }

        @Override
        public ArrowBuf allocateBuffer() {
            BaseAllocator.this.assertOpen();
            Preconditions.checkState(!this.closed, "Attempt to allocate after closed");
            Preconditions.checkState(!this.used, "Attempt to allocate more than once");
            ArrowBuf arrowBuf = this.allocate(this.nBytes);
            this.used = true;
            return arrowBuf;
        }

        @Override
        public int getSize() {
            return this.nBytes;
        }

        @Override
        public boolean isUsed() {
            return this.used;
        }

        @Override
        public boolean isClosed() {
            return this.closed;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void close() {
            BaseAllocator.this.assertOpen();
            if (this.closed) {
                return;
            }
            if (DEBUG && !this.isClosed()) {
                Object object;
                Object object2 = BaseAllocator.this.DEBUG_LOCK;
                synchronized (object2) {
                    object = BaseAllocator.this.reservations != null ? BaseAllocator.this.reservations.remove(this) : null;
                }
                if (object == null) {
                    StringBuilder sb = new StringBuilder();
                    BaseAllocator.this.print(sb, 0, Verbosity.LOG_WITH_STACKTRACE);
                    logger.debug(sb.toString());
                    throw new IllegalStateException(String.format("Didn't find closing reservation[%d]", System.identityHashCode(this)));
                }
                if (this.historicalLog != null) {
                    this.historicalLog.recordEvent("closed", new Object[0]);
                }
            }
            if (!this.used) {
                this.releaseReservation(this.nBytes);
            }
            this.closed = true;
        }

        @Override
        public boolean reserve(int nBytes) {
            BaseAllocator.this.assertOpen();
            AllocationOutcome outcome = BaseAllocator.this.allocateBytes(nBytes);
            if (this.historicalLog != null) {
                this.historicalLog.recordEvent("reserve(%d) => %s", nBytes, Boolean.toString(outcome.isOk()));
            }
            return outcome.isOk();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private ArrowBuf allocate(int nBytes) {
            BaseAllocator.this.assertOpen();
            boolean success = false;
            try {
                ArrowBuf arrowBuf = BaseAllocator.this.bufferWithoutReservation(nBytes, null);
                BaseAllocator.this.listener.onAllocation(nBytes);
                if (this.historicalLog != null) {
                    this.historicalLog.recordEvent("allocate() => %s", String.format("ArrowBuf[%d]", arrowBuf.getId()));
                }
                success = true;
                ArrowBuf arrowBuf2 = arrowBuf;
                return arrowBuf2;
            }
            finally {
                if (!success) {
                    BaseAllocator.this.releaseBytes(nBytes);
                }
            }
        }

        private void releaseReservation(int nBytes) {
            BaseAllocator.this.assertOpen();
            BaseAllocator.this.releaseBytes(nBytes);
            if (this.historicalLog != null) {
                this.historicalLog.recordEvent("releaseReservation(%d)", nBytes);
            }
        }
    }

    @Value.Immutable
    static abstract class Config {
        Config() {
        }

        @Value.Default
        AllocationManager.Factory getAllocationManagerFactory() {
            return DefaultAllocationManagerOption.getDefaultAllocationManagerFactory();
        }

        @Value.Default
        AllocationListener getListener() {
            return AllocationListener.NOOP;
        }

        @Value.Default
        long getInitReservation() {
            return 0L;
        }

        @Value.Default
        long getMaxAllocation() {
            return Long.MAX_VALUE;
        }

        @Value.Default
        RoundingPolicy getRoundingPolicy() {
            return DefaultRoundingPolicy.DEFAULT_ROUNDING_POLICY;
        }
    }

    public static enum Verbosity {
        BASIC(false, false),
        LOG(true, false),
        LOG_WITH_STACKTRACE(true, true);

        public final boolean includeHistoricalLog;
        public final boolean includeStackTraces;

        private Verbosity(boolean includeHistoricalLog, boolean includeStackTraces) {
            this.includeHistoricalLog = includeHistoricalLog;
            this.includeStackTraces = includeStackTraces;
        }
    }
}

