/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.aws.glue;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.iceberg.Schema;
import org.apache.iceberg.TableMetadata;
import org.apache.iceberg.catalog.Namespace;
import org.apache.iceberg.catalog.TableIdentifier;
import org.apache.iceberg.common.DynMethods;
import org.apache.iceberg.exceptions.ValidationException;
import org.apache.iceberg.relocated.com.google.common.collect.ImmutableList;
import org.apache.iceberg.relocated.com.google.common.collect.ImmutableMap;
import org.apache.iceberg.relocated.com.google.common.collect.Lists;
import org.apache.iceberg.relocated.com.google.common.collect.Maps;
import org.apache.iceberg.relocated.com.google.common.collect.Sets;
import org.apache.iceberg.types.Type;
import org.apache.iceberg.types.Types;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.services.glue.model.Column;
import software.amazon.awssdk.services.glue.model.DatabaseInput;
import software.amazon.awssdk.services.glue.model.StorageDescriptor;
import software.amazon.awssdk.services.glue.model.Table;
import software.amazon.awssdk.services.glue.model.TableInput;

class IcebergToGlueConverter {
    private static final Logger LOG = LoggerFactory.getLogger(IcebergToGlueConverter.class);
    private static final Pattern GLUE_DB_PATTERN = Pattern.compile("^[a-z0-9_]{1,252}$");
    private static final Pattern GLUE_TABLE_PATTERN = Pattern.compile("^[a-z0-9_]{1,255}$");
    public static final String GLUE_DB_LOCATION_KEY = "location";
    public static final String GLUE_DESCRIPTION_KEY = "comment";
    public static final String ICEBERG_FIELD_ID = "iceberg.field.id";
    public static final String ICEBERG_FIELD_OPTIONAL = "iceberg.field.optional";
    public static final String ICEBERG_FIELD_CURRENT = "iceberg.field.current";
    private static final List<String> ADDITIONAL_LOCATION_PROPERTIES = ImmutableList.of((Object)"write.data.path", (Object)"write.metadata.path", (Object)"write.object-storage.path", (Object)"write.folder-storage.path");
    private static final DynMethods.UnboundMethod SET_ADDITIONAL_LOCATIONS = DynMethods.builder((String)"additionalLocations").hiddenImpl("software.amazon.awssdk.services.glue.model.StorageDescriptor$Builder", new Class[]{Collection.class}).orNoop().build();

    private IcebergToGlueConverter() {
    }

    static boolean isValidNamespace(Namespace namespace) {
        if (namespace.levels().length != 1) {
            return false;
        }
        String dbName = namespace.level(0);
        return dbName != null && GLUE_DB_PATTERN.matcher(dbName).find();
    }

    static void validateNamespace(Namespace namespace) {
        ValidationException.check((boolean)IcebergToGlueConverter.isValidNamespace(namespace), (String)"Cannot convert namespace %s to Glue database name, because it must be 1-252 chars of lowercase letters, numbers, underscore", (Object[])new Object[]{namespace});
    }

    static String toDatabaseName(Namespace namespace, boolean skipNameValidation) {
        if (!skipNameValidation) {
            IcebergToGlueConverter.validateNamespace(namespace);
        }
        return namespace.level(0);
    }

    static String getDatabaseName(TableIdentifier tableIdentifier, boolean skipNameValidation) {
        return IcebergToGlueConverter.toDatabaseName(tableIdentifier.namespace(), skipNameValidation);
    }

    static DatabaseInput toDatabaseInput(Namespace namespace, Map<String, String> metadata, boolean skipNameValidation) {
        DatabaseInput.Builder builder = DatabaseInput.builder().name(IcebergToGlueConverter.toDatabaseName(namespace, skipNameValidation));
        HashMap parameters = Maps.newHashMap();
        metadata.forEach((k, v) -> {
            if (GLUE_DESCRIPTION_KEY.equals(k)) {
                builder.description(v);
            } else if (GLUE_DB_LOCATION_KEY.equals(k)) {
                builder.locationUri(v);
            } else {
                parameters.put(k, v);
            }
        });
        return (DatabaseInput)builder.parameters((Map)parameters).build();
    }

    static boolean isValidTableName(String tableName) {
        return tableName != null && GLUE_TABLE_PATTERN.matcher(tableName).find();
    }

    static void validateTableName(String tableName) {
        ValidationException.check((boolean)IcebergToGlueConverter.isValidTableName(tableName), (String)"Cannot use %s as Glue table name, because it must be 1-255 chars of lowercase letters, numbers, underscore", (Object[])new Object[]{tableName});
    }

    static String getTableName(TableIdentifier tableIdentifier, boolean skipNameValidation) {
        if (!skipNameValidation) {
            IcebergToGlueConverter.validateTableName(tableIdentifier.name());
        }
        return tableIdentifier.name();
    }

    static void setTableInputInformation(TableInput.Builder tableInputBuilder, TableMetadata metadata) {
        IcebergToGlueConverter.setTableInputInformation(tableInputBuilder, metadata, null);
    }

    static void setTableInputInformation(TableInput.Builder tableInputBuilder, TableMetadata metadata, Table existingTable) {
        try {
            String description;
            Map properties = metadata.properties();
            StorageDescriptor.Builder storageDescriptor = StorageDescriptor.builder();
            if (!SET_ADDITIONAL_LOCATIONS.isNoop()) {
                Object[] objectArray = new Object[1];
                objectArray[0] = ADDITIONAL_LOCATION_PROPERTIES.stream().map(properties::get).filter(Objects::nonNull).collect(Collectors.toSet());
                SET_ADDITIONAL_LOCATIONS.invoke((Object)storageDescriptor, objectArray);
            }
            if ((description = (String)properties.get(GLUE_DESCRIPTION_KEY)) != null) {
                tableInputBuilder.description(description);
            } else if (existingTable != null) {
                Optional.ofNullable(existingTable.description()).ifPresent(arg_0 -> ((TableInput.Builder)tableInputBuilder).description(arg_0));
            }
            Map<String, String> existingColumnMap = null;
            if (existingTable != null) {
                List existingColumns = existingTable.storageDescriptor().columns();
                existingColumnMap = existingColumns.stream().filter(column -> column.comment() != null).collect(Collectors.toMap(Column::name, Column::comment));
            } else {
                existingColumnMap = Collections.emptyMap();
            }
            List<Column> columns = IcebergToGlueConverter.toColumns(metadata, existingColumnMap);
            tableInputBuilder.storageDescriptor((StorageDescriptor)storageDescriptor.location(metadata.location()).columns(columns).build());
        }
        catch (RuntimeException e) {
            LOG.warn("Encountered unexpected exception while converting Iceberg metadata to Glue table information", (Throwable)e);
        }
    }

    private static String toTypeString(Type type) {
        switch (type.typeId()) {
            case BOOLEAN: {
                return "boolean";
            }
            case INTEGER: {
                return "int";
            }
            case LONG: {
                return "bigint";
            }
            case FLOAT: {
                return "float";
            }
            case DOUBLE: {
                return "double";
            }
            case DATE: {
                return "date";
            }
            case TIME: 
            case STRING: 
            case UUID: {
                return "string";
            }
            case TIMESTAMP: {
                return "timestamp";
            }
            case FIXED: 
            case BINARY: {
                return "binary";
            }
            case DECIMAL: {
                Types.DecimalType decimalType = (Types.DecimalType)type;
                return String.format("decimal(%s,%s)", decimalType.precision(), decimalType.scale());
            }
            case STRUCT: {
                Types.StructType structType = type.asStructType();
                String nameToType = structType.fields().stream().map(f -> String.format("%s:%s", f.name(), IcebergToGlueConverter.toTypeString(f.type()))).collect(Collectors.joining(","));
                return String.format("struct<%s>", nameToType);
            }
            case LIST: {
                Types.ListType listType = type.asListType();
                return String.format("array<%s>", IcebergToGlueConverter.toTypeString(listType.elementType()));
            }
            case MAP: {
                Types.MapType mapType = type.asMapType();
                return String.format("map<%s,%s>", IcebergToGlueConverter.toTypeString(mapType.keyType()), IcebergToGlueConverter.toTypeString(mapType.valueType()));
            }
        }
        return type.typeId().name().toLowerCase(Locale.ENGLISH);
    }

    private static List<Column> toColumns(TableMetadata metadata, Map<String, String> existingColumnMap) {
        ArrayList columns = Lists.newArrayList();
        HashSet addedNames = Sets.newHashSet();
        for (Types.NestedField field : metadata.schema().columns()) {
            IcebergToGlueConverter.addColumnWithDedupe(columns, addedNames, field, true, existingColumnMap);
        }
        for (Schema schema : metadata.schemas()) {
            if (schema.schemaId() == metadata.currentSchemaId()) continue;
            for (Types.NestedField field : schema.columns()) {
                IcebergToGlueConverter.addColumnWithDedupe(columns, addedNames, field, false, existingColumnMap);
            }
        }
        return columns;
    }

    private static void addColumnWithDedupe(List<Column> columns, Set<String> dedupe, Types.NestedField field, boolean isCurrent, Map<String, String> existingColumnMap) {
        if (!dedupe.contains(field.name())) {
            Column.Builder builder = Column.builder().name(field.name()).type(IcebergToGlueConverter.toTypeString(field.type())).parameters((Map)ImmutableMap.of((Object)ICEBERG_FIELD_ID, (Object)Integer.toString(field.fieldId()), (Object)ICEBERG_FIELD_OPTIONAL, (Object)Boolean.toString(field.isOptional()), (Object)ICEBERG_FIELD_CURRENT, (Object)Boolean.toString(isCurrent)));
            if (field.doc() != null && !field.doc().isEmpty()) {
                builder.comment(field.doc());
            } else if (existingColumnMap != null && existingColumnMap.containsKey(field.name())) {
                builder.comment(existingColumnMap.get(field.name()));
            }
            columns.add((Column)builder.build());
            dedupe.add(field.name());
        }
    }
}

