/*
 * Decompiled with CFR 0.152.
 */
package io.smallrye.faulttolerance.core.circuit.breaker;

import io.smallrye.faulttolerance.core.Completer;
import io.smallrye.faulttolerance.core.FaultToleranceContext;
import io.smallrye.faulttolerance.core.FaultToleranceStrategy;
import io.smallrye.faulttolerance.core.Future;
import io.smallrye.faulttolerance.core.circuit.breaker.CircuitBreakerEvents;
import io.smallrye.faulttolerance.core.circuit.breaker.CircuitBreakerLogger;
import io.smallrye.faulttolerance.core.circuit.breaker.RollingWindow;
import io.smallrye.faulttolerance.core.stopwatch.RunningStopwatch;
import io.smallrye.faulttolerance.core.stopwatch.Stopwatch;
import io.smallrye.faulttolerance.core.timer.Timer;
import io.smallrye.faulttolerance.core.util.ExceptionDecision;
import io.smallrye.faulttolerance.core.util.Preconditions;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import org.eclipse.microprofile.faulttolerance.exceptions.CircuitBreakerOpenException;

public class CircuitBreaker<V>
implements FaultToleranceStrategy<V> {
    public static final int STATE_CLOSED = 0;
    public static final int STATE_OPEN = 1;
    public static final int STATE_HALF_OPEN = 2;
    private final FaultToleranceStrategy<V> delegate;
    private final String description;
    private final ExceptionDecision exceptionDecision;
    private final long delayInMillis;
    private final int rollingWindowSize;
    private final int failureThreshold;
    private final int successThreshold;
    private final Stopwatch stopwatch;
    private final Timer timer;
    private final AtomicReference<State> state;

    public CircuitBreaker(FaultToleranceStrategy<V> delegate, String description, ExceptionDecision exceptionDecision, long delayInMillis, int requestVolumeThreshold, double failureRatio, int successThreshold, Stopwatch stopwatch, Timer timer) {
        this.delegate = Preconditions.checkNotNull(delegate, "Circuit breaker delegate must be set");
        this.description = Preconditions.checkNotNull(description, "Circuit breaker description must be set");
        this.exceptionDecision = Preconditions.checkNotNull(exceptionDecision, "Exception decision must be set");
        this.delayInMillis = Preconditions.check(delayInMillis, delayInMillis >= 0L, "Circuit breaker delay must be >= 0");
        this.successThreshold = Preconditions.check(successThreshold, successThreshold > 0, "Circuit breaker success threshold must be > 0");
        this.stopwatch = Preconditions.checkNotNull(stopwatch, "Stopwatch must be set");
        this.timer = Preconditions.checkNotNull(timer, "Timer must be set");
        this.failureThreshold = Preconditions.check((int)Math.ceil(failureRatio * (double)requestVolumeThreshold), failureRatio >= 0.0 && failureRatio <= 1.0, "Circuit breaker rolling window failure ratio must be >= 0 && <= 1");
        this.rollingWindowSize = Preconditions.check(requestVolumeThreshold, requestVolumeThreshold > 0, "Circuit breaker rolling window size must be > 0");
        this.state = new AtomicReference<State>(State.closed(this.rollingWindowSize, this.failureThreshold));
    }

    @Override
    public Future<V> apply(FaultToleranceContext<V> ctx) {
        CircuitBreakerLogger.LOG.trace("CircuitBreaker started");
        try {
            State currentState = this.state.get();
            switch (currentState.id) {
                case 0: {
                    Future<V> future = this.inClosed(ctx, currentState);
                    return future;
                }
                case 1: {
                    Future<V> future = this.inOpen(ctx, currentState);
                    return future;
                }
                case 2: {
                    Future<V> future = this.inHalfOpen(ctx, currentState);
                    return future;
                }
            }
            throw new AssertionError((Object)("Invalid circuit breaker state: " + currentState.id));
        }
        finally {
            CircuitBreakerLogger.LOG.trace("CircuitBreaker finished");
        }
    }

    private Future<V> inClosed(FaultToleranceContext<V> ctx, State state) {
        try {
            CircuitBreakerLogger.LOG.trace("Circuit breaker closed, invocation allowed");
            Completer result = Completer.create();
            this.delegate.apply(ctx).then((value, error) -> {
                if (error == null) {
                    this.inClosedHandleResult(true, ctx, state);
                    result.complete(value);
                } else {
                    this.inClosedHandleResult(this.exceptionDecision.isConsideredExpected((Throwable)error), ctx, state);
                    result.completeWithError((Throwable)error);
                }
            });
            return result.future();
        }
        catch (Throwable e) {
            this.inClosedHandleResult(this.exceptionDecision.isConsideredExpected(e), ctx, state);
            return Future.ofError(e);
        }
    }

    private void inClosedHandleResult(boolean isSuccess, FaultToleranceContext<V> ctx, State state) {
        boolean failureThresholdReached;
        ctx.fireEvent(isSuccess ? CircuitBreakerEvents.Finished.SUCCESS : CircuitBreakerEvents.Finished.FAILURE);
        boolean bl = failureThresholdReached = isSuccess ? state.rollingWindow.recordSuccess() : state.rollingWindow.recordFailure();
        if (failureThresholdReached) {
            CircuitBreakerLogger.LOG.trace("Failure threshold reached, circuit breaker moving to open");
            this.toOpen(ctx, state);
        }
    }

    private Future<V> inOpen(FaultToleranceContext<V> ctx, State state) {
        if (state.runningStopwatch.elapsedTimeInMillis() < this.delayInMillis) {
            CircuitBreakerLogger.LOG.debugOrTrace(this.description + " invocation prevented by circuit breaker", "Circuit breaker open, invocation prevented");
            ctx.fireEvent(CircuitBreakerEvents.Finished.PREVENTED);
            return Future.ofError((Throwable)new CircuitBreakerOpenException(this.description + " circuit breaker is open"));
        }
        CircuitBreakerLogger.LOG.trace("Delay elapsed synchronously, circuit breaker moving to half-open");
        this.toHalfOpen(ctx, state);
        State currentState = this.state.get();
        switch (currentState.id) {
            case 0: {
                return this.inClosed(ctx, currentState);
            }
            case 1: {
                return this.inOpen(ctx, currentState);
            }
            case 2: {
                return this.inHalfOpen(ctx, currentState);
            }
        }
        throw new AssertionError((Object)("Invalid circuit breaker state: " + currentState.id));
    }

    private Future<V> inHalfOpen(FaultToleranceContext<V> ctx, State state) {
        if (state.probeAttempts.incrementAndGet() > this.successThreshold) {
            CircuitBreakerLogger.LOG.debugOrTrace(this.description + " invocation prevented by circuit breaker", "Circuit breaker half-open, invocation prevented");
            ctx.fireEvent(CircuitBreakerEvents.Finished.PREVENTED);
            return Future.ofError((Throwable)new CircuitBreakerOpenException(this.description + " circuit breaker is half-open"));
        }
        try {
            CircuitBreakerLogger.LOG.trace("Circuit breaker half-open, probe invocation allowed");
            Completer result = Completer.create();
            this.delegate.apply(ctx).then((value, error) -> {
                if (error == null) {
                    this.inHalfOpenHandleResult(true, ctx, state);
                    result.complete(value);
                } else {
                    this.inHalfOpenHandleResult(this.exceptionDecision.isConsideredExpected((Throwable)error), ctx, state);
                    result.completeWithError((Throwable)error);
                }
            });
            return result.future();
        }
        catch (Throwable e) {
            this.inHalfOpenHandleResult(this.exceptionDecision.isConsideredExpected(e), ctx, state);
            return Future.ofError(e);
        }
    }

    private void inHalfOpenHandleResult(boolean isSuccess, FaultToleranceContext<V> ctx, State state) {
        ctx.fireEvent(isSuccess ? CircuitBreakerEvents.Finished.SUCCESS : CircuitBreakerEvents.Finished.FAILURE);
        if (isSuccess) {
            int successes = state.consecutiveSuccesses.incrementAndGet();
            if (successes >= this.successThreshold) {
                CircuitBreakerLogger.LOG.trace("Success threshold reached, circuit breaker moving to closed");
                this.toClosed(ctx, state);
            }
        } else {
            CircuitBreakerLogger.LOG.trace("Failure while in half-open, circuit breaker moving to open");
            this.toOpen(ctx, state);
        }
    }

    void toClosed(FaultToleranceContext<V> ctx, State state) {
        State newState = State.closed(this.rollingWindowSize, this.failureThreshold);
        boolean moved = this.state.compareAndSet(state, newState);
        if (moved) {
            ctx.fireEvent(CircuitBreakerEvents.StateTransition.TO_CLOSED);
        }
    }

    void toOpen(FaultToleranceContext<V> ctx, State state) {
        State newState = State.open(this.stopwatch);
        boolean moved = this.state.compareAndSet(state, newState);
        if (moved) {
            ctx.fireEvent(CircuitBreakerEvents.StateTransition.TO_OPEN);
            this.timer.schedule(this.delayInMillis, () -> {
                CircuitBreakerLogger.LOG.trace("Delay elapsed asynchronously, circuit breaker moving to half-open");
                this.toHalfOpen(ctx, newState);
            });
        }
    }

    void toHalfOpen(FaultToleranceContext<V> ctx, State state) {
        State newState = State.halfOpen();
        boolean moved = this.state.compareAndSet(state, newState);
        if (moved) {
            ctx.fireEvent(CircuitBreakerEvents.StateTransition.TO_HALF_OPEN);
        }
    }

    public int currentState() {
        return this.state.get().id;
    }

    public void reset() {
        State newState = State.closed(this.rollingWindowSize, this.failureThreshold);
        this.state.set(newState);
    }

    static final class State {
        final int id;
        RollingWindow rollingWindow;
        RunningStopwatch runningStopwatch;
        AtomicInteger probeAttempts;
        AtomicInteger consecutiveSuccesses;

        private State(int id) {
            this.id = id;
        }

        static State closed(int rollingWindowSize, int failureThreshold) {
            State result = new State(0);
            result.rollingWindow = RollingWindow.create(rollingWindowSize, failureThreshold);
            return result;
        }

        static State open(Stopwatch stopwatch) {
            State result = new State(1);
            result.runningStopwatch = stopwatch.start();
            return result;
        }

        static State halfOpen() {
            State result = new State(2);
            result.probeAttempts = new AtomicInteger(0);
            result.consecutiveSuccesses = new AtomicInteger(0);
            return result;
        }
    }
}

