/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.polaris.core.identity.dpo;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonSubTypes;
import com.fasterxml.jackson.annotation.JsonTypeInfo;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import jakarta.annotation.Nullable;
import org.apache.polaris.core.admin.model.ServiceIdentityInfo;
import org.apache.polaris.core.identity.ServiceIdentityType;
import org.apache.polaris.core.identity.credential.ServiceIdentityCredential;
import org.apache.polaris.core.identity.provider.ServiceIdentityProvider;
import org.apache.polaris.core.secrets.SecretReference;

/**
 * The internal persistence-object counterpart to ServiceIdentityInfo defined in the API model.
 * Important: JsonSubTypes must be kept in sync with {@link ServiceIdentityType}.
 *
 * <p>This DPO stores only the identity type and a {@link SecretReference} that serves as a unique
 * identifier for the service identity instance. The reference is used at runtime by a {@link
 * ServiceIdentityProvider} to look up the configuration and retrieve the full {@link
 * ServiceIdentityCredential} with credentials and metadata.
 */
@JsonTypeInfo(
    use = JsonTypeInfo.Id.NAME,
    include = JsonTypeInfo.As.EXISTING_PROPERTY,
    property = "identityTypeCode")
@JsonSubTypes({@JsonSubTypes.Type(value = AwsIamServiceIdentityInfoDpo.class, name = "1")})
public abstract class ServiceIdentityInfoDpo {

  @JsonProperty(value = "identityTypeCode")
  private final int identityTypeCode;

  @JsonProperty(value = "identityInfoReference")
  private final SecretReference identityInfoReference;

  public ServiceIdentityInfoDpo(
      @JsonProperty(value = "identityTypeCode", required = true) int identityTypeCode,
      @JsonProperty(value = "identityInfoReference", required = false) @Nullable
          SecretReference identityInfoReference) {
    this.identityTypeCode = identityTypeCode;
    this.identityInfoReference = identityInfoReference;
  }

  public int getIdentityTypeCode() {
    return identityTypeCode;
  }

  @JsonIgnore
  public ServiceIdentityType getIdentityType() {
    return ServiceIdentityType.fromCode(identityTypeCode);
  }

  @JsonProperty
  public SecretReference getIdentityInfoReference() {
    return identityInfoReference;
  }

  /**
   * Converts this persistence object to the corresponding API model.
   *
   * <p>The conversion uses the provided {@link ServiceIdentityProvider} to retrieve the user-facing
   * identity information (e.g., AWS IAM ARN) without exposing sensitive credentials. The credential
   * reference stored in this DPO is not included in the API model.
   *
   * @param serviceIdentityProvider the service identity provider used to retrieve display
   *     information
   * @return the API model representation, or null if the provider is null or cannot resolve the
   *     identity
   */
  public @Nullable ServiceIdentityInfo asServiceIdentityInfoModel(
      ServiceIdentityProvider serviceIdentityProvider) {
    Preconditions.checkNotNull(
        serviceIdentityProvider,
        "Need ServiceIdentityProvider to inject service identity info, should not be null");
    return serviceIdentityProvider.getServiceIdentityInfo(this).orElse(null);
  }

  @Override
  public String toString() {
    return MoreObjects.toStringHelper(this)
        .add("identityTypeCode", getIdentityTypeCode())
        .toString();
  }
}
