/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.flink.agents.runtime.python.event;

import org.apache.flink.agents.api.Event;
import org.apache.flink.shaded.jackson2.com.fasterxml.jackson.annotation.JsonCreator;
import org.apache.flink.shaded.jackson2.com.fasterxml.jackson.annotation.JsonIgnore;
import org.apache.flink.shaded.jackson2.com.fasterxml.jackson.annotation.JsonProperty;

import java.util.Arrays;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;

/**
 * An event generated by the framework, passing a Python event to the Java agent runner.
 *
 * <p>This class stores Python objects as serialized byte arrays for processing, and also maintains
 * a human-readable string representation (eventString) for logging purposes. The eventString is
 * generated on the Python side when the event is created.
 */
public class PythonEvent extends Event {
    private final byte[] event;
    private final String eventType;
    private final String eventString;

    public PythonEvent(byte[] event, String eventType, String eventString) {
        super();
        this.event = event;
        this.eventType = eventType;
        this.eventString = eventString;
    }

    @JsonCreator
    public PythonEvent(
            @JsonProperty("id") UUID id,
            @JsonProperty("attributes") Map<String, Object> attributes,
            @JsonProperty("event") byte[] event,
            @JsonProperty("eventType") String eventType,
            @JsonProperty("eventString") String eventString) {
        super(id, attributes);
        this.event = event;
        this.eventType = eventType;
        this.eventString = eventString;
    }

    @JsonIgnore // Don't serialize byte array in logs - used for processing only
    public byte[] getEvent() {
        return event;
    }

    public String getEventType() {
        return eventType;
    }

    /**
     * Returns the human-readable string representation of this event.
     *
     * <p>This string is generated on the Python side when the event is created and is primarily
     * used for logging purposes.
     *
     * @return the string representation of the event, or null if not available
     */
    public String getEventString() {
        return eventString;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        if (!super.equals(o)) {
            return false;
        }
        PythonEvent other = (PythonEvent) o;
        return Arrays.equals(event, other.event)
                && Objects.equals(this.eventType, other.eventType)
                && Objects.equals(this.eventString, other.eventString);
    }

    @Override
    public int hashCode() {
        return Objects.hash(super.hashCode(), Arrays.hashCode(event), eventType, eventString);
    }
}
