/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.quarkus.component.xj.it;

import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.stream.Stream;

import io.quarkus.test.junit.QuarkusTest;
import io.restassured.RestAssured;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;
import org.junit.jupiter.params.provider.ValueSource;

@QuarkusTest
public class XJTest {
    private static final String JSON = "{\"hello\":\"world!\"}";

    private static final String XML = "<hello>world!</hello>";

    private static Stream<Arguments> identityTestSets() {
        return Stream.of(
                Arguments.of("XML2JSON", XML, "{\"#text\":\"world!\"}"),
                Arguments.of("JSON2XML", JSON,
                        "<?xml version=\"1.0\" encoding=\"UTF-8\"?>"
                                + "<object xmlns:xj=\"http://camel.apache.org/component/xj\" xj:type=\"object\">"
                                + "<object xj:name=\"hello\" xj:type=\"string\">world!</object>"
                                + "</object>"));
    }

    @Test
    public void loadComponentXj() {
        /* A simple autogenerated test */
        RestAssured.get("/xj/load/component/xj")
                .then()
                .statusCode(200);
    }

    @ParameterizedTest
    @ValueSource(strings = { "string", "bytes", "dom", "file" })
    public void testJsonToXML(String output) throws Exception {
        String actual = RestAssured.given()
                .body(JSON)
                .post("/xj/j2x?output=" + output)
                .then()
                .statusCode(200)
                .extract().body().asString();

        if (output.equals("dom")) {
            Assertions.assertEquals(XML, actual);
        } else if (output.equals("file")) {
            Assertions.assertEquals("<?xml version=\"1.0\" encoding=\"UTF-8\"?>" + XML,
                    Files.readString(Path.of("target/J2XOutputFileTest.xml"), Charset.forName("UTF-8")));
        } else {
            Assertions.assertEquals("<?xml version=\"1.0\" encoding=\"UTF-8\"?>" + XML, actual);
        }
    }

    @ParameterizedTest
    @ValueSource(strings = { "string", "bytes", "file" })
    public void testXMLToJson(String output) throws Exception {
        String actual = RestAssured.given()
                .body(XML)
                .post("/xj/x2j?output=" + output)
                .then()
                .statusCode(200)
                .extract().body().asString();

        if (output.equals("file")) {
            Assertions.assertEquals(JSON, Files.readString(Path.of("target/X2JOutputFileTest.xml"), Charset.forName("UTF-8")));
        } else {
            Assertions.assertEquals(JSON, actual);
        }
    }

    @ParameterizedTest
    @MethodSource("identityTestSets")
    public void testIdentity(String direction, String body, String expected) {
        String actual = RestAssured.given()
                .body(body)
                .post("/xj/identity?direction=" + direction)
                .then()
                .statusCode(200)
                .extract().body().asString();

        Assertions.assertEquals(expected, actual);
    }
}
