use crate::iter::plumbing::{bridge_unindexed, Folder, UnindexedConsumer, UnindexedProducer};
use crate::prelude::*;
use std::iter::once;

#[derive(Debug)]
struct WalkTreePrefixProducer<'b, S, B> {
    to_explore: Vec<S>, 
    seen: Vec<S>,       
    children_of: &'b B, 
}

impl<S, B, I> UnindexedProducer for WalkTreePrefixProducer<'_, S, B>
where
    S: Send,
    B: Fn(&S) -> I + Send + Sync,
    I: IntoIterator<Item = S>,
    I::IntoIter: DoubleEndedIterator,
{
    type Item = S;

    fn split(mut self) -> (Self, Option<Self>) {
        
        while self.to_explore.len() == 1 {
            let front_node = self.to_explore.pop().unwrap();
            self.to_explore
                .extend((self.children_of)(&front_node).into_iter().rev());
            self.seen.push(front_node);
        }
        
        let right_children = split_vec(&mut self.to_explore);
        let right = right_children
            .map(|mut c| {
                std::mem::swap(&mut c, &mut self.to_explore);
                WalkTreePrefixProducer {
                    to_explore: c,
                    seen: Vec::new(),
                    children_of: self.children_of,
                }
            })
            .or_else(|| {
                
                let right_seen = split_vec(&mut self.seen);
                right_seen.map(|s| WalkTreePrefixProducer {
                    to_explore: Default::default(),
                    seen: s,
                    children_of: self.children_of,
                })
            });
        (self, right)
    }

    fn fold_with<F>(mut self, mut folder: F) -> F
    where
        F: Folder<Self::Item>,
    {
        
        folder = folder.consume_iter(self.seen);
        if folder.full() {
            return folder;
        }
        
        while let Some(e) = self.to_explore.pop() {
            self.to_explore
                .extend((self.children_of)(&e).into_iter().rev());
            folder = folder.consume(e);
            if folder.full() {
                return folder;
            }
        }
        folder
    }
}



#[derive(Debug)]
pub struct WalkTreePrefix<S, B> {
    initial_state: S,
    children_of: B,
}

impl<S, B, I> ParallelIterator for WalkTreePrefix<S, B>
where
    S: Send,
    B: Fn(&S) -> I + Send + Sync,
    I: IntoIterator<Item = S>,
    I::IntoIter: DoubleEndedIterator,
{
    type Item = S;

    fn drive_unindexed<C>(self, consumer: C) -> C::Result
    where
        C: UnindexedConsumer<Self::Item>,
    {
        let producer = WalkTreePrefixProducer {
            to_explore: once(self.initial_state).collect(),
            seen: Vec::new(),
            children_of: &self.children_of,
        };
        bridge_unindexed(producer, consumer)
    }
}






/// # Ordering



















/// # Example
























/// # Example



















































pub fn walk_tree_prefix<S, B, I>(root: S, children_of: B) -> WalkTreePrefix<S, B>
where
    S: Send,
    B: Fn(&S) -> I + Send + Sync,
    I: IntoIterator<Item = S>,
    I::IntoIter: DoubleEndedIterator,
{
    WalkTreePrefix {
        initial_state: root,
        children_of,
    }
}



#[derive(Debug)]
struct WalkTreePostfixProducer<'b, S, B> {
    to_explore: Vec<S>, 
    seen: Vec<S>,       
    children_of: &'b B, 
}

impl<S, B, I> UnindexedProducer for WalkTreePostfixProducer<'_, S, B>
where
    S: Send,
    B: Fn(&S) -> I + Send + Sync,
    I: IntoIterator<Item = S>,
{
    type Item = S;

    fn split(mut self) -> (Self, Option<Self>) {
        
        while self.to_explore.len() == 1 {
            let front_node = self.to_explore.pop().unwrap();
            self.to_explore
                .extend((self.children_of)(&front_node).into_iter());
            self.seen.push(front_node);
        }
        
        let right_children = split_vec(&mut self.to_explore);
        let right = right_children
            .map(|c| {
                let right_seen = std::mem::take(&mut self.seen); 
                WalkTreePostfixProducer {
                    to_explore: c,
                    seen: right_seen,
                    children_of: self.children_of,
                }
            })
            .or_else(|| {
                
                let right_seen = split_vec(&mut self.seen);
                right_seen.map(|mut s| {
                    std::mem::swap(&mut self.seen, &mut s);
                    WalkTreePostfixProducer {
                        to_explore: Default::default(),
                        seen: s,
                        children_of: self.children_of,
                    }
                })
            });
        (self, right)
    }

    fn fold_with<F>(self, mut folder: F) -> F
    where
        F: Folder<Self::Item>,
    {
        
        for e in self.to_explore {
            folder = consume_rec_postfix(&self.children_of, e, folder);
            if folder.full() {
                return folder;
            }
        }
        
        folder.consume_iter(self.seen.into_iter().rev())
    }
}

fn consume_rec_postfix<F, S, B, I>(children_of: &B, s: S, mut folder: F) -> F
where
    F: Folder<S>,
    B: Fn(&S) -> I,
    I: IntoIterator<Item = S>,
{
    let children = (children_of)(&s).into_iter();
    for child in children {
        folder = consume_rec_postfix(children_of, child, folder);
        if folder.full() {
            return folder;
        }
    }
    folder.consume(s)
}



#[derive(Debug)]
pub struct WalkTreePostfix<S, B> {
    initial_state: S,
    children_of: B,
}

impl<S, B, I> ParallelIterator for WalkTreePostfix<S, B>
where
    S: Send,
    B: Fn(&S) -> I + Send + Sync,
    I: IntoIterator<Item = S>,
{
    type Item = S;

    fn drive_unindexed<C>(self, consumer: C) -> C::Result
    where
        C: UnindexedConsumer<Self::Item>,
    {
        let producer = WalkTreePostfixProducer {
            to_explore: once(self.initial_state).collect(),
            seen: Vec::new(),
            children_of: &self.children_of,
        };
        bridge_unindexed(producer, consumer)
    }
}




fn split_vec<T>(v: &mut Vec<T>) -> Option<Vec<T>> {
    if v.len() <= 1 {
        None
    } else {
        let n = v.len() / 2;
        Some(v.split_off(n))
    }
}






/// # Ordering





















/// # Example
























/// # Example



















































pub fn walk_tree_postfix<S, B, I>(root: S, children_of: B) -> WalkTreePostfix<S, B>
where
    S: Send,
    B: Fn(&S) -> I + Send + Sync,
    I: IntoIterator<Item = S>,
{
    WalkTreePostfix {
        initial_state: root,
        children_of,
    }
}



#[derive(Debug)]
pub struct WalkTree<S, B>(WalkTreePostfix<S, B>);






/// # Ordering






/// # Example























pub fn walk_tree<S, B, I>(root: S, children_of: B) -> WalkTree<S, B>
where
    S: Send,
    B: Fn(&S) -> I + Send + Sync,
    I: IntoIterator<Item = S>,
    I::IntoIter: DoubleEndedIterator,
{
    let walker = WalkTreePostfix {
        initial_state: root,
        children_of,
    };
    WalkTree(walker)
}

impl<S, B, I> ParallelIterator for WalkTree<S, B>
where
    S: Send,
    B: Fn(&S) -> I + Send + Sync,
    I: IntoIterator<Item = S> + Send,
    I::IntoIter: DoubleEndedIterator,
{
    type Item = S;

    fn drive_unindexed<C>(self, consumer: C) -> C::Result
    where
        C: UnindexedConsumer<Self::Item>,
    {
        self.0.drive_unindexed(consumer)
    }
}
