use std::iter::{Fuse,Peekable, FusedIterator};






#[must_use = "iterator adaptors are lazy and do nothing unless consumed"]
pub struct WithPosition<I>
    where I: Iterator,
{
    handled_first: bool,
    peekable: Peekable<Fuse<I>>,
}

impl<I> Clone for WithPosition<I>
    where I: Clone + Iterator,
          I::Item: Clone,
{
    clone_fields!(handled_first, peekable);
}


pub fn with_position<I>(iter: I) -> WithPosition<I>
    where I: Iterator,
{
    WithPosition {
        handled_first: false,
        peekable: iter.fuse().peekable(),
    }
}





#[derive(Copy, Clone, Debug, PartialEq)]
pub enum Position<T> {
    
    First(T),
    
    Middle(T),
    
    Last(T),
    
    Only(T),
}

impl<T> Position<T> {
    
    pub fn into_inner(self) -> T {
        match self {
            Position::First(x) |
            Position::Middle(x) |
            Position::Last(x) |
            Position::Only(x) => x,
        }
    }
}

impl<I: Iterator> Iterator for WithPosition<I> {
    type Item = Position<I::Item>;

    fn next(&mut self) -> Option<Self::Item> {
        match self.peekable.next() {
            Some(item) => {
                if !self.handled_first {
                    
                    self.handled_first = true;
                    
                    
                    match self.peekable.peek() {
                        Some(_) => Some(Position::First(item)),
                        None => Some(Position::Only(item)),
                    }
                } else {
                    
                    
                    match self.peekable.peek() {
                        Some(_) => Some(Position::Middle(item)),
                        None => Some(Position::Last(item)),
                    }
                }
            }
            
            None => None,
        }
    }

    fn size_hint(&self) -> (usize, Option<usize>) {
        self.peekable.size_hint()
    }
}

impl<I> ExactSizeIterator for WithPosition<I>
    where I: ExactSizeIterator,
{ }

impl<I: Iterator> FusedIterator for WithPosition<I> 
{}
