% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/surprisals_data.R
\name{surprisals.numeric}
\alias{surprisals.numeric}
\alias{surprisals.matrix}
\alias{surprisals.data.frame}
\alias{surprisals_prob.numeric}
\alias{surprisals_prob.matrix}
\alias{surprisals_prob.data.frame}
\title{Surprisals and surprisal probabilities computed from data}
\usage{
\method{surprisals}{numeric}(object, distribution = dist_kde(object, ...), loo = FALSE, ...)

\method{surprisals}{matrix}(object, distribution = dist_kde(object, ...), loo = FALSE, ...)

\method{surprisals}{data.frame}(object, distribution = dist_kde(object, ...), loo = FALSE, ...)

\method{surprisals_prob}{numeric}(
  object,
  approximation = c("none", "gpd", "rank"),
  threshold_probability = 0.1,
  distribution = dist_kde(object, ...),
  loo = FALSE,
  ...
)

\method{surprisals_prob}{matrix}(
  object,
  approximation = c("none", "gpd", "rank"),
  threshold_probability = 0.1,
  distribution = dist_kde(object, ...),
  loo = FALSE,
  ...
)

\method{surprisals_prob}{data.frame}(
  object,
  approximation = c("none", "gpd", "rank"),
  threshold_probability = 0.1,
  distribution = dist_kde(object, ...),
  loo = FALSE,
  ...
)
}
\arguments{
\item{object}{A numerical data set (either a vector, matrix, or a data.frame
containing only numerical columns).}

\item{distribution}{A distribution object. By default, a kernel density
estimate is computed from the data \code{object}.}

\item{loo}{Should leave-one-out surprisals be computed?}

\item{...}{Other arguments are passed to the appropriate method.}

\item{approximation}{Character string specifying the method to use in
computing the surprisal probabilities. See Details below. For a multivariate
data set, it needs to be set to either "gpd" or "rank".}

\item{threshold_probability}{Probability threshold when computing the GPD
approximation. This is the probability below which the GPD is fitted. Only
used if \code{approximation = "gpd"}.}
}
\value{
A numerical vector containing the surprisals or surprisal probabilities.
}
\description{
A surprisal is given by \eqn{s = -\log f(y)} where \eqn{f} is the
density or probability mass function of the estimated or assumed distribution,
and \eqn{y} is an observation. This is returned by \code{surprisals()}.
A surprisal probability is the probability of a surprisal at least as extreme
as \eqn{s}. This is returned by \code{surprisals_prob()}
}
\details{
The surprisal probabilities may be computed in three different ways.
\enumerate{
\item When \code{approximation = "none"} (the default), the surprisal probabilities are computed
using the same distribution that was used to compute the surprisal values.
Under this option, surprisal probabilities are equal to 1 minus the
coverage probability of the largest HDR that contains each value. Surprisal
probabilities smaller than 1e-6 are returned as 1e-6.
\item When \code{approximation = "gdp"}, the surprisal probabilities are
computed using a Generalized Pareto Distribution fitted to the most extreme
surprisal values (those with probability less than \code{threshold_probability}).
For surprisal probabilities greater than \code{threshold_probability}, the value of
\code{threshold_probability} is returned. Under this option, the distribution is
used for computing the surprisal values but not for determining their
probabilities. Due to extreme value theory, the resulting probabilities should
be relatively insensitive to the distribution used in computing the surprisal
values.
\item When \code{approximation = "rank"}, the surprisal probability of each
observation is estimated using the proportion of observations with
greater surprisal values; i.e., 1 - rank(s)/n where \code{rank(s)} is the rank
of the surprisal value \code{s} among all surprisal values, and \code{n} is the
number of observations. This is a nonparametric approach that is also
insensitive to the distribution used in computing the surprisal values.
}
}
\examples{
# Univariate data
tibble(
  y = c(5, rnorm(49)),
  p_kde = surprisals_prob(y, loo = TRUE),
  p_normal = surprisals_prob(y, distribution = dist_normal()),
  p_zscore = 2 * (1 - pnorm(abs(y)))
)
tibble(
  y = n01$v1,
  prob1 = surprisals_prob(y),
  prob2 = surprisals_prob(y, loo = TRUE),
  prob3 = surprisals_prob(y, distribution = dist_normal()),
  prob4 = surprisals_prob(y, distribution = dist_normal(), approximation = "gpd")
) |>
  arrange(prob1)
# Bivariate data
tibble(
  x = rnorm(50),
  y = c(5, rnorm(49)),
  prob = surprisals_prob(cbind(x, y), approximation = "gpd")
)
oldfaithful |>
  mutate(
    s = surprisals(cbind(duration, waiting), loo = TRUE),
    p = surprisals_prob(cbind(duration, waiting), loo = TRUE, approximation = "gpd")
  ) |>
  arrange(p)
}
\references{
Rob J Hyndman (2026) "That's weird: Anomaly detection using R", Chapter 6,
\url{https://OTexts.com/weird/}.
}
\seealso{
\code{\link{dist_kde}}
}
\author{
Rob J Hyndman
}
