% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/umx_build_high_level_models.R
\name{umxTwoStage}
\alias{umxTwoStage}
\alias{umxMR}
\title{Build a SEM implementing the instrumental variable design}
\usage{
umxTwoStage(
  formula = Y ~ X,
  instruments = ~qtl,
  data,
  std = FALSE,
  subset,
  contrasts = NULL,
  name = "IV_model",
  tryHard = c("no", "yes", "ordinal", "search"),
  ...
)
}
\arguments{
\item{formula}{The structural equation to be estimated (default = Y ~ X). A constant is implied if not explicitly deleted.}

\item{instruments}{A one-sided formula specifying instrumental variables (default = qtl).}

\item{data}{Frame containing the variables in the model.}

\item{std}{Standardize the manifests before running model (default is FALSE)}

\item{subset}{(optional) vector specifying a subset of observations to be used in fitting the model.}

\item{contrasts}{An optional list (not supported)}

\item{name}{The model name (default is "IVmodel")}

\item{tryHard}{Default ('no') uses normal mxRun. "yes" uses mxTryHard. Other options: "ordinal", "search"}

\item{...}{arguments to be passed along. (not supported)}
}
\value{
\itemize{
\item \code{\link[OpenMx:mxModel]{OpenMx::mxModel()}}
}
}
\description{
\code{umxMR} (\code{umxTwoStage}) implements a Mendelian randomization or instrumental variable Structural Equation Model.
For ease of learning, the parameters follow the \code{tsls()} function in the sem package.
}
\details{
The example is a \href{https://en.wikipedia.org/wiki/Mendelian_randomization}{Mendelian Randomization}
analysis showing the utility of SEM over two-stage regression.

The following figure shows how the MR model appears as a path diagram:

\if{html}{\figure{TSLS.png}{options: style="width: 50\%;" alt="Figure: Mendelian Randomization analysis.png"}}
\if{latex}{\figure{TSLS.pdf}{options: width=7cm}}
}
\examples{
\dontrun{
# ====================================
# = Mendelian Randomization analysis =
# ====================================

library(umx)
df = umx_make_MR_data(10e4, Vqtl = 0.02, bXY = 0.1, bUX = 0.5, bUY = 0.5, pQTL = 0.5)
m1 = umxMR(Y ~ X, instruments = ~ qtl, data = df)
parameters(m1)
plot(m1, means = FALSE, min="") # help DiagrammR layout the plot.
m2 = umxModify(m1, "qtl_to_X", comparison=TRUE, tryHard="yes", name="QTL_affects_X") # yip
m3 = umxModify(m1, "X_to_Y"  , comparison=TRUE, tryHard="yes", name="X_affects_Y") # yip
plot(m3, means = FALSE)

# Errant analysis using ordinary least squares regression (WARNING this result is CONFOUNDED!!)
ols1 = lm(Y ~ X    , data = df); coef(ols1) # Inflated .35 effect of X on Y
ols2 = lm(Y ~ X + U, data = df); coef(ols2) # Controlling U reveals the true 0.1 beta weight

# Simulate date with no causal X -> Y effect.
df = umx_make_MR_data(10e4, Vqtl = 0.02, bXY = 0, bUX = 0.5, bUY = 0.5, pQTL = 0.5)
m1 = umxMR(Y ~ X, instruments = ~ qtl, data = df)
parameters(m1)

# ======================
# = Now with sem::tsls =
# ======================
# libs("sem")
m2 = sem::tsls(formula = Y ~ X, instruments = ~ qtl, data = df)
coef(m2)

# Try with missing value for one subject: A benefit of the FIML approach in OpenMx.
m3 = tsls(formula = Y ~ X, instruments = ~ qtl, data = (df[1, "qtl"] = NA))
}
}
\references{
\itemize{
\item Fox, J. (1979) Simultaneous equation models and two-stage least-squares. In Schuessler, K. F. (ed.) \emph{Sociological Methodology}, Jossey-Bass.
}
\itemize{
\item Greene, W. H. (1993) \emph{Econometric Analysis}, Second Edition, Macmillan.
\item Sekula, P., Del Greco, M. F., Pattaro, C., & Kottgen, A. (2016). Mendelian Randomization as an Approach to
Assess Causality Using Observational Data. \emph{Journal of the American Society of Nephrology}, \strong{27}), 3253-3265. \doi{10.1681/ASN.2016010098}
}
}
\seealso{
\itemize{
\item \code{\link[=umx_make_MR_data]{umx_make_MR_data()}}, \code{\link[=umxDiffMZ]{umxDiffMZ()}}, \code{\link[=umxDoC]{umxDoC()}}, \code{\link[=umxDiscTwin]{umxDiscTwin()}}
}

Other Super-easy helpers: 
\code{\link{umx}},
\code{\link{umxEFA}()}
}
\concept{Super-easy helpers}
