% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/crosswalk.R
\name{find_billable}
\alias{find_billable}
\title{Find Billable ICD Codes from CMS GEMs}
\usage{
find_billable(dx, version = 10, year = 2018,
              match_all = TRUE, decimal = FALSE,
              output = c("character", "list", "tidy-data"), ...)
}
\arguments{
\item{dx}{A character vector representing diagnosis codes.  Each element
of the vector can either represent individual diagnosis code or a set
of diagnosis codes that are concartenated by commas in between.}

\item{version}{A numeric value specifying the version of the diagnosis
codes that should be either \code{9} for ICD-9 codes or \code{10} for
ICD-10 codes.}

\item{year}{A numeric value specifying the year of the CMS GEMs.  The
currently available options are \code{2017} and \code{2018}.  By
default, 2018 CMS GEMs is used.}

\item{match_all}{A logical value specifying the strategy for finding
billable codes based on the input diagnosis category.  If \code{TRUE}
(the default), the function will add the regular expression
\code{"[[[[:alnum:]]{1,4}]]"} to the tail of diagnosis category so
that all the billable diagnosis codes under the given category will be
matched.  If \code{FALSE}, the function will add the regular
experssion \code{"[[:alnum:]]"} repeatedly at most four times until
any set of billable codes are matched.}

\item{decimal}{A logical value.  If \code{TRUE}, the diagnosis codes would
be returned with decimal points.  The default is \code{FALSE}.}

\item{output}{A character value specifying the format of the output.  The
avaiable options are \code{"character"}, \code{"list"}, and
\code{"tidy-data"}.  By default, option \code{"character"} is used and
results in a character vector that consists of element-wise
concatenatation by commas of all the translated diagnosis codes from
the original codes.  If \code{"list"} is specified, all the translated
codes will not be concartenated and a list of character vectors will
be returned by the function.  Similarly, if \code{"tidy-data"} is
specified, a data frame in a tidy format will be returned.  The first
column of the data frame consists of the original diagnosis codes; the
second column consists of the translated diagnosis codes.}

\item{...}{Other arguments for future usage.  A warning will be thrown out
if any argument goes into \code{...} accidentally.}
}
\value{
A character vector of the same length with the input vector will
    be returned by default or if \code{output = "charactor"}.  A list of
    character vectors will be returned if \code{output = "list"}; A data
    frame in tidy-format will be returned if \code{output = "tidy-data"}.
    See argument \code{output} for details.
}
\description{
This function tries to find all the billable ICD codes that can be
translated by CMS GEMs for each of the input diagnosis codes representing
a major category.
}
\details{
It is designed to be used with the function \code{\link{icd_map}} for
translating the diagnosis codes that are not billable but representing
major categories.  Notice that only the character vector output can be
directly passed to the function \code{\link{icd_map}} for translation.
}
\examples{
library(touch)

### for ICD-9 codes
icd9_major <- c("001", "316", "808", NA, "not_a_dx")

## find all billable codes under the major category
find_billable(icd9_major, version = 9)

## find the billable codes right under the major category
(icd9_billable <- find_billable(icd9_major, version = 9,
                                match_all = FALSE))

## compare the translation results
icd_map(icd9_major, nomatch = NA)
icd_map(icd9_billable, nomatch = NA)

### for ICD-10 codes
icd10_major <- c("T36.0X2", "T36.3X2", "T38.6X2")

## find all billable codes and output in different formats
find_billable(icd10_major, version = 10)
find_billable(icd10_major, version = 10, output = "list")
find_billable(icd10_major, version = 10, output = "tidy-data")

## add decimal if wanted
(icd10_billable <- find_billable(icd10_major, version = 10, decimal = TRUE))

## compare the translation results
icd_map(icd10_major, from = 10, to = 9, nomatch = NA)
icd_map(icd10_billable, from = 10, to = 9)
}
\seealso{
icd_map
}
\author{
Wenjie Wang <wenjie.2.wang@uconn.edu>
}
