\name{grammar_probs}
\alias{grammar_probs}
\title{Posterior Probabilities of Grammatical Types for Each Sign}
\description{
For each cuneiform sign in a sentence, computes Bayesian posterior
probabilities for all grammatical types, combining prior beliefs from
\code{\link{prior_probs}} with observed dictionary frequencies. The
dictionary counts are corrected for verb underrepresentation using the
\code{sentence_prob} stored in the prior.
}
\usage{
grammar_probs(sg, prior, dic, alpha0 = 1)
}
\arguments{
  \item{sg}{A data frame as returned by \code{\link{sign_grammar}}.}

  \item{prior}{A named numeric vector as returned by
    \code{\link{prior_probs}}, with a \code{sentence_prob} attribute.}

  \item{dic}{A dictionary data frame as returned by
    \code{\link{read_dictionary}}.}

  \item{alpha0}{Numeric (>= 0). Strength of the prior (pseudo sample
    size). Larger values pull the posterior towards the prior. When
    \code{alpha0 = 0}, the result is purely data-driven. Default: 1.}
}
\details{
For each sign at position \eqn{i} in the sentence, the function computes:

\enumerate{
  \item The raw dictionary counts \eqn{n_k} for each grammar type \eqn{k}.
  \item A correction factor \eqn{x_k = 1 / \mathrm{sentence\_prob}} for
    verb-like types, \eqn{x_k = 1} otherwise. The corrected counts are
    \eqn{m_k = n_k \cdot x_k} with total \eqn{M = \sum_k m_k}.
  \item The posterior probability (Dirichlet-Multinomial model):
    \deqn{\theta_k = \frac{\alpha_0 \, p_k + m_k}{\alpha_0 + M}}
    where \eqn{p_k} is the prior probability from \code{prior_probs()}.
}

For signs not in the dictionary (\eqn{M = 0}), the posterior equals the
prior. For signs with many observations (\eqn{M \gg \alpha_0}), the
posterior is dominated by the data.
}
\value{
A data frame with columns:
\describe{
  \item{position}{Integer. Position of the sign in the sentence.}
  \item{sign_name}{Character. The sign name.}
  \item{cuneiform}{Character. The cuneiform character.}
  \item{type}{Character. The grammar type (e.g., \code{"S"}, \code{"V"},
    \code{"Sx->S"}).}
  \item{prob}{Numeric. Posterior probability for this type at this
    position.}
  \item{n}{Numeric. Number of counts in the dictionary.}
}
}
\examples{
dic   <- read_dictionary()
sg    <- sign_grammar("a-ma-ru ba-ur3 ra", dic)
prior <- prior_probs(dic, sentence_prob = 0.25)
gp    <- grammar_probs(sg, prior, dic, alpha0 = 1)
print(gp)
}
\seealso{
\code{\link{prior_probs}} for computing the prior,
\code{\link{sign_grammar}} for the input data,
\code{\link{plot_sign_grammar}} for visualisation.
}
\keyword{utilities}
