\name{sim_n_datasets}
\alias{sim_n_datasets}

\title{
Simulate multiple datasets from a single \code{DAG} object
}
\description{
This function takes a single \code{DAG} object and generates a list of multiple datasets, possible using parallel processing
}
\usage{
sim_n_datasets(dag, n_sim, n_repeats, n_cores=1,
               data_format="raw", data_format_args=list(),
               seed=NULL, progressbar=TRUE, ...)
}
\arguments{
  \item{dag}{
A \code{DAG} object created using the \code{\link{empty_dag}} function with nodes added to it using the \code{+} syntax. See \code{?empty_dag} or \code{?node} for more details. If the \code{dag} contains time-varying nodes added using the \code{\link{node_td}} function, the \code{\link{sim_discrete_time}} or \code{\link{sim_discrete_event}} functions will be used to generate the data (depending on the \code{type}s of the included time-dependent nodes). Otherwise, the \code{\link{sim_from_dag}} function will be used.
  }
  \item{n_sim}{
A single number specifying how many observations per dataset should be generated.
  }
  \item{n_repeats}{
A single number specifying how many datasets should be generated.
  }
  \item{n_cores}{
A single number specifying the amount of cores that should be used. If \code{n_cores = 1}, a simple for loop is used to generate the datasets with no parallel processing. If \code{n_cores > 1} is used, the \pkg{doSNOW} package is used in conjunction with the \pkg{doRNG} package to generate the datasets in parallel. By using the \pkg{doRNG} package, the results are completely reproducible by setting a \code{seed}.
  }
  \item{data_format}{
An optional character string specifying the output format of the generated datasets, or a function. If \code{"raw"} (default), the dataset will be returned as generated by the respective data generation function. If the \code{dag} contains time-varying nodes added using the \code{\link{node_td}} function that are appropriate for discrete-time simulation and this argument is set to either \code{"start_stop"}, \code{"long"} or \code{"wide"}, the \code{\link{sim2data}} function will be called to transform the dataset into the defined format. If any other string is supplied, regardless of whether time-varying nodes are included in the \code{dag} or not, the function with the name given in the string is called to transform the data. If a function is supplied directly, it will also be applied. This can be any function. The only requirement is that it has a named argument called \code{data}. Arguments to the function can be set using the \code{data_format_args} argument (see below).
  }
  \item{data_format_args}{
An optional list of named arguments passed to the function specified by \code{data_format}. Set to \code{list()} to use no arguments. Ignored if \code{data_format="raw"}.
  }
  \item{seed}{
A seed for the random number generator. By supplying a value to this argument, the results will be replicable, even if parallel processing is used to generate the datasets (using \code{n_cores > 1}), thanks to the magic performed by the \pkg{doRNG} package. See details.
  }
  \item{progressbar}{
Either \code{TRUE} (default) or \code{FALSE}, specifying whether a progressbar should be used. Currently only works if \code{n_cores > 1}, ignored otherwise.
  }
  \item{...}{
Further arguments passed to the \code{\link{sim_from_dag}} function (if the \code{dag} does not contain time-varying nodes) or the \code{\link{sim_discrete_time}} / \code{\link{sim_discrete_event}} function (if the \code{dag} contains time-varying nodes).
  }
}
\details{
Generating a number of datasets from a single defined \code{dag} object is usually the first step when conducting Monte-Carlo simulation studies. This is simply a convenience function which automates this process using parallel processing (if specified).

Note that for more complex Monte-Carlo simulations this function may not be ideal, because it does not allow the user to vary aspects of the data-generation mechanism inside the main \code{for} loop, because it can only handle a single \code{dag}. For example, if the user wants to simulate \code{n_repeats} datasets with confounding and \code{n_repeats} datasets without confounding, he/she has to call this function twice. This is not optimal, because setting up the clusters for parallel processing takes some processing time. If many different \code{dag}s should be used, it would make more sense to write a single function that generates the \code{dag} itself for each of the desired settings. This can sadly not be automated by us though.
}
\note{
In previous versions (< 0.4.1) the \code{seed} argument was set to \code{stats::runif(1)}, which is equivalent to using \code{seed=0}. This was a mistake, because it results in the same output being generated regardless of any \code{\link{set.seed}} call used before calling \code{sim_n_datasets()}. This default has been changed to \code{NULL}, which is equivalent to not setting a \code{seed}. To obtain the same results as in versions < 0.4.1 (when no `seed` was specified), use \code{seed=0}.
}
\author{
Robin Denz
}
\value{
Returns a list of length \code{n_repeats} containing datasets generated according to the supplied \code{dag} object.
}
\seealso{
\code{\link{empty_dag}}, \code{\link{node}}, \code{\link{node_td}}, \code{\link{sim_from_dag}}, \code{\link{sim_discrete_time}}, \code{\link{sim2data}}, \code{\link{sim_discrete_event}}
}
\examples{
library(simDAG)

# some example DAG
dag <- empty_dag() +
  node("death", type="binomial", parents=c("age", "sex"), betas=c(1, 2),
       intercept=-10) +
  node("age", type="rnorm", mean=10, sd=2) +
  node("sex", parents="", type="rbernoulli", p=0.5) +
  node("smoking", parents=c("sex", "age"), type="binomial",
       betas=c(0.6, 0.2), intercept=-2)

# generate 10 datasets without parallel processing
out <- sim_n_datasets(dag, n_repeats=10, n_cores=1, n_sim=100)

if (requireNamespace("doSNOW") & requireNamespace("doRNG") &
    requireNamespace("foreach")) {

# generate 10 datasets with parallel processing
out <- sim_n_datasets(dag, n_repeats=10, n_cores=2, n_sim=100)
}

# generate 10 datasets and transforming the output
# (using the sim2data function internally)
dag <- dag + node_td("CV", type="time_to_event", prob_fun=0.01)
out <- sim_n_datasets(dag, n_repeats=10, n_cores=1, n_sim=100,
                      max_t=20, data_format="start_stop")
}
