% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get-range-edge.R
\name{get_range_edge}
\alias{get_range_edge}
\title{Calculate range edges via simulation from the joint precision matrix}
\usage{
get_range_edge(
  obj,
  axis,
  quantiles = c(0.025, 0.975),
  level = 0.95,
  return_sims = FALSE
)
}
\arguments{
\item{obj}{\code{\link[=predict.sdmTMB]{predict.sdmTMB()}} output with \code{nsim > 0}. The prediction object
should include predictions on a spatial grid that covers the area of interest.}

\item{axis}{Numeric vector of the same length as the prediction data,
representing the spatial axis along which to calculate range edges
(e.g., latitude, coastal distance values). This should align with
the rows of the prediction matrix.}

\item{quantiles}{Numeric vector of quantiles to calculate. Default is
\code{c(0.025, 0.975)} for lower and upper 1\% range edges. Common alternatives
include \code{c(0.01, 0.99)} for 1\% edges or \code{c(0.05, 0.5, 0.95)} to
include the median.}

\item{level}{The confidence level for uncertainty intervals.}

\item{return_sims}{Logical. Return simulation draws? The default (\code{FALSE})
returns a quantile summary of the simulation draws.}
}
\value{
A data frame. If \code{return_sims = FALSE}:
\itemize{
\item name of time column (e.g., \code{year}) that was supplied to \code{\link[=sdmTMB]{sdmTMB()}} time argument
\item \code{quantile}: the quantile value (from \code{quantiles} argument)
\item \code{est}: estimated range edge position
\item \code{lwr}: lower confidence interval
\item \code{upr}: upper confidence interval
\item \code{se}: standard error
}

If \code{return_sims = TRUE}, simulation draws from range edge positions in long format:
\itemize{
\item name of time column (e.g., \code{year})
\item \code{quantile}: the quantile value
\item \code{.value}: simulated range edge position
\item \code{.iteration}: simulation number
}
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#experimental}{\figure{lifecycle-experimental.svg}{options: alt='[Experimental]'}}}{\strong{[Experimental]}}

Calculate range edges as density-weighted quantiles along a spatial axis.
Range edges are calculated as the positions along a user-supplied spatial
axis (e.g., latitude, coastal distance) where the cumulative proportion of
density equals specified quantiles (e.g., 0.01 and 0.99 for the lower and
upper 1\% range edges). Uncertainty is calculated via simulation from the
joint precision matrix.
}
\details{
This function implements a similar approach to VAST's range edge
calculations, following methods from Fredston et al. (2021) and similar
studies. The method:
\enumerate{
\item Orders spatial locations by position along the specified axis
\item Calculates cumulative proportion of total density along that axis
\item Finds positions where cumulative proportion equals target quantiles
\item Uses simulation from the joint precision to quantify uncertainty
}

To find the exact position where the cumulative proportion equals a target
quantile, the function uses linear interpolation between adjacent grid
points. This provides more accurate range edge estimates than selecting the
closest grid point, especially on coarser grids or for extreme quantiles
(e.g., 0.01, 0.99).
}
\examples{
\donttest{
# Fit a spatiotemporal model
mesh <- make_mesh(pcod, c("X", "Y"), n_knots = 100)
m <- sdmTMB(
  density ~ 0 + as.factor(year),
  data = pcod, mesh = mesh, family = tweedie(link = "log"),
  time = "year", spatiotemporal = "iid", spatial = "on"
)

# Create prediction grid
nd <- replicate_df(qcs_grid, "year", unique(pcod$year))

# Get predictions with simulations
p <- predict(m, newdata = nd, nsim = 100)

# Calculate range edges along latitude (Y coordinate)
edges <- get_range_edge(p, axis = nd$Y)
edges

# Plot range edges over time
if (require("ggplot2", quietly = TRUE)) {
  ggplot(edges, aes(year, est, colour = as.factor(quantile))) +
    geom_line() +
    geom_ribbon(aes(ymin = lwr, ymax = upr, fill = as.factor(quantile)),
      alpha = 0.2
    ) +
    labs(y = "Latitude", colour = "Quantile", fill = "Quantile")
}

# Get simulation draws for further analysis
edges_sims <- get_range_edge(p, axis = nd$Y, return_sims = TRUE)
}
}
\references{
Fredston, A. L., Pinsky, M., Selden, R. L., Szuwalski, C., Thorson, J. T.,
Gaines, S. D., & Halpern, B. S. (2021). Range edges of North American
marine species are tracking temperature over decades. Global Change Biology,
27(13), 3145-3156. \doi{10.1111/gcb.15614}
}
