#' Adjust adjacent insertions and deletions in the realignment
#' 
#' @param realn.ref An aligned reference sequence 
#' @param realn.reads An aligned read sequence
#' @param reference.genome A BSgenome or DNAStringSet object of reference sequences 
#' @param chr.name The chromosome name or sequence name of the reference sequence
#' @param ref.start The start position of the aligned reference sequence
#' @param ref.end The end position of the aligned reference sequence
#' 
#' @return A list of realignment after indel-adjustments including reference sequence, read sequence, cigar, start position of reference sequence, end position of reference sequence
#' 
#' @importFrom BSgenome getSeq
#' @importFrom GenomicRanges GRanges
#' @importFrom IRanges IRanges
#' 
#' @examples
#' \dontrun{
#'     adjustIndels()
#' }
#' 
#' @noRd
adjustIndels <- function(realn.ref, realn.reads, reference.genome, chr.name, ref.start, ref.end){
    
    chr.length <- length( unlist( BSgenome::getSeq(reference.genome, chr.name) ) )
    
	cigar <- encodeCIGAR(realn.ref, realn.reads)
	op <- parseCIGAR(cigar)
	op$start <- c(1, cumsum(op$length)[setdiff(1:nrow(op), nrow(op))]+1 )
	op$end <- cumsum(op$length)
	df <- data.frame(
		ref_seq = unlist( strsplit(realn.ref, "") ),
		reads_seq = unlist( strsplit(realn.reads, "") ),
		stringsAsFactors = FALSE
	)
	while( grepl("ID", paste(op$code, collapse="")) ){
		n <- regexpr( "ID", paste(op$code, collapse="") )
		r1 <- op$start[n]
		r2 <- op$end[n]
		r3 <- op$start[n+1]
		r4 <- op$end[n+1]
		df <- rbind(
			df[setdiff(1:r1, r1), ],
			df[r3:r4, ],
			df[r1:r2, ],
			df[setdiff(r4:nrow(df),r4), ]
		)
		cigar <- encodeCIGAR(
			paste(df$ref_seq, collapse = ""),
			paste(df$reads_seq, collapse = "")
		)
		op <- parseCIGAR(cigar)
		op$start <- c(1, cumsum(op$length)[setdiff(1:nrow(op), nrow(op))]+1 )
		op$end <- cumsum(op$length)
	}
	if( grepl("DI", paste(op$code, collapse="")) ){
	    DI.n <- unlist( gregexpr("DI", paste(op$code, collapse="")) )
	    for(i in 1:length(DI.n)){
	        DI.n.i <- DI.n[i] 
    	    DI.r1 <- op$start[DI.n.i]
    	    DI.r2 <- op$end[DI.n.i]
    	    DI.r3 <- op$start[DI.n.i+1]
    	    DI.r4 <- op$end[DI.n.i+1]
            seq.D <- paste( df$ref_seq[DI.r1:DI.r2], collapse="" )	  
            seq.I <- paste( df$reads_seq[DI.r3:DI.r4], collapse="" )
	        if(seq.D==seq.I){
	            df$reads_seq[DI.r1:DI.r2] <- df$reads_seq[DI.r3:DI.r4]
	            df$reads_seq[DI.r3:DI.r4] <- "-"
	        }
	    }
	}
	df <- df[ df$ref_seq!="-" | df$reads_seq!="-" ,]
	
	realn.ref.adj <- paste(df$ref_seq, collapse = "")
	realn.reads.adj <- paste(df$reads_seq, collapse = "")
	cigar.adj <- encodeCIGAR(realn.ref.adj, realn.reads.adj)
	op.adj <- parseCIGAR(cigar.adj)
	op.adj.string <- paste(op.adj$code, collapse="")
	
	ref.seq <- paste( df$ref_seq[df$ref_seq!="-"], collapse="")
	reads.seq <- paste( df$reads_seq[df$reads_seq!="-"], collapse="")
	reads.seq.orig <- reads.seq
	
	if( grepl("^DI", op.adj.string) ){
	    I.left.len <- op.adj$length[2]
	    if( I.left.len < ref.start ){
	        ref.ext.left <- as.character( BSgenome::getSeq(reference.genome, GenomicRanges::GRanges(chr.name, IRanges::IRanges(ref.start-I.left.len, ref.start-1) ) ) )
	        if( ref.ext.left == substr(reads.seq, 1, I.left.len) ){
	            op.I.left <- data.frame(
	                code = "M",
	                length = I.left.len,
	                stringsAsFactors = FALSE
	            )
	            op.adj <- rbind( op.I.left, op.adj[-2,] )
	            ref.seq <- paste( ref.ext.left, ref.seq, sep="" )
	            ref.start <- ref.start-I.left.len
	            op.adj.string <- paste(op.adj$code, collapse="")
	        }
	    }
	}
	
	if( grepl("DI$", op.adj.string) ){
	    I.right.len <- op.adj$length[nrow(op.adj)]
	    if( I.right.len < (chr.length-ref.end) ){
	        ref.ext.right <- as.character( BSgenome::getSeq(reference.genome, GenomicRanges::GRanges(chr.name, IRanges::IRanges(ref.end+1, ref.end+I.right.len) ) ) )
	        if( ref.ext.right == substr(reads.seq, ref.end-I.right.len+1, ref.end) ){
	            op.adj$code[ nrow(op.adj) ] <- "M"
	            ref.seq <- paste( ref.seq, ref.ext.right, sep="" )
	            ref.end <- ref.end+I.right.len
	            op.adj.string <- paste(op.adj$code, collapse="")
	        }
	    }
	}
	
	if( grepl("^DI", op.adj.string) ){
	    del.gap.len <- op.adj$length[1]
	    ref.seq <- substring( ref.seq, del.gap.len+1 )
	    ref.start <- ref.start + del.gap.len
	    op.adj$code[2] <- "S"
	    op.adj <- op.adj[-1, ]
	}
	if( grepl("^D", op.adj.string) & !grepl("^DI", op.adj.string) ){
	    del.gap.len <- op.adj$length[1]
	    ref.seq <- substring( ref.seq, del.gap.len+1 )
	    ref.start <- ref.start + del.gap.len
	    op.adj <- op.adj[-1, ]
	}
	if( grepl("^I", op.adj.string) ){
	    op.adj$code[1] <- "S"
	}
	
	if( grepl("DI$", op.adj.string) ){
	    del.gap.len <- op.adj$length[ nrow(op.adj)-1 ]
	    ref.seq <- substring( ref.seq, 1, nchar(ref.seq)-del.gap.len )
	    ref.end <- ref.end - del.gap.len
	    op.adj$code[ nrow(op.adj) ] <- "S"
	    op.adj <- op.adj[-(nrow(op.adj)-1), ]
	}
	if( grepl("D$", op.adj.string) ){
	    del.gap.len <- op.adj$length[ nrow(op.adj) ]
	    ref.seq <- substring( ref.seq, 1, nchar(ref.seq)-del.gap.len )
	    ref.end <- ref.end - del.gap.len
	    op.adj <- op.adj[-nrow(op.adj), ]
	}
	if( grepl("I$", op.adj.string) & !grepl("DI$", op.adj.string) ){
	    op.adj$code[ nrow(op.adj) ] <- "S"
	}
	
	cigar.adj2 <- paste( paste(op.adj$length, op.adj$code, sep="") , collapse="" )
	
	realn.adj <- list(
	    ref = ref.seq,
	    reads = reads.seq,
	    cigar = cigar.adj2,
	    ref.start = ref.start,
	    ref.end = ref.end
	)
	
	return(realn.adj)
}
