#' @title Zero-Order Drug Release Kinetic Model
#' @name zero_order_release
#' @description
#' Fits experimental cumulative drug release data to a zero-order kinetic model
#' using linear regression. The function supports optional grouping variables
#' (e.g., formulation, batch) and optional pH-dependent analysis. It can generate
#' publication-quality plots with fitted straight lines and annotations for
#' zero-order rate constant (k0), intercept, coefficient of determination (R^2),
#' and time required for 50-percent drug release (t50).
#'
#' @param data A data frame containing experimental drug release data.
#' @param time_col Character string specifying the column name for time
#'   (e.g., minutes or hours).
#' @param release_col Character string specifying the column name for cumulative
#'   drug release (typically percentage).
#' @param group_col Optional character string specifying a column name used for
#'   grouping (e.g., formulation, batch). Default is NULL.
#' @param pH_col Optional character string specifying a column name containing
#'   pH values. If provided, zero-order models are fitted separately for each pH.
#' @param plot Logical; if TRUE, generates a plot of experimental data with
#'   zero-order fitted straight lines (default is TRUE).
#' @param annotate Logical; if TRUE, annotates the plot with k0, intercept, R^2,
#'   and t50 values for each group (default is TRUE).
#'
#' @import stats
#' @import ggplot2
#' @importFrom stats lm na.omit
#' @importFrom ggplot2 ggplot aes geom_point geom_line geom_smooth
#'   geom_text labs theme theme_bw element_text element_blank
#'
#' @return A list containing:
#' \describe{
#'   \item{\code{fitted_parameters}}{A data frame with k0, intercept, R^2, and t50
#'         values for each group or pH condition.}
#'   \item{\code{data}}{The processed data used for model fitting and plotting.}
#' }
#' @examples
#' # Example I: Single formulation
#' df_1 <- data.frame(
#'   time = c(0, 15, 30, 45, 60, 90, 120, 150, 180),
#'   release = c(0, 11.4, 20.8, 30.8, 39.8, 57.8, 72, 84.8, 93.5)
#' )
#' zero_order_release(
#'   data = df_1,
#'   time_col = "time",
#'   release_col = "release"
#' )
#'
#' # Example II: Two formulations (grouped, not pH-dependent)
#' df_2 <- data.frame(
#'   time = rep(c(0, 30, 60, 90, 120, 150), 2),
#'   release = c(
#'     0, 18, 35, 55, 72, 88,   # Formulation A
#'     0, 12, 26, 40, 58, 70    # Formulation B
#'   ),
#'   formulation = rep(c("Formulation A", "Formulation B"), each = 6)
#' )
#' zero_order_release(
#'   data = df_2,
#'   time_col = "time",
#'   release_col = "release",
#'   group_col = "formulation"
#' )
#'
#' # Example III: pH-dependent release
#' df_pH <- data.frame(
#'   time = rep(c(0, 60, 120, 180), 2),
#'   release = c(0, 40, 75, 95, 0, 30, 60, 80),
#'   pH = rep(c(7.4, 4.5), each = 4)
#' )
#' zero_order_release(
#'   data = df_pH,
#'   time_col = "time",
#'   release_col = "release",
#'   pH_col = "pH"
#' )
#'
#' # Example IV: Two formulations under two pH conditions
#' df1 <- data.frame(
#'   time = rep(c(0, 30, 60, 90, 120, 150, 180), 2),
#'   release = c(
#'     0, 12, 25, 38, 52, 65, 78,   # pH 4.5
#'     0, 15, 30, 47, 63, 78, 90    # pH 7.6
#'   ),
#'   pH = rep(c(4.5, 7.6), each = 7)
#' )
#' df2 <- data.frame(
#'   time = rep(c(0, 20, 40, 60, 80, 100, 120), 2),
#'   release = c(
#'     0, 10, 22, 35, 50, 64, 77,   # pH 4.5
#'     0, 14, 28, 45, 61, 76, 88    # pH 7.6
#'   ),
#'   pH = rep(c(4.5, 7.6), each = 7)
#' )
#' df_all <- rbind(
#'   cbind(dataset = "Dataset 1", df1),
#'   cbind(dataset = "Dataset 2", df2)
#' )
#' zero_order_release(
#'   data = df_all,
#'   time_col = "time",
#'   release_col = "release",
#'   group_col = "dataset",
#'   pH_col = "pH"
#' )
#' @references Higuchi, T. (1961) <doi:10.1002/jps.2600501018> Rate of release
#' of medicaments from ointment bases containing drugs in suspension. Journal of
#' Pharmaceutical Sciences, 50(10), 874–875.
#' @author Paul Angelo C. Manlapaz
#' @export

utils::globalVariables(c("time", "release", "group", "k0", "intercept", "R2",
                         "t50", "label", "x_pos", "y_pos", "hjust", "vjust"))

zero_order_release <- function(data,
                               time_col = "time",
                               release_col = "release",
                               group_col = NULL,
                               pH_col = NULL,
                               plot = TRUE,
                               annotate = TRUE) {

  if (!requireNamespace("ggplot2", quietly = TRUE)) {
    stop("Package 'ggplot2' is required.")
  }

  # -------------------------
  # Prepare data
  # -------------------------
  df <- data[, c(time_col, release_col, group_col, pH_col), drop = FALSE]
  df <- stats::na.omit(df)

  colnames(df)[1:2] <- c("time", "release")

  # Grouping
  if (!is.null(group_col) && !is.null(pH_col)) {
    df$group <- paste0(df[[group_col]], " | pH ", df[[pH_col]])
  } else if (!is.null(pH_col)) {
    df$group <- paste0("pH ", df[[pH_col]])
  } else if (!is.null(group_col)) {
    df$group <- as.factor(df[[group_col]])
  } else {
    df$group <- "Experimental"
  }

  df$group <- as.factor(df$group)

  # -------------------------
  # Zero-order fitting
  # -------------------------
  fit_results <- do.call(rbind, lapply(split(df, df$group), function(d) {

    fit <- stats::lm(release ~ time, data = d)
    s <- summary(fit)

    k0 <- coef(fit)[2]
    intercept <- coef(fit)[1]
    R2 <- s$r.squared
    t50 <- if (k0 > 0 && intercept < 50) {
      (50 - intercept) / k0
    } else {
      NA_real_
    }

    data.frame(
      group = unique(d$group),
      k0 = k0,
      intercept = intercept,
      R2 = R2,
      t50 = t50
    )
  }))

  # -------------------------
  # Plot
  # -------------------------
  if (plot) {

    p <- ggplot2::ggplot(df, ggplot2::aes(x = time, y = release, color = group)) +
      ggplot2::geom_point(size = 3) +
      ggplot2::geom_line(linewidth = 1.2) +

      # fitted zero-order line (black, separate)
      ggplot2::geom_smooth(
        method = "lm",
        se = FALSE,
        color = "black",
        linewidth = 1
      ) +

      ggplot2::labs(
        title = "Zero-Order Drug Release Kinetics",
        subtitle = "Experimental data with zero-order linear fit",
        x = "Time (minutes)",
        y = "Cumulative % Drug Released",
        color = "Condition"
      ) +
      ggplot2::theme_bw(base_size = 14) +
      ggplot2::theme(
        plot.title = ggplot2::element_text(face = "bold", hjust = 0.5),
        plot.subtitle = ggplot2::element_text(hjust = 0.5),
        panel.grid.major = ggplot2::element_blank(),
        panel.grid.minor = ggplot2::element_blank()
      )

    # -------------------------
    # Conditional annotations
    # -------------------------
    num_groups <- nlevels(df$group)

    if (annotate && num_groups <= 2) {
      ann <- fit_results

      ann$label <- paste0(
        "k0 = ", round(ann$k0, 3), " %/min\n",
        "Intercept = ", round(ann$intercept, 2), "\n",
        "R^2 = ", round(ann$R2, 3), "\n",
        "t[50] = ", round(ann$t50, 1), " min"
      )

      x_min <- min(df$time, na.rm = TRUE)
      x_max <- max(df$time, na.rm = TRUE)
      y_min <- min(df$release, na.rm = TRUE)
      y_max <- max(df$release, na.rm = TRUE)

      ann$x_pos <- c(
        x_min + 0.05 * (x_max - x_min),
        x_max - 0.05 * (x_max - x_min)
      )[seq_len(nrow(ann))]

      ann$y_pos <- c(
        y_max - 0.05 * (y_max - y_min),
        y_min + 0.05 * (y_max - y_min)
      )[seq_len(nrow(ann))]

      ann$hjust <- c(0, 1)[seq_len(nrow(ann))]
      ann$vjust <- c(1, 0)[seq_len(nrow(ann))]

      p <- p +
        ggplot2::geom_text(
          data = ann,
          ggplot2::aes(
            x = x_pos,
            y = y_pos,
            label = label,
            color = group
          ),
          hjust = ann$hjust,
          vjust = ann$vjust,
          size = 4,
          show.legend = FALSE
        )
    } else if (num_groups > 2 && annotate) {
      message("More than 2 groups detected - annotations have been disabled to prevent overlap.")
    }

    print(p)
  }

  return(list(
    fitted_parameters = fit_results,
    data = df
  ))
}
