#' Create the path elements for MeteoCat API
#'
#' Path vectors for MeteoCat API to use with httr2
#'
#' @section Dates
#' In this case as MeteoCat is capped to one date: one day for hourly, one month for daily, one year for
#' monthly and all years for yearly (no date needed). So we only use the start_date
#'
#' @section Variables
#' MeteoCat API only return one variable for all stations, so we need to iterate the desired variables. This
#' means that \code{.create_meteocat_path} should return a vector of paths for which iterate the get function.
#' Desired variables are:
#' \itemize{
#'   \item{instant & hourly: 32 (temp), 33 (humidity), 35 (precip), 36 (rad), 46 (windspeed), 47 (winddir)}
#'   \item{daily: 1000:1002 (temp), 1100:1102 (humidity), 1300 (precip), 1400 (rad), 1505 (windspeed), 1511(winddir)}
#'   \item{monthly: 2000:2004 (temp), 2100:2104 (humidity), 2300 (precip), 2400 (rad), 2505 (windspeed), 2511(winddir)}
#'   \item{yearly: 3000:3004 (temp), 3100:3104 (humidity), 3300 (precip), 3400 (rad), 3505 (windspeed), 3511(winddir)}
#' }
#'
#' @param api_options Option list as generated by \link{\code{meteocat_options}}
#'
#' @noRd
.create_meteocat_path <- function(api_options) {

  # we need the resolution to create the corresponding path
  resolution <- api_options$resolution

  # depending on resolution, the variables list is different
  variables_list <- switch(
    api_options$resolution,
    "instant" = c(
      "1", "2", "3", "32", "33", "34", "35", "36", "38",
      "40", "42", "44", "46", "47", "56", "57", "59", "72"
    ),
    "hourly" = c(
      "1", "2", "3", "32", "33", "34", "35", "36", "38",
      "40", "42", "44", "46", "47", "56", "57", "59", "72"
    ),
    "daily" = c(
      "1000", "1001", "1002", "1003", "1004", "1100", "1101",
      "1102", "1200", "1201", "1202", "1300", "1301", "1302",
      "1303", "1304", "1305", "1400", "1505", "1511", "1514",
      "1517", "1600", "1601", "1602", "1603", "1700"
    ),
    "monthly" = c(
      "2000", "2001", "2002", "2003", "2004", "2005", "2006",
      "2007", "2008", "2009", "2100", "2101", "2102", "2103",
      "2104", "2200", "2201", "2202", "2203", "2204", "2300",
      "2301", "2302", "2303", "2304", "2305", "2306", "2307",
      "2308", "2309", "2400", "2505", "2511", "2514", "2517",
      "2520", "2600", "2601", "2602"
    ),
    "yearly" = c(
      "3000", "3001", "3002", "3003", "3004", "3005", "3006",
      "3007", "3008", "3009", "3010", "3100", "3101", "3102",
      "3103", "3104", "3200", "3201", "3202", "3203", "3204",
      "3300", "3301", "3302", "3303", "3304", "3305", "3306",
      "3307", "3308", "3309", "3400", "3505", "3511", "3514",
      "3517", "3520", "3600", "3601", "3602"
    )
  )

  # now the path vectors for the resolutions
  paths_resolution <- switch(
    api_options$resolution,
    'instant' = purrr::map(
      variables_list,
      function(variable) { c('xema', 'v1', 'variables', 'mesurades', variable, 'ultimes') }
    ),
    'hourly' = purrr::map(
      variables_list,
      function(variable) {
        c(
          'xema', 'v1', 'variables', 'mesurades', variable, lubridate::year(api_options$start_date),
          format(api_options$start_date,"%m"), format(api_options$start_date,"%d")
        )
      }
    ),
    # for daily and monthly, dates are query parameters not path ones.
    'daily' = purrr::map(
      variables_list,
      function(variable) { c('xema', 'v1', 'variables', 'estadistics', 'diaris', variable) }
    ),
    'monthly' = purrr::map(
      variables_list,
      function(variable) { c('xema', 'v1', 'variables', 'estadistics', 'mensuals', variable) }
    ),
    'yearly' = purrr::map(
      variables_list,
      function(variable) { c('xema', 'v1', 'variables', 'estadistics', 'anuals', variable) }
    ),
    list()
  )

  # not recognised resolution
  if (length(paths_resolution) < 1) {
    cli::cli_abort(c(
      "{.arg {api_options$resolution}} is not a valid temporal resolution for MeteoCat.\nPlease see meteocat_options help for more information"
    ))
  }

  return(paths_resolution)
}

#' Create the query element for MeteoCat API
#'
#' Query string for MeteoCat API to use with httr2
#'
#' MeteoCat needs dates for daily and monthly resolutions as query parameters (broken down in year and month).
#'
#' @param api_options Option list as generated by \link{\code{meteocat_options}}
#'
#' @noRd
.create_meteocat_query <- function(api_options) {

  # dates supplied must be broken down and stored in year and month values to create the query parameters
  year_query_par <- lubridate::year(api_options$start_date)
  month_query_par <- format(api_options$start_date, '%m')

  dates_query <- switch(
    api_options$resolution,
    'daily' = list(any = year_query_par, mes = month_query_par),
    'monthly' = list(any = year_query_par),
    list()
  )

  return(dates_query)
}

#' Create request for MeteoCat API
#'
#' Create the request for meteocat based on api_options (for key), query and
#' paths
#'
#' This function creates the request, deal with errors and retries if needed
#' and access the data of the response.
#'
#' @param paths list of character vectors with the paths as obtained by
#'   \link{\code{.create_meteocat_path}}
#' @param api_options Option list as generated by \link{\code{meteocat_options}}
#' @param query List of name value pairs to query dates, needed for the request
#'   on some API endpoints
#'
#' @return a tibble converted from the response json.
#'
#' @noRd
.create_meteocat_request <- function(paths, api_options, query = NULL) {

  # Here is a little convoluted. In some cases, like when retrieving data for
  # all stations, MeteoCat only accepts a path with a fixed single variable, so
  # we need to create requests for each variable (each path) and retrieve them
  # with req_perform_sequential.
  # TO DO: explore req_perform_parallel.
  meteocat_requests <- purrr::map(
    paths,
    \(path) {
      httr2::request("https://api.meteo.cat") |>
        httr2::req_url_path_append(path) |>
        httr2::req_url_query(!!!query) |>
        httr2::req_headers_redacted("x-api-key" = api_options$api_key) |>
        httr2::req_user_agent(
          "meteospain R package (https://emf.creaf.cat/software/meteospain/)"
        ) |>
        httr2::req_error(
          body = \(resp) {
            # fallback
            message <- httr2::resp_body_string(resp)
            # more verbose known errors
            if (httr2::resp_status(resp) == 403L) {
              message <- c("API key invalid")
            }
            if (httr2::resp_status(resp) == 400L) {
              message <-
                httr2::resp_body_json(resp, simplifyVector = TRUE)$message
            }
            if (httr2::resp_status(resp) == 429L) {
              message <- c(
                httr2::resp_status(resp),
                httr2::resp_body_json(resp, simplifyVector = TRUE)$message,
                "i" = "This usually means you have reached your API key limit for this month and you are out of requests"
              )
            }

            message
          }
        ) |>
        httr2::req_retry(
          max_tries = 3,
          retry_on_failure = TRUE,
          is_transient = \(resp) {
            if (httr2::resp_status(resp) == 429L && httr2::resp_body_json(resp, simplifyVector = TRUE)$message == "Limit Exceeded") {
              return(FALSE)
            }
            httr2::resp_status(resp) %in% c(429L, 500L, 503L)
          },
          backoff = \(resp) {
            60
          },
          after = \(resp) {
            NA
          }
        )
    }
  )

  # now we use req_perform_sequential to get the results.
  # Here there is catch. As we are using sequential perform, but some
  # API endpoints have only one request, the on_error argument
  # must be "continue" when more than one request, but "stop" when
  # only one request
  on_error_value <- "stop"
  if (length(meteocat_requests) > 1) {
    on_error_value <- "continue"
  }
  meteocat_responses <- httr2::req_perform_sequential(
    meteocat_requests, on_error = on_error_value, progress = TRUE
  )

  res <- meteocat_responses |>
    httr2::resps_successes() |>
    httr2::resps_data(
      \(resp) {
        httr2::resp_body_json(resp, simplifyVector = TRUE) |>
          dplyr::as_tibble()
      }
    )

  # Check if any failure happened
  failures <- meteocat_responses |>
    httr2::resps_failures()

  if (length(failures) > 0) {
    urls <- purrr::map(httr2::resps_requests(failures), "url")
    messages <- purrr::map_chr(
      failures,
      \(resp) {
        httr2::resp_body_json(resp, simplifyVector = TRUE)$message
      }
    )

    # If all is failures, stop
    if (identical(length(failures), length(meteocat_responses))) {
      cli::cli_abort(c(
        "x" = "No data was retrieved",
        unique(messages)
      ))
    }

    cli::cli_warn(purrr::map2_chr(urls, messages, \(u, m) { paste0(u, ": ", m) }))
  }

  return(res)
}

#' Get quota
#'
#' User quota
#'
#' @param api_options Option list as generated by \link{\code{meteocat_options}}
#' @noRd
.get_quota_meteocat <- function(api_options) {

  # path
  path_quota <- list(c('quotes', 'v1', 'consum-actual'))

  # retrieve the data
  .create_meteocat_request(path_quota, api_options) |>
    unnest_safe(c("client", "plans"))
}

#' Get variables metadata
#'
#' Variables metadata
#'
#' @param api_options Option list as generated by \link{\code{meteocat_options}}
#' @noRd
.get_variables_meteocat <- function(api_options) {

  # path
  path_resolution <- switch(
    api_options$resolution,
    'instant' = list(c('xema', 'v1', 'variables', 'mesurades', 'metadades')),
    'hourly' = list(c('xema', 'v1', 'variables', 'mesurades', 'metadades')),
    'daily' = list(c('xema', 'v1', 'variables', 'estadistics', 'diaris', 'metadades')),
    'monthly' = list(c('xema', 'v1', 'variables', 'estadistics', 'mensuals', 'metadades')),
    'yearly' = list(c('xema', 'v1', 'variables', 'estadistics', 'anuals', 'metadades'))
  )
  # cache
  cache_ref <- rlang::hash(c(path_resolution, api_options$api_key))

  # get the data from cache or from API if new
  .get_cached_result(cache_ref, {
    # retrieve the data
    .create_meteocat_request(path_resolution, api_options) |>
      dplyr::mutate(meteospain_names = .meteocat_var_codes_2_names(.data$codi)) |>
      dplyr::filter(!is.na(.data$meteospain_names))
  })
}

#' Get info for the meteocat stations
#'
#' Get info for the meteocat stations
#'
#' @noRd

.get_info_meteocat <- function(api_options) {

  # GET parts needed --------------------------------------------------------------------------------------
  # path
  path_resolution <- list(c('xema', 'v1', 'estacions', 'metadades'))
  # cache
  cache_ref <- rlang::hash(c(path_resolution, api_options$api_key))

  # get data from cache or from API if new
  info_meteocat <- .get_cached_result(cache_ref, {
    # retrieve the data
    .create_meteocat_request(path_resolution, api_options) |>
      unnest_safe(
        dplyr::any_of(c("coordenades", "provincia")),
        names_sep = "_"
      ) |>
      dplyr::mutate(service = "meteocat") |>
      dplyr::select(
        "service", station_id = "codi", station_name = "nom",
        station_province = "provincia_nom",
        altitude = "altitud",
        "coordenades_longitud", "coordenades_latitud"
      ) |>
      dplyr::distinct() |>
      dplyr::mutate(
        altitude = units::set_units(.data$altitude, 'm')
      ) |>
      sf::st_as_sf(
        coords = c('coordenades_longitud', 'coordenades_latitud'),
        crs = 4326
      )
  })

  return(info_meteocat)
}

#' Get data from MeteoCat
#'
#' Get data from MeteoCat service
#'
#' For all resolutions, there is no need of supply the stations_id in the query,
#' as the data is not so big. So, in case of stations provided, we can filter later, after getting
#' the data. This also has the advantage of using only one query, reducing the probability of reaching
#' the API limit per minute or total.
#'
#' @param api_options Option list as generated by \link{\code{meteocat_options}}
#'
#' @noRd
.get_data_meteocat <- function(api_options) {

  # All necessary things for the GET ----------------------------------------------------------------------
  # create api paths
  paths_resolution <- .create_meteocat_path(api_options)
  # create query if needed (only use it when length is > 0)
  query_resolution <- .create_meteocat_query(api_options)
  if (length(query_resolution) < 1) {
    query_resolution <- NULL
  }
  # cache (in this case with path and query to get the date also)
  cache_ref <- rlang::hash(c(paths_resolution, query_resolution, api_options$api_key))

  # if resolution less than daily, remove the cache
  if (api_options$resolution %in% c("instant", "hourly")) {
    apis_cache$remove(cache_ref)
  }

  data_meteocat <- .get_cached_result(cache_ref, {

    # Resolution specific carpentry -------------------------------------------------------------------------
    # Now, instant/hourly and daily/monthly/yearly differs in the unnest step, as the column names are called
    # differently. It also differs in the select step as in the latter group there is no repetition of column
    # names after the unnest step.
    resolution_specific_unnest <- .meteocat_short_carpentry
    radiation_units <- "W/m^2"
    var_names <- switch(
      api_options$resolution,
      "instant" = c(
        "temperature", "min_temperature", "max_temperature",
        "relative_humidity", "min_relative_humidity", "max_relative_humidity",
        "precipitation", "max_precipitation_minute",
        "wind_direction", "wind_speed", "max_wind_direction", "max_wind_speed",
        "global_solar_radiation", "net_solar_radiation",
        "snow_cover",
        "atmospheric_pressure", "min_atmospheric_pressure", "max_atmospheric_pressure"
      ),
      "hourly" = c(
        "temperature", "min_temperature", "max_temperature",
        "relative_humidity", "min_relative_humidity", "max_relative_humidity",
        "precipitation", "max_precipitation_minute",
        "wind_direction", "wind_speed", "max_wind_direction", "max_wind_speed",
        "global_solar_radiation", "net_solar_radiation",
        "snow_cover",
        "atmospheric_pressure", "min_atmospheric_pressure", "max_atmospheric_pressure"
      ),
      "daily" = c(
        "mean_temperature", "max_temperature", "min_temperature",
        "mean_temperature_classic", "thermal_amplitude", "mean_relative_humidity",
        "max_relative_humidity", "min_relative_humidity", "mean_atmospheric_pressure",
        "max_atmospheric_pressure", "min_atmospheric_pressure", "precipitation",
        "precipitation_8h_8h", "max_precipitation_minute", "max_precipitation_hour",
        "max_precipitation_30m", "max_precipitation_10m", "global_solar_radiation",
        "mean_wind_speed", "mean_wind_direction", "max_wind_speed",
        "max_wind_direction", "mean_snow_cover", "max_snow_cover", "new_snow_cover",
        "min_snow_cover", "reference_evapotranspiration"
      ),
      "monthly" = c(
        "mean_temperature", "max_temperature_absolute", "min_temperature_absolute",
        "max_temperature_mean", "min_temperature_mean", "mean_temperature_classic",
        "frost_days", "max_thermal_amplitude", "mean_thermal_amplitude",
        "extreme_thermal_amplitude", "mean_relative_humidity",
        "max_relative_humidity_absolute", "min_relative_humidity_absolute",
        "max_relative_humidity_mean", "min_relative_humidity_mean",
        "mean_atmospheric_pressure", "max_atmospheric_pressure_absolute",
        "min_atmospheric_pressure_absolute", "max_atmospheric_pressure_mean",
        "min_atmospheric_pressure_mean", "precipitation", "precipitation_8h_8h",
        "max_precipitation_minute", "max_precipitation_24h",
        "max_precipitation_24h_8h_8h", "rain_days_0", "rain_days_02",
        "max_precipitation_hour", "max_precipitation_30m", "max_precipitation_10m",
        "global_solar_radiation", "mean_wind_speed", "mean_wind_direction",
        "max_wind_speed", "max_wind_direction", "max_wind_speed_mean", "mean_snow_cover",
        "max_snow_cover", "new_snow_cover"
      ),
      "yearly" = c(
        "mean_temperature", "max_temperature_absolute", "min_temperature_absolute",
        "max_temperature_mean", "min_temperature_mean", "mean_temperature_classic",
        "frost_days", "max_thermal_amplitude", "mean_thermal_amplitude",
        "extreme_thermal_amplitude", "thermal_oscillation", "mean_relative_humidity",
        "max_relative_humidity_absolute", "min_relative_humidity_absolute",
        "max_relative_humidity_mean", "min_relative_humidity_mean",
        "mean_atmospheric_pressure", "max_atmospheric_pressure_absolute",
        "min_atmospheric_pressure_absolute", "max_atmospheric_pressure_mean",
        "min_atmospheric_pressure_mean", "precipitation", "precipitation_8h_8h",
        "max_precipitation_minute", "max_precipitation_24h",
        "max_precipitation_24h_8h_8h", "rain_days_0", "rain_days_02",
        "max_precipitation_hour", "max_precipitation_30m", "max_precipitation_10m",
        "global_solar_radiation", "mean_wind_speed", "mean_wind_direction",
        "max_wind_speed", "max_wind_direction", "max_wind_speed_mean",
        "mean_snow_cover", "max_snow_cover", "new_snow_cover"
      )
    )
    resolution_date_floor <- switch(
      api_options$resolution,
      "instant" = "second",
      "hourly" = "second",
      "daily" = "day",
      "monthly" = "month",
      "yearly" = "year"
    )

    if (api_options$resolution %in% c('daily', 'monthly', 'yearly')) {
      resolution_specific_unnest <- .meteocat_long_carpentry
      radiation_units <- "MJ/m^2"
    }

    # Stations info for getting coords ----------------------------------------------------------------------
    stations_info <- .get_info_meteocat(api_options)

    # Data transformation -----------------------------------------------------------------------------------
    raw_data <-
      .create_meteocat_request(paths_resolution, api_options, query_resolution)
    
    response_transformed <- raw_data |>
      # resolution specific unnesting of raw data
      resolution_specific_unnest() |>
      # transform variable codes to standard names
      dplyr::mutate(variable_name = .meteocat_var_codes_2_names(.data$variable_code)) |>
      # for daily, monthly and yearly, sometimes there are duplicated rows, remove them
      dplyr::distinct() |>
      # each variable in its own column
      tidyr::pivot_wider(
        id_cols = -"variable_code",
        names_from = "variable_name", values_from = "valor"
      ) |>
      .create_missing_vars(var_names = var_names) |>
      # set service, date and units
      dplyr::mutate(
        service = 'meteocat',
        timestamp = lubridate::parse_date_time(
          .data$timestamp,
          orders = c('ymdHMS', 'Ymz'),
          truncated = 5
        ) |>
          lubridate::floor_date(resolution_date_floor),
        dplyr::across(dplyr::contains('temperature'), ~ units::set_units(.x, 'degree_C')),
        dplyr::across(dplyr::contains('humidity'), ~ units::set_units(.x, '%')),
        dplyr::across(dplyr::contains('precipitation'), ~ units::set_units(.x, 'L/m^2')),
        # standard mode to avoid interpreting radiation_units as a symbol (default)
        dplyr::across(
          dplyr::contains('radiation'),
          ~ units::set_units(.x, radiation_units, mode = "standard")
        ),
        dplyr::across(dplyr::contains('speed'), ~ units::set_units(.x, 'm/s')),
        dplyr::across(dplyr::contains('direction'), ~ units::set_units(.x, 'degree')),
        dplyr::across(dplyr::contains('pressure'), ~ units::set_units(.x, 'hPa')),
        dplyr::across(dplyr::contains('snow'), ~ units::set_units(.x, 'cm')),
        dplyr::across(dplyr::contains('days'), ~ units::set_units(.x, 'days')),
        dplyr::across(dplyr::contains('thermal'), ~ units::set_units(.x, 'degree_C')),
        dplyr::across(dplyr::contains('evapotranspiration'), ~ units::set_units(.x, 'L/m^2'))
      )

    res <- response_transformed |>
      # join stations_info
      dplyr::left_join(stations_info, by = c('service', 'station_id')) |>
      # arrange data
      dplyr::arrange(.data$timestamp, .data$station_id) |>
      # reorder variables to be consistent among all services
      relocate_vars() |>
      # ensure we have an sf
      sf::st_as_sf()

    # Copyright message -------------------------------------------------------------------------------------
    cli::cli_inform(c(
      i = copyright_style("Data provided by meteo.cat \u00A9 Servei Meteorol\u00F2gic de Catalunya"),
      legal_note_style("https://www.meteo.cat/wpweb/avis-legal/#info")
    ))

    res
  })

  # Filter expression for stations ------------------------------------------------------------------------
  # In case stations were supplied, we need also to filter them
  filter_expression <- TRUE
  # update filter if there is stations supplied
  if (!rlang::is_null(api_options$stations)) {
    filter_expression <- rlang::expr(.data$station_id %in% api_options$stations)
  }

  data_meteocat_fil <- data_meteocat |>
    # remove unwanted stations
    dplyr::filter(!! filter_expression)

  # Check if any stations were returned -------------------------------------------------------------------
  if ((!is.null(api_options$stations)) & nrow(data_meteocat_fil) < 1) {
    cli::cli_abort(c(
      x = "Station(s) provided have no data for the dates selected.",
      "Available stations with data for the actual query are:",
      glue::glue_collapse(unique(data_meteocat$station_id), sep = ', ', last = ' and ')
    ))
  }

  return(data_meteocat_fil)
}



# resolution_specific_carpentry -------------------------------------------------------------------------
.meteocat_short_carpentry <- function(data) {
  data |>
    unnest_safe(
      cols = "variables",
      # names_repair = 'universal'
      names_repair = ~ vctrs::vec_as_names(.x, repair = 'universal', quiet = TRUE)
    ) |>
    unnest_safe(cols = "lectures", names_repair = 'universal') |>
    dplyr::select(
      timestamp = "data", station_id = "codi...1", variable_code = "codi...2", "valor"
    )
}

.meteocat_long_carpentry <- function(data) {
  data |>
    unnest_safe(cols = "valors", names_repair = "universal") |>
    dplyr::select(
      timestamp = "data", station_id = "codiEstacio", variable_code = "codiVariable",
      "valor"
    )
}
