#' Rastrigin Function (F09)
#'
#' @description
#' A highly multimodal test function with many local minima arranged in a
#' regular lattice pattern. The global minimum is at the origin.
#'
#' @param x Numeric vector of input values.
#'
#' @return Numeric scalar representing the function value.
#'
#' @details
#' \strong{Formula:}
#' \deqn{f(x) = 10n + \sum_{i=1}^{n} \left[x_i^2 - 10\cos(2\pi x_i)\right]}{
#'   f(x) = 10 n + sum(x_i^2 - 10 cos(2 pi x_i))}
#'
#' \strong{Global minimum:} \eqn{f(0, 0, ..., 0) = 0}
#'
#' \strong{Characteristics:}
#' \itemize{
#'   \item Type: Multimodal
#'   \item Separable: Yes
#'   \item Differentiable: Yes
#'   \item Number of local minima: \eqn{\approx 10^n}
#'   \item Default bounds: \eqn{[-5.12, 5.12]^n}
#'   \item Default dimensions: 50
#' }
#'
#' The Rastrigin function is a typical example of non-linear multimodal function.
#' The large number of local minima makes it difficult for optimization algorithms
#' to find the global minimum.
#'
#' @seealso
#' \code{\link{test-functions}} for an overview of all test functions,
#' \code{\link{get_function_details}} to retrieve function parameters.
#'
#' @examples
#' F09(c(0, 0))    # Returns 0 (global minimum)
#' F09(c(1, 1))    # Returns approximately 2
#' F09(rep(0, 10)) # Returns 0 in 10 dimensions
#'
#' @export
F09 <- function(x) {
  # Rastrigin function
  dim <- length(x)
  return(sum(x^2 - 10 * cos(2 * pi * x)) + 10 * dim)
}
