% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pathway_heatmap.R
\name{pathway_heatmap}
\alias{pathway_heatmap}
\title{Create pathway heatmap with support for multiple grouping variables}
\arguments{
\item{abundance}{A matrix or data frame of pathway abundance data, where columns
correspond to samples and rows correspond to pathways. Must contain at least
two samples.}

\item{metadata}{A data frame of metadata, where each row corresponds to a sample
and each column corresponds to a metadata variable.}

\item{group}{A character string specifying the column name in the metadata data frame
that contains the primary group variable. Must contain at least two groups.}

\item{secondary_groups}{A character vector specifying additional grouping variables
for creating nested faceted heatmaps. If NULL, only the primary group will be used.
These variables will be used as secondary levels in the faceting hierarchy.}

\item{colors}{A vector of colors used for the background of the facet labels in the
heatmap. If NULL or not provided, a default color set is used for the facet strips.}

\item{font_size}{A numeric value specifying the font size for the heatmap.}

\item{show_row_names}{A logical value indicating whether to show row names in the heatmap.}

\item{show_legend}{A logical value indicating whether to show the legend in the heatmap.}

\item{custom_theme}{A custom theme for the heatmap.}

\item{low_color}{A character string specifying the color for low values in the heatmap gradient. Default is "#0571b0" (blue).}

\item{mid_color}{A character string specifying the color for middle values in the heatmap gradient. Default is "white".}

\item{high_color}{A character string specifying the color for high values in the heatmap gradient. Default is "#ca0020" (red).}

\item{cluster_rows}{A logical value indicating whether to cluster rows (pathways). Default is FALSE.}

\item{cluster_cols}{A logical value indicating whether to cluster columns (samples). Default is FALSE.}

\item{clustering_method}{A character string specifying the clustering method. Options: "complete", "average", "single", "ward.D", "ward.D2", "mcquitty", "median", "centroid". Default is "complete".}

\item{clustering_distance}{A character string specifying the distance metric. Options: "euclidean", "maximum", "manhattan", "canberra", "binary", "minkowski", "correlation", "spearman". Default is "euclidean".}

\item{dendro_line_size}{A numeric value specifying the line width of dendrogram branches. Default is 0.5.}

\item{dendro_labels}{A logical value indicating whether to show dendrogram labels. Default is FALSE.}

\item{facet_by}{\strong{[Deprecated]} A character string specifying an additional grouping variable for creating faceted heatmaps.
This parameter is deprecated and will be removed in future versions. Use \code{secondary_groups} instead.}

\item{colorbar_title}{A character string specifying the title for the color bar. Default is "Z Score".}

\item{colorbar_position}{A character string specifying the position of the color bar. Options: "right", "left", "top", "bottom". Default is "right".}

\item{colorbar_width}{A numeric value specifying the width of the color bar. Default is 0.6.}

\item{colorbar_height}{A numeric value specifying the height of the color bar. Default is 9.}

\item{colorbar_breaks}{An optional numeric vector specifying custom breaks for the color bar.}
}
\value{
A ggplot heatmap object representing the heatmap of the predicted functional
  pathway abundance data.
}
\description{
This function creates a heatmap of the predicted functional pathway abundance data
with support for single or multiple grouping variables. The function first performs
z-score normalization on the abundance data, then converts it to a long format and
orders the samples based on the grouping information. The heatmap supports nested
faceting for multiple grouping variables and is created using the `ggplot2` library.
}
\examples{
\donttest{
library(ggpicrust2)
library(ggh4x)
library(dplyr)
library(tidyr)
library(tibble)
library(magrittr)

# Create example functional pathway abundance data
kegg_abundance_example <- matrix(rnorm(30), nrow = 3, ncol = 10)
colnames(kegg_abundance_example) <- paste0("Sample", 1:10)
rownames(kegg_abundance_example) <- c("PathwayA", "PathwayB", "PathwayC")

# Create example metadata
metadata_example <- data.frame(
  sample_name = colnames(kegg_abundance_example),
  group = factor(rep(c("Control", "Treatment"), each = 5)),
  batch = factor(rep(c("Batch1", "Batch2"), times = 5))
)

# Custom colors for facet strips
custom_colors <- c("skyblue", "salmon")

# Example 1: Basic heatmap
pathway_heatmap(kegg_abundance_example, metadata_example, "group", colors = custom_colors)

# Example 2: Heatmap with row clustering
pathway_heatmap(
  abundance = kegg_abundance_example,
  metadata = metadata_example,
  group = "group",
  cluster_rows = TRUE,
  clustering_method = "complete",
  clustering_distance = "euclidean",
  dendro_line_size = 0.8
)

# Example 3: Heatmap with column clustering using correlation distance
pathway_heatmap(
  abundance = kegg_abundance_example,
  metadata = metadata_example,
  group = "group",
  cluster_cols = TRUE,
  clustering_method = "ward.D2",
  clustering_distance = "correlation"
)

# Example 4: Multi-level grouping with secondary_groups (NEW FEATURE)
pathway_heatmap(
  abundance = kegg_abundance_example,
  metadata = metadata_example,
  group = "group",
  secondary_groups = "batch",
  colors = c("lightblue", "lightcoral", "lightgreen", "lightyellow")
)

# Example 5: Custom colorbar settings
pathway_heatmap(
  abundance = kegg_abundance_example,
  metadata = metadata_example,
  group = "group",
  colorbar_title = "Expression Level",
  colorbar_position = "bottom",
  colorbar_width = 8,
  colorbar_height = 0.8,
  colorbar_breaks = c(-2, -1, 0, 1, 2)
)

# Example 6: Advanced heatmap with clustering and custom aesthetics
pathway_heatmap(
  abundance = kegg_abundance_example,
  metadata = metadata_example,
  group = "group",
  cluster_rows = TRUE,
  cluster_cols = FALSE,  # Don't cluster columns to preserve group order
  clustering_method = "average",
  clustering_distance = "manhattan",
  dendro_line_size = 1.0,
  low_color = "#053061",     # Dark blue
  mid_color = "#f7f7f7",     # Light gray
  high_color = "#67001f",    # Dark red
  colorbar_title = "Z-Score",
  colorbar_position = "left"
)

# Use real dataset
data("metacyc_abundance")
data("metadata")
metacyc_daa_results_df <- pathway_daa(
  abundance = metacyc_abundance \%>\% column_to_rownames("pathway"),
  metadata = metadata,
  group = "Environment",
  daa_method = "LinDA"
)
annotated_metacyc_daa_results_df <- pathway_annotation(
  pathway = "MetaCyc",
  daa_results_df = metacyc_daa_results_df,
  ko_to_kegg = FALSE
)
feature_with_p_0.05 <- metacyc_daa_results_df \%>\% filter(p_adjust < 0.05)

# Example 7: Real data with hierarchical clustering
pathway_heatmap(
  abundance = metacyc_abundance \%>\%
    right_join(
      annotated_metacyc_daa_results_df \%>\%
      select(all_of(c("feature","description"))),
      by = c("pathway" = "feature")
    ) \%>\%
    filter(pathway \%in\% feature_with_p_0.05$feature) \%>\%
    select(-"pathway") \%>\%
    filter(!is.na(description)) \%>\%
    distinct(description, .keep_all = TRUE) \%>\%
    column_to_rownames("description"),
  metadata = metadata,
  group = "Environment",
  cluster_rows = TRUE,
  clustering_method = "ward.D2",
  clustering_distance = "correlation",
  colors = custom_colors,
  low_color = "#2166ac",  # Custom blue for low values
  mid_color = "#f7f7f7",  # Light gray for mid values
  high_color = "#b2182b", # Custom red for high values
  colorbar_title = "Standardized Abundance"
)

# Example 8: Multiple grouping variables (NEW FEATURE)
# Create extended metadata with additional grouping variables
metadata_extended <- metadata_example \%>\%
  mutate(
    sex = factor(rep(c("Male", "Female"), times = 5)),
    age_group = factor(rep(c("Young", "Old"), each = 5))
  )

# Multi-level grouping with three variables
pathway_heatmap(
  abundance = kegg_abundance_example,
  metadata = metadata_extended,
  group = "group",                    # Primary grouping
  secondary_groups = c("batch", "sex"), # Secondary groupings
  colors = c("lightblue", "lightcoral")
)

# Example 9: Migration from facet_by to secondary_groups
# OLD WAY (deprecated, will show warning):
# pathway_heatmap(abundance, metadata, group = "Environment", facet_by = "Group")

# NEW WAY (recommended):
# pathway_heatmap(abundance, metadata, group = "Environment", secondary_groups = "Group")

# Example 10: Real data with multiple grouping variables
pathway_heatmap(
  abundance = metacyc_abundance \%>\%
    right_join(
      annotated_metacyc_daa_results_df \%>\%
      select(all_of(c("feature","description"))),
      by = c("pathway" = "feature")
    ) \%>\%
    filter(pathway \%in\% feature_with_p_0.05$feature) \%>\%
    select(-"pathway") \%>\%
    filter(!is.na(description)) \%>\%
    distinct(description, .keep_all = TRUE) \%>\%
    column_to_rownames("description"),
  metadata = metadata,
  group = "Environment",              # Primary: Pro-survival vs others
  secondary_groups = "Group",         # Secondary: Broad Institute vs Jackson Labs
  cluster_rows = TRUE,
  clustering_method = "ward.D2",
  clustering_distance = "correlation"
)
}
}
