#' Cosine-Correlation Coefficient Computation
#'
#' Computes the cosine-correlation coefficient for a vector of p-1 variables,
#' measuring the degree of linear dependence among variables in a multidimensional context.
#'
#' @param x A numeric vector of length p >= 2, where x[1] should be 0 (representing t_1).
#'   The remaining elements x[2], x[3], ..., x[p] represent the p-1 variables (t_2, t_3, ..., t_p).
#' @param na.rm Logical. If TRUE, NA values are removed before computation. Default is FALSE.
#'
#' @return A numeric value representing the cosine-correlation coefficient
#'   \eqn{\rho_{p-1}}{rho}.
#'   The coefficient ranges from 0 to 1, where higher values indicate greater
#'   linear dependence among the variables.
#'
#' @details
#' The cosine-correlation coefficient is defined as:
#'
#' \eqn{\rho_{p-1} = \frac{(p-1) \prod_{i=2}^{p} |t_i|}{\sum_{i=2}^{p} |t_i|^{p-1}}}{rho = [(p-1) * prod(|t_i|)] / sum(|t_i|^(p-1))}
#'
#' where \eqn{t_1 = 0}{t_1 = 0} and \eqn{t_2, \ldots, t_p}{t_2, ..., t_p} are the variables within the system.
#' The coefficient serves as a measure of the degree of linear dependence among
#' the p-1 variables, providing a quantitative assessment of their interrelationships
#' within a multidimensional context.
#'
#' @examples
#' # Example 1: Simple vector
#' x <- c(0, 2, 3, 4)
#' cosCorr(x)
#'
#' # Example 2: Vector with 5 variables
#' x <- c(0, 1, 2, 3, 4, 5)
#' cosCorr(x)
#'
#' @export
cosCorr <- function(x, na.rm = FALSE) {

  # Remove NA values if requested
  if (na.rm) {
    x <- x[!is.na(x)]
  }

  # Input validation
  if (!is.numeric(x)) {
    stop("Input must be a numeric vector.")
  }

  if (length(x) < 2) {
    stop("Vector must have at least 2 elements (p >= 2).")
  }

  if (any(is.na(x))) {
    stop("Vector contains NA values. Use na.rm = TRUE to remove them.")
  }

  # Check if first element is 0 (or very close to 0)
  if (abs(x[1]) > 1e-10) {
    warning("The first element should be 0 (t_1 = 0). Proceeding with computation anyway.")
  }

  # Extract variables t_2, t_3, ..., t_p (from index 2 onwards)
  t_values <- x[-1]
  p <- length(t_values) + 1
  p_minus_1 <- p - 1

  # Check for zero values that would make the product zero
  if (any(t_values == 0)) {
    warning("Some variables contain zero values, which will result in a zero product.")
  }

  # Compute absolute values
  abs_t <- abs(t_values)

  # Compute the product of |t_i|
  product_term <- prod(abs_t)

  # Compute the sum of |t_i|^{p-1}
  sum_term <- sum(abs_t^p_minus_1)

  # Handle division by zero
  if (sum_term == 0) {
    stop("Sum of |t_i|^{p-1} equals zero. Cannot compute correlation coefficient.")
  }

  # Compute rho_{p-1}
  rho <- (p_minus_1 * product_term) / sum_term

  # Ensure result is between 0 and 1
  if (rho < 0 || rho > 1) {
    warning("Computed coefficient is outside [0,1] range. Check input values.")
  }

  return(rho)
}

