% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bwFo.R
\name{bwFo}
\alias{bwFo}
\title{Optimal Bandwidth Selection via Fourier Plug-in Method}
\usage{
bwFo(x, C1 = 0.25, C2 = 25, gamma = 0.5)
}
\arguments{
\item{x}{Data from which the smoothing parameter is to be computed. The object is
coerced to a numeric vector in radians using \code{\link[circular]{circular}}.
Can be a numeric vector or an object of class \code{circular}.}

\item{C1}{Numeric scalar (default 0.25) representing the lower bound constant for
determining the range of Fourier coefficients. Used to compute the lower bound
\eqn{L_n = \lfloor C_1 \cdot n^{1/11} \rfloor + 1} for the optimal number of
Fourier terms. Must be positive and less than \code{C2}.}

\item{C2}{Numeric scalar (default 25) representing the upper bound constant for
determining the range of Fourier coefficients. Used to compute the upper bound
\eqn{U_n = \lfloor C_2 \cdot n^{1/11} \rfloor} for the optimal number of
Fourier terms. Must be positive and greater than \code{C1}.}

\item{gamma}{Numeric scalar between 0 and 1 (default 0.5) representing the penalty
parameter in the criterion function \eqn{H(m)} used for selecting the optimal
number of Fourier coefficients. Controls the trade-off between bias and variance
in the functional estimation.}
}
\value{
The computed optimal smoothing parameter \code{kappa}, a numeric concentration
parameter (analogous to inverse radians) derived from the Fourier method for circular
kernel density estimation. Higher values indicate sharper,  more concentrated kernels
and less smoothing; lower values indicate broader kernels and more smoothing.
}
\description{
This function computes the optimal smoothing parameter (bandwidth) for circular data
using the Fourier series-based direct plug-in approach based on delta sequence estimators (see \doi{10.1080/10485252.2022.2057974}).
}
\details{
The Fourier-based plug-in estimator computes the optimal bandwidth using the formula:
\deqn{\hat{h}_{FO} := (4\pi)^{-1/10} \hat{\theta}_{2,\hat{m}}^{-1/5} n^{-1/5}}
where \eqn{\hat{\theta}_{2,\hat{m}}} is the estimator of the second-order functional
\eqn{\theta_2(f)} based on the selected number of Fourier coefficients \eqn{\hat{m}}.

Under the assumption of von Mises density, this formula becomes:
\deqn{\hat{h}_{VM} = (4\pi)^{-1/10} \left(\frac{3\hat{\kappa}^2 I_0(2\hat{\kappa}) - \hat{\kappa}I_1(2\hat{\kappa})}{8\pi I_0(\hat{\kappa})^2}\right)^{-1/5} n^{-1/5}}
where \eqn{I_0} and \eqn{I_1} are the modified Bessel functions of the first kind of orders 0 and 1,
and \eqn{\hat{\kappa}} is the estimated concentration parameter of the von Mises distribution.
}
\examples{
# Example with circular data
library(circular)
set.seed(123)
x <- rvonmises(100, mu = circular(0), kappa = 2)
bw <- bwFo(x)
print(bw)

x <- rwrappednormal(100, mu = circular(1), rho = 0.7)
bw <- bwFo(x)
y <- density(x, bw=bw) 
plot(y, main="KDE with Fourier Plug-in Bandwidth")

}
\references{
Tenreiro, C. (2022). Kernel density estimation for circular data: a Fourier
series-based plug-in approach for bandwidth selection. \emph{Journal of
Nonparametric Statistics}, 34(2):377--406.
\doi{10.1080/10485252.2022.2057974}
}
\seealso{
\link{bwScv}, \link{bwLscvg}, \link{bwCcv}
}
